/**
 * $Id: Admin.java 179359 2017-03-13 19:58:16Z cviana $
 */
package csbase.console;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.rmi.RemoteException;
import java.rmi.UnmarshalException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Vector;
import java.util.regex.Matcher;

import csbase.exception.PermissionException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.CommonProjectInfo;
import csbase.logic.SGAInfo;
import csbase.logic.SGASet;
import csbase.logic.User;
import csbase.logic.UserOutline;
import csbase.logic.UserProjectInfo;
import csbase.logic.diagnosticservice.PropertyInfo;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ServerServiceInterface;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Cliente sem interface, para ser executado diretamente da linha de comando.
 *
 * @author Tecgraf
 */
class Admin extends AbstractConsoleApp {

  /**
   * Encoding usado nas transferncias de/para arquivos remotos.
   */
  private static final String REMOTE_ENCODING = "ISO-8859-1";

  /**
   * Indica se o servidor j foi encerrado (via parmetro "--shutdown").
   */
  boolean serverHasBeenShutdown = false;

  /**
   * Construtor. Obtm a senha interativamente.
   *
   * @param args parmetros fornecidos na linha de comando
   */
  Admin(String[] args) {
    super(args);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected BasicParams createParams() {
    return new AdminParams();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void processExtraParams() {
    // vazio
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getLogin() {
    AdminParams params = (AdminParams) getParams();
    String login = (String) (params.userLogin == null ? User.getAdminId()
      : params.userLogin);
    return login;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExitCode execute() {
    AdminParams params = (AdminParams) getParams();
    try {
      if (params.version) {
        showSystemVersion();
      }
      else if (params.listLoggedUsers) {
        listLoggedUsers();
      }
      else if (params.shutdown) {
        shutdownServer();
      }
      else if (params.listSGAs) {
        listSGAs();
      }
      else if (params.stopAllSGAs) {
        stopAllSGAs();
      }
      else if (params.restartAllSGAs) {
        restartAllSGAs();
      }
      else if (params.sgaToStop != null) {
        stopSGA(params.sgaToStop);
      }
      else if (params.sgaToRestart != null) {
        restartSGA(params.sgaToRestart);
      }
      else if (params.sgaToGetInfo != null) {
        showInfoFromSGA(params.sgaToGetInfo);
      }
      else if (params.cmdIdToKill != null) {
        killSGAcmd(params.cmdIdToKill);
      }
      else if (params.blockCmdQueue != null) {
        blockCmdsQueue(params.blockCmdQueue);
      }
      else if (params.listProps) {
        listProperties();
      }
      else if (params.msgToAll != null) {
        sendMsgToAll();
      }
      else if (params.msgToLogged != null) {
        sendMsgToLogged();
      }
      else if (params.msgToUser != null) {
        sendMsgToUser(params.msgRecipient);
      }
      else if (params.remoteSrcPath != null) {
        copyRemoteFile(params.remoteSrcPath, params.localDestDir);
      }
      else if (params.localSrcPath != null) {
        if (params.remoteDestDir == null) {
          printError("diretrio remoto no foi especificado");
        }
        copyLocalFile(params.localSrcPath, params.remoteDestDir);
      }
      else if (params.dirToCreate != null) {
        createDir(params.dirToCreate);
      }
      else if (params.fileToremove != null) {
        removeRemote(params.fileToremove, params.force);
      }
      else if (params.prjToCreate != null) {
        createProject(params.prjToCreate);
      }
      // FIXME desabilitado temporariamente
      //      else if (params.prjToRemove != null) {
      //        removeProject(params.prjToRemove, params.force);
      //      }
    }
    catch (Exception e) {
      String msg = e.getMessage();
      if (msg == null) {
        printStackTrace(e);
      }
      else {
        printError(msg);
      }
      // Exit code de falha.
      return ExitCode.FAILURE;
    }
    // Exit code de sucesso.
    return ExitCode.SUCCESS;
  }

  /**
   * Cria um diretrio remoto.
   *
   * @param dirPath path para o diretrio no fomato <code>[usr:]prj/path</code>
   * @throws RemoteException
   */
  private void createDir(String dirPath) throws RemoteException {
    String[] fullPath = FileUtils.splitPath(dirPath, '/');
    if (fullPath.length == 1) {
      printError("no foi especificado um diretrio");
      return;
    }
    CommonClientProject project = openProject(fullPath[0]);
    if (project == null) {
      return;
    }
    String[] dirPathArray = Arrays.copyOfRange(fullPath, 1, fullPath.length);
    ClientRemoteLocator.projectService.createDirectory(project.getId(),
      dirPathArray);
  }

  /**
   * Cria um projeto.
   *
   * @param prjSpec especificao do projeto no formato <code>[usr]:prj</code>.
   * @throws Exception
   */
  private void createProject(String prjSpec) throws Exception {
    User user = getUser(prjSpec);
    if (user == null) {
      return;
    }
    String projectName = getProjectName(prjSpec);
    CommonProjectInfo projectInfo = new CommonProjectInfo();
    projectInfo.userId = user.getId();
    projectInfo.name = projectName;
    projectInfo.description = "Projeto criado por " + getLogin();
    ClientRemoteLocator.projectService.createProject(projectInfo);
  }

  /**
   * Remove um arquivo/diretrio remoto.
   *
   * @param path path remoto no formato <code>[usr:]prj/path</code>.
   *
   * @param force <code>true</code> para no pedir confirmao na remoo de
   *        diretrios
   * @throws RemoteException
   */
  private void removeRemote(String path, boolean force) throws RemoteException {
    String[] fullPath = FileUtils.splitPath(path, '/');
    if (fullPath.length == 1) {
      printError("este comando no remove projetos");
      return;
    }
    CommonClientProject project = openProject(fullPath[0]);
    if (project == null) {
      return;
    }
    String[] pathArray = Arrays.copyOfRange(fullPath, 1, fullPath.length);
    ClientProjectFile file =
      ClientRemoteLocator.projectService.getChild(project.getId(), pathArray);
    if (!force) {
      String prompt;
      if (file.isDirectory()) {
        prompt =
          "\nATENO: a remoo deste diretrio causar erro caso ele esteja sendo\n"
            + "usado por algum usurio.\n\nConfirma sua remoo e de todo o seu contedo?";
      }
      else {
        prompt =
          "\nATENO: a remoo deste arquivo causar erro caso ele esteja sendo\n"
            + "usado por algum usurio.\n\nConfirma sua remoo?";
      }
      if (!confirm(prompt)) {
        printInfo("operao cancelada");
        return;
      }
    }
    ClientRemoteLocator.projectService.removeFile(project.getId(), pathArray);
  }

  // FIXME desabilitado temporariamente
  //  /**
  //   * Remove um projeto.
  //   *
  //   * @param prjSpec especificao do projeto no formato <code>[usr]:prj</code>.
  //   * @param force <code>true</code> para no pedir confirmao
  //   * @throws Exception
  //   */
  //  private void removeProject(String prjSpec, boolean force) throws Exception {
  //    User user = getUser(prjSpec);
  //    if (user == null) {
  //      return;
  //    }
  //    String projectName = getProjectName(prjSpec);
  //    if (!force) {
  //      String option =
  //        ask("Confirma remoo do projeto '%s' do usurio '%s'? [s/n]",
  //          projectName, user.getLogin());
  //      if (!option.equalsIgnoreCase("s")) {
  //        printInfo("operao cancelada");
  //        return;
  //      }
  //    }
  //    Object userId = user.getId();
  //    // FIXME precisamos garantir que o projeto no esteja aberto
  //    ClientRemoteLocator.projectService.removeProject(userId, projectName);
  //  }

  /**
   * Copia um arquivo local para um diretrio remoto.
   *
   * @param localSrcPath path local para o arquivo
   * @param remoteDirPath path para o diretrio remoto
   * @throws IOException
   */
  private void copyLocalFile(String localSrcPath, String remoteDirPath)
    throws IOException {
    String[] fullPath = FileUtils.splitPath(remoteDirPath, '/');
    // FIXME verificar se o usurio pode escrever no projeto, para dar uma mensagem mais alto-nvel
    CommonClientProject project = openProject(fullPath[0]);
    if (project == null) {
      return;
    }
    File localFile = getLocalSrcFile(localSrcPath);
    if (localFile == null) {
      return;
    }

    String[] filePath;
    if (fullPath.length > 1) {
      filePath = Arrays.copyOfRange(fullPath, 1, fullPath.length);
    }
    else {
      filePath = new String[0];
    }
    Object prjId = project.getId();
    ClientRemoteLocator.projectService.createFile(prjId, filePath,
      localFile.getName(), "UNKNOWN");
    ClientProjectFile remoteFile = ClientRemoteLocator.projectService
      .getChild(prjId, filePath, localFile.getName());

    try (OutputStream outputStream = remoteFile.getOutputStream();
      BufferedWriter writer = new BufferedWriter(
        new OutputStreamWriter(outputStream, REMOTE_ENCODING));
      BufferedReader reader = new BufferedReader(new FileReader(localFile))) {
      copyData(reader, writer);
      ClientRemoteLocator.projectService.closeProject(prjId, false);
    }
  }

  /**
   * Obtm o arquivo local que ser copiado para o diretrio remoto.
   *
   * @param localSrcPath path para o arquivo local
   * @return o arquivo local, ou <code>null</code> caso este no exista ou seja
   *         um diretrio
   */
  private File getLocalSrcFile(String localSrcPath) {
    File localFile;
    if (localSrcPath.charAt(0) != '/') {
      localFile = new File(getCurrentDir(), localSrcPath);
    }
    else {
      localFile = new File(localSrcPath);
    }
    if (!localFile.exists()) {
      printError(localSrcPath + " no existe");
      return null;
    }
    if (localFile.isDirectory()) {
      printError(localSrcPath + "  um diretrio");
      return null;
    }
    return localFile;
  }

  /**
   * Copia um arquivo remoto para o disco local.
   *
   * @param remotePath path para o arquivo remoto
   * @param localDirPath path local para onde ser copiado o arquivo
   * @throws IOException
   */
  private void copyRemoteFile(String remotePath, String localDirPath)
    throws IOException {
    String[] fullPath = FileUtils.splitPath(remotePath, '/');
    String[] filePath = Arrays.copyOfRange(fullPath, 1, fullPath.length - 1);
    String fileName = fullPath[fullPath.length - 1];
    CommonClientProject project = openProject(fullPath[0]);
    if (project == null) {
      return;
    }

    ClientProjectFile remoteFile = ClientRemoteLocator.projectService
      .getChild(project.getId(), filePath, fileName);
    if (remoteFile.isDirectory()) {
      printError(remotePath + "  um diretrio");
      return;
    }

    File localFile = getTargetDir(localDirPath, fileName);
    if (localFile == null) {
      return;
    }

    /*
     * fixamos o encoding em ISO-8859-1 para arquivos remotos
     */
    try (InputStream inputStream = remoteFile.getInputStream();
      BufferedReader reader =
        new BufferedReader(new InputStreamReader(inputStream, REMOTE_ENCODING));
      BufferedWriter writer = new BufferedWriter(new FileWriter(localFile))) {
      copyData(reader, writer);
    }
  }

  /**
   * Abre um projeto.
   *
   * @param prjSpec especificao do projeto a ser aberto. Pode ser apenas o
   *        nome do projeto, ou pode conter o nome do usurio (user:prj).
   * @return projeto aberto, ou <code>null</code> em caso de erro
   */
  private CommonClientProject openProject(String prjSpec) {
    try {
      User user = getUser(prjSpec);
      if (user == null) {
        return null;
      }
      String projectName = getProjectName(prjSpec);
      Object prjUserId = user.getId();
      Object loggedUserId = User.getLoggedUser().getId();
      String login = user.getLogin();
      boolean sameUser = loggedUserId.equals(prjUserId);

      List<UserProjectInfo> projects = ClientRemoteLocator.projectService
        .getProjectsSharedWithUser(loggedUserId);
      if (sameUser) {
        /*
         * queremos tambm os projetos do prprio usurio em questo
         */
        projects.addAll(
          ClientRemoteLocator.projectService.getProjectsFromUser(prjUserId));
      }
      List<UserProjectInfo> matches = new ArrayList<>();
      /*
       * primeiro identificamos todos os projetos com o nome especificado
       * acessveis pelo usurio
       */
      for (UserProjectInfo prjInfo : projects) {
        if (projectName.equals(prjInfo.getProjectName())) {
          matches.add(prjInfo);
        }
      }
      /*
       * se temos apenas um match,  ele que queremos
       */
      if (matches.size() == 1) {
        return openProject(matches.get(0));
      }
      /*
       * no temos nenhum match
       */
      if (matches.isEmpty()) {
        if (sameUser) {
          printError("usurio %s no possui projeto com nome %s", login,
            projectName);
        }
        else {
          printError("usurio %s no participa do projeto %s do usurio %s",
            User.getLoggedUser().getLogin(), projectName, login);
        }
      }
      else if (projectSpecIncludesUser(prjSpec)) {
        /*
         * dono do projeto foi especificado, temos um alvo especfico
         */
        for (UserProjectInfo prjInfo : matches) {
          if (prjUserId.equals(prjInfo.getOwnerId())) {
            return openProject(prjInfo);
          }
        }
        printError("usurio %s no possui projeto com nome %s", login,
          projectName);
      }
      else {
        printError(
          "o usurio %s possui acesso a mais de um projeto com nome '%s'",
          login, projectName);
      }
    }
    catch (Exception e) {
      printError("usurio e/ou projeto invlidos");
    }
    return null;
  }

  /**
   * Verifica se a especificao de um projeto inclui o login do seu dono.
   *
   * @param prjSpec especificao do projeto no formato <code>[usr:]prj</code>
   * @return <code>true</code> se a especificao inclui o nome do usurio
   */
  private static boolean projectSpecIncludesUser(String prjSpec) {
    return prjSpec.indexOf(':') != -1;
  }

  /**
   * Abre um projeto, sem enviar notificaes.
   *
   * @param prjInfo dados do projeto
   * @return projeto aberto
   * @throws RemoteException
   */
  private static CommonClientProject openProject(UserProjectInfo prjInfo)
    throws RemoteException {
    return ClientRemoteLocator.projectService
      .openProject(prjInfo.getProjectId(), false);
  }

  /**
   * Obtm o usurio associado ao projeto a ser aberto.
   *
   * @param prjSpec especificao do projeto a ser aberto. Pode ser apenas o
   *        nome do projeto, ou pode conter o nome do usurio (user:prj).
   * @return usurio especificado, ou o usurio corrente (caso no tenha sido
   *         especificado)
   * @throws Exception
   */
  private User getUser(String prjSpec) throws Exception {
    int pos = prjSpec.indexOf(':');
    String login;
    if (pos != -1) {
      login = prjSpec.substring(0, pos);
    }
    else {
      login = getLogin();
    }
    User user = User.getUserByLogin(login);
    if (user == null) {
      printError("usurio invlido: %s", login);
    }
    return user;
  }

  /**
   * Obtm o nome do projeto.
   *
   * @param prjSpec especificao do projeto a ser aberto. Pode ser apenas o
   *        nome do projeto, ou pode conter o nome do usurio (user:prj).
   * @return nome do projeto
   */
  private static String getProjectName(String prjSpec) {
    int pos = prjSpec.indexOf(':');
    if (pos != -1) {
      return prjSpec.substring(pos + 1, prjSpec.length());
    }
    return prjSpec;
  }

  /**
   * Copia dados, linha-a-linha, de um leitor para um escritor. <b>ATENO:</b>
   * no fecha os steams.
   *
   * @param reader leitor
   * @param writer escritor
   * @throws IOException
   */
  private static void copyData(BufferedReader reader, BufferedWriter writer)
    throws IOException {
    String line = reader.readLine();
    while (line != null) {
      writer.write(line);
      writer.newLine();
      line = reader.readLine();
    }
  }

  /**
   * Obtm o diretrio local que ser o destino da cpia.
   *
   * @param localDirPath path local (pode ser <code>null</code> se o usurio no
   *        indicou um diretrio de destino -- neste caso, gravaremos no disco
   *        local)
   * @param fileName nome do arquivo local
   * @return diretrio local, ou <code>null</code> se o diretrio no existe ou
   *         se j possui um arquivo com o nome especificado
   */
  private File getTargetDir(String localDirPath, String fileName) {
    File localFile;
    if (localDirPath != null) {
      /*
       * usurio especificou dir local de destino
       */
      File localDir = getLocalSrcFile(localDirPath);
      if (!localDir.exists()) {
        printError(localDirPath + " no  um diretrio local vlido");
        return null;
      }
      localFile = new File(localDir, fileName);
    }
    else {
      localFile = new File(getCurrentDir(), fileName);
    }
    if (localFile.exists()) {
      printError(localFile.getAbsolutePath() + " j existe");
      return null;
    }
    return localFile;
  }

  /**
   * Exibe o identificador da verso do servidor.
   *
   * @throws RemoteException
   */
  private void showSystemVersion() throws RemoteException {
    println(getSystemVersion());
    if (isAdmin(false)) {
      println("\n" + getDeploymentInfo());
    }
  }

  /**
   * Envia uma mensagem para todos os usurios. A mensagem aparecer como popup
   * e no ser voltil.
   *
   * @see #sendMsgToLogged()
   * @see #sendMsgToUser(String)
   *
   * @throws RemoteException
   */
  private void sendMsgToAll() throws RemoteException {
    AdminParams params = (AdminParams) getParams();
    ClientRemoteLocator.notificationService.notifyTo(null, params.msgToAll,
      true, false);
  }

  /**
   * Envia uma mensagem apenas para os usurios logados. A mensagem aparecer
   * como popup e ser voltil.
   *
   * @see #sendMsgToAll()
   * @see #sendMsgToUser(String)
   *
   * @throws RemoteException
   */
  private void sendMsgToLogged() throws RemoteException {
    UserOutline[] loggedUsers = ClientRemoteLocator.server.getLoggedUsers();
    Set<Object> ids = new HashSet<>();
    for (int i = 0; i < loggedUsers.length; i++) {
      ids.add(loggedUsers[i].getId());
    }
    AdminParams params = (AdminParams) getParams();
    ClientRemoteLocator.notificationService.notifyTo(ids.toArray(),
      params.msgToLogged, true, true);
  }

  /**
   * Envia uma mensagem para um usurio especfico. A mensagem aparecer como
   * popup e ser voltil.
   *
   * @see #sendMsgToAll()
   * @see #sendMsgToLogged()
   *
   * @param msgRecipient
   */
  private void sendMsgToUser(String msgRecipient) {
    if (msgRecipient == null) {
      printError("destinatrio da mensagem no foi fornecido");
      return;
    }
    try {
      User recipient = User.getUserByLogin(msgRecipient);
      if (recipient != null) {
        AdminParams params = (AdminParams) getParams();
        ClientRemoteLocator.notificationService.notifyTo(
          new Object[] { recipient.getId() }, params.msgToUser, true, true);
      }
      else {
        printError("usurio " + msgRecipient + " no existe");
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * @throws RemoteException
   *
   */
  private void listProperties() throws RemoteException {
    if (!isAdmin(true)) {
      return;
    }
    ServerServiceInterface serverService = ClientRemoteLocator.serverService;
    if (serverService == null) {
      printError(
        "servio 'ServerService' no foi habilitado para este servidor");
      return;
    }
    String properties = serverService.listRuntimeProperties();
    AdminParams params = (AdminParams) getParams();
    if (params.propsPattern == null) {
      println(properties);
    }
    else {
      Matcher matcher = params.propsPattern.matcher(properties);
      while (matcher.find()) {
        println(matcher.group());
      }
    }
    println("=========");
    Collection<PropertyInfo> infos =
      ClientRemoteLocator.diagnosticService.getPropertiesInfo().values();
    for (PropertyInfo info : infos) {
      println(info.key + "=" + info.runtimeValue + " (" + info.serviceValue
        + ", " + info.serverValue + ", " + info.systemValue + ", "
        + info.commandLineValue + ")");
    }
  }

  /**
   * Bloqueia ou libera a fila de comandos.
   *
   * @param blockCmdQueue <code>true</code> para bloquear a fila,
   *        <code>false</code> para desbloque-la
   * @throws RemoteException
   */
  private void blockCmdsQueue(String blockCmdQueue) throws RemoteException {
    if (!isAdmin(true)) {
      return;
    }
    ClientRemoteLocator.schedulerService
      .setBlocked(Boolean.valueOf(blockCmdQueue));
  }

  /**
   * Interrompe a execuo de um comando.
   *
   * @param cmdIdToKill ID do comando
   * @throws RemoteException
   */
  private void killSGAcmd(String cmdIdToKill) throws RemoteException {
    if (!ClientRemoteLocator.sgaService.killCommand(cmdIdToKill)) {
      printError("o comando no foi encerrado");
      printError(
        "cerifique-se de que o comando existe e que voc tem permisso para interromp-lo");
    }
  }

  /**
   * Exibe as informaes de um SGA.
   *
   * @param sgaToGetInfo nome do SGA
   * @throws RemoteException
   */
  private void showInfoFromSGA(String sgaToGetInfo) throws RemoteException {
    SGAInfo info = getSGAInfo(sgaToGetInfo);
    if (info != null) {
      println("host                      = " + info.getHostName());
      println("plataforma                = " + info.getPlatformId());
      println("ativo                     = " + info.getAlive());
      println("jobs em execuo          = " + info.getNumberOfJobs());
      println("requisitos                = " + info.getRequirements());
      println("sandbox de execuo       = " + FileUtils.joinPath(true,
        info.getFileSeparator(), info.getSandboxRootDirectory()));

      // FIXME se o SGA estiver usando CSFS, mostrar a info do CSFS no lugar da 2 infos abaixo
      println("repositrio de algoritmos = " + FileUtils.joinPath(true,
        info.getFileSeparator(), info.getAlgorithmRootDirectory()));
      println("repositrio de projetos   = " + FileUtils.joinPath(true,
        info.getFileSeparator(), info.getProjectRootDirectory()));

    }
    else {
      printError("no existe SGA registrado com nome " + sgaToGetInfo);
    }
  }

  /**
   * Obtm as informaes de um SGA.
   *
   * @param sgaToGetInfo nome do SGA
   * @return informaes do SGA
   * @throws RemoteException em caso de erro de comunicao.
   */
  protected static SGAInfo getSGAInfo(String sgaToGetInfo)
    throws RemoteException {
    SGAInfo info = ClientRemoteLocator.sgaService.getInfo(sgaToGetInfo, 0);
    return info;
  }

  /**
   * Encerra execuo do servidor. Pede confirmao caso haja algum usurio
   * logado.
   *
   * @throws PermissionException se o usurio no tiver permisso para o
   *         shutdown (na prtica no vai acontecer, pois garantimos antes que o
   *         usurio  o admin)
   * @throws RemoteException em caso de erro de comunicao.
   */
  private void shutdownServer() throws PermissionException, RemoteException {
    if (!isAdmin(true)) {
      return;
    }
    UserOutline[] loggedUsers = getLoggedUsers();
    AdminParams params = (AdminParams) getParams();
    if (loggedUsers.length > 1 && !params.force) {
      String msg =
        "Existe(m) %d usurio(s) logado(s). Confirma shutdown do servidor? (s/n) ";
      String answer = readLine(msg, loggedUsers.length - 1);
      if (!answer.equalsIgnoreCase("s")) {
        System.out.println("operao cancelada pelo usurio");
        return;
      }
    }
    try {
      ClientRemoteLocator.killServerService.shutdown();
    }
    catch (UnmarshalException e) {
      /*
       * esta exceo  esperada dado que o servidor  terminado antes que a
       * chamada RMI seja concluda
       */
    }
    serverHasBeenShutdown = true;
  }

  /**
   * Reinicia todos os SGAs.
   *
   * @throws RemoteException
   */
  private void restartAllSGAs() throws RemoteException {
    if (!isAdmin(true)) {
      return;
    }
    ClientRemoteLocator.sgaService.restartAllSGAs();
  }

  /**
   * Encerra execuo de todos os SGAs.
   *
   * @throws RemoteException
   */
  private void stopAllSGAs() throws RemoteException {
    if (!isAdmin(true)) {
      return;
    }
    ClientRemoteLocator.sgaService.shutdownAllSGAs();
  }

  /**
   * Reinicia a execuo de um SGA especfico.
   *
   * @param sgaToRestart nome do SGA
   * @throws RemoteException
   */
  private void restartSGA(String sgaToRestart) throws RemoteException {
    if (!isAdmin(true)) {
      return;
    }
    ClientRemoteLocator.sgaService.restartSGA(sgaToRestart);
  }

  /**
   * Encerra a execuo de um SGA especfico.
   *
   * @param sgaToStop nome do SGA
   * @throws RemoteException
   */
  private void stopSGA(String sgaToStop) throws RemoteException {
    if (!isAdmin(true)) {
      return;
    }
    ClientRemoteLocator.sgaService.shutdownSGA(sgaToStop);
  }

  /**
   * Lista os SGAs atualmente cadastrados no servidor.
   *
   * @throws RemoteException
   */
  private void listSGAs() throws RemoteException {
    Vector<String> sgaNames = ClientRemoteLocator.sgaService.getAllSGANames();
    if (sgaNames.isEmpty()) {
      printInfo("no h nenhum SGA registrado no servidor");
    }
    else {
      println();
      int maxLen = getMaxStrLen(sgaNames);
      Collections.sort(sgaNames);
      for (String sga : sgaNames) {
        SGASet sgaSet = ClientRemoteLocator.sgaService.getSGASet(sga);
        String extraInfo;
        if (sgaSet.isCluster()) {
          extraInfo = getClusterInfo(sgaSet);
        }
        else {
          extraInfo = sgaSet.getAlive() ? "disponvel" : "indisponvel";
        }
        String fmt = "%-" + (maxLen + 3) + "s%s";
        println(fmt, sga, extraInfo);
      }
    }
    boolean blocked = ClientRemoteLocator.schedulerService.isBlocked();
    if (blocked) {
      println();
      printInfo("a fila de comandos est bloqueada");
    }
  }

  /**
   * Obtm as informaes de disponibilidade de um cluster (quantidade de ns
   * atvos e inativos).
   *
   * @param sga SGA
   * @return string informando que se trata de um cluster, e quantos ns esto
   *         ativos e inativos
   * @throws RemoteException
   */
  private static String getClusterInfo(SGASet sga) throws RemoteException {
    SGAInfo[] infos = ClientRemoteLocator.sgaService.getAllInfo(sga.getName());
    int numInactive = 0;
    for (SGAInfo info : infos) {
      if (!info.getAlive()) {
        numInactive++;
      }
    }
    if (numInactive > 0) {
      return String.format("cluster (%d ns, %d indisponveis)", infos.length,
        numInactive);
    }
    return String.format("cluster %d ns disponveis)", infos.length);
  }

  /**
   * Lista usurios conectados.
   *
   * @throws RemoteException
   */
  private void listLoggedUsers() throws RemoteException {
    UserOutline[] loggedUsers = getLoggedUsers();
    for (UserOutline userOutline : loggedUsers) {
      println("%-10s%s", userOutline.getLogin(), userOutline.getName());
    }
  }

  /**
   * Obtm um vetor com os usurios logados, ordenado alfabeticamente pelo
   * login.
   *
   * @return vetor com os usurios logados
   * @throws RemoteException
   */
  static UserOutline[] getLoggedUsers() throws RemoteException {
    UserOutline[] loggedUsers = ClientRemoteLocator.server.getLoggedUsers();
    Arrays.sort(loggedUsers, new Comparator<UserOutline>() {

      @Override
      public int compare(UserOutline o1, UserOutline o2) {
        return o1.getLogin().compareTo(o2.getLogin());
      }
    });
    return loggedUsers;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean preLogout() {
    return !serverHasBeenShutdown;
  }
}
