package csbase.rest.adapter.algorithm.v1;

import java.io.File;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;

import csbase.logic.AlgorithmExecutionPermission;
import csbase.logic.CategoryAlgorithmsExecutionPermission;
import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.CategorySet;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.conditions.Condition;
import csbase.logic.algorithms.parameters.conditions.SimpleCondition;
import csbase.logic.algorithms.parameters.triggers.Trigger;
import csbase.remote.AlgorithmServiceInterface;
import csbase.remote.ClientRemoteLocator;
import ibase.common.ServiceAdapter;
import ibase.exception.InternalServiceException;
import ibase.rest.api.algorithm.v1.adapter.AlgorithmNotFoundException;
import ibase.rest.api.algorithm.v1.adapter.AlgorithmServiceAdapter;
import ibase.rest.api.algorithm.v1.adapter.VersionNotFoundException;
import ibase.rest.model.algorithm.v1.Action;
import ibase.rest.model.algorithm.v1.Algorithm;
import ibase.rest.model.algorithm.v1.AlgorithmConfiguration;
import ibase.rest.model.algorithm.v1.AlgorithmVersion;

/**
 * Created by mjulia on 08/06/16.
 */
public class CSBaseAlgorithmServiceAdapter implements AlgorithmServiceAdapter {

  private final static String CURRENT_SYSTEM_ID = ".";

  private static final Logger logger = Logger.getLogger(
    CSBaseAlgorithmServiceAdapter.class.getName());

  private ParameterMapper parameterMapper = new ParameterMapper();
  private ActionMapper actionMapper = new ActionMapper();


  @Override
  public void setLocale(Locale locale) {
    ClientRemoteLocator.administrationService.setLocale(locale);
  }

  @Override
  public boolean existsAlgorithm(String algorithmId) {
    AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
    return getAlgorithmInfo(service, algorithmId) != null;
  }

  @Override
  public boolean existsAlgorithmVersion(String algorithmId, String versionId) throws AlgorithmNotFoundException {
    AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
    try {
      AlgorithmInfo algoInfo = getAlgorithmInfo(service, algorithmId);
      if (algoInfo == null) {
        throw new AlgorithmNotFoundException();
      }
      AlgorithmVersionId algoVersionId = AlgorithmVersionId.create(versionId);
      AlgorithmVersionInfo version = algoInfo.getVersionInfo(algoVersionId);
      return version != null;
    } catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  @Override
  public List<Algorithm> getAlgorithms(Predicate<Algorithm> predicate) {
    AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
    AlgorithmInfo[] algoInfos;
    try {
      algoInfos = service.getAllInfo();
    } catch (Throwable e) {
      throw new InternalServiceException(e);
    }
    List<Algorithm> algorithms = new ArrayList<Algorithm>();
    csbase.logic.User user;
    try {
      user = User.getUserByLogin(ServiceAdapter.getCurrenUser());
    } catch (Exception e) {
      throw new InternalServiceException(e);
    }
    Arrays.asList(algoInfos).stream().sorted().forEach(i -> {
      if (userHasPermissionToExecuteAlgorithm(user, i)) {
        try {
          Algorithm a = buildAlgorithm(i);
          if (predicate == null) {
            algorithms.add(a);
          }
          else if (predicate.test(a)) {
            algorithms.add(a);
          }
        } catch (Throwable e) {
          e.printStackTrace();
        }
      }
    });
    return algorithms;

  }

  @Override
  public Optional<Algorithm> getAlgorithm(String algorithmId) {
    AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
    try {
      Optional<Algorithm> algorithm = Optional.empty();
      AlgorithmInfo algoInfo = getAlgorithmInfo(service, algorithmId);
      if (algoInfo != null) {
        algorithm = Optional.of(buildAlgorithm(algoInfo));
      }
      return algorithm;
    } catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  @Override
  public List<AlgorithmVersion> getAlgorithmVersions(String algorithmId) throws AlgorithmNotFoundException {
    AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
    AlgorithmInfo algoInfo;
    try {
      algoInfo = getAlgorithmInfo(service, algorithmId);
    } catch (Throwable e) {
      throw new InternalServiceException(e);
    }
    if (algoInfo == null) {
      throw new AlgorithmNotFoundException();
    }
    List<AlgorithmVersionInfo> versionInfos = algoInfo.getVersions();
    if (versionInfos != null) {
      List<AlgorithmVersion> versions = versionInfos.stream().map(v -> buildAlgorithmVersion(algoInfo, v))
        .collect(Collectors.toList());
      return versions;
    }
    return null;
  }

  @Override
  public Optional<AlgorithmVersion> getAlgorithmVersion(String algorithmId, String versionId)
    throws AlgorithmNotFoundException {
    AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
    AlgorithmInfo algoInfo;
    AlgorithmVersionId algoVersionId = AlgorithmVersionId.create(versionId);
    try {
      algoInfo = getAlgorithmInfo(service, algorithmId);
    } catch (Throwable e) {
      throw new InternalServiceException(e);
    }
    if (algoInfo == null) {
      throw new AlgorithmNotFoundException();
    }
    Optional<AlgorithmVersion> algorithmVersion = Optional.empty();
    AlgorithmVersionInfo version = algoInfo.getVersionInfo(algoVersionId);
    if (version != null) {
      algorithmVersion = Optional.of(buildAlgorithmVersion(algoInfo, version));
    }
    return algorithmVersion;
  }

  @Override
  public AlgorithmConfiguration getAlgorithmConfiguration(String algorithmId, String versionId)
    throws AlgorithmNotFoundException, VersionNotFoundException {
    if (!existsAlgorithm(algorithmId)) {
      throw new AlgorithmNotFoundException();
    }
    if (!existsAlgorithmVersion(algorithmId, versionId)) {
      throw new VersionNotFoundException();
    }
    AlgorithmConfiguration algorithmConfig = new AlgorithmConfiguration();
    AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
    AlgorithmVersionId algoVersionId = AlgorithmVersionId.create(versionId);
    try {
      AlgorithmConfigurator configurator = service.createAlgorithmConfigurator(algorithmId, algoVersionId);
      if (SimpleAlgorithmConfigurator.class.isInstance(configurator)) {
        SimpleAlgorithmConfigurator simpleConfigurator = SimpleAlgorithmConfigurator.class.cast(configurator);
        List<csbase.logic.algorithms.parameters.ParameterGroup> groups = simpleConfigurator.getGroups();
        algorithmConfig
        .setGroups(groups.stream().map(g -> buildParameterGroup(g)).collect(Collectors.toList()));
        algorithmConfig.setCommand(simpleConfigurator.getCommandBinaryName());
        algorithmConfig.setExecutionType(getExecutionType(simpleConfigurator.getExecutionType()));
        Set<Trigger<?>> triggers = simpleConfigurator.getTriggers();
        if (triggers != null && !triggers.isEmpty()) {
          List<Action> actions = new ArrayList();
          for (Trigger trigger : triggers) {
            Action action = buildTriggerAction(trigger);
            actions.add(action);
          }
          algorithmConfig.setActions(actions);
        }
        algorithmConfig.setLoadParameters(simpleConfigurator.allowParametersLoad());
      }
    } catch (Throwable e) {
      throw new InternalServiceException(e);
    }
    return algorithmConfig;
  }

  private Action buildTriggerAction(Trigger trigger) {
    Action action = new Action();
    Action.NameEnum actionType = actionMapper.getActionFor(trigger.getClass());
    action.setName(actionType);
    final Parameter parameter = trigger.getParameter();
    action.setTargetId(parameter.getName());
    final Condition condition = trigger.getCondition();
    if (SimpleCondition.class.isInstance(condition)) {
      ibase.rest.model.algorithm.v1.Condition jsonCondition = new ibase.rest.model.algorithm.v1.Condition();
      SimpleCondition simpleCondition = SimpleCondition.class.cast(condition);
      jsonCondition.setParameterId(simpleCondition.getParameterName());
      Object value = simpleCondition.getValue();
      jsonCondition.setValue(value);
      action.setCondition(jsonCondition);
    }
    return action;
  }

  /**
   * Build a JSON algorithm object model.
   * 
   * @param info
   *            The CSBase algorithm info
   * @return the JSON object representing the algorithm
   */
  private Algorithm buildAlgorithm(AlgorithmInfo info) {
    Algorithm algorithm = new Algorithm();
    algorithm.setName(info.getName());
    algorithm.setId(info.getId());
    String owner = info.getOwner();
    if (owner != null) {
      algorithm.setWhoCreated(buildRestUser(owner));
    }
    List<AlgorithmVersionInfo> versions = info.getVersions();
    if (versions != null) {
      algorithm.setVersions(
        versions.stream().map(v -> buildAlgorithmVersion(info, v)).collect(Collectors.toList()));
      AlgorithmVersionInfo lastVersion = info.getLastVersion();
      if (lastVersion != null) {
        algorithm.setLastVersion(buildAlgorithmVersion(info, lastVersion));
      }
    }
    return algorithm;
  }

  private ibase.rest.model.algorithm.v1.User buildRestUser(String userId) {
    Client client = ClientBuilder.newClient();
    WebTarget target = client.target(ServiceAdapter.getURI());
    Response response = target.path("users").path(userId).queryParam("locale", ClientRemoteLocator.administrationService.getCurrentLocale()).
      request("application/json;charset=UTF-8")
      .header(HttpHeaders.AUTHORIZATION, ServiceAdapter.getAutheticationHeader()).get();
    if (response.getStatus() != 200) {
      return null;
    }
    ibase.rest.model.algorithm.v1.User restUser = response.readEntity(ibase.rest.model.algorithm.v1.User.class);
    return restUser;
  }

  private ibase.rest.model.algorithm.v1.AlgorithmVersion buildAlgorithmVersion(AlgorithmInfo algoInfo,
    AlgorithmVersionInfo versionInfo) {
    AlgorithmVersion version = new AlgorithmVersion();
    version.setId(versionInfo.getId().toString());
    try {
      AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
      AlgorithmConfigurator configurator = service.createAlgorithmConfigurator(algoInfo.getId(),
        versionInfo.getId());
      version.setDescription(configurator.getDescription());
    } catch (RemoteException e) {
      // Nothing to do -- it is not a remote call
    }
    RequirementsBuilder mapper = new RequirementsBuilder(algoInfo.getId(), versionInfo.getId(),
      ClientRemoteLocator.administrationService.getCurrentLocale());
    version.setRequirements(mapper.requirements);
    return version;
  }

  private ibase.rest.model.algorithm.v1.ParameterGroup buildParameterGroup(
    csbase.logic.algorithms.parameters.ParameterGroup group) {
    ibase.rest.model.algorithm.v1.ParameterGroup parameterGroup = new ibase.rest.model.algorithm.v1.ParameterGroup();
    parameterGroup.setParameters(group.getSimpleParameters().stream()
      .map(p -> parameterMapper.getFactoryFor(p.getType()).buildParameter(p)).collect(Collectors.toList()));
    parameterGroup.setId(group.getName());
    parameterGroup.setLabel(group.getLabel());
    parameterGroup.setCollapsable(group.isCollapsible());
    return parameterGroup;
  }

  private ibase.rest.model.algorithm.v1.AlgorithmConfiguration.ExecutionTypeEnum getExecutionType(
    ExecutionType type) {
    switch (type) {
      case SIMPLE:
        return AlgorithmConfiguration.ExecutionTypeEnum.SIMPLE;
      case MULTIPLE:
        return AlgorithmConfiguration.ExecutionTypeEnum.MULTIPLE;
    }
    return null;
  }

  private AlgorithmInfo getAlgorithmInfo(AlgorithmServiceInterface service, String algorithmId)
    throws InternalServiceException {
    try {
      AlgorithmInfo algoInfo = service.getInfo((Object) algorithmId);
      return (algoInfo != null
        && !userHasPermissionToExecuteAlgorithm(User.getUserByLogin(ServiceAdapter.getCurrenUser()),
          algoInfo)) ? null : algoInfo;
    } catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  private boolean userHasPermissionToExecuteAlgorithm(csbase.logic.User user, AlgorithmInfo algoInfo) {
    try {
      String algorithmName = algoInfo.getName();
      CategorySet allCategories = ClientRemoteLocator.algorithmService.getAllCategories();
      List<String> categoriesFullNames = allCategories.getAlgorithmCategoriesFullNames(algoInfo);
      return user.isAdmin()
        || AlgorithmExecutionPermission.checkSystemAndAlgorithmExecPermission(user, CURRENT_SYSTEM_ID,
          algorithmName)
        || CategoryAlgorithmsExecutionPermission.checkSystemAndCategoriesExecPermission(user,
          CURRENT_SYSTEM_ID, categoriesFullNames);
    } catch (Exception e) {
      logger.log(Level.SEVERE,
        "Erro no metodo userHasPermissionToExecuteAlgorithm", e);
      return false;
    }
  }

  @Override
  public String getAlgorithmDirPath(Object algorithmId,
    int versionMajor, int versionMinor, int versionPath, String platformId) {
    try {
      AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
      AlgorithmInfo info = service.getInfo(algorithmId);
      return info.getAlgorithmRepositoryPath() + File.separator + info
        .getDirectory() + File.separator + "versions" + File.separator
        + AlgorithmVersionInfo.getDirectoryFor(versionMajor, versionMinor,
          versionPath) + File.separator + "bin" + File.separator + platformId;
    }
    catch (Exception e) {
      logger.log(Level.SEVERE, "Erro no metodo getAlgorithmDirPath", e);
      return null;
    }
  }
}
