package csbase.rest.adapter.drmaa2.v1;

import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.ResourceBundle;

import org.json.JSONArray;

import csbase.exception.ParseException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.Priority;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.BooleanParameter;
import csbase.logic.algorithms.parameters.DoubleListParameter;
import csbase.logic.algorithms.parameters.DoubleParameter;
import csbase.logic.algorithms.parameters.EnumerationListParameter;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.InputFileListParameter;
import csbase.logic.algorithms.parameters.InputFileParameter;
import csbase.logic.algorithms.parameters.IntegerListParameter;
import csbase.logic.algorithms.parameters.IntegerParameter;
import csbase.logic.algorithms.parameters.OutputFileListParameter;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TextListParameter;
import csbase.logic.algorithms.parameters.TextParameter;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.remote.AlgorithmServiceInterface;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;
import ibase.common.ServiceUtil;
import ibase.exception.InternalServiceException;
import ibase.rest.model.drmaa2.v1.JobTemplate;

/**
 * Valida um job template que descreve os parmetros para execuo de um
 * algoritmo.
 *
 * @author Tecgraf PUC-Rio
 */
public class JobTemplateValidator {

  /**
   * Language bundle
   */
  public static final String RESOURCE_BUNDLE = "JobTemplateValidator";

  private static final String ALGORITHM_ID_KEY = "algorithmId";
  private static final String VERSION_ID_KEY = "versionId";
  private static final String PROJECT_ID_KEY = "projectId";

  /**
   * Identificador da raiz do sistema de arquivos
   **/
  private static final String ROOT = "root";

  /**
   * O job template a ser validado
   */
  private JobTemplate jobTemplate;

  /**
   * O configurador do algoritmo descrito pelo job template
   */
  private AlgorithmConfigurator configurator;

  /**
   * O identificador da verso do algoritmo
   */
  private AlgorithmVersionId algorithmVersionId;

  /**
   * O identificador projeto sobre o qual esse job template atua
   */
  private String projectId;

  /**
   * O identificador do algoritmo
   */
  private String algorithmId;

  /**
   * Identificador do usurio.
   */
  private String userId;

  /**
   * Idioma a ser usado.
   */
  private Locale locale;

  private ResourceBundle getBundle() {
    ResourceBundle bundle = ResourceBundle
      .getBundle(RESOURCE_BUNDLE, locale, this.getClass().getClassLoader());
    return bundle;
  }

  /**
   * Constri um validador de um <code>OpenDreamsJobTemplate</code>
   *
   * @param userId o identificador do usuario.
   * @param jt o job template a ser validado
   */
  public JobTemplateValidator(String userId, JobTemplate jt, Locale locale) {
    this.userId = userId;
    this.jobTemplate = jt;
    this.locale = locale;
    validateRemoteCommand();
    validateProject();
    validateArgs();
    validateJobPriority();
  }

  /**
   * Faz a validao do atributo <b>remoteCommand</b> do job template.
   **/
  private boolean validateRemoteCommand() {
    Object remoteCommand = jobTemplate.getRemoteCommand();
    if (remoteCommand == null) {
      String message = ServiceUtil.getTranslator(getBundle()).
        message("remoteCommand.missing.error");
      throw new InternalServiceException(message);
    }
    if (!LinkedHashMap.class.isInstance(remoteCommand)) {
      String message = ServiceUtil.getTranslator(getBundle()).
        message("remoteCommand.invalid.format.error");
      throw new InternalServiceException(message);
    }
    LinkedHashMap<String, Object> remoteCommandMap =
      LinkedHashMap.class.cast(remoteCommand);
    this.algorithmId = (String) remoteCommandMap.get(ALGORITHM_ID_KEY);
    if (algorithmId == null) {
      String message = ServiceUtil.getTranslator(getBundle()).
        message("algorithmId.missing.error");
      throw new InternalServiceException(message);
    }
    String versionId = (String) remoteCommandMap.get(VERSION_ID_KEY);
    if (versionId == null) {
      String message = ServiceUtil.getTranslator(getBundle()).
        message("versionId.missing.error");
      throw new InternalServiceException(message);
    }
    this.algorithmVersionId = AlgorithmVersionId.create(versionId);
    String projectParam = (String) remoteCommandMap.get(PROJECT_ID_KEY);
    if (projectParam == null) {
      String message = ServiceUtil.getTranslator(getBundle()).
        message("projectId.missing.error");
      throw new InternalServiceException(message);
    }
    try {
      this.projectId = ServiceUtil.decodeFromBase64(projectParam);
      AlgorithmServiceInterface algorithmService =
        ClientRemoteLocator.algorithmService;
      this.configurator = algorithmService
        .createAlgorithmConfigurator(algorithmId, algorithmVersionId);
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
    return true;
  }


  /**
   * Valida se o projeto existe e se o usurio tem permisso de escrita no
   * projeto.
   */
  private boolean validateProject() {
    try {
      ProjectServiceInterface projectService =
        ClientRemoteLocator.projectService;
      if (projectService.openProject(projectId, false) == null) {
        String message = ServiceUtil.getTranslator(getBundle()).
          message("projectId.not.found.error", projectId);
        throw new InternalServiceException(message);
      }
      if (!projectService.userCanWrite(projectId, userId)) {
        String message = ServiceUtil.getTranslator(getBundle()).
          message("projectId.user.can.not.write", userId, projectId);
        throw new InternalServiceException(message);
      }
      return true;
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }


  /**
   * Valida os parmetros para execuo do algoritmo.
   */
  private void validateArgs() {
    LinkedHashMap<String, Object> argsMap =
      LinkedHashMap.class.cast(jobTemplate.getArgs());
    argsMap.forEach((k, v) -> setParameterValue(k, v));
    try {
      ValidationContext context =
        new ValidationContext(ValidationMode.FULL, projectId, userId);
      Validation result = configurator.validate(context);
      if (!result.isWellSucceeded()) {
        String message = ServiceUtil.getTranslator(getBundle()).
          message("args.invalid.format");
        throw new InternalServiceException(message);
      }
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * Atribui o valor do parmetro.
   *
   * @param key chave do parmetro.
   * @param value o valor do parmetro.
   */
  private void setParameterValue(String key, Object value) {
    try {
      String type = configurator.getParameterType(key);
      if (SimpleAlgorithmConfigurator.class.isInstance(configurator)) {
        SimpleAlgorithmConfigurator config =
          SimpleAlgorithmConfigurator.class.cast(configurator);
        SimpleParameter<?> simpleParameter = config.getSimpleParameter(key);
        setParameterValue(simpleParameter, type, value);
      }
      else if (FlowAlgorithmConfigurator.class.isInstance(configurator)) {
        FlowAlgorithmConfigurator config =
          FlowAlgorithmConfigurator.class.cast(configurator);
        {
          SimpleParameter<?> simpleParameter = config.getNodeParameter(key);
          setParameterValue(simpleParameter, type, value);
        }
      }
    }
    catch (ParameterNotFoundException e) {
      throw new InternalServiceException(e);
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * Atribui o valor do parmetro.
   *
   * @param simpleParameter o parmetro.
   * @param type o tipo do parmetro.
   * @param value o valor do parmetro.
   * @throws ParseException em caso de erro no formato do valor.
   */
  private void setParameterValue(SimpleParameter<?> simpleParameter,
    String type, Object value) throws ParseException {
    Boolean useRawParameterValues = jobTemplate.getUseRawParameterValues();
    if(useRawParameterValues == Boolean.TRUE) {
      simpleParameter.setValueAsText((String) value);
    } else {
      switch (type) {
        case IntegerListParameter.TYPE: {
          IntegerListParameter integerListParameter = IntegerListParameter.class.cast(simpleParameter);
          JSONArray list = new JSONArray(value);
          list.forEach(i -> integerListParameter.addElement((Integer) i));
          break;
        }
        case IntegerParameter.TYPE: {
          IntegerParameter integerParameter = IntegerParameter.class.cast(simpleParameter);
          integerParameter.setValue((Integer) value);
          break;
        }
        case DoubleListParameter.TYPE: {
          DoubleListParameter doubleListParameter = DoubleListParameter.class
            .cast(simpleParameter);
          JSONArray list = new JSONArray(value);
          list.forEach(i -> doubleListParameter.addElement((Double) i));
          break;
        }
        case DoubleParameter.TYPE: {
          DoubleParameter doubleParameter = DoubleParameter.class.cast(simpleParameter);
          doubleParameter.setValue(Double.parseDouble("" + value));
          break;
        }
        case BooleanParameter.TYPE: {
          BooleanParameter booleanParameter = BooleanParameter.class.cast(simpleParameter);
          booleanParameter.setValue((Boolean) value);
          break;
        }
        case TextListParameter.TYPE: {
          TextListParameter textListParameter = TextListParameter.class.cast(simpleParameter);

          JSONArray list = new JSONArray(value);
          list.forEach(i -> textListParameter.addElement((String) i));
          break;
        }
        case TextParameter.TYPE: {
          TextParameter textParameter = TextParameter.class.cast(simpleParameter);
          textParameter.setValue((String) value);
          break;
        }
        case EnumerationListParameter.TYPE: {
          EnumerationListParameter enumListParameter = EnumerationListParameter.class.cast(simpleParameter);
          JSONArray list = new JSONArray(value);
          list.forEach(item -> addEnumItem(enumListParameter, (String) item));
          break;
        }
        //TODO: Enumerao simples?
        case EnumerationParameter.TYPE: {
          EnumerationParameter enumParameter = EnumerationParameter.class
            .cast(simpleParameter);
          enumParameter.setValue(enumParameter.getItem((String) value));
        }
        case InputFileParameter.TYPE: {
          if (InputFileParameter.class.isInstance(simpleParameter)) {
            InputFileParameter inputFileParameter = InputFileParameter.class
              .cast(simpleParameter);
            String filePath = ROOT.equalsIgnoreCase((String) value) ? "." :
              (String) value;
            inputFileParameter.setValueAsText(filePath);
            break;
          }
          if (InputFileListParameter.class.isInstance(simpleParameter)) {
            InputFileListParameter inputFileListParameter =
              InputFileListParameter.class.cast(simpleParameter);
            // TODO ver aqui ainda como fazer
            break;
          }
        }
        case OutputFileParameter.TYPE: {
          if (OutputFileParameter.class.isInstance(simpleParameter)) {
            OutputFileParameter outputFileParameter = OutputFileParameter
              .class.cast(simpleParameter);
            String filePath = ROOT.equalsIgnoreCase((String) value) ? "." : (String) value;
            outputFileParameter.setValueAsText(filePath);
            break;
          }
          if (OutputFileListParameter.class.isInstance(simpleParameter)) {
            OutputFileListParameter outputFileListParameter =
              OutputFileListParameter.class.cast(simpleParameter);
            // TODO ver aqui ainda como fazer
            break;
          }
        }
        default: {
          simpleParameter.setValueAsText((String) value);
        }
      }
    }
  }

  private String getFileId(String value) {
    return ROOT.equalsIgnoreCase(value) ? "" : ServiceUtil
      .decodeFromBase64(value);
  }

  private void addEnumItem(EnumerationListParameter enumListParameter,
    String item) {
    try {
      enumListParameter
        .addElement(enumListParameter.getItemValueFromText(item));
    }
    catch (ParseException e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * Valida o atributo <code>jobPriority</code> do job template.
   */
  private void validateJobPriority() {
    if (this.jobTemplate.getPriority() < 0 || this.jobTemplate
      .getPriority() >= Priority.values().length) {
      String message = ServiceUtil.getTranslator(getBundle()).
        message("priority.error");
      throw new InternalServiceException(message);
    }
  }

  /**
   * Obtm o nome do algoritmo.
   *
   * @return o nome do algoritmo
   */
  public String getAlgorithmId() {
    return this.algorithmId;
  }

  /**
   * Obtm a verso do algoritmo.
   *
   * @return a verso do algoritmo
   */
  public AlgorithmVersionId getAlgorithmmVersionId() {
    return this.algorithmVersionId;
  }

  /**
   * Obtm o configurador do algoritmo correspondente ao job template.
   *
   * @return o configurador do algoritmo criado a partir do job template.
   */
  public AlgorithmConfigurator getAlgorithmConfigurator() {
    return this.configurator;
  }

  /**
   * Obtm o identificador do projeto.
   *
   * @return project id.
   */
  public Object getProjectId() {
    return projectId;
  }
}
