package csbase.rest.adapter.job.v1;

import csbase.logic.*;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.CommandPersistenceServiceInterface;
import csbase.remote.ProjectServiceInterface;
import csbase.remote.SchedulerServiceInterface;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import csbase.server.services.projectservice.ProjectService;
import ibase.common.ServiceAdapter;
import ibase.common.ServiceUtil;
import ibase.exception.InternalServiceException;
import ibase.rest.api.job.v1.adapter.JobsServiceAdapter;
import ibase.rest.api.job.v1.adapter.JobDAO;
import ibase.rest.api.job.v1.adapter.JobMonitorListener;
import ibase.rest.api.job.v1.factories.JobsDAOFactory;
import ibase.rest.model.job.v1.Job;
import ibase.rest.model.job.v1.JobSession;
import ibase.rest.model.job.v1.JobTemplate;
import ibase.rest.model.job.v1.StatusType;
import tecgraf.javautils.core.timestamp.TStamp32;

import java.io.*;
import java.rmi.RemoteException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by Tecgraf/PUC-Rio on 11/07/16.
 */
public class CSBaseJobsServiceAdapter implements JobsServiceAdapter {

  /** Language bundle */
  public static final String RESOURCE_BUNDLE = "CSBaseJobsServiceAdapter";

  /** Listerner que ouve mudancas do estado dos jobs */
  private final JobStateListener jobStateListener;

  /** Monitor de mudanas de info de jobs */
  private final JobInfoMonitor jobInfoMonitor;

  /** Cache para evitar ir desnecessariamente ao banco */
  private final LocalCache<String, Job> localCache;
  /** Tempo que ser usado o cache acima. Depois disso, devera ir no banco de */
  private final long cacheTime = 300000; //5m

  /**
   * @return
   */
  private JobDAO getJobDAO() {
    JobsDAOFactory factory = ServiceAdapter.getDAOFactory(JobsDAOFactory.class);
    if (factory==null) {
      String message = ServiceUtil.getTranslator(
        getBundle(ClientRemoteLocator.administrationService.getCurrentLocale())).
        message("job.dao.not.found");
      throw new InternalServiceException(message);
    }
    return factory.getJobDAO();
  }

  /**
   * @param sessionName
   * @return
   */
  private boolean isUnique(String sessionName) {
    return !getJobDAO().containsJobSession(sessionName);
  }

  /**
   * @param userId
   * @return
   */
  private String generateUniqueSessionName(String userId) {
    return String.format("%s@%s", userId, new TStamp32().toString());
  }

  /**
   * @param locale
   * @return
   */
  private ResourceBundle getBundle(Locale locale) {
    ResourceBundle bundle = ResourceBundle.getBundle(RESOURCE_BUNDLE, locale, this.getClass().getClassLoader());
    return bundle;
  }

  /**
   * Retorna um estatus de comando, dado a causa
   * 
   * @param type
   * @param cause
   * @return status do comando
   */
  @SuppressWarnings("incomplete-switch")
  private Job.ExitStatusEnum getExitStatus(CommandFinalizationType type, FailureFinalizationType cause) {
    switch (type) {
      case END:
      case NO_EXIT_CODE :
        return Job.ExitStatusEnum.UNKNOWN;
      case SUCCESS :
        return Job.ExitStatusEnum.SUCCESS;
      case EXECUTION_ERROR :
        return Job.ExitStatusEnum.EXECUTION_ERROR;
      case FAILED :
        switch (cause) {
          case UNKNOWN :
            return Job.ExitStatusEnum.UNKNOWN;
          case COMMAND_IDENTIFIER_NOT_FOUND :
            return Job.ExitStatusEnum.JOB_IDENTIFIER_NOT_FOUND;
          case SGA_EXECUTION_ERROR :
            return Job.ExitStatusEnum.UNEXPECTED_MACHINE_ERROR;
          case FAILED_SETUP_EXECUTION_ENVIRONMENT :
            return Job.ExitStatusEnum.FAILED_SETUP_EXECUTION_ENVIRONMENT;
          case NO_SGA_AVAILABLE_TO_ROOT_COMMAND :
          case SGA_IS_NOT_AVAILABLE :
            return Job.ExitStatusEnum.NO_MACHINE_AVAILABLE;
          case PROJECT_NOT_FOUND :
            return Job.ExitStatusEnum.PROJECT_NOT_FOUND;
          case USER_WITHOUT_PERMISSION_FOR_EXECUTION :
            return Job.ExitStatusEnum.NO_PERMISSION;
        }
      case KILLED :
        return Job.ExitStatusEnum.KILLED;
      case LOST :
        return Job.ExitStatusEnum.LOST;
      default :
        return Job.ExitStatusEnum.UNDEFINED;
    }
  }

  /**
   * @param cmdInfo
   * @param job
   * @return Job com as informacoes atualizadas
   */
  private Job updateJobInfo(CommandInfo cmdInfo, Job job) {
    job.setState(getJobState(cmdInfo));
    job.setNumberOfAttempts(cmdInfo.getSubmissionAttempts());
    CommandFinalizationInfo finalizationInfo = cmdInfo.getFinalizationInfo();
    if (finalizationInfo!=null) {
      job.setExitCode(finalizationInfo.getExitCode());
      job.setExitStatus(getExitStatus(finalizationInfo.getFinalizationType(),
        finalizationInfo.getFailureCause()));
      job.setCpuTime(cmdInfo.getCpuTimeSec());
      job.setWallclockTime(cmdInfo.getWallTimeSec());
      job.setExecutionMachine(cmdInfo.getSGAName() != null ? cmdInfo.getSGAName() : "");
      job.setRamMemory(cmdInfo.getRAMMemoryMB());
    }
    return job;
  }

  // TODO REMOVER
  /**
   * @param cmdInfo
   */
  private void printDetailedJobInfo(CommandInfo cmdInfo) {
    System.out.println("---- DADOS DETALHADOS DO JOB " + cmdInfo.getId() + " ----");
    System.out.println("---- CpuTimeSec: " + cmdInfo.getCpuTimeSec());
    System.out.println("---- CpuPerc: " + cmdInfo.getCpuPerc());
    System.out.println("---- WallTimeSec: " + cmdInfo.getWallTimeSec());
    System.out.println("---- BytesInKB: " + cmdInfo.getBytesInKB());
    System.out.println("---- BytesOutKB: " + cmdInfo.getBytesOutKB());
    System.out.println("---- DiskBytesReadKB: " + cmdInfo.getDiskBytesReadKB());
    System.out.println("---- DiskBytesWriteKB: " + cmdInfo.getDiskBytesWriteKB());
    System.out.println("---- GlobalPosition: " + cmdInfo.getGlobalPosition());
    System.out.println("---- Priority: " + cmdInfo.getPriority());
    System.out.println("---- RAMMemoryMB: " + cmdInfo.getRAMMemoryMB());
    System.out.println("---- RAMMemoryPerc: " + cmdInfo.getRAMMemoryPerc());
    System.out.println("---- SwapMemoryMB: " + cmdInfo.getSwapMemoryMB());
    System.out.println("---- SwapMemoryPerc: " + cmdInfo.getSwapMemoryPerc());
    System.out.println("---- UserTimeSec: " + cmdInfo.getUserTimeSec());
  }

  /**
   * @param cmdInfo
   * @return
   */
  private StatusType getJobState(CommandInfo cmdInfo) {
    CommandStatus cmdStatus = cmdInfo.getStatus();
    if (cmdInfo.isQueued()) {
      return StatusType.QUEUED;
    }
    switch (cmdStatus) {
      case INIT:
      case SCHEDULED:
        return StatusType.SCHEDULED;
      case UPLOADING:
        return  StatusType.UPLOADING;
      case EXECUTING:
        return StatusType.EXECUTING;
      case DOWNLOADING:
        return StatusType.DOWNLOADING;
      case FINISHED:
        return StatusType.FINISHED;
      default:
        return StatusType.UNKNOWN;
    }
  }


  ///////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // MTODOS PBLICOS
  //
  ///////////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Contrutor padro
   */
  CSBaseJobsServiceAdapter() {
    CommandPersistenceService persistenceService = CommandPersistenceService.getInstance();
    jobStateListener = new JobStateListener(this.getJobDAO());
    localCache = new LocalCache<String, Job>(cacheTime);//5m
    jobStateListener.addJobMonitorLister(new JobMonitorListener() {
      @Override
      public void statusChanged(Job job) {
        localCache.put(job.getJobId(), job);
      }

      @Override
      public void infoChanged(List<Job> jobs) {
        jobs.stream().forEach(job -> localCache.put(job.getJobId(), job));
      }
    });
    persistenceService.addCommandStatusListener(jobStateListener);
    jobInfoMonitor = new JobInfoMonitor(this.getJobDAO());
  }

  /**
   * Atribui o locale a ser usado nas requisies ao servio.
   *
   * @param locale o locale com o idioma
   */
  @Override
  public void setLocale(Locale locale) {
    ClientRemoteLocator.administrationService.setLocale(locale);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobSession createJobSession(String sessionName) {
    if (!isUnique(sessionName)) {
      String message = ServiceUtil.getTranslator(
        getBundle(ClientRemoteLocator.administrationService.getCurrentLocale())).
        message("createJobSession.name.exists.error", sessionName);
      throw new InternalServiceException(message);
    }
    String currentUser = ServiceAdapter.getCurrenUser();
    sessionName = sessionName == null ? generateUniqueSessionName(currentUser) : sessionName;
    JobSession jobSession = new JobSession();
    jobSession.setSessionName(sessionName);
    getJobDAO().insertJobSession(jobSession);
    return jobSession;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobSession getJobSession(String name) {
    return getJobDAO().findJobSessionById(name);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<JobSession> getAllJobSessions() {
    return getJobDAO().findAllJobSessions();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String runJob(String jobSessionId, JobTemplate jobTemplate) {
    if (jobTemplate==null) {
      String message =  ServiceUtil.getTranslator(
        getBundle(ClientRemoteLocator.administrationService.getCurrentLocale())).
        message("runJob.jobtemplate.missing.error");
      throw new InternalServiceException(message);
    }
    JobTemplateValidator validator = new JobTemplateValidator(ServiceAdapter.getCurrenUser(),
      jobTemplate, ClientRemoteLocator.administrationService.getCurrentLocale() );
    AlgorithmConfigurator configurator = validator.getAlgorithmConfigurator();
    CommandSubmission submission = new CommandSubmission(configurator, validator.getProjectId());
    submission.setDescription(jobTemplate.getDescription());
    submission.setPriority(Priority.values()[jobTemplate.getPriority()]);
    List<String> emails = jobTemplate.getEmail();
    if (emails!=null) {
      submission.setEmailList(emails.toArray(new String[0]));
    }
    Boolean emailOnTerminated = jobTemplate.getEmailOnTerminated();
    if (emailOnTerminated!=null){
      submission.setMailAtEnd(emailOnTerminated);
    }
    // Atualmente assume-se que s tenha uma mquina candidata
    List<String> candidateMachines = jobTemplate.getCandidateMachines();
    if (candidateMachines!=null && candidateMachines.size()>0) {
      submission.configureSimpleExecution(candidateMachines.get(0));
    }
    // As propriedades extras so um Json de mapa chave/valor
    Object extraProperties = jobTemplate.getExtraProperties();
    if (extraProperties!=null) {
      @SuppressWarnings("unchecked")
      LinkedHashMap<String, Object> extraPropertiesMap = LinkedHashMap.class.cast(extraProperties);
      extraPropertiesMap.forEach((k, v) -> submission.addExtraInfo(k, (String) v));
    }
    Set<CommandInfo> commandInfos = null;
    SchedulerServiceInterface schedulerService = ClientRemoteLocator.schedulerService;
    try {
      commandInfos = schedulerService.submitCommand(submission);
    } catch (Throwable e) {
      throw new InternalServiceException(e);
    }
    CommandInfo cmdInfo = commandInfos.toArray(new CommandInfo[0])[0];
    return cmdInfo.getId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Job getJob(String jobId) {
    Job job = getJobDAO().findJobById(jobId);
    return job;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Job> getJobs(String projectId, Long modifiedAfter){
    List<Job> allJobs = null;
    if ((projectId==null || projectId.isEmpty()) &&
      (modifiedAfter==null )) {
      return getJobDAO().findAllJobs();
    }

    allJobs = getJobDAO().findJobs(projectId, modifiedAfter);

    return allJobs;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Collection<Job> getLastJobs(Long modifiedAfter, String projectId,
    String jobId) {
    long currentTime = System.currentTimeMillis();
    if (currentTime < modifiedAfter + cacheTime) //significa que as informacoes ainda estao no cache
    {
      return localCache.valuesAfter(modifiedAfter).stream().filter(j -> {
        if (jobId != null) {
          return j.getJobId().equals(jobId);
        }
        if (projectId != null) {
          return j.getProjectId().equals(projectId);
        }
        return true;
      }).collect(Collectors.toList());
    }
    if (jobId != null) {
      LocalDateTime after = LocalDateTime.ofInstant(Instant.ofEpochMilli(
        modifiedAfter), ZoneId.systemDefault());
      Job job = getJobDAO().findJobById(jobId);
      if (job != null && LocalDateTime.parse(job.getLastModifiedTime()).isAfter(
        after)) {
        return Collections.singletonList(job);
      }
      return new ArrayList<Job>();
    }

    return getJobDAO().findJobs(projectId, modifiedAfter);
  }

  /**
   * {@inheritDoc}
   * 
   * @throws IOException
   */
  @Override
  public String getLogDir(String jobId) throws IOException {

    ProjectService projectService = ProjectService.getInstance();

    CommandPersistenceService commandService = CommandPersistenceService
      .getInstance();

    Job job = getJobDAO().findJobById(jobId);
    if (job == null) {
      return null;
    }
    String projectId = ServiceUtil.decodeFromBase64(job.getProjectId());

    String path = ProjectService.getInstance().getProjectRepositoryPath();
    path += File.separator + String.join(File.separator, projectService
      .getProjectPath(projectId));
    path += File.separator + String.join(File.separator, commandService
      .getLogDirectory(jobId));

    return path;

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public StringBuffer getLogFile(String jobId) {
    CommandPersistenceServiceInterface commandService = ClientRemoteLocator.commandPersistenceService;
    Job job = getJobDAO().findJobById(jobId);
    String projectId = ServiceUtil.decodeFromBase64(job.getProjectId());
    try {

      CommandInfo cmdInfo = commandService.getCommandInfo(projectId, job
        .getJobId());
      Set<FileURLValue> files = cmdInfo.getConfigurator()
        .getStandardOutputFiles();
      return logOutputFile(files, projectId);
    } catch (RemoteException e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * 
   * @param stdOutputFiles
   * @param projectId
   * @return log de saida
   */
  private StringBuffer logOutputFile(Set<FileURLValue> stdOutputFiles, Object projectId) {
    ProjectServiceInterface projectService = ClientRemoteLocator.projectService;
    StringWriter sw = new StringWriter();
    PrintWriter printWriter = new PrintWriter(sw);
    for (final FileURLValue stdOutputFile : stdOutputFiles) {
      String[] stdOutputFilePath = stdOutputFile.getPathAsArray();
      InputStream fileFrom = null;
      try {
        if (stdOutputFiles.size() > 1) {
          printWriter.printf("\n---- %s ----\n", stdOutputFile.getPath());
        }
        if (!projectService.existsFile(projectId, stdOutputFilePath)) {
          continue;
        }
        ClientProjectFile file = projectService.getChild(projectId, stdOutputFilePath);
        InputStream inputStream = file.getInputStream();
        BufferedReader bufferedReader =
          new BufferedReader(new InputStreamReader(inputStream));
        String line = bufferedReader.readLine();
        while (line != null) {
          printWriter.println(line);
          line = bufferedReader.readLine();
        }
        printWriter.flush();
      } catch (IOException e) {
        e.printStackTrace();
      } catch (Exception e) {
        e.printStackTrace();
      } finally {
        try {
          if (fileFrom != null) {
            fileFrom.close();
          }
          printWriter.close();
        } catch (IOException e) {
          e.printStackTrace();
        }
      }
    }
    return sw.getBuffer();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addJobMonitorListener(JobMonitorListener listener) {
    jobStateListener.addJobMonitorLister(listener);
    jobInfoMonitor.addJobMonitorListener(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeJobMonitorListener(JobMonitorListener listener) {
    jobStateListener.removeJobMonitorLister(listener);
    jobInfoMonitor.removeJobMonitorListener(listener);

  }
}
