package csbase.server.services.schedulerservice;

import java.io.File;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import tecgraf.javautils.core.io.FileUtils;
import csbase.logic.CommandInfo;
import csbase.logic.SGASet;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.server.Server;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.sgaservice.SGAService;

/**
 * Poltica de escalonamento que privilegia os SGAs com a melhor combinao
 * entre as capacidades de leitura/escrita em disco e transferncia de dados na
 * rede. Os melhores SGAs so definidos com base em uma equao com variveis
 * alimentadas pelos resultados dos benchmarks aplicados aos SGAs.
 * 
 * @author valeria
 * 
 */
public class IOCapacityPolicy implements SchedulerPolicyInterface {
  private SGAService sgaService;

  /**
   * Obtm uma instncia do SGAService.
   */
  public IOCapacityPolicy() {
    this.sgaService = SGAService.getInstance();
  }

  public String chooseServer(CommandInfo command, List<String> serversNames) {
    List<String> sgasNames;
    sgasNames = getSortedSGASetByIOCapacity(command, serversNames);

    if ((sgasNames != null) && (sgasNames.size() != 0)) {
      return sgasNames.get(0);
    }
    return null;
  }

  public List<String> chooseServers(CommandInfo command,
    List<String> serversNames, int numServers) {
    List<String> sgasNames;

    sgasNames = getSortedSGASetByIOCapacity(command, serversNames);

    if (sgasNames != null && sgasNames.size() >= numServers) {
      return sgasNames.subList(0, numServers);
    }
    return null;
  }

  /**
   * Ordena os SGAs de acordo com as suas capacidades de E/S de dados.
   * 
   * @param command Comando para execuo
   * @param serversNames Lista de servidores disponveis
   * 
   * @return Lista de servidores ordenados decrescentemente pela capacidade de
   *         E/S
   */
  private List<String> getSortedSGASetByIOCapacity(CommandInfo command,
    List<String> serversNames) {
    long inputFilesLength;
    Hashtable<String, Long> platfBinFilesLengthList;
    List<SGASet> sgasSets = new ArrayList<SGASet>();
    List<String> sgasNames = new ArrayList<String>();

    AlgorithmConfigurator algorithmConfigurator;
    try {
      algorithmConfigurator = command.getConfigurator();
    }
    catch (RemoteException e) {
      Server.logSevereMessage(
        "No foi possvel obter o configurador do comando " + command.getId(),
        e);
      return null;
    }

    inputFilesLength =
      getInputFilesLength(algorithmConfigurator, command.getId());
    platfBinFilesLengthList =
      getExecFilesLength(algorithmConfigurator, serversNames);

    for (int i = 0; i < serversNames.size(); i++) {
      SGASet sgaSet = sgaService.getSGASet(serversNames.get(i));
      if (sgaSet != null) {
        sgasSets.add(sgaSet);
      }
    }

    Collections.sort(sgasSets, new SGASetIOComparator(inputFilesLength,
      platfBinFilesLengthList));
    for (SGASet sga : sgasSets) {
      sgasNames.add(sga.getName());
    }
    return sgasNames;
  }

  /**
   * Calcula a soma dos tamanhos de todos os arquivos de entrada do algoritmo a
   * ser executado.
   * 
   * @param algorithmConfigurator o configurador do comando.
   * @param commandId o identificador do comando.
   * 
   * @return A soma dos tamanhos de todos os arquivos de entrada do algoritmo
   */
  private long getInputFilesLength(AlgorithmConfigurator algorithmConfigurator,
    String commandId) {
    Set<csbase.logic.algorithms.FileParameterValue> inputFiles =
      algorithmConfigurator.getInputFiles();
    Iterator<csbase.logic.algorithms.FileParameterValue> inputFileIterator =
      inputFiles.iterator();
    long filesLength = 0;
    ProjectService projectService = ProjectService.getInstance();
    for (int i = 0; i < inputFiles.size(); i++) {
      String[] path = FileUtils.splitPath(inputFileIterator.next().getPath());
      if (projectService.existsFile(commandId, path)) {
        filesLength += projectService.fileSize(commandId, path);
      }
    }
    filesLength /= 1024;
    return filesLength;
  }

  /**
   * XXX - [CSBASE-601] Rever o tratamento para o caso de fluxo nas polticas de
   * escalonamento.
   * 
   * Obtm o tamanho dos arquivos executveis do algoritmo para cada plataforma
   * existente entre os servidores disponveis.
   * 
   * @param algoConfigurator o configurador do comando.
   * @param serversNames Servidores disponveis
   * 
   * @return Uma lista com todos os tamanhos dos arquivos binrios de acordo com
   *         cada plataforma de execuo
   */
  private Hashtable<String, Long> getExecFilesLength(
    AlgorithmConfigurator algoConfigurator, List<String> serversNames) {
    long filesLength;
    Hashtable<String, Long> platformsList = new Hashtable<String, Long>();
    AlgorithmVersionInfo algoVersionInfo =
      algoConfigurator.getAlgorithmVersion();

    sgaService = SGAService.getInstance();

    for (int i = 0; i < serversNames.size(); i++) {
      SGASet sgaSet = sgaService.getSGASet(serversNames.get(i));
      if (sgaSet != null) {
        if (!platformsList.containsKey(sgaSet.getPlatformId())) {
          filesLength = 0;
          String binDirPath =
            algoVersionInfo.getDirPath() + File.separator
              + algoVersionInfo.getBinDirName() + File.separator
              + sgaSet.getPlatformId();
          java.io.File binDir = new java.io.File(binDirPath);
          File binFiles[] = binDir.listFiles();
          for (int j = 0; j < binFiles.length; j++) {
            filesLength += binFiles[j].length();
          }
          filesLength /= 1024;
          platformsList.put(sgaSet.getPlatformId(), new Long(filesLength));
        }
      }
    }

    return platformsList;
  }
}
