/*
 * $Id: CommandPersistenceService.java 149543 2014-02-11 13:02:45Z oikawa $
 */
package csbase.server.services.commandpersistenceservice;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;

import tecgraf.javautils.core.io.FileUtils;
import csbase.exception.OperationFailureException;
import csbase.exception.ParseException;
import csbase.exception.ServiceFailureException;
import csbase.exception.algorithms.AlgorithmNotFoundException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.Priority;
import csbase.logic.ProjectFileInfo;
import csbase.logic.ProjectFileType;
import csbase.logic.SimpleCommandFinalizationInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.algorithms.FileParameterValue;
import csbase.logic.algorithms.commands.CommandPersistenceNotification;
import csbase.logic.algorithms.commands.CommandPersistenceNotification.Type;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.serializer.DefaultAlgorithmConfigurationSerializer;
import csbase.logic.algorithms.serializer.FlowAlgorithmConfigurationSerializer;
import csbase.logic.algorithms.serializer.IAlgorithmConfigurationSerializer;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerException;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.CommandPersistenceServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.projectservice.ServerFileLockListener;
import csbase.util.messages.Message;
import csbase.util.messages.MessageBroker;

/**
 * Servio de persistncia de comandos.
 * 
 * @see CommandPersistenceServiceInterface
 * @author TecGraf/PUC-Rio
 */
public final class CommandPersistenceService extends Service implements
  CommandPersistenceServiceInterface {

  /**
   * Codificao utilizada para persistir comando em arquivo.
   */
  private static final String ENCODING = "ISO-8859-1";

  /**
   * Fbrica de threads utilizada pelo mtodo
   * {@link #requestCommandInfos(Object, long)}. Ao ser chamado, este mtodo
   * cria uma thread permitindo retornar imediatamente. Essa thread fica
   * responsvel por carregar os comandos e enviar em blocos ao usurio atravs
   * do servio de mensagens.
   */
  private static ThreadFactory threadFactory = Executors.defaultThreadFactory();

  /**
   * Nome do arquivo de redirecionamento da sada da execuo do comando.
   */
  private static final String OUT_LOG = "out.log";

  /**
   * Nome do arquivo de captura do cdigo de sada do comando.
   */
  private static final String EXIT_CODE_LOG = "exit_code.log";

  /**
   * Nome do arquivo de captura o identificador do n que gerou erro na execuo
   * de um fluxo (se houver).
   */
  private static final String FLOW_GUILTY_NODE_LOG = "flow_guilty_id.log";

  /**
   * Tipo ({@link ProjectFileType}) do arquivo de script.
   */
  private static final String SCRIPT_FILE_TYPE = "SCRIPT";

  /**
   * Tempo mximo de espera pelo lock exclusivo em milesegundos.
   */
  private static final int LOCK_EXCLUSIVE_TIMEOUT = 4000;

  /**
   * Tempo mximo de espera pelo lock compartilhado em milesegundos.
   */
  private static final int LOCK_SHARED_TIMEOUT = 1000;

  /**
   * Nome da propriedade que indica o nome do diretrio que ser usado para
   * armazenar os dados dos comandos executados.
   */
  private static final String COMMANDS_DIRECTORY_NAME_PROPERTY =
    "commandsDirectoryName";

  /**
   * Propriedade que indica o nome do diretrio que ser usado para armazenar os
   * dados dos comandos executados.
   */
  private String commandsDirectoryName;

  /**
   * O nome do diretrio de logs.
   */
  private static final String LOGS_DIRECTORY_NAME = "logs";

  /**
   * O nome do arquivo de propriedades do comando.
   */
  private static final String COMMAND_PROPERTIES_FILE_NAME = "cmd.properties";

  /**
   * O nome do arquivo contendo a configuraco de um algoritmo instalado na
   * rvore de algoritmos.
   */
  private static final String DEFAULT_CONFIGURATION_FILE_NAME =
    "cmd.parameters";

  /**
   * O tipo do arquivo contendo a configuraco de um algoritmo instalado na
   * rvore de algoritmos.
   */
  private static final String DEFAULT_CONFIGURATION_FILE_TYPE = "PARAMETERS";

  /**
   * O nome do arquivo contendo a configuraco de um fluxo.
   */
  private static final String FLOW_CONFIGURATION_FILE_NAME = "cmd.flx";

  /**
   * O tipo do arquivo contendo a configuraco de um fluxo.
   */
  private static final String FLOW_CONFIGURATION_FILE_TYPE = "FLX";

  /**
   * Thread para limpeza dos arquivos de persistncia das execues, de acordo
   * com um tempo mximo de vida. Para mais detalhes, ver documentao da classe
   * {@link CommandPersistenceCleanupThread}.
   */
  private CommandPersistenceCleanupThread cleanupThread = null;

  /**
   * Referncia para o servio de mensagens
   */
  private MessageService messageService;

  /**
   * Constri a instncia do servio de escalonamento.
   * 
   * @throws ServerException se houver erro na inicializao.
   */
  public static void createService() throws ServerException {
    new CommandPersistenceService();
  }

  /**
   * Retorno da instncia do servio.
   * 
   * @return o servio.
   */
  public static CommandPersistenceService getInstance() {
    return (CommandPersistenceService) Service
      .getInstance(CommandPersistenceServiceInterface.SERVICE_NAME);
  }

  /**
   * Fecha um fluxo de dados sem gerar exceo.
   * 
   * @param stream O fluxo de dados. Se for {@code null}, o mtodo no faz nada.
   */
  private void close(Closeable stream) {
    if (stream != null) {
      try {
        stream.close();
      }
      catch (IOException e) {
        Server.logSevereMessage(e.getMessage(), e);
      }
    }
  }

  /**
   * Cria os diretrios e arquivos onde so persistidas as informaes dos
   * comandos.
   * 
   * @param projectId identificador do projeto
   * @param command informaes do comando
   * @param configurationFileName nome do arquivo onde so gravadas as
   *        configuraes do comando
   * @param configurationFileType tipo do arquivo de configurao
   * @throws OperationFailureException em caso de falha.
   */
  private void createProjectFiles(final Object projectId,
    final CommandInfo command, final String configurationFileName,
    final String configurationFileType) throws OperationFailureException {
    AlgorithmConfigurator configurator;
    try {
      configurator = command.getConfigurator();
    }
    catch (final RemoteException e) {
      throw new IllegalArgumentException(String.format(
        "No foi possvel determinar um nome para o arquivo de metadados do "
          + "configurador do comando %s.\n", command.getId()));
    }
    final String commandId = command.getId();
    final String commandDirectoryName = getCommandDirectoryName(commandId);
    final List<ProjectFileInfo> projectFileInfos =
      new LinkedList<ProjectFileInfo>();
    checkCommandRepositoryDirectory(projectId);
    projectFileInfos.add(new ProjectFileInfo(new String[] {
        commandsDirectoryName, commandDirectoryName },
      ProjectFileType.DIRECTORY_TYPE));
    projectFileInfos.add(new ProjectFileInfo(new String[] {
        commandsDirectoryName, commandDirectoryName, configurationFileName },
      configurationFileType));
    projectFileInfos.add(new ProjectFileInfo(new String[] {
        commandsDirectoryName, commandDirectoryName,
        COMMAND_PROPERTIES_FILE_NAME }, "TEXT"));
    projectFileInfos.add(new ProjectFileInfo(new String[] {
        commandsDirectoryName, commandDirectoryName, LOGS_DIRECTORY_NAME },
      ProjectFileType.DIRECTORY_TYPE));
    // Atribui valor ao log de sada padro do algoritmo.
    final String outputFilePath =
      FileUtils.joinPath(File.separatorChar, commandsDirectoryName,
        commandDirectoryName, LOGS_DIRECTORY_NAME,
        CommandPersistenceService.OUT_LOG);
    final FileParameterValue outputFile =
      new FileParameterValue(outputFilePath, "LOG");
    configurator.setStandardOutputFile(outputFile);
    /*
     * Cria os arquivos de log. Pede-se os arquivos a serem criados ao
     * configurador pois, se ele for de fluxo, ele ter criado arquivos para
     * cada um de seus ns a partir do arquivo atribuido acima.
     */
    for (final FileParameterValue stdout : configurator
      .getStandardOutputFiles()) {
      projectFileInfos.add(new ProjectFileInfo(stdout.getPathAsArray(), stdout
        .getType()));
    }
    if (configurator.hasExitCode()) {
      final String exitCodeLogFilePath =
        FileUtils.joinPath(File.separatorChar, commandsDirectoryName,
          commandDirectoryName, LOGS_DIRECTORY_NAME, EXIT_CODE_LOG);
      final FileParameterValue exitCodeLogFile =
        new FileParameterValue(exitCodeLogFilePath, "LOG");
      configurator.setExitCodeLogFile(exitCodeLogFile);

      for (final FileParameterValue exitLog : configurator
        .getExitCodeLogFiles()) {
        projectFileInfos.add(new ProjectFileInfo(exitLog.getPathAsArray(),
          exitLog.getType()));
      }
      if (configurator.getConfiguratorType() == ConfiguratorType.FLOW) {
        FlowAlgorithmConfigurator flowConfigurator =
          (FlowAlgorithmConfigurator) configurator;

        final String guiltyNodeLog =
          FileUtils.joinPath(File.separatorChar, commandsDirectoryName,
            commandDirectoryName,
            CommandPersistenceService.LOGS_DIRECTORY_NAME,
            CommandPersistenceService.FLOW_GUILTY_NODE_LOG);
        final FileParameterValue guiltyNodeLogFile =
          new FileParameterValue(guiltyNodeLog, "LOG");

        flowConfigurator.setGuiltyNodeLog(guiltyNodeLogFile);
        projectFileInfos.add(new ProjectFileInfo(guiltyNodeLogFile
          .getPathAsArray(), guiltyNodeLogFile.getType()));
      }
    }
    ProjectService projectService = ProjectService.getInstance();
    String[] root = new String[] {};
    projectService.createFiles(projectId, root, projectFileInfos);
  }

  /**
   * Verifica se o diretrio de persistncia existe. Caso no exista, cria.
   * 
   * @param projectId Identificador do projeto.
   */
  private void checkCommandRepositoryDirectory(Object projectId) {
    ProjectService projectService = ProjectService.getInstance();
    String[] cmdsDirPath = new String[] { commandsDirectoryName };
    projectService.createDirectory(projectId, cmdsDirPath);
  }

  /**
   * Obtm o diretrio aonde foram/sero salvos os dados do comando.
   * 
   * @param commandId Identificador do comando.
   * @return O diretrio do comando ou {@code null} se o comando no existir.
   */
  private String[] getCommandDirectory(String commandId) {
    return new String[] { commandsDirectoryName,
        getCommandDirectoryName(commandId) };
  }

  /**
   * Obtm o caminho do diretrio onde devem ser gerados os arquivos de log.
   * Este caminho  dado a partir do diretrio do projeto (no includo).
   * 
   * @param commandId Identificador do comando.
   * @return O caminho do diretrio.
   */
  public String[] getLogDirectory(String commandId) {
    final String commandDirectoryName = getCommandDirectoryName(commandId);
    return new String[] { commandsDirectoryName, commandDirectoryName,
        LOGS_DIRECTORY_NAME };
  }

  /**
   * Obtm o nome do diretrio do comando a partir do identificador do comando.
   * 
   * @param commandId Identificador do comando.
   * @return O nome do diretrio.
   */
  private String getCommandDirectoryName(String commandId) {
    return FileUtils.fixDirectoryName(commandId);
  }

  /**
   * Obtm o identificador de um comando a partir de seu diretrio.<br>
   * Ex.:<br>
   * getCommandId("admi_proj_GAASauRbdV") == "admi@proj.GAASauRbdV"<br>
   * getCommandId("admi_FLOW_GAASauRz_0") == "admi@FLOW.GAASauRz-0"
   * 
   * @param commandDirectoryName Nome do diretrio aonde o comando est sendo
   *        persistido.
   * 
   * @return O identificador do comando.
   */
  private String getCommandId(String commandDirectoryName) {
    String id =
      commandDirectoryName.replaceAll("^([^_]+)_([^_]+)_(.*)$", "$1@$2.$3");
    return id.replaceAll("_", "-");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommandInfo getCommandInfo(Object projectId, String commandId) {
    if (projectId == null) {
      throw new IllegalArgumentException("Parmetro projectId nulo.");
    }
    if (commandId == null) {
      throw new IllegalArgumentException("Parmetro commandId nulo.");
    }
    String[] commandDirectory = getCommandDirectory(commandId);
    return readCommandInfo(projectId, commandDirectory);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<CommandInfo> getCommandInfos(Object projectId) {
    if (projectId == null) {
      throw new IllegalArgumentException("Parmetro projectId nulo.");
    }
    Set<CommandInfo> cmdsInfo = new HashSet<CommandInfo>();
    List<String[]> cmdDirectories = getCommandsDirectories(projectId);
    for (String[] cmdDirectory : cmdDirectories) {
      CommandInfo commandInfo = readCommandInfo(projectId, cmdDirectory);
      cmdsInfo.add(commandInfo);
    }
    return Collections.unmodifiableSet(cmdsInfo);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long requestCommandInfos(final Object projectId,
    final long desiredResponseTime) {
    if (projectId == null) {
      throw new IllegalArgumentException("Parmetro projectId nulo.");
    }
    if (1 > desiredResponseTime) {
      throw new IllegalArgumentException(
        "Parmetro avgResponseTime tem que ser maior que 0.");
    }

    final MessageBroker broker = Server.getInstance().getMessageBroker();
    final String caller = getUser().getLogin();
    final List<String[]> cmdDirectories = getCommandsDirectories(projectId);
    final Object userId = Service.getUser().getId();
    CommandPersistenceService.threadFactory.newThread(new Runnable() {
      @Override
      public void run() {
        try {
          Service.setUserId(userId);
          int index = 0;
          long now = System.currentTimeMillis();
          /*
           * Enquanto existir comandos a serem lidos, l e envia para o cliente.
           */
          while (index < cmdDirectories.size()) {
            final long end = now + desiredResponseTime;
            /*
             * L as propriedades dos comandos at que os comandos terminem ou o
             * tempo de resposta seja alcanado. Caso haja erro ao ler algum
             * comando, este ser ignorado.
             */
            final Set<CommandInfo> cmdsInfo = new HashSet<CommandInfo>();
            for (; index < cmdDirectories.size() && now < end; index++) {
              String[] cmdDirectory = cmdDirectories.get(index);
              final CommandInfo commandInfo =
                readCommandInfo(projectId, cmdDirectory);
              cmdsInfo.add(commandInfo);
              now = System.currentTimeMillis();
            }
            /*
             * Envia para o usurio os comandos lidos at o momento para o
             * cliente atravs do servio de mensagens.
             */
            if (cmdsInfo.size() > 0) {
              try {
                long total = cmdDirectories.size();
                long missing = total - index;
                CommandInfosRetrived data =
                  new CommandInfosRetrived(cmdsInfo, missing, total);

                Message message = new Message(data);
                broker.send(message, 0, caller);
              }
              catch (final Exception e) {
                final String message =
                  "Erro ao enviar as informaes de comandos ao usurio.";
                Server.logSevereMessage(message, e);
              }
            }
          }
        }
        finally {
          Service.setUserId(null);
        }
      }
    }).start();

    return cmdDirectories.size();
  }

  /**
   * Obtm os diretrios de comandos do repositrio de comandos do projeto
   * fornecido.
   * 
   * @param projectId Identificador do projeto.
   * @return A lista de diretrios (se no houver comandos, a lista estar
   *         vazia).
   */
  private List<String[]> getCommandsDirectories(Object projectId) {
    ProjectService projectService = ProjectService.getInstance();
    List<String[]> result = new ArrayList<String[]>();
    String[] cmdsDirPath = new String[] { commandsDirectoryName };
    if (!projectService.existsFile(projectId, cmdsDirPath)) {
      return result;
    }
    ClientProjectFile[] cmdsDirs =
      projectService.getChildren(projectId, cmdsDirPath);
    for (ClientProjectFile cmdDir : cmdsDirs) {
      if (cmdDir.isDirectory()) {
        result.add(cmdDir.getPath());
      }
    }
    return result;
  }

  /**
   * Obtm uma mensagem no mecanismo de internacionalizao.
   * 
   * @param keySuffix O sufixo da chave (o nome da classe ser pr-fixado ao
   *        sufixo) da mensagem a ser procurada no servico de linguagem.
   * @param args Possveis argumentos utilizados na formataco da mensagem
   *        utilizando para isso, o mtodo
   *        {@link String#format(String, Object...)}.
   * 
   * @return A mensagem formatada.
   */
  private String getMessage(final String keySuffix, final Object... args) {
    final String className = CommandPersistenceService.class.getSimpleName();
    final String key = className + "." + keySuffix;
    final String format = getString(key);
    return String.format(format, args);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(final Object arg, final Object event) {
    return true;
  }

  /**
   * Inicializao do servio
   * 
   * @throws ServerException Caso ocorra erro na inicializao do servio.
   */
  @Override
  public void initService() throws ServerException {
    createCleanupThread();
    messageService = MessageService.getInstance();
  }

  /**
   * Notifica usurios sobre eventos ocorridos com comandos.
   * 
   * @param allUsers Usurios a serem notificados alm do admin.
   * @param notification Notificao a ser enviada indicando se um comando foi
   *        salvo, removido ou atualizado na persistncia.
   */
  public void notifyUsers(final Collection<Object> allUsers,
    final CommandPersistenceNotification notification) {
    try {
      String[] users =
        ProjectService.getInstance().getUserToNotify(
          notification.getProjectId());
      messageService.send(new Message(notification), users);

    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao notificar usurios sobre o comando "
        + notification.getCommandId() + ".", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AlgorithmConfigurator readAlgorithmConfigurator(Object projectId,
    String commandId) {
    if (projectId == null) {
      throw new IllegalArgumentException("Parmetro projectId nulo.");
    }
    if (commandId == null) {
      throw new IllegalArgumentException("Parmetro commandId nulo.");
    }
    String[] cmdDir = getCommandDirectory(commandId);
    ProjectService projectService = ProjectService.getInstance();
    /*
     * Obtm o desserializador e o arquivo que contm o configurador
     * serializado.
     */
    IAlgorithmConfigurationSerializer algConfSerializer;
    String[] algConfFilePath =
      extendPath(cmdDir, DEFAULT_CONFIGURATION_FILE_NAME);
    if (projectService.existsFile(projectId, algConfFilePath)) {
      algConfSerializer = new DefaultAlgorithmConfigurationSerializer();
    }
    else {
      algConfFilePath = extendPath(cmdDir, FLOW_CONFIGURATION_FILE_NAME);
      if (!projectService.existsFile(projectId, algConfFilePath)) {
        String path = path2Text(cmdDir);
        String err = getMessage("error_configuration_file_not_found", path);
        throw new ServiceFailureException(err);
      }
      algConfSerializer = new FlowAlgorithmConfigurationSerializer();
    }
    Object lockId = null;
    InputStream inputStream = null;
    try {
      ServerFileLockListener lockListener = new ServerFileLockListener();
      lockId =
        projectService.acquireSharedLock(projectId, algConfFilePath,
          lockListener, LOCK_SHARED_TIMEOUT);
      if (lockListener.getLock()) {
        inputStream = projectService.getInputStream(projectId, algConfFilePath);
        AlgorithmConfigurator configurator =
          algConfSerializer.read(inputStream);
        return configurator;
      }
      String message = getMessage("error.get.shared.lock.config", commandId);
      throw new ServiceFailureException(message);
    }
    catch (final AlgorithmConfigurationSerializerException e) {
      String path = path2Text(algConfFilePath);
      String message = getMessage("error_create_configurator", path);
      throw new ServiceFailureException(message, e);
    }
    catch (final AlgorithmNotFoundException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
    finally {
      close(inputStream);
      if (lockId != null) {
        projectService.releaseLock(projectId, algConfFilePath, lockId);
      }
    }
  }

  /**
   * L as informaes do comando de um diretrio do comando fornecido. Apenas
   * as propriedades do comando so carregadas.
   * 
   * @param projectId Identificador do comando.
   * @param commandDirectory O diretrio do comando.
   * @return Um comando com apenas suas propriedades carregadas. Caso no seja
   *         possvel carregar as propriedades, ser retornado um comando com o
   *         status {@link CommandStatus#SYSTEM_FAILURE}. Neste as nicas
   *         propriedades garantidas so o identificador do comando e o
   *         identificador do projeto, todas as demais devem ser
   *         desconsideradas.
   */
  private CommandInfo readCommandInfo(Object projectId,
    String[] commandDirectory) {
    try {
      return readCommandInfoProperties(projectId, commandDirectory);
    }
    catch (Exception e) {
      // Registra o erro...
      String path = path2Text(commandDirectory);
      String message = getMessage("error_read_command_dir", path);
      Server.logSevereMessage(message, e);
      // ... e retorna um comando com o status SYSTEM_FAILURE
      String commandId =
        getCommandId(commandDirectory[commandDirectory.length - 1]);
      return createSystemFailureCommand(projectId, commandId);
    }
  }

  /**
   * <p>
   * Valida o diretrio do comando quanto  existncia do arquivo de
   * propriedades e do arquivo de parmetros e que estes tem contedo.
   * </p>
   * <p>
   * A validao deve ser executada dentro de um lock, para evitar concorrncia
   * de leitura e escrita no diretrio.
   * </p>
   * 
   * @param projectId Identificador do projeto.
   * @param cmdDir O diretrio do comando.
   * @throws OperationFailureException se o diretrio dado no contiver o
   *         arquivo de propriedades ou o arquivo de parmetros do comando, ou
   *         algum deles for vazio.
   */
  private void validateCommandInfoDirectory(Object projectId, String[] cmdDir)
    throws OperationFailureException {
    ProjectService projectService = ProjectService.getInstance();
    // Verifica se o arquivo de propriedades existe e no  vazio.
    String[] propFile = extendPath(cmdDir, COMMAND_PROPERTIES_FILE_NAME);
    if (!projectService.existsFile(projectId, propFile)) {
      final String path = path2Text(cmdDir);
      final String err =
        getMessage("error_command_properties_not_found", path,
          COMMAND_PROPERTIES_FILE_NAME);
      throw new OperationFailureException(err);
    }
    else if (projectService.fileSize(projectId, propFile) == 0) {
      final String path = path2Text(cmdDir);
      final String err =
        getMessage("error_command_properties_is_empty", path,
          COMMAND_PROPERTIES_FILE_NAME);
      throw new OperationFailureException(err);
    }
    // Verifica se o arquivo de parmetros existe e no  vazio.
    String[] paramFile = extendPath(cmdDir, DEFAULT_CONFIGURATION_FILE_NAME);
    if (!projectService.existsFile(projectId, paramFile)) {
      paramFile = extendPath(cmdDir, FLOW_CONFIGURATION_FILE_NAME);
    }
    if (!projectService.existsFile(projectId, paramFile)) {
      final String path = path2Text(cmdDir);
      final String err = getMessage("error_configuration_file_not_found", path);
      throw new OperationFailureException(err);
    }
    else if (projectService.fileSize(projectId, paramFile) == 0) {
      final String path = path2Text(cmdDir);
      final String err = getMessage("error_configuration_file_is_empty", path);
      throw new OperationFailureException(err);
    }
  }

  /**
   * Cria um comando com status {@link CommandStatus#SYSTEM_FAILURE} para ser
   * retornado sempre que no for possvel recuperar as propriedades de um
   * comando. Neste caso, as nicas propriedades garantidas so o identificador
   * do comando e o identificador do projeto, todas as demais devem ser
   * desconsideradas.
   * 
   * @param projectId Identificador do projeto no qual o comando foi executado.
   * @param commandId Identificador do comando.
   * 
   * @return Um comando com status {@link CommandStatus#SYSTEM_FAILURE}.
   */
  private CommandInfo createSystemFailureCommand(Object projectId,
    String commandId) {
    return new CommandInfo(commandId, 0, false, projectId, null, null,
      Priority.getDefault(), CommandStatus.SYSTEM_FAILURE,
      new SimpleCommandFinalizationInfo(CommandFinalizationType.NOT_FINISHED),
      null);
  }

  /**
   * L as informaes do comando de um diretrio do comando fornecido. Apenas
   * as propriedades do comando so carregadas.
   * 
   * @param projectId Identificador do projeto.
   * @param cmdDir Diretrio do comando.
   * @return As informaes.
   * @throws OperationFailureException Se houver um erro ao ler as informaes.
   */
  private CommandInfo readCommandInfoProperties(Object projectId,
    String[] cmdDir) throws OperationFailureException {
    ProjectService projectService = ProjectService.getInstance();
    String[] propFilePath = extendPath(cmdDir, COMMAND_PROPERTIES_FILE_NAME);
    InputStream inputStream = null;
    Object lockId = null;
    try {
      /*
       * No  necessrio validar e existncia do arquivo sendo 'lockado'. O
       * mtodo ProjectService.acquireSharedLock(...) j faz isso, lanando uma
       * exceo quando o arquivo no existe.
       * 
       * Esse  o mesmo tratamento dado a ausncia do arquivo de propriedades,
       * pelo mtodo validateCommandInfoDirectory(...).
       */
      ServerFileLockListener lockListener = new ServerFileLockListener();
      lockId =
        projectService.acquireSharedLock(projectId, propFilePath, lockListener,
          LOCK_SHARED_TIMEOUT);
      if (lockListener.getLock()) {
        /*
         * A validao deve estar protegida pelo lock para evitar ser executada
         * durante a escrita do diretrio.
         */
        validateCommandInfoDirectory(projectId, cmdDir);
        inputStream = projectService.getInputStream(projectId, propFilePath);
        CommandInfo command =
          readCommandInfoPropertiesFromStream(projectId, inputStream);
        command.setPersistencyPath(cmdDir);
        return command;
      }
      String path = path2Text(propFilePath);
      String message = getMessage("error.get.shared.lock", path);
      throw new OperationFailureException(message);
    }
    catch (final IOException e) {
      String path = path2Text(propFilePath);
      String message = getMessage("error_read_file", path);
      throw new OperationFailureException(message, e);
    }
    catch (final ParseException e) {
      String path = path2Text(propFilePath);
      String message = getMessage("error_file_format", path);
      throw new OperationFailureException(message, e);
    }
    finally {
      close(inputStream);
      if (lockId != null) {
        projectService.releaseLock(projectId, propFilePath, lockId);
      }
    }
  }

  /**
   * Cria um comando a partir da leitura de seu arquivo de propriedades.
   * 
   * @param projectId O identificador do projeto (No aceita {@code null}).
   * @param inputStream O stream de entrada com o contedo do arquivo (No
   *        aceita {@code null}).
   * @return commandInfo O comando lido.
   * 
   * @throws ParseException Em caso de erro no formato do arquivo.
   * @throws IOException Se houver um erro de ES ao ler no stream.
   */
  private CommandInfo readCommandInfoPropertiesFromStream(
    final Object projectId, final InputStream inputStream)
    throws ParseException, IOException {
    final CommandPropertyParser parser = new CommandPropertyParser();
    parser.loadProperties(inputStream);

    final String id = parser.getId();
    final String userId = parser.getUserId();
    final Date submittedDate = parser.getSubmittedDate();
    final int globalPosition = parser.getGlobalPosition();
    final boolean mailAtEnd = parser.getMailAtEnd();
    final Priority priority = parser.getPriority();
    final CommandStatus status = parser.getStatus();
    final CommandFinalizationInfo finalizationInfo =
      parser.getFinalizationInfo();
    final String tip = parser.getTip();
    final CommandInfo commandInfo =
      new CommandInfo(id, globalPosition, mailAtEnd, projectId, submittedDate,
        userId, priority, status, finalizationInfo, tip);
    final Integer wallTimeSec = parser.getWallTimeSec();
    if (wallTimeSec != null) {
      commandInfo.setWallTimeSec(wallTimeSec);
    }
    final String description = parser.getDescription();
    if (description != null) {
      commandInfo.setDescription(description);
    }
    final String sgaName = parser.getSGAName();
    if (sgaName != null) {
      commandInfo.setSGAName(sgaName);
    }
    final String platformFilter = parser.getPlatformFilter();
    if (platformFilter != null) {
      commandInfo.setPlatformFilter(platformFilter);
    }

    final boolean isAutomatic = parser.getIsAutomatic();
    final ExecutionType executionType = parser.getExecutionType();
    switch (executionType) {
      case MULTIPLE:
        if (isAutomatic) {
          final int executionCount =
            parser.getExecutionCountForMultipleExecution();
          commandInfo.configureMultipleExecution(executionCount);
        }
        else {
          final int executionCountPerSGA =
            parser.getExecutionCountPerSGAForMultipleExecution();
          final List<String> selectedSGAsNames = parser.getSelectedSGAsNames();
          commandInfo.configureMultipleExecution(selectedSGAsNames,
            executionCountPerSGA);
        }
        break;
      case SIMPLE:
        if (isAutomatic) {
          commandInfo.configureSimpleExecution();
        }
        else {
          final String selectedSgaName = parser.getSelectedSGAName();
          commandInfo.configureSimpleExecution(selectedSgaName);
        }
        break;
      default:
        final String fmt =
          "Tipo de execuo invlido no arquivo de persistncia %s.";
        final String err = String.format(fmt, executionType);
        throw new IllegalStateException(err);
    }

    return commandInfo;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeCommandInfo(Object projectId, String commandId) {
    ProjectService projectService = ProjectService.getInstance();
    String[] cmdDirectory = getCommandDirectory(commandId);
    if (!projectService.existsFile(projectId, cmdDirectory)) {
      return false;
    }
    projectService.removeFile(projectId, cmdDirectory);
    Collection<Object> allUsers = projectService.getAllUsers(projectId);
    allUsers.add(projectService.getOwnerId(projectId));
    notifyUsers(allUsers, new CommandPersistenceNotification(getSenderName(),
      projectId, commandId, Type.REMOVED));
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean[] removeCommandInfos(final List<Object> projectIds,
    final List<String> commandIds) {
    final boolean[] result = new boolean[commandIds.size()];
    Arrays.fill(result, false);
    for (int inx = 0; inx < commandIds.size(); inx++) {
      result[inx] = removeCommandInfo(projectIds.get(inx), commandIds.get(inx));
    }
    return result;
  }

  /**
   * Salva as propriedades de um comando e os seus parmetros
   * 
   * @param command Informaes do comando.
   * @param configurationFileName Nome do arquivo de parmetros do comando.
   * @param configurationSerializer Responsvel por persistir as configuraes
   *        do comando.
   * @throws OperationFailureException em caso de falha.
   */
  private void saveAlgorithmConfiguration(CommandInfo command,
    String configurationFileName,
    IAlgorithmConfigurationSerializer configurationSerializer)
    throws OperationFailureException {
    ProjectService projectService = ProjectService.getInstance();
    Object projectId = command.getProjectId();
    String commandId = command.getId();
    String[] cmdDirectory = getCommandDirectory(commandId);
    if (!projectService.existsFile(projectId, cmdDirectory)) {
      String tag = "error_command_directory_not_found";
      String message = getMessage(tag, commandId);
      throw new OperationFailureException(message);
    }
    String[] configFilePath = extendPath(cmdDirectory, configurationFileName);
    if (!projectService.existsFile(projectId, configFilePath)) {
      String tag = "error_file_not_found";
      String path = path2Text(cmdDirectory);
      String message = getMessage(tag, path, configurationFileName);
      throw new OperationFailureException(message);
    }
    OutputStream output = null;
    Object lockId = null;
    try {
      ServerFileLockListener lockListener = new ServerFileLockListener();
      lockId =
        projectService.acquireExclusiveLock(projectId, configFilePath,
          lockListener, LOCK_EXCLUSIVE_TIMEOUT);
      if (lockListener.getLock()) {
        output = projectService.getOutputStream(projectId, configFilePath);
        AlgorithmConfigurator configurator = command.getConfigurator();
        configurationSerializer.write(configurator, output);
      }
      else {
        String path = path2Text(configFilePath);
        String message = getMessage("error.get.exclusive.lock", path);
        throw new OperationFailureException(message);
      }
    }
    catch (final AlgorithmConfigurationSerializerException e) {
      final String fmt = getMessage("error.parse_configuration_file");
      final String err = String.format(fmt, commandId, configurationFileName);
      throw new OperationFailureException(err);
    }
    catch (final RemoteException e) {
      final String fmt = getMessage("error.no_configurator");
      final String err = String.format(fmt, commandId, configurationFileName);
      throw new OperationFailureException(err);
    }
    finally {
      close(output);
      if (lockId != null) {
        projectService.releaseLock(projectId, configFilePath, lockId);
      }
    }
  }

  /**
   * Salva o comando informado.
   * 
   * Caso o comando tenha sido salvo com sucesso, gera uma notificao
   * {@link CommandPersistenceNotification} do tipo {@link Type#SAVED}.
   * 
   * @param command O comando a ser salvo.
   * @throws OperationFailureException Se houver uma falha ao salvar o comando.
   */
  public void saveCommandInfo(CommandInfo command)
    throws OperationFailureException {
    if (command == null) {
      throw new IllegalArgumentException("Parmetro command nulo.");
    }
    AlgorithmConfigurator configurator;
    String cmdId = command.getId();
    try {
      configurator = command.getConfigurator();
    }
    catch (final RemoteException e) {
      final String fmt =
        "No foi possvel obter o configurador do comando %s.\n";
      throw new IllegalArgumentException(String.format(fmt, cmdId));
    }
    String configurationFileName;
    String configurationFileType;
    IAlgorithmConfigurationSerializer configurationSerializer;
    String errorMsg =
      String
        .format(
          "No foi possvel determinar o arquivo de metadados do configurador do comando %s.\n",
          cmdId);
    switch (configurator.getConfiguratorType()) {
      case SIMPLE:
        if ((configurator.getAlgorithmName() != null)
          && (configurator.getAlgorithmVersionId() != null)) {
          configurationFileName = DEFAULT_CONFIGURATION_FILE_NAME;
          configurationFileType = DEFAULT_CONFIGURATION_FILE_TYPE;
          configurationSerializer =
            new DefaultAlgorithmConfigurationSerializer();
        }
        else {
          throw new IllegalArgumentException(errorMsg);
        }
        break;
      case FLOW: {
        configurationFileName = FLOW_CONFIGURATION_FILE_NAME;
        configurationFileType = FLOW_CONFIGURATION_FILE_TYPE;
        configurationSerializer = new FlowAlgorithmConfigurationSerializer();
      }
        break;
      default: {
        throw new IllegalArgumentException(errorMsg);
      }
    }
    Object projectId = command.getProjectId();
    try {
      createProjectFiles(projectId, command, configurationFileName,
        configurationFileType);
    }
    catch (OperationFailureException ofe) {
      // Tratamento para que saia no log do command-persistence-service em
      // correlao ao que aparece na inexistncia do arquivo (peridico).
      String fmt = "Erro ao criar arquivos de persistncia do comando [%s].\n";
      Server.logSevereMessage(String.format(fmt, cmdId), ofe);
      throw ofe;
    }
    try {
      saveCommandInfoProperties(command);
      saveAlgorithmConfiguration(command, configurationFileName,
        configurationSerializer);
    }
    catch (OperationFailureException ofe) {
      try {
        removeCommandInfo(projectId, cmdId);
      }
      catch (ServiceFailureException sfe) {
        String fmt =
          "Erro ao remover arquivos de persistncia do comando %s.\n";
        Server.logSevereMessage(String.format(fmt, cmdId), sfe);
      }
      throw ofe;
    }
    // Notifica a todos os interessados que o comando foi salvo com sucesso.
    ProjectService projectService = ProjectService.getInstance();
    Collection<Object> allUsers = projectService.getAllUsers(projectId);
    allUsers.add(projectService.getOwnerId(projectId));
    CommandPersistenceNotification notif =
      new CommandPersistenceNotification(getSenderName(), projectId, cmdId,
        Type.SAVED);
    notifyUsers(allUsers, notif);
  }

  /**
   * Salva o arquivo {@value #COMMAND_PROPERTIES_FILE_NAME} de um comando
   * fornecido.
   * 
   * @param command O comando.
   * @throws OperationFailureException Se houver algum erro ao escrever no
   *         arquivo.
   */
  private void saveCommandInfoProperties(CommandInfo command)
    throws OperationFailureException {
    ProjectService projectService = ProjectService.getInstance();
    Object projectId = command.getProjectId();
    String commandId = command.getId();
    String[] cmdDir = getCommandDirectory(commandId);
    if (!projectService.existsFile(projectId, cmdDir)) {
      String key = "error_command_directory_not_found";
      String message = getMessage(key, commandId);
      throw new ServiceFailureException(message);
    }
    String[] propFilePath = extendPath(cmdDir, COMMAND_PROPERTIES_FILE_NAME);
    if (!projectService.existsFile(projectId, propFilePath)) {
      String path = path2Text(cmdDir);
      String message =
        getMessage("error_file_not_found", path, COMMAND_PROPERTIES_FILE_NAME);
      throw new OperationFailureException(message);
    }
    command.setPersistencyPath(cmdDir);
    Object lockId = null;
    try {
      ServerFileLockListener lockListener = new ServerFileLockListener();
      lockId =
        projectService.acquireExclusiveLock(projectId, propFilePath,
          lockListener, LOCK_EXCLUSIVE_TIMEOUT);
      if (lockListener.getLock()) {
        saveCommandInfoPropertiesToStream(propFilePath, command);
      }
      else {
        String path = path2Text(propFilePath);
        String message = getMessage("error.get.exclusive.lock", path);
        throw new OperationFailureException(message);
      }
    }
    catch (final IOException e) {
      String path = path2Text(propFilePath);
      String message =
        getMessage("error_write_command_properties", commandId, path);
      throw new OperationFailureException(message, e);
    }
    finally {
      if (lockId != null) {
        projectService.releaseLock(projectId, propFilePath, lockId);
      }
    }
  }

  /**
   * Persiste as propriedades do comando.
   * 
   * @param propFilePath arquivo de sada
   * @param command informaes do comando
   * @throws IOException caso ocorra falha na persistncia do comando
   * @throws OperationFailureException em caso de falha na escrita do
   *         server-project-file associado.
   */
  private void saveCommandInfoPropertiesToStream(String[] propFilePath,
    CommandInfo command) throws IOException, OperationFailureException {
    CommandPropertyParser parser = new CommandPropertyParser();
    String commandId = command.getId();
    parser.setId(commandId);
    parser.setUserId(command.getUserId());
    parser.setSubmittedDate(command.getSubmittedDate());
    parser.setWallTimeSec(command.getWallTimeSec());
    parser.setGlobalPosition(command.getGlobalPosition());
    parser.setMailAtEnd(command.isMailAtEnd());
    parser.setPriority(command.getPriority());
    parser.setStatus(command.getStatus());
    parser.setFinalizationInfo(command.getFinalizationInfo());
    parser.setDescription(command.getDescription());
    parser.setSGAName(command.getSGAName());
    parser.setPlatformFilter(command.getPlatformFilter());
    parser.setTip(command.getTip());
    boolean isAutomatic = command.isAutomatic();
    parser.setIsAutomatic(isAutomatic);
    ExecutionType executionType = command.getExecutionType();
    parser.setExecutionType(executionType);
    switch (executionType) {
      case MULTIPLE:
        if (isAutomatic) {
          parser.setExecutionCountForMultipleExecution(command
            .getExecutionCountForMultipleExecution());
        }
        else {
          parser.setExecutionCountPerSGAForMultipleExecution(command
            .getExecutionCountPerSGAForMultipleExecution());
          parser.setSelectedSGAsNames(command.getSelectedSGAsNames());
        }
        break;
      case SIMPLE:
        if (!isAutomatic) {
          parser.setSelectedSGAsNames(command.getSelectedSGAsNames());
        }
        break;
      default:
        throw new IllegalStateException(String.format(
          "Tipo de execuo invlido no arquivo de persistncia %s.",
          executionType));
    }
    OutputStream outputStream = null;
    try {
      ProjectService projectService = ProjectService.getInstance();
      Object projectId = command.getProjectId();
      outputStream = projectService.getOutputStream(projectId, propFilePath);
      parser.saveProperties(outputStream);
    }
    finally {
      if (outputStream != null) {
        outputStream.close();
      }
    }
  }

  /**
   * Mtodo invocado no trmino do servio.
   */
  @Override
  public void shutdownService() {
    if (cleanupThread != null) {
      cleanupThread.shutdown();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateCommandDescription(Object projectId, String commandId,
    String description) {
    if (projectId == null) {
      throw new IllegalArgumentException("Parmetro projectId nulo.");
    }
    if (commandId == null) {
      throw new IllegalArgumentException("Parmetro commandId nulo.");
    }
    CommandInfo command = getCommandInfo(projectId, commandId);
    command.setDescription(description);
    updateCommandInfo(command);

    ProjectService projectService = ProjectService.getInstance();
    Collection<Object> allUsers = projectService.getAllUsers(projectId);
    allUsers.add(projectService.getOwnerId(projectId));
    notifyUsers(allUsers, new CommandPersistenceNotification(getSenderName(),
      projectId, commandId, Type.UPDATED));
  }

  /**
   * Atualiza as informaes do comando.
   * 
   * @param command comando a ser atualizado.
   */
  public void updateCommandInfo(CommandInfo command) {
    if (command == null) {
      throw new IllegalArgumentException("Parmetro command nulo.");
    }
    String commandId = command.getId();
    try {
      saveCommandInfoProperties(command);
    }
    catch (final OperationFailureException e) {
      String message = getMessage("error_save_command_info", commandId);
      throw new ServiceFailureException(message, e);
    }
  }

  /**
   * Construtor da classe.
   * 
   * @throws ServerException se houver erro na inicializao.
   */
  private CommandPersistenceService() throws ServerException {
    super(CommandPersistenceServiceInterface.SERVICE_NAME);
    // XXX - Temporrio. Aguardando a soluo do Bug #5847.
    ClientRemoteLocator.commandPersistenceService = this;
    commandsDirectoryName = getStringProperty(COMMANDS_DIRECTORY_NAME_PROPERTY);
    /*
     * Remove caracteres invlidos do nome do projeto. - Remove os espaos antes
     * e depois do nome. - Caractres invlidos so substituidos por '_'. -
     * Somente ser mantido o '.' se este for a primeira letra do nome.
     */
    commandsDirectoryName = commandsDirectoryName.trim();
    if (commandsDirectoryName.startsWith(".")) {
      final String substring = commandsDirectoryName.substring(1);
      commandsDirectoryName = '.' + FileUtils.fixDirectoryName(substring);
    }
    else {
      commandsDirectoryName = FileUtils.fixDirectoryName(commandsDirectoryName);
    }
  }

  /**
   * Obtm o conjunto de scripts de um comando especfico.
   * 
   * @param commandId Identificador do comando.
   * @param projectId Identificador do projeto.
   * @return Os caminhos para cada script do comando.
   * @throws OperationFailureException se no for possvel encontrar o diretrio
   *         onde se encontram os scripts.
   */
  public String[][] getCommandScripts(String commandId, Object projectId)
    throws OperationFailureException {
    if (projectId == null) {
      throw new IllegalArgumentException("Parmetro projectId nulo.");
    }
    if (commandId == null) {
      throw new IllegalArgumentException("Parmetro commandId nulo.");
    }
    ProjectService projectService = ProjectService.getInstance();
    String[] cmdDirPath = getCommandDirectory(commandId);
    if (!projectService.existsFile(projectId, cmdDirPath)) {
      throw new OperationFailureException(getMessage(
        "error_command_directory_not_found", commandId));
    }
    ClientProjectFile[] scripts =
      projectService.getChildren(projectId, cmdDirPath);
    String[][] paths = new String[scripts.length][];
    for (int i = 0; i < scripts.length; i++) {
      paths[i] = scripts[i].getPath();
    }
    return paths;
  }

  /**
   * Persiste a linha de comando em um arquivo de script.
   * 
   * @param fileName o nome do arquivo de script.
   * @param commandId o identificador do comando.
   * @param projectId o identificador do projeto.
   * @param commandLine a linha de comando.
   * @throws OperationFailureException Se a linha de comando no puder ser
   *         persistida.
   */
  public void saveCommandScript(String fileName, String commandId,
    Object projectId, String commandLine) throws OperationFailureException {
    if (fileName == null) {
      throw new IllegalArgumentException("Parmetro fileName nulo.");
    }
    if (commandId == null) {
      throw new IllegalArgumentException("Parmetro commandId nulo.");
    }
    if (projectId == null) {
      throw new IllegalArgumentException("Parmetro projectId nulo.");
    }
    if (commandLine == null) {
      throw new IllegalArgumentException("Parmetro commandLine nulo.");
    }
    ProjectService projectService = ProjectService.getInstance();
    String[] cmdDirPath = getCommandDirectory(commandId);
    if (!projectService.existsFile(projectId, cmdDirPath)) {
      String message =
        getMessage("error_command_directory_not_found", commandId);
      throw new OperationFailureException(message);
    }
    projectService
      .createFile(projectId, cmdDirPath, fileName, SCRIPT_FILE_TYPE);
    String[] filePath =
      projectService.getChild(projectId, cmdDirPath, fileName).getPath();
    if (!projectService.existsFile(projectId, filePath)) {
      String path = path2Text(cmdDirPath);
      String message = getMessage("error_file_not_found", path, fileName);
      throw new OperationFailureException(message);
    }
    OutputStream output = null;
    Object lockId = null;
    try {
      ServerFileLockListener lockListener = new ServerFileLockListener();
      lockId =
        projectService.acquireExclusiveLock(projectId, filePath, lockListener,
          LOCK_EXCLUSIVE_TIMEOUT);
      if (lockListener.getLock()) {
        output = projectService.getOutputStream(projectId, filePath);
        output.write(commandLine.getBytes(ENCODING));
      }
      else {
        String path = path2Text(filePath);
        String message = getMessage("error.get.exclusive.lock", path);
        throw new ServiceFailureException(message);
      }
    }
    catch (IOException e) {
      String path = path2Text(filePath);
      String message = getMessage("error_write_script_file", path, commandId);
      throw new OperationFailureException(message, e);
    }
    finally {
      close(output);
      if (lockId != null) {
        projectService.releaseLock(projectId, filePath, lockId);
      }
    }
  }

  /**
   * O mtodo createUniqueCommandDirectory deve garantir que o identificador do
   * comando seja nico. Estamos considerando o caso em que 2 ou mais servidores
   * podem escrever na mesma rea de projetos. Por isso, o mtodo verifica a
   * existncia do diretrio que representa o comando. Caso ele exista, o mtodo
   * lanar uma exceo, indicando que j existe outro comando com este
   * identificador. Caso o diretrio no exista, ele  criado e o identificador
   *  considerado nico.  importante que a criao do diretrio seja uma
   * operao atmica, de modo a evitar possveis erros de concorrncia.
   * 
   * Preferimos lanar a exceo ao invs de retornar um boolean, pois no
   * podemos garantir que o diretrio no tenha sido criado por causa da
   * existncia de um diretrio com mesmo nome. Desta forma, a exceo conter a
   * razo do erro que poder ser logado no servidor.
   * 
   * @param projectId Indentificador do projeto
   * @param commandId Indentificador de um comando
   */
  public void createUniqueCommandDirectory(Object projectId, String commandId) {
    checkCommandRepositoryDirectory(projectId);
    ProjectService projectService = ProjectService.getInstance();
    // No adianta verificar a priori se o diretrio existe. 
    // ATENO:
    // **No** usar o mtodo:
    // projectService.createDirectory(projectId, getCommandDirectory(commandId));
    // pois ele no lanar exceo caso o diretrio j exista!
    projectService.createDirectory(projectId,
      new String[] { getCommandsDirectoryName() },
      getCommandDirectoryName(commandId));
  }

  /**
   * Obtm o nome do diretrio de persistncia dos arquivos associados s
   * execues dos comandos (default = <code>.cmds</code>).
   * 
   * @return nome do diretrio de persistncia dos arquivos associados s
   *         execues dos comandos
   */
  public String getCommandsDirectoryName() {
    return commandsDirectoryName;
  }

  /**
   * Cria e inicia a thread para limpeza dos arquivos de persistncia.
   * 
   * @throws ServerException em caso de erro (propriedades obrigatrias
   *         indefinidas, valores incorretos)
   */
  private void createCleanupThread() throws ServerException {
    cleanupThread = new CommandPersistenceCleanupThread();
    cleanupThread.start();
  }

  /**
   * Transforma um caminho no formato String[] em um texto.
   * 
   * @param path O caminho a ser transformado em texto.
   * @return Uma verso textual do caminho fornecido.
   */
  private String path2Text(String[] path) {
    StringBuffer text = new StringBuffer("");
    for (String part : path) {
      text.append(part).append("/");
    }
    return text.substring(0, text.length() - 1);
  }

  /**
   * Estende um caminho no formato String[] com o filho especificado.
   * 
   * @param path O caminho a ser estendido.
   * @param part A parte adicional a ser includa ao trmino do caminho.
   * @return Um novo caminho, no formato String[], que estende o original com a
   *         parte fornecida.
   */
  private String[] extendPath(String[] path, String part) {
    String[] newPath = new String[path.length + 1];
    System.arraycopy(path, 0, newPath, 0, path.length);
    newPath[path.length] = part;
    return newPath;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Service[] getInitializationDependencies() {
    return new Service[] { ProjectService.getInstance() };
  }
}
