package csbase.server.services.diagnosticservice.monitors.openbus;

import java.util.Locale;

import org.omg.CORBA.OBJECT_NOT_EXIST;
import org.omg.CORBA.ORB;

import scs.core.IComponentHelper;
import csbase.logic.diagnosticservice.StatusCode;
import csbase.logic.diagnosticservice.ValidationStatus;
import csbase.server.services.diagnosticservice.MessageTranslator;
import csbase.server.services.diagnosticservice.monitors.Validation;
import csbase.server.services.openbusservice.OpenBusService;

/**
 * Validao da verso do barramento.
 */
public class BusVersionValidation implements Validation {
  /** Nome que identifica a validao */
  public static final String NAME = "BusVersionValidation";
  /**
   * O host do barramento
   */
  private String host;
  /**
   * A porta do barramento
   */
  private int port;

  /**
   * Construtor
   * 
   * @param host a mquina onde o barramento est escutando
   * @param port a porta onde o barramento est escutando
   */
  public BusVersionValidation(String host, int port) {
    this.host = host;
    this.port = port;
  }

  /**
   * {@inheritDoc}
   * 
   * Verifica a verso do barramento.
   */
  @Override
  public ValidationStatus check(Locale locale) {
    OpenBusService service = OpenBusService.getInstance();
    ORB orb = service.getORB();
    if (orb == null) {
      return new ValidationStatus(StatusCode.ERROR, MessageTranslator
        .getString("server.openbusmonitor.version.error", locale));
    }

    String str =
      String.format("corbaloc::1.0@%s:%d/%s", host, port, "OpenBus_2_0");
    org.omg.CORBA.Object obj = orb.string_to_object(str);
    try {
      if (obj == null || obj._non_existent()) {
        return new ValidationStatus(StatusCode.OK, MessageTranslator.getString(
          "server.openbusmonitor.version.1.5.check", locale));
      }
    }
    catch (OBJECT_NOT_EXIST e) {
      return new ValidationStatus(StatusCode.OK, MessageTranslator.getString(
        "server.openbusmonitor.version.1.5.check", locale));
    }
    if (obj._is_a(IComponentHelper.id())) {
      return new ValidationStatus(StatusCode.OK, MessageTranslator.getString(
        "server.openbusmonitor.version.2.0.check", locale));
    }
    return new ValidationStatus(StatusCode.OK, MessageTranslator.getString(
      "server.openbusmonitor.version.1.5.check", locale));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String id() {
    return NAME;
  }

  /**
   * {@inheritDoc}
   * 
   * Depende do resultado da validao de {@link RegistryServiceValidation}.
   */
  @Override
  public String[] requires() {
    return new String[] { RegistryServiceValidation.NAME };
  }
}