/*
 * $Id: ServerEntryPointImpl.java 167902 2015-09-02 00:19:31Z isabella $
 */
package csbase.server;

import java.io.Serializable;
import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.Remote;
import java.security.PublicKey;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;

import csbase.logic.EncryptedPassword;
import csbase.logic.PreLoginData;
import csbase.logic.ServerURI;
import csbase.logic.Session;
import csbase.logic.UserOutline;
import csbase.logic.openbus.OpenBusLoginToken;
import csbase.logic.url.URLParameters;
import csbase.remote.HttpServiceInterface;
import csbase.remote.ServerEntryPoint;
import csbase.remote.ServiceInterface;
import csbase.server.keystore.CSKeyStore;
import csbase.server.services.httpservice.HttpService;
import csbase.server.services.loginservice.LoginService;
import csbase.util.rmi.Pingable;

/**
 * Implementa a porta de entrada para acesso as funcionalidade do servidor.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ServerEntryPointImpl extends Pingable implements ServerEntryPoint {
  /**
   * Nome do parmetro que indica a porta RMI onde o servidor est rodando.
   */
  private static final String SERVER_PORT_PARAM = "server_port";

  /**
   * Nome do parmetro que indica a pgina principal do sistema que ser
   * inserida na URL do sistema
   */
  private static final String PAGE_PARAM = "page";

  /**
   * Sufixo que ser adicionado ao nome do sistema no parmetro
   * {@link #PAGE_PARAM}
   */
  private static final String PAGE_SUFFIX = ".jnlp";

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canChangePasswords() {
    final LoginService loginService = LoginService.getInstance();
    return loginService.canChangePasswords();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public HttpServiceInterface fetchHttpService() {
    final HttpService httpService = HttpService.getInstance();
    return httpService;
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  public <T extends ServiceInterface> T fetchService(final Object sessionKey,
    final String serviceName) {
    final ServiceManager serviceManager = ServiceManager.getInstance();
    return (T) serviceManager.getRemoteService(sessionKey, serviceName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, ServiceInterface> fetchServices(final Object sessionKey,
    final Set<String> servicesNames) {
    final ServiceManager serviceManager = ServiceManager.getInstance();
    return serviceManager.getRemoteServices(sessionKey, servicesNames);
  }

  /**
   * Gera a URL de acesso ao sistema.
   * 
   * @param locale A Locale que o usurio escolheu no login.
   * @param token Identificao de um preLogin.
   * 
   * @return A URL de acesso ao sistema.
   */
  private String generateSystemURL(final String token, final Locale locale) {
    return this.generateSystemURL(token, locale, null);
  }

  /**
   * Gera a URL de acesso ao sistema.
   * 
   * @param locale A Locale que o usurio escolheu no login.
   * @param token Identificao de um preLogin.
   * @param systemName o nome do sistema que se deseja obter a URL. Este nome 
   *        usado para definir o parmetro <code>page</code> da URL, que ser
   *        acrescido da extenso do tipo de aplicativo web. (i.e.: ".jnlp")
   * 
   * @return A URL de acesso ao sistema.
   */
  private String generateSystemURL(final String token, final Locale locale,
    final String systemName) {
    final String LOCALE_PARAM = URLParameters.LOCALE_PARAMETER;
    final String TOKEN_PARAM = URLParameters.PRE_LOGIN_TOKEN_PARAMETER;

    String url = Server.getInstance().getSystemURL();
    final String port =
      Integer.toString(Server.getInstance().getRegistryPort());
    url =
      MessageFormat.format("{0}?{1}={2}", new Object[] { url,
          SERVER_PORT_PARAM, port });
    url =
      MessageFormat.format("{0}&{1}={2}", new Object[] { url, TOKEN_PARAM,
          token });
    url =
      MessageFormat.format("{0}&{1}={2}", new Object[] { url, LOCALE_PARAM,
          locale });
    if (systemName != null) {
      url =
        MessageFormat.format("{0}&{1}={2}{3}", new Object[] { url, PAGE_PARAM,
            systemName, PAGE_SUFFIX });
    }
    return url;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public UserOutline[] getLoggedUsers() {
    final LoginService loginService = LoginService.getInstance();
    return loginService.getLoggedUsers();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getSystemName() {
    final Server server = Server.getInstance();
    final String serverName = server.getSystemName();
    return serverName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getSystemURLWithRMIPort() {
    final Server server = Server.getInstance();
    String url = server.getSystemURL();
    final int port = server.getRegistryPort();
    final String portTxt = Integer.toString(port);
    final Object[] args = new Object[] { url, SERVER_PORT_PARAM, portTxt };
    url = MessageFormat.format("{0}?{1}={2}", args);
    return url;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getSystemURLWithRMIPort(final String systemName) {
    if (systemName == null) {
      throw new IllegalArgumentException("systemName = null");
    }
    if (systemName.trim().equals("")) {
      throw new IllegalArgumentException("systemName vazio");
    }
    String url = this.getSystemURLWithRMIPort();
    url =
      MessageFormat.format("{0}&{1}={2}{3}", new Object[] { url, PAGE_PARAM,
          systemName, PAGE_SUFFIX });
    return url;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getVersionName() {
    return Server.getInstance().getVersion();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isLocalLogin() {
    final LoginService loginService = LoginService.getInstance();
    return loginService.isLocalLogin();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Serializable> isValidSession(final Object sessionKey,
    final String serverName, final byte[] signedServerName) {

    if (sessionKey == null || serverName == null || signedServerName == null) {
      throw new IllegalArgumentException(
        "sessionKey == null || serverName == null || signedServerName == null");
    }

    try {
      final CSKeyStore keyStore = CSKeyStore.getInstance();
      if (keyStore == null) {
        final String msg =
          "O repositrio de chaves/certificados no existe. Sesso no validada.";
        Server.logWarningMessage(msg);
        return null;
      }

      if (!keyStore.containsAlias(serverName)) {
        final String fmt =
          "O servidor %s no possui certificado cadastrado. Sesso no validada.";
        final String message = String.format(fmt, serverName);
        Server.logWarningMessage(message);
        return null;
      }

      if (!keyStore.verify(serverName, serverName, signedServerName)) {
        final String fmt =
          "A assinatura do servidor %s est invlida. Sesso no validada.";
        final String message = String.format(fmt, serverName);
        Server.logWarningMessage(message);
        return null;
      }
    }
    catch (final Exception e) {
      Server.logSevereMessage("Erro validando sesso .");
      return null;
    }

    final LoginService loginService = LoginService.getInstance();
    if (loginService.isValidSession(sessionKey)) {
      return loginService.getSessionAttributes(sessionKey);
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Session login(final ServerURI referedServerURI,
    final Map<String, Serializable> attr, final boolean copyServerSessionAttrs,
    final Object sessionkey, final String login, final String delegatedLogin,
    final Locale locale, final TimeZone tz) {
    final LoginService loginService = LoginService.getInstance();
    return loginService.login(referedServerURI, attr, copyServerSessionAttrs,
      sessionkey, login, delegatedLogin, locale, tz);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Session login(final String token) {
    final LoginService loginService = LoginService.getInstance();
    return loginService.login(token);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Session login(final String localServerName,
    final byte[] signedLocalServerName, final Locale locale) {
    final LoginService loginService = LoginService.getInstance();
    return loginService.login(localServerName, signedLocalServerName, locale);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Session login(final String login, final EncryptedPassword encryptedPassword,
    final Locale locale) {
    final LoginService loginService = LoginService.getInstance();
    return loginService.login(login, encryptedPassword, locale);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Session login(final String login, final EncryptedPassword encryptedPassword,
    final Locale locale, final String delegatedLogin) {
    final LoginService loginService = LoginService.getInstance();
    return loginService.login(login, encryptedPassword, locale, delegatedLogin);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Session login(final String login, final EncryptedPassword encryptedPassword,
    final Locale locale, final TimeZone timeZone) {
    final LoginService loginService = LoginService.getInstance();
    return loginService.login(login, encryptedPassword, locale, timeZone);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Session login(final String login, final EncryptedPassword encryptedPassword,
    final Locale locale, final TimeZone tz, final String delegatedLogin) {
    final LoginService loginService = LoginService.getInstance();
    return loginService
      .login(login, encryptedPassword, locale, delegatedLogin, tz, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Session login(final String login, final EncryptedPassword encryptedPassword,
    final Locale locale, final TimeZone tz, final String delegatedLogin,
    final Map<String, Serializable> params) {
    final LoginService loginService = LoginService.getInstance();
    return loginService.login(login, encryptedPassword, locale, delegatedLogin, tz,
      params);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Session login(final String token, final TimeZone timeZone) {
    final LoginService loginService = LoginService.getInstance();
    return loginService.login(token, timeZone);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Session login(OpenBusLoginToken token, Locale locale, TimeZone timeZone) {
    final LoginService loginService = LoginService.getInstance();
    return loginService.login(token, locale, timeZone);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void logout(final Object sessionKey) {
    final LoginService loginService = LoginService.getInstance();
    loginService.logout(sessionKey);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PreLoginData preLogin(final String login, final String password,
    final Locale locale) {
    final LoginService loginService = LoginService.getInstance();
    final PreLoginData data = loginService.preLogin(login, password, locale);
    if (data == null) {
      return null;
    }
    final String token = data.getToken();
    try {
      final String urlText = generateSystemURL(token, locale);
      final URL url = new URL(urlText);
      data.setUrl2Login(url);
    }
    catch (final MalformedURLException e) {
      e.printStackTrace();
      return null;
    }
    return data;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PreLoginData preLogin(final String login, final String password,
    final Locale locale, final Map<String, Serializable> attributes) {
    final LoginService loginService = LoginService.getInstance();
    final PreLoginData data =
      loginService.preLogin(login, password, locale, attributes);
    if (data == null) {
      return null;
    }
    final String token = data.getToken();
    try {
      final String urlText = generateSystemURL(token, locale);
      final URL url = new URL(urlText);
      data.setUrl2Login(url);
    }
    catch (final MalformedURLException e) {
      e.printStackTrace();
      return null;
    }
    return data;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PreLoginData preLogin(final String login, final String password,
    final Locale locale, final String delegatedLogin, final Remote control,
    final Serializable userData) {
    final LoginService loginService = LoginService.getInstance();
    final PreLoginData data =
      loginService.preLogin(login, password, locale, delegatedLogin, control,
        userData);
    if (data == null) {
      return null;
    }
    final String token = data.getToken();
    try {
      final String urlText = generateSystemURL(token, locale);
      final URL url = new URL(urlText);
      data.setUrl2Login(url);
    }
    catch (final MalformedURLException e) {
      e.printStackTrace();
      return null;
    }
    return data;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String preLogin2Web(final String login, final String password,
    final Locale locale) {
    final LoginService loginService = LoginService.getInstance();
    final PreLoginData data = loginService.preLogin(login, password, locale);
    if (data == null) {
      return null;
    }
    final String token = data.getToken();
    return generateSystemURL(token, locale);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String preLogin2Web(final String login, final String password,
    final Locale locale, final Map<String, Serializable> attributes) {
    final LoginService loginService = LoginService.getInstance();
    final PreLoginData data =
      loginService.preLogin(login, password, locale, attributes);
    if (data == null) {
      return null;
    }
    final String token = data.getToken();
    return generateSystemURL(token, locale);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String preLogin2Web(final String login, final String password,
    final Locale locale, final String systemName) {
    final LoginService loginService = LoginService.getInstance();
    final PreLoginData data = loginService.preLogin(login, password, locale);
    if (data == null) {
      return null;
    }
    final String token = data.getToken();
    return this.generateSystemURL(token, locale, systemName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String preLogin2Web(final String login, final String password,
    final Locale locale, final String delegatedLogin, final String systemName) {
    final LoginService loginService = LoginService.getInstance();
    final PreLoginData data =
      loginService.preLogin(login, password, locale, delegatedLogin);
    if (data == null) {
      return null;
    }
    final String token = data.getToken();
    return this.generateSystemURL(token, locale, systemName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setServerSessionProperty(final Object sessionKey,
    final String propertyName, final Serializable propertyValue) {
    final LoginService loginService = LoginService.getInstance();
    if (propertyValue == null) {
      loginService.removeSessionProperty(sessionKey, propertyName);
    }
    else {
      loginService.setSessionProperty(sessionKey, propertyName, propertyValue);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setSystemName(final Object sessionKey, final String systemName) {
    final LoginService loginService = LoginService.getInstance();
    loginService.setSystemName(sessionKey, systemName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public String getSystemDefaultCharsetName() {
    final Server server = Server.getInstance();
    return server.getSystemDefaultCharsetName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PublicKey getPublicKey() {
    return LoginService.getInstance().getPublicKey();
  }
}
