/*
 * $Id: UserIO.java 154918 2014-08-01 00:49:36Z fpina $
 */
package csbase.server.services.administrationservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Calendar;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import tecgraf.javautils.core.io.FileUtils;
import csbase.logic.MDigest;
import csbase.logic.Permission;
import csbase.logic.Role;
import csbase.logic.User;
import csbase.logic.UserInfo;
import csbase.server.Server;
import csbase.server.ServerException;

/**
 * A classe <code>UserIO</code>  responsvel pela persistncia dos registros
 * dos usurios (objetos User) do sistema. Cada User  mantido em um arquivo
 * cujo o nome  o login do usurio que  atualizado sempre que as funes
 * User.createUser, User.ModifyUser e User.RemoveUser so invocadas. 
 * responsvel tambm pela gerao do identificador de um usurio. O
 * identificador de um usurio  o seu prprio login. Isso facilita a consulta
 * de usurios durante o login do sistema e deixa transparente a referncia por
 * identificador.
 * 
 * @author Tecgraf
 */
public class UserIO {

  /** Extenso dos arquivos dos usurios. */
  private static String USERS_EXTENSION = ".dat";

  private volatile boolean isReadingAll = false;

  /**
   * Constri um objeto responsvel pela persistncia de usurios.
   */
  public UserIO() {
    createAdmin();
  }

  /**
   * Mtodo de consulta ao nome do diretrio de usurios. Faz parte dos dados
   * persistidos.
   * 
   * @return uma string com o nome do diretrio-raiz.
   */
  final private String getPersistencyUsersDirectoryName() {
    final Server server = Server.getInstance();
    final String persistDirPath = server.getPersistencyRootDirectoryName();
    final String userDirPath = persistDirPath + File.separator + "users";
    try {
      Server.checkDirectory(userDirPath);
    }
    catch (ServerException e) {
      final String err = "Falha de criao de diretrio: " + userDirPath;
      throw new IllegalStateException(err);
    }
    return userDirPath;
  }

  /**
   * Insere no arquivo de dados o usurio Administrador do sistema.
   */
  private void createAdmin() {
    final String usersDirPath = getPersistencyUsersDirectoryName();
    String adminFileName =
      usersDirPath + File.separator + User.getAdminId() + USERS_EXTENSION;
    if ((new File(adminFileName).exists())) {
      return;
    }
    // Criar as permisses do admin.
    UserInfo info =
      new UserInfo((String) User.getAdminId(), "Administrador", new String[0],
        null, null, 0);
    try {
      info.setAttribute(User.PASSWORD, "1234");
      write(info);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro criando conta admin!");
      Server.getInstance().shutdown();
    }
  }

  /**
   * Imprime na tela o contedo do arquivo de dados
   * 
   * @param args .
   * 
   * @throws Exception
   */
  public static void main(String[] args) throws Exception {
    new UserIO().printDataFile();
  }

  /**
   * Imprime na tela o contedo do arquivo de usurios.
   * 
   * @throws Exception
   */
  public void printDataFile() throws Exception {
    Vector<User> users = readAll();
    for (Enumeration<User> e = users.elements(); e.hasMoreElements();) {
      User user = e.nextElement();
      System.out.println(user);
    }
  }

  /**
   * Verifica se h senha definida e se h mudana de senha. Caso o UserInfo
   * venha com uma senha, calcula seu digest, armazena do campo apropriado do
   * UserInfo e apaga a senha.
   * 
   * @param info o UserInfo a ser analisado
   * 
   * @throws Exception se no houver senha nem digest
   */
  private void checkPassword(UserInfo info) throws Exception {
    String password = (String) info.getAttribute(User.PASSWORD);
    if ((info.getAttribute(User.PASSWORD_DIGEST) == null) && (password == null)) {
      throw new Exception("senha no pode ser vazia");
    }
    if (password != null) {
      info.setAttribute(User.PASSWORD_DIGEST, MDigest.getDigest(password));
      info.removeAttribute(User.PASSWORD);
    }
  }

  /**
   * Cria um novo usurio e grava no arquivo.
   * 
   * @param info os dados a serem usados na criao do usurio
   * 
   * @return um novo usurio
   * 
   * @throws Exception inconsistncia nos dados ou erro de I/O
   */
  public synchronized User write(UserInfo info) throws Exception {
    checkPassword(info);
    String login = info.getLogin();
    long current = Calendar.getInstance().getTimeInMillis();
    info.setAttribute(User.LAST_UPDATE, new Long(current));
    User user = new User(login, info);
    Hashtable<String, Object> attributes = info.getAttributes();

    final String usersDirPath = getPersistencyUsersDirectoryName();
    String fileName = usersDirPath + File.separator + login + USERS_EXTENSION;
    ObjectOutputStream out = null;
    try {
      out =
        new ObjectOutputStream(new DataOutputStream(new BufferedOutputStream(
          new FileOutputStream(fileName))));
      out.writeObject(attributes);
    }
    finally {
      FileUtils.close(out);
    }
    return user;
  }

  /**
   * Remove o usurio. Apaga o seu arquivo de persistncia.
   * 
   * @param id o identificador do usurio a ser removido
   * 
   * @throws Exception se esse usurio no existir ou ocorrer erro de I/O
   */
  public synchronized void delete(Object id) throws Exception {
    User user = User.getUser(id);
    if (user == null) {
      return;
    }
    final String usersDirPath = getPersistencyUsersDirectoryName();
    File file = new File(usersDirPath + File.separator + id + USERS_EXTENSION);
    if (!file.exists()) {
      throw new Exception("No existe o arquivo do usurio " + id);
    }
    if (!file.delete()) {
      throw new Exception("No foi possvel remover o arquivo do usurio " + id);
    }
  }

  /**
   * Devolve um usurio que possua um determinado identificador.
   * 
   * @param id o identificador do usurio procurado
   * 
   * @return o usurio procurado ou null caso esse usurio no exista
   * 
   * @throws Exception erro de I/O
   */
  public synchronized User read(Object id) throws Exception {
    for (User user : User.getAllUsers()) {
      if (user.getId().equals(id)) {
        return user;
      }
    }
    return null;
  }

  /**
   * L e retorna todos os usurios existentes no arquivo de dados.
   * 
   * @return um vetor com todos os usurios
   * 
   * @throws Exception caso ocorra algum erro na leitura.
   */
  public synchronized Vector<User> readAll() throws Exception {
    final String usersDirPath = getPersistencyUsersDirectoryName();
    if (isReadingAll) {
      return null;
    }
    isReadingAll = true;
    Vector<User> users = new Vector<User>();
    try {
      final File usersDirectory = new File(usersDirPath);
      File[] userFiles = usersDirectory.listFiles();
      for (File userFile : userFiles) {
        ObjectInputStream in = null;
        try {
          in =
            new ObjectInputStream(new DataInputStream(new BufferedInputStream(
              new FileInputStream(userFile))));
          Hashtable<String, Object> attributes =
            (Hashtable<String, Object>) in.readObject();
          in.close();
          UserInfo info = new UserInfo(attributes);
          checkPermissions(info);
          checkRoles(info);
          User user = new User(info.getLogin(), info);
          users.add(user);
        }
        catch (Exception e) {
          AdministrationService srv = AdministrationService.getInstance();
          Server.logSevereMessage("Erro na leitura do arquivo do usurio "
            + userFile.getName());
        }
        finally {
          FileUtils.close(in);
        }
      }
    }
    finally {
      isReadingAll = false;
    }
    return users;
  }

  /**
   * Verifica se as permisses do UserInfo existem. As que no existirem mais
   * sero removidas e ser gerado um registro (log) do ocorrido. Esse teste 
   * feito para evitar que um usurio seja "perdido" caso ocorra algum erro na
   * gerncia de permisses.
   * 
   * @param info dados do usurio
   */
  private void checkPermissions(UserInfo info) {
    Object[] ids = (Object[]) info.getAttribute(User.PERMISSION_IDS);
    int lostPermissions = 0;
    for (int i = 0; i < ids.length; i++) {
      Permission p = null;
      try {
        p = Permission.getPermission(ids[i]);
      }
      catch (Exception e) {
        Server.logSevereMessage("Erro ao obter permisso " + ids[i], e);
      }
      if (p == null) {
        Server
          .logSevereMessage("Permisso inexistente sendo removida do usurio "
            + info.getLogin() + ": " + ids[i]);
        ids[i] = null;
        lostPermissions++;
      }
    }
    if (lostPermissions > 0) {
      Object[] newIds = new Object[ids.length - lostPermissions];
      int n = 0;
      for (int i = 0; i < ids.length; i++) {
        if (ids[i] != null) {
          newIds[n++] = ids[i];
        }
      }
      info.setAttribute(User.PERMISSION_IDS, newIds);
      try {
        // persiste o usuario com as permissoes validas
        Server.logInfoMessage("Usurio " + info.getLogin()
          + " sendo persistido " + "apenas com as permisses vlidas");
        write(info);
      }
      catch (Exception e) {
        Server.logSevereMessage("O usurio " + info.getLogin()
          + " possui permisso inexistente. No foi possvel persist-lo "
          + " com apenas as permisses vlidas.");
      }
    }
  }

  /**
   * Verifica se os perfis do UserInfo existem. Os que no existirem mais sero
   * removidos e ser gerado um registro (log) do ocorrido. Esse teste  feito
   * para evitar que um usurio seja "perdido" caso ocorra algum erro na
   * gerncia de perfis.
   * 
   * @param info dados do usurio.
   */
  private void checkRoles(UserInfo info) {
    Object[] ids = (Object[]) info.getAttribute(User.ROLE_IDS);
    int lostRoles = 0;
    for (int i = 0; i < ids.length; i++) {
      Role r = null;
      try {
        r = Role.getRole(ids[i]);
      }
      catch (Exception e) {
        Server.logSevereMessage("Erro ao obter perfil " + ids[i], e);
      }
      if (r == null) {
        Server.logSevereMessage("Perfil inexistente sendo removido do usurio "
          + info.getLogin() + ": " + ids[i]);
        ids[i] = null;
        lostRoles++;
      }
    }
    if (lostRoles > 0) {
      Object[] newIds = new Object[ids.length - lostRoles];
      int n = 0;
      for (int i = 0; i < ids.length; i++) {
        if (ids[i] != null) {
          newIds[n++] = ids[i];
        }
      }
      info.setAttribute(User.ROLE_IDS, newIds);
      try {
        // persiste o usuario com os perfis validos
        Server.logInfoMessage("Usurio sendo persistido apenas com os "
          + "perfis vlidos");
        write(info);
      }
      catch (Exception e) {
        Server.logSevereMessage("O usurio " + info.getAttribute(User.LOGIN)
          + " possui perfil inexistente. No foi possvel persist-lo "
          + " com apenas os perfis vlidos.");
      }
    }
  }
}
