/**
 * $Id: AbstractMonitor.java 141791 2013-06-12 14:54:50Z fpina $
 */

package csbase.server.services.diagnosticservice.monitors;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import csbase.logic.diagnosticservice.ResourceException;
import csbase.logic.diagnosticservice.ResourceProperty;
import csbase.logic.diagnosticservice.Status;
import csbase.logic.diagnosticservice.StatusCode;
import csbase.logic.diagnosticservice.ValidationStatus;

/**
 * Classe abstrata de um monitor.
 * 
 * @author Tecgraf PUC-Rio
 */
public abstract class AbstractMonitor implements Monitor {
  /**
   * Nome do recurso a ser monitorado.
   */
  protected String resourceName;
  /**
   * Lista de validaes utilizada pelo monitor.
   */
  private List<Validation> validationList;

  /**
   * Propriedades
   */
  private List<ResourceProperty> properties;

  /**
   * Construtor bsico de um monitor.
   * 
   * @param resourceName nome do recurso monitorado
   * 
   * @throws ResourceException caso algum parametro informado na criao do
   *         monitor for invlido
   */
  public AbstractMonitor(String resourceName) throws ResourceException {
    if ((resourceName == null) || (resourceName.length() == 0)) {
      throw new ResourceException(
        "O nome do resource no pode ser null ou vazio: " + resourceName);
    }
    this.resourceName = resourceName;
    this.validationList = new LinkedList<Validation>();
    this.properties = new LinkedList<ResourceProperty>();
  }

  /**
   * {@inheritDoc}
   * 
   */
  @Override
  public String getResourceName() {
    return resourceName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void addValidation(Validation validation) {
    if (validation == null) {
      throw new NullPointerException();
    }
    validationList.add(validation);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void addValidation(int pos, Validation validation) {
    try {
      validationList.add(pos, validation);
    }
    catch (IndexOutOfBoundsException e) {
      throw new IndexOutOfBoundsException();
    }
  }

  /**
   * Remove todas as propriedades cadastradas.
   */
  public void cleanProperties() {
    properties.clear();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void addProperty(String name, String value, String source) {
    properties.add(new ResourceProperty(name, value, source));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized Status checkResource(Locale locale)
    throws ResourceException {
    StatusCode result = StatusCode.OK;
    Map<String, ValidationStatus> validationResult =
      new HashMap<String, ValidationStatus>();
    List<ValidationStatus> validationStatusList =
      new LinkedList<ValidationStatus>();
    for (Validation validation : validationList) {
      ValidationStatus validationStatus;
      String[] requires = validation.requires();
      if (requires != null) {
        boolean dependenciesAreOk = true;
        for (String id : requires) {
          if (validationResult.get(id) == null
            || validationResult.get(id).code != StatusCode.OK) {
            dependenciesAreOk = false;
            break;
          }
        }
        if (!dependenciesAreOk) {
          continue;
        }
      }
      validationStatus = validation.check(locale);
      if (validationStatus.code.equals(StatusCode.ERROR)) {
        result = StatusCode.ERROR;
      }
      if (validationStatus.code.equals(StatusCode.WARNING)
        && !result.equals(StatusCode.ERROR)) {
        result = StatusCode.WARNING;
      }
      validationStatusList.add(validationStatus);
      validationResult.put(validation.id(), validationStatus);
    }
    return new Status(resourceName, result,
      validationStatusList.toArray(new ValidationStatus[0]), properties);
  }
}
