/*
 * $Id: SharedObjectService.java 181439 2018-04-13 14:19:12Z clinio $
 */

package csbase.server.services.sharedobjectservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;

import tecgraf.javautils.core.io.FileUtils;
import csbase.exception.ServiceFailureException;
import csbase.logic.SharedObject;
import csbase.logic.User;
import csbase.remote.SharedObjectServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;

/**
 * Servio para gerenciamento de objetos compartilhados.
 * 
 * @see SharedObject
 * 
 * @author Tecgraf
 */
public class SharedObjectService extends Service implements
  SharedObjectServiceInterface {

  /**
   * Tipos de contedo de um shared-object.
   */
  private enum ContentType {
    /**
     * Tabela interna com os dados do shared-object.
     */
    HASHTABLE(1),
    /**
     * Contedo do shared-object.
     */
    CONTENT(2);

    /**
     * ndice do respectivo contedo no shared-object (os contedos so
     * armazenados sequencialmente).
     */
    private final int contentIndex;

    /**
     * Construtor.
     * 
     * @param contentIndex ndice do contedo
     */
    private ContentType(int contentIndex) {
      this.contentIndex = contentIndex;
    }
  }

  /**
   * Nome do subdiretrio utilizado pelo servio. (dentro da persistncia)
   */

  public static final String OBJECTS_SUBDIR_NAME = "shared-objects";
  /*
   * Indices usados nas Hashtables para armazenar os atributos e o objeto.
   */
  private static final String CATEGORY = "category";
  private static final String OWNER_USER_ID = "ownerUserId";
  private static final String NAME = "name";
  private static final String ALLOWED_READING_USERS_ID = "allowedUsersId";
  private static final String ALLOWED_WRITING_USERS_ID =
    "allowedWritingUsersId";
  private static final String CREATED = "created";
  private static final String LAST_MODIFIED = "lastModified";
  private static final String CONTENTS = "contents";
  private static final String GLOBAL = "global";

  /**
   * Armazena a hierarquia completa de diretrios e arquivos, permitindo que
   * esta hierarquia seja manipulada.
   */
  private FileTree catsTree;

  /**
   * Relaciona nomes de categorias a uma tabela que por sua vez contm um
   * mapeamento de userID para uma terceira tabela que mapeia sharedObjectID
   * para sharedObjects. Resumindo:
   * <p>
   * <code>tabela(categoria, tabela(userID, tabela(soID, so)))</code>
   * <p>
   * O uso de SharedObjectId  necessrio para podermos colocar todos os objetos
   * de uma dada categoria acessveis por um usurio na mesma tabela. O
   * SharedObjectId usa o identificador do usurio e o nome do objeto como seu
   * prprio identificador. Se ns no usssemos o SharedObjectId, dois objetos
   * de usurios diferentes mas com o mesmo nome iriam se fundir nesta tabela.
   */
  private Hashtable<String, Hashtable<Object, Hashtable<Object, SharedObject>>> catsTable;

  /**
   * Mapeamento dos objetos globais. Mapeia nome de categoria em um mapeamento
   * de SharedObjectId de objeto em SharedObject. Os objetos globais ficam na
   * estrutura do catsTable e na global. Se deixam de ser global, ficam apenas
   * na catsTable.
   */
  private Hashtable<String, Hashtable<Object, SharedObject>> globalTable;

  /**
   * Armazena uma hierarquia de diretrios onde nomes livres so mapeados para
   * nomes compatveis com o sistema de arquivos. Em uma FileTree, cada nome 
   * mapeado para um diretrio e para uma tabela de associaes que mapeia nomes
   * livres (de filhos) em outras FileTree's. Dessa forma,  possvel criar uma
   * hierarquia cada nvel possui nomes livres e nomes que respeitam as regras
   * do sistema de arquivos. O ltimo nvel possui arquivos no lugar de
   * diretrios. O nmero de nveis  fixado no construtor.
   */
  private class FileTree {
    private final File root;
    private final int levels;
    private final Hashtable<String, File> name2File;
    private Hashtable<String, FileTree> name2Tree;

    FileTree(File root, int levels) {
      this.root = root;
      this.levels = levels;
      if (levels < 1) {
        throw new ServiceFailureException("Nmero de nveis invlido: "
          + levels);
      }
      name2File = new Hashtable<String, File>();
      if (levels > 1) {
        name2Tree = new Hashtable<String, FileTree>();
      }
    }

    boolean isEmpty() {
      return name2File.isEmpty();
    }

    void remove(String name) {
      File file = name2File.get(name);
      name2File.remove(name);
      if (name2Tree != null) {
        name2Tree.remove(name);
      }
      if (!file.delete()) {
        Server.logWarningMessage("Erro ao remover: " + file);
        return;
      }
    }

    void putFile(String name, File file) {
      if (!file.getParentFile().equals(root)) {
        throw new ServiceFailureException("Hierarquia incorreta: root=" + root
          + ", child=" + file);
      }
      name2File.put(name, file);
    }

    /**
     * Este mtodo cria um novo nome de arquivo caso este ainda no exista. O
     * nome  baseado no texto livre fornecido mas adaptado para atender os
     * requisitos do sistema de arquivos. Alm disso, caso haja conflito com um
     * arquivo existente, o nome  numerado para ficar diferente. Por este
     * motivo,  preciso sincronizar este mtodo, evitando que uma condio de
     * corrida faa com que um nremo seja pulado.
     * 
     * @param name .
     * 
     * @return .
     * 
     * @throws ServiceFailureException
     */
    synchronized File getFile(String name) {
      File file = name2File.get(name);
      if (file != null) {
        return file;
      }
      String filename = FileUtils.fixDirectoryName(name);
      file = new File(root, filename);
      for (int i = 0; file.exists(); i++) {
        file = new File(root, filename + i);
      }
      if (levels > 1 && !file.mkdir()) {
        throw new ServiceFailureException("Erro ao criar: " + file);
      }
      putFile(name, file);
      return file;
    }

    FileTree getTree(String name) {
      if (levels == 1) {
        throw new ServiceFailureException("Nvel folha em " + name);
      }
      FileTree tree = name2Tree.get(name);
      if (tree == null) {
        tree = new FileTree(getFile(name), levels - 1);
        name2Tree.put(name, tree);
      }
      return tree;
    }
  }

  public static void createService() throws ServerException {
    new SharedObjectService();
  }

  protected SharedObjectService() throws ServerException {
    super(SERVICE_NAME);
  }

  /**
   * @return instncia do servio
   */
  public static SharedObjectService getInstance() {
    return (SharedObjectService) getInstance(SERVICE_NAME);
  }

  /**
   * L o contedo de um arquivo associado a um share-object.
   * 
   * @param soFile arquivo
   * @param contentType qual contedo deve ser lido
   * @param throwException <code>true</code> se erros devem gerar uma
   *        {@link ServiceFailureException}. Se for <code>false</code>, erros
   *        so apenas logados.
   * @return objeto lido do arquivo
   */
  private Object readObjectFrom(File soFile, ContentType contentType,
    boolean throwException) {
    if (!soFile.exists()) {
      return null;
    }
    Object o = null;
    ObjectInputStream in = null;
    try {
      in =
        new ObjectInputStream(new BufferedInputStream(new FileInputStream(
          soFile)));
      for (int n = contentType.contentIndex; n > 0; n--) {
        o = in.readObject();
      }
    }
    catch (Exception e) {
      if (throwException) {
        String msg =
          String.format("Erro ao ler shared object %s [%s]",
            debugFileToString(soFile), contentType.name());
        throw new ServiceFailureException(msg, e);
      }
      else {
        Server.logSevereMessage("Erro ao ler: " + soFile.getPath(), e);
      }
      return null;
    }
    finally {
      FileUtils.close(in);
    }
    return o;
  }

  /**
   * Gera uma string no formato <i>categoria &gt; usurio &gt; nome</i> a partir
   * de um arquivo relacionado a um SO. O nome  o ltimo componente do path, o
   * usurio o penltimo e a categoria o antepenltimo.
   * 
   * @param soFile arquivo associado a um SO
   * @return string no formato categoria &gt; usurio &gt; nome
   */
  private static String debugFileToString(File soFile) {
    String[] pathComponents = soFile.getPath().split(File.separator);
    int last = pathComponents.length - 1;
    return String.format("%s > %s > %s", pathComponents[last - 2],
      pathComponents[last - 1], pathComponents[last]);
  }

  /**
   * Gera uma string no formato <i>categoria &gt; usurio &gt; nome</i> a partir
   * de um SO.
   * 
   * @param so shared-object
   * @return string no formato categoria &gt; usurio &gt; nome
   */
  private static String debugSOToString(SharedObject so) {
    return String.format("%s > %s > %s", so.getCategory(), SharedObject
      .getUserId(so.getId()), so.getName());
  }

  private Hashtable<String, Object> sharedObject2Hash(SharedObject object) {
    Hashtable<String, Object> table = new Hashtable<String, Object>();
    table.put(CATEGORY, object.getCategory());
    table.put(OWNER_USER_ID, object.getOwnerUserId());
    table.put(NAME, object.getName());
    table.put(ALLOWED_READING_USERS_ID, object.getUsersRO());
    table.put(ALLOWED_WRITING_USERS_ID, object.getUsersRW());
    table.put(CREATED, object.getCreated());
    table.put(LAST_MODIFIED, object.getLastModified());
    table.put(GLOBAL, new Boolean(object.isGlobal()));
    return table;
  }

  private SharedObject hash2SharedObject(Hashtable<?, ?> table) {
    SharedObject object = null;
    try {
      String category = (String) table.get(CATEGORY);
      Object ownerUserId = table.get(OWNER_USER_ID);
      String name = (String) table.get(NAME);
      Object[] allowedReadingUsersId =
        (Object[]) table.get(ALLOWED_READING_USERS_ID);
      Object[] allowedWritingUsersId =
        (Object[]) table.get(ALLOWED_WRITING_USERS_ID);
      /*
       * Caso de objetos antigos, sem atributo de ids pra escrita, coloca um
       * array vazio, no lugar de um objeto null
       */
      if (allowedWritingUsersId == null) {
        allowedWritingUsersId = new Object[0];
      }
      boolean global = ((Boolean) table.get(GLOBAL)).booleanValue();
      object =
        new SharedObject(category, ownerUserId, name, global,
          allowedReadingUsersId, allowedWritingUsersId, null);
      object.setCreated((Date) table.get(CREATED));
      object.setLastModified((Date) table.get(LAST_MODIFIED));
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na converso.", e);
      return null;
    }
    return object;
  }

  private void unregisterSharedObject(Hashtable<?, ?> usrsTable, Object userId,
    Object id, SharedObject object) {
    Hashtable<?, ?> objsTable = (Hashtable<?, ?>) usrsTable.get(userId);
    if (objsTable == null) {
      throw new ServiceFailureException("Usurio no cadastrado: " + userId);
    }
    objsTable.remove(id);
  }

  private void unregisterSharedObject(SharedObject object) {
    if (object.isGlobal()) {
      unregisterGlobalObject(object);
    }
    Hashtable<Object, Hashtable<Object, SharedObject>> usrsTable =
      catsTable.get(object.getCategory());
    if (usrsTable == null) {
      throw new ServiceFailureException("Categoria no cadastrada: "
        + object.getCategory());
    }
    Object id = object.getId();
    unregisterSharedObject(usrsTable, object.getOwnerUserId(), id, object);
    /*
     * Os usurios a serem registrados so os de leitura e escrita, por isso a
     * cpia abaixo: usersId tem ambos os tipos de usurios
     */
    Object[] usersId = object.getAllUsers();
    for (int i = 0; i < usersId.length; i++) {
      unregisterSharedObject(usrsTable, usersId[i], id, object);
    }
  }

  private void unregisterGlobalObject(SharedObject object) {
    Hashtable<Object, SharedObject> objsTable =
      globalTable.get(object.getCategory());
    if (objsTable == null) {
      throw new ServiceFailureException("Categoria no cadastrada: "
        + object.getCategory());
    }
    Object id = object.getId();
    objsTable.remove(id);
  }

  private void registerSharedObject(
    Hashtable<Object, Hashtable<Object, SharedObject>> usrsTable,
    Object userId, SharedObject object) {
    Hashtable<Object, SharedObject> objsTable = usrsTable.get(userId);
    if (objsTable == null) {
      objsTable = new Hashtable<Object, SharedObject>();
      usrsTable.put(userId, objsTable);
    }
    Object id = object.getId();
    SharedObject oldObject = objsTable.get(id);
    if (oldObject != null) {
      if (userId.equals(oldObject.getOwnerUserId())) {
        unregisterSharedObject(oldObject);
      }
    }
    objsTable.put(id, object);
  }

  private void registerSharedObject(SharedObject object) {
    Hashtable<Object, Hashtable<Object, SharedObject>> usrsTable =
      catsTable.get(object.getCategory());
    if (usrsTable == null) {
      usrsTable = new Hashtable<Object, Hashtable<Object, SharedObject>>();
      catsTable.put(object.getCategory(), usrsTable);
    }
    Object objectOwner = object.getOwnerUserId();
    registerSharedObject(usrsTable, objectOwner, object);
    /*
     * Os usurios a serem registrados so os de leitura e escrita, por isso a
     * cpia abaixo: usersId tem ambos os tipos de usurios
     */
    Object[] usersId = object.getAllUsers();
    for (int i = 0; i < usersId.length; i++) {
      /*
       * se o usurio  o prprio owner do objeto, ele no deve ser adicionado 
       * lista de compartilhamento
       */
      if (usersId[i].equals(objectOwner)) {
        continue;
      }
      registerSharedObject(usrsTable, usersId[i], object);
    }
    if (object.isGlobal()) {
      registerGlobalObject(object);
    }
  }

  private void registerGlobalObject(SharedObject object) {
    Hashtable<Object, SharedObject> objsTable =
      globalTable.get(object.getCategory());
    if (objsTable == null) {
      objsTable = new Hashtable<Object, SharedObject>();
      globalTable.put(object.getCategory(), objsTable);
    }
    Object id = object.getId();
    objsTable.put(id, object);
  }

  /**
   * Processa a estrutura dos diretrios dos shared-objects, criando as relaes
   * entre eles em memria. Diretrios referentes a usurios inexistentes so
   * automaticamente removidos, o mesmo acontecendo com arquivos invlidos
   * (categoria ou owner registrados no objeto inconsistentes com a estrutura de
   * diretrios).
   * 
   * @param rootDir - diretrio-raiz a partir do qual ser feito o percorrimento
   */
  private void readSharedObjectsStructure(File rootDir) {
    catsTable =
      new Hashtable<String, Hashtable<Object, Hashtable<Object, SharedObject>>>();
    globalTable = new Hashtable<String, Hashtable<Object, SharedObject>>();
    File[] catFiles = rootDir.listFiles();
    for (int i = 0; i < catFiles.length; i++) {
      if (!catFiles[i].isDirectory()) {
        catFiles[i].delete();
        continue;
      }
      /*
       * diretrios associados a cada usurio dentro da categoria em questo
       */
      File[] usrFiles = catFiles[i].listFiles();
      boolean catUsed = false;
      for (int j = 0; j < usrFiles.length; j++) {
        if (!usrFiles[j].isDirectory()) {
          usrFiles[j].delete();
          continue;
        }
        final String userName = usrFiles[j].getName();
        try {
          if (User.getUserByLogin(userName) == null) {
            /*
             * usurio no est mais cadastrado no sistema, ns o removemos
             * desta categoria junto com seus respectivos objetos
             */
            Server.logWarningMessage(String.format(
              "categoria=%s: usuario %s inexistente. Removendo objetos...",
              catFiles[i].getName(), userName));
            if (!deleteUserDir(usrFiles[j])) {
              Server.logWarningMessage("Erro removendo objetos de " + userName);
            }
            continue;
          }
        }
        catch (Exception e) {
          Server.logSevereMessage("Erro em User.getUser para " + userName, e);
          continue;
        }
        /*
         * arquivos (shared-objects) dentro do diretrio do usurio em questo
         */
        File[] objFiles = usrFiles[j].listFiles();
        boolean usrUsed = false;
        for (int k = 0; k < objFiles.length; k++) {
          /*
           * shared-objects s podem ser arquivos
           */
          if (!objFiles[k].isFile()) {
            objFiles[k].delete();
            continue;
          }
          Object o = readObjectFrom(objFiles[k], ContentType.HASHTABLE, false);
          if (!(o instanceof Hashtable<?, ?>)) {
            Server.logWarningMessage("Arquivo " + objFiles[k]
              + " no possui Hashtable: " + o);
            continue;
          }
          Hashtable<?, ?> table = (Hashtable<?, ?>) o;
          SharedObject object = hash2SharedObject(table);
          if (object == null) {
            Server.logWarningMessage("Arquivo " + objFiles[k]
              + " possui Hashtable incorreta.");
            continue;
          }
          User owner = null;
          final Object ownerId = object.getOwnerUserId();
          try {
            owner = User.getUser(ownerId);
            if (owner == null) {
              Server.logWarningMessage(String.format(
                "Desconsiderando %s: usuario inexistente (%s)", objFiles[k],
                ownerId.toString()));
              continue;
            }
          }
          catch (Exception e) {
            Server.logSevereMessage("Erro em User.getUser para " + objFiles[k],
              e);
            continue;
          }
          if (!object.getCategory().equals(catFiles[i].getName())) {
            /*
             * categoria registrada no corresponde  categoria real.
             * Registramos no log e removemos o objeto.
             */
            Server
              .logWarningMessage(String
                .format(
                  "categoria incorreta [%s]: registrada=%s real=%s. Removendo arquivo...",
                  objFiles[k], object.getCategory(), catFiles[i].getName()));
            objFiles[k].delete();
            continue;
          }
          catsTree.putFile(object.getCategory(), catFiles[i]);
          FileTree usrsTree = catsTree.getTree(object.getCategory());
          usrsTree.putFile(owner.getLogin(), usrFiles[j]);
          FileTree objsTree = usrsTree.getTree(owner.getLogin());
          objsTree.putFile(object.getName(), objFiles[k]);
          registerSharedObject(object);
          usrUsed = true;
        }
        if (!usrUsed) {
          /*
           * nenhum objeto do usurio em questo foi registrado com sucesso,
           * removemos o diretrio associado ao usurio para esta categoria
           */
          usrFiles[j].delete();
        }
        else {
          catUsed = true;
        }
      }
      if (!catUsed) {
        /*
         * nenhum objeto de nenhum usurio desta categoria foi registrado com
         * sucesso, removemos a categoria
         */
        catFiles[i].delete();
      }
    }
  }

  /**
   * Remove o diretrio associado a um usurio, removendo antes todos os
   * arquivos dentro dele. Assume que, por construo, no existe nenhum
   * diretrio dentro do diretrio do usurio (caso exista a operao vai
   * falhar).
   * 
   * @param userDir - diretrio a ser removido. Por construo deve possuir
   *        apenas arquivos
   * @return true se o diretrio foi removido com sucesso
   */
  private boolean deleteUserDir(File userDir) {
    File[] objectFiles = userDir.listFiles();
    for (File file : objectFiles) {
      if (!file.delete()) {
        return false;
      }
    }
    return userDir.delete();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() throws ServerException {
    final Server server = Server.getInstance();
    final String persistDirPath = server.getPersistencyRootDirectoryName();
    final String sep = File.separator;
    final String objectsDirPath = getRootPath();
    server.checkDirectory(objectsDirPath);
    final File rootDir = getRootDir();
    catsTree = new FileTree(rootDir, 3); // Categorias -> usurios -> objetos.
    readSharedObjectsStructure(rootDir);
  }

  /**
   * Obtmm o rootPath do servio
   * @return path
   */
  private String getRootPath() {
    final Server server = Server.getInstance();
    final String persistDirPath = server.getPersistencyRootDirectoryName();
    final String sep = File.separator;
    final String objectsDirPath = persistDirPath + sep + OBJECTS_SUBDIR_NAME;
    return objectsDirPath;
  }

  /**
   * Obtem diretrio raiz do servio.
   * @return diretrio
   */
  private File getRootDir() {
    final File rootDir = new File(getRootPath());
    return rootDir;
  }


  /**
   * {@inheritDoc}
   */
  @Override
  public void shutdownService() throws ServerException {
    // vazio
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return false;
  }

  /**
   * Obtm o arquivo associado a um determinado shared-object.
   * 
   * @param object shared-object
   * @param checkOwner <code>true</code> se o ID do usurio corrente deve ser o
   *        dono do shared-object
   * @return arquivo associado ao shared-object
   * @throws ServiceFailureException em caso de erro
   */
  private File getSharedObjectFile(SharedObject object, boolean checkOwner) {
    User user = Service.getUser();
    if (user == null) {
      throw new ServiceFailureException("Usurio no identificado.");
    }
    if (checkOwner && !user.getId().equals(object.getOwnerUserId())) {
      throw new ServiceFailureException("Usurio no  dono do objeto.");
    }
    FileTree usrsTree = catsTree.getTree(object.getCategory());
    User ownerUser = null;
    try {
      ownerUser = User.getUser(object.getOwnerUserId());
    }
    catch (Exception e) {
      throw new ServiceFailureException(
        "No foi possvel obter o usurio dono do objeto.", e);
    }
    FileTree objsTree = usrsTree.getTree(ownerUser.getLogin());
    return objsTree.getFile(object.getName());
  }

  /**
   * Grava um objeto em um arquivo, junto com suas propriedades na forma de
   * Hashtable. Este mtodo precisa ser sincronizado para evitar uma condio de
   * corrida na gerao de nomes para arquivos novos. Outro motivo para a
   * sincronizao  evitar uma condio de corrida com a remoo, que apaga os
   * diretrios que ficam vazios.
   * 
   * @param so .
   * 
   * @throws ServiceFailureException
   */
  private synchronized void writeSharedObjectFile(SharedObject so) {
    /*
     * Chama getSharedObjectFile com false (no considerar owner, porque o
     * mtodo que chama write... (saveShared...) j faz essa checagem
     */
    File objFile = getSharedObjectFile(so, false);
    ObjectOutputStream out = null;
    try {
      objFile.getParentFile().mkdirs();
      out =
        new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(
          objFile)));
      out.writeObject(sharedObject2Hash(so));
      out.writeObject(so.getContents());
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro ao gravar: " + objFile.getPath(), e);
      throw new ServiceFailureException("Erro ao gravar shared object: "
        + debugSOToString(so), e);
    }
    finally {
      FileUtils.close(out);
    }
  }

  /**
   * Obtm o contedo de um shared-object.
   * 
   * @param object shared-object
   * @return contedo do shared-object
   */
  private Object readSharedObjectContent(SharedObject object) {
    return readObjectFrom(getSharedObjectFile(object, false),
      ContentType.CONTENT, true);
  }

  /**
   * Remove um arquivo de objeto. Note que este mtodo precisa ser sincronizado
   * para evitar uma condio de corrida com a escrita, que gera arquivos. Essa
   * sincronizao com a escrita  necessria porque este mtodo remove os
   * diretrios que ficam vazios.
   * 
   * @param object .
   * 
   * @throws ServiceFailureException
   */
  private synchronized void removeSharedObjectFile(SharedObject object) {
    User user = Service.getUser();
    if (user == null) {
      throw new ServiceFailureException("Usurio no identificado.");
    }
    FileTree usrsTree = catsTree.getTree(object.getCategory());
    FileTree objsTree = usrsTree.getTree(user.getLogin());
    objsTree.remove(object.getName());
    if (objsTree.isEmpty()) {
      usrsTree.remove(user.getLogin());
    }
    if (usrsTree.isEmpty()) {
      catsTree.remove(object.getCategory());
    }
  }

  /**
   * Obtm uma referncia para um objeto.
   * 
   * @param category categoria
   * @param userId id do usurio
   * @param name nome do objeto
   * @return referncia para o objeto, ou <code>null</code> se a categoria era
   *         invlida, ou se o usurio no possua objetos na cateoria, ou se o
   *         usurio no possua o objeto com o nome solicitado
   */
  private SharedObject getSharedObjectRef(String category, Object userId,
    String name) {
    Hashtable<Object, Hashtable<Object, SharedObject>> usrsTable =
      catsTable.get(category);
    if (usrsTable == null) {
      /*
       * categoria no existe
       */
      return null;
    }
    Hashtable<Object, SharedObject> objsTable = usrsTable.get(userId);
    if (objsTable == null) {
      /*
       * usurio no possui objetos na categoria especificada
       */
      return null;
    }
    Object id = SharedObject.getId(userId, name);
    SharedObject object = objsTable.get(id);
    /*
     * OBS.: object ainda pode ser null se no existir um objeto com o nome
     * solicitado
     */
    return object;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SharedObject saveSharedObject(SharedObject object) {
    if (object.getCategory() == null || object.getCategory().isEmpty()) {
      throw new ServiceFailureException("Nome de categoria invlido.");
    }
    if (object.getName() == null || object.getName().isEmpty()) {
      throw new ServiceFailureException("Nome de objeto invlido.");
    }
    if (object.getUsersRO() == null || object.getUsersRW() == null) {
      throw new ServiceFailureException("Lista de usurios nula.");
    }
    if (object.getContents() == null) {
      throw new ServiceFailureException("Objeto vazio (contedo == null).");
    }
    User callerUser = Service.getUser();
    if (callerUser == null) {
      throw new ServiceFailureException("Usurio no identificado.");
    }
    SharedObject oldObject =
      getSharedObjectRef(object.getCategory(), object.getOwnerUserId(), object
        .getName());
    /*
     * Se for um objeto novo, ento atribui a propriedade ao usurio corrente,
     * caso contrrio, verifica se usurio pode escrever. Se no puder, ento
     * joga exceo
     */
    if (oldObject == null) {
      object.setCreated(new Date());
      object.setOwnerUserId(callerUser.getId());
      object.setLastModified(object.getCreated());
    }
    else {
      if (!oldObject.userHasAccessRW(callerUser.getId())) {
        throw new ServiceFailureException(
          "Usurio no tem permisso de escrita.");
      }
      object.setCreated(oldObject.getCreated());
      object.setLastModified(new Date());
    }
    writeSharedObjectFile(object);
    object.setContents(null);
    registerSharedObject(object);
    return object;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeSharedObject(String category, String name) {
    User user = Service.getUser();
    if (user == null) {
      throw new ServiceFailureException("Usurio no identificado.");
    }
    SharedObject object = getSharedObjectRef(category, user.getId(), name);
    if (object == null) {
      // TODO no deveramos lanar ServiceFailureException neste caso?
      return;
    }
    unregisterSharedObject(object);
    removeSharedObjectFile(object);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SharedObject[] getSharedObjectAttributes(String category) {
    User user = Service.getUser();
    if (user == null) {
      throw new ServiceFailureException("Usurio no identificado.");
    }
    Hashtable<Object, SharedObject> resultTable =
      new Hashtable<Object, SharedObject>();

    // Obtm os objetos globais, em seguida, os que o usurio tem acesso.
    // No repete os elementos.
    Hashtable<Object, SharedObject> globalObjsTable = globalTable.get(category);
    if (globalObjsTable != null) {
      resultTable.putAll(globalObjsTable);
    }
    Hashtable<Object, Hashtable<Object, SharedObject>> usrsTable =
      catsTable.get(category);
    if (usrsTable != null) {
      Hashtable<Object, SharedObject> objsTable =
        usrsTable.get(user.getLogin());
      if (objsTable != null) {
        resultTable.putAll(objsTable);
      }
    }
    SharedObject[] result = new SharedObject[resultTable.size()];
    int i = 0;
    for (Enumeration<SharedObject> e = resultTable.elements(); e
      .hasMoreElements();) {
      result[i++] = e.nextElement();
    }
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SharedObject getSharedObject(String category, Object userId,
    String name) {
    User user = Service.getUser();
    if (user == null) {
      throw new ServiceFailureException("Usurio no identificado.");
    }
    SharedObject object = getSharedObjectRef(category, userId, name);
    if (object == null) {
      return null;
    }
    if (!object.isGlobal() && !user.getId().equals(object.getOwnerUserId())) {
      boolean allowedUser = false;
      // Rene em usersId todos que podem ler
      Object[] usersId = object.getAllUsers();
      for (int i = 0; i < usersId.length; i++) {
        if (user.getId().equals(usersId[i])) {
          allowedUser = true;
          break;
        }
      }
      if (!allowedUser) {
        return null;
      }
    }
    String cat = object.getCategory();
    Object ownerUserId = object.getOwnerUserId();
    String resultName = object.getName();
    Object[] allowedReadingUsersId = new Object[object.getUsersRO().length];
    for (int i = 0; i < allowedReadingUsersId.length; i++) {
      allowedReadingUsersId[i] = object.getUsersRO()[i];
    }
    Object[] allowedWritingUsersId = new Object[object.getUsersRW().length];
    for (int i = 0; i < allowedWritingUsersId.length; i++) {
      allowedWritingUsersId[i] = object.getUsersRW()[i];
    }
    boolean global = object.isGlobal();
    Object content = readSharedObjectContent(object);
    SharedObject result =
      new SharedObject(cat, ownerUserId, resultName, global,
        allowedReadingUsersId, allowedWritingUsersId, content);
    result.setCreated(object.getCreated());
    result.setLastModified(object.getLastModified());
    return result;
  }

  @Override
  public boolean removeAllFromUser(Object userId) {
    final String login = (String) userId;
    final File rootDir = getRootDir();
    File[] catFiles = rootDir.listFiles();
    boolean shrRemoved = true;
    for (int i = 0; i < catFiles.length; i++) {
      File[] usrFiles = catFiles[i].listFiles();
      for (int j = 0; j < usrFiles.length; j++) {
        final String fileName = usrFiles[j].getName();
        if (login.equalsIgnoreCase(fileName)) {
          final String fmt = "categoria=%s: usuario %s remoo de dados do usurio. Removendo objetos...";
          Server.logInfoMessage(String.format(fmt, catFiles[i].getName(), login));
          if (!deleteUserDir(usrFiles[j])) {
            Server.logWarningMessage("Erro removendo dados shared-obj de " + login);
            shrRemoved = false;
          }
        }
      }
    }
    return shrRemoved;
  }
}
