/*
 * $Id: ORBDealer.java 154918 2014-08-01 00:49:36Z fpina $
 */
package csbase.server.services.wioservice;

import java.util.Hashtable;
import java.util.Properties;
import java.util.Set;

import org.omg.CORBA.ORB;
import org.omg.CORBA.Policy;
import org.omg.PortableServer.IdAssignmentPolicyValue;
import org.omg.PortableServer.LifespanPolicyValue;
import org.omg.PortableServer.POA;
import org.omg.PortableServer.POAHelper;
import org.omg.PortableServer.Servant;

import csbase.server.Server;
import csbase.server.services.wioservice.idl.WIOServiceException;

/**
 * A classe <code>ORBDealer</code> implementa um servidor CORBA persistente,
 * isto , um servidor cujos objetos possuem referncias persistentes.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ORBDealer extends Thread {
  /**
   * Instncia de ORB
   */
  private ORB orb = null;

  /**
   * Root POA
   */
  private POA rootPOA = null;

  /**
   * POA persistente com id para os objetos CORBA ativados pelo servidor
   */
  private POA persistentIdPOA = null;

  /**
   * POA comum sem id para os objetos CORBA ativados pelo servidor
   */
  private POA persistentAnonPOA = null;

  /**
   * Identificadores de objetos ativados
   */
  private final Hashtable<String, String> hashIds =
    new Hashtable<String, String>();

  /**
   * Indicao de servidor CORBA est ativo pois o servidor CORBA associado a
   * uma instncia de ORBDealer  considerado desativado (por precauo) aps a
   * execuo de um shutdown.
   */
  private boolean isActive;

  /**
   * Ativa um objeto CORBA associado ao servidor.
   * 
   * @param oid id do objeto
   * @param object implementao do objeto (servant)
   * 
   * @return o POA utilizado na ativao.
   * 
   * @throws WIOServiceException em caaso de falha.
   */
  public POA activateObject(final String oid, final Servant object)
    throws WIOServiceException {
    if (!isActive) {
      final String err = "CORBA desativado na ativacao de objeto";
      final WIOServiceException exception = new WIOServiceException(err);
      Server.logSevereMessage(err);
      throw exception;
    }

    // Ativa o objeto atraves do POA adequado
    POA myPOA = null;
    try {
      if (oid == null) {
        myPOA = persistentAnonPOA;
        myPOA.activate_object(object);
      }
      else {
        myPOA = persistentIdPOA;
        if (hashIds.get(oid) == null) {
          myPOA.activate_object_with_id(oid.getBytes(), object);
          hashIds.put(oid, oid);
          final org.omg.CORBA.Object o = myPOA.servant_to_reference(object);
          final String ior = orb.object_to_string(o);
          Server.logFineMessage("Ativao: Object IOR [" + ior + "]");
        }
      }
      return myPOA;
    }
    catch (final org.omg.PortableServer.POAPackage.ObjectAlreadyActive e) {
      final String err =
        e.getMessage() + "\n" + "ORB dealer: Dupla ativao de objeto CORBA "
          + "[" + object.toString() + "] - Object ID: [" + oid + "]";
      Server.logSevereMessage(err);
      return myPOA;
    }
    catch (final Exception e) {
      final String err =
        "ORB dealer: Falha na ativacao de CORBA " + "[" + object.toString()
          + "] - Object ID: [" + oid + "] ->> " + e.getMessage();
      final WIOServiceException exception = new WIOServiceException(err);
      Server.logSevereMessage(err);
      e.printStackTrace();
      throw exception;
    }
  }

  /**
   * Desativa um objeto CORBA associado ao servidor.
   * 
   * @param object implementao do objeto (servant)
   * 
   * @throws WIOServiceException em caso de falha.
   */
  public void deactivateObject(final Servant object) throws WIOServiceException {
    if (!isActive) {
      final String err = "CORBA desativado para desativacao de objeto";
      final WIOServiceException exception = new WIOServiceException(err);
      Server.logSevereMessage(err);
      throw exception;
    }
    try {
      final POA poa = object._poa();
      poa.deactivate_object(object._object_id());
    }
    catch (final Exception e) {
      final String err = e.getMessage();
      final WIOServiceException exception = new WIOServiceException(err);
      Server.logSevereMessage(err);
      e.printStackTrace();
      throw exception;
    }
  }

  /**
   * Inicializa o servidor CORBA persistente. Instancia um ORB para o servidor e
   * cria um POA persistente para os objetos CORBA que sero associados ao
   * servidor. O nome do servidor persistente, fornecido como parmetro, ser
   * utilizado para nomear o POA persistente. Em alguns ORBs (como o JacORB)
   * tanto o nome do servidor como o nome do POA so utilizados na composio
   * das referncias aos objetos CORBA. A atribuio de uma porta "bem
   * conhecida" ao servidor (fornecida tambm como parmetro) permite a
   * construo de referncias aos objetos CORBA ativados pelo servidor sem que
   * seja necessria a utilizao de um servio de nomes.
   * 
   * @param props propriedades ajustadas externamente ao ORB.
   * 
   * @throws Exception em caso de falha.
   */
  private void init(final Properties props) throws Exception {

    //Propriedade preventiva para evitar bug de no recuperao aps queda.
    props.setProperty(
      "jacorb.connection.client.disconnect_after_systemexception", "false");

    final Set<Object> keySet = props.keySet();
    for (final Object key : keySet) {
      final String value = props.getProperty((String) key);
      Server.logInfoMessage("Propriedade ORB: " + key + " = " + value);
    }

    // Instancia o ORB e cria POAs persistentes
    try {
      final String[] args = null;
      orb = ORB.init(args, props);

      final String serverName = "WIOMgr";
      rootPOA = POAHelper.narrow(orb.resolve_initial_references("RootPOA"));
      final org.omg.CORBA.Policy[] idPolicies = new org.omg.CORBA.Policy[2];
      idPolicies[0] =
        rootPOA.create_id_assignment_policy(IdAssignmentPolicyValue.USER_ID);
      idPolicies[1] =
        rootPOA.create_lifespan_policy(LifespanPolicyValue.PERSISTENT);

      // idPolicies[2] = rootPOA.create_id_uniqueness_policy(
      // IdUniquenessPolicyValue.UNIQUE_ID);
      // idPolicies[3] = rootPOA.create_servant_retention_policy(
      // ServantRetentionPolicyValue.RETAIN);
      persistentIdPOA =
        rootPOA.create_POA(serverName + "POA", rootPOA.the_POAManager(),
          idPolicies);
      final org.omg.CORBA.Policy[] id2Policies = new org.omg.CORBA.Policy[1];
      id2Policies[0] =
        rootPOA.create_lifespan_policy(LifespanPolicyValue.PERSISTENT);
      persistentAnonPOA =
        rootPOA.create_POA(serverName + "AnonPOA", rootPOA.the_POAManager(),
          id2Policies);
      for (final Policy idPolicie : idPolicies) {
        idPolicie.destroy();
      }
      for (final Policy id2Policie : id2Policies) {
        id2Policie.destroy();
      }
      rootPOA.the_POAManager().activate();
      persistentIdPOA.the_POAManager().activate();
      persistentAnonPOA.the_POAManager().activate();

      // Cria a thread fornecida ao ORB para atendimento s
      // requisies de servio aos objetos CORBA
      start();
    }
    catch (final org.omg.CORBA.UserException e) {
      throw new Exception("Erro na inicializao de servidor CORBA", e);
    }
  }

  /**
   * Dedica uma thread ao procedimento de atendimento a requisies de servio
   * do ORB. Ao receber um comando de shutdown, o ORB devolve o controle. Nesse
   * caso, a instncia de ORB ser destruda.
   */
  @Override
  public void run() {
    Server.logFineMessage("Thread iniciada: ORBDealer");
    orb.run();
    orb.destroy();
    Server.logFineMessage("Thread terminada: ORBDealer");
  }

  /**
   * Solicita a desativao do servidor CORBA. Comanda o shutdown do ORB
   * instanciado para o servidor, o que provocar a terminao da thread
   * dedicada ao ORB.
   * 
   * @throws Exception em caso de falha.
   */
  public void shutdown() throws Exception {
    if (!isActive) {
      throw new Exception("Servidor CORBA desativado");
    }

    // Comanda o shutdown do ORB
    orb.shutdown(false);
    isActive = false;
  }

  /**
   * Constri um servidor CORBA persistente associado ao nome e porta fornecidos
   * como parmetros. Uma instncia de ORB e um POA persistente so criados para
   * este servidor.
   * 
   * @param props propriedades ajustadas externamente
   * 
   * @throws Exception em caso de falha.
   */
  public ORBDealer(final Properties props) throws Exception {
    orb = null;
    rootPOA = null;
    persistentIdPOA = null;
    persistentAnonPOA = null;
    init(props);
    isActive = true;
  }
}
