/*
 * 
 * $Id: Service.java 175407 2016-08-10 20:31:03Z clinio $
 */
package csbase.server;

import java.io.File;
import java.io.InputStream;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.SortedSet;

import org.slf4j.MDC;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.properties.PropertiesUtils;
import csbase.exception.ServiceFailureException;
import csbase.logic.Permission;
import csbase.logic.SecureKey;
import csbase.logic.User;
import csbase.remote.ServiceInterface;
import csbase.server.services.loginservice.LoginService;

/**
 * Classe que implementa o framework para definio de um servio. Nem todos os
 * servios so servidores RMI: os que precisam de validao de usurio so
 * exportados por meio de proxies. Cada servio tem recursos de log e um
 * conjunto de propriedades especficos. Os nomes do arquivo de log e do arquivo
 * de propriedades so construdos a partir do nome do servio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class Service extends ServerRemoteObjectObservable implements
  ServiceInterface {

  /**
   * Constante que define o nome padro do arquivo de idiomas.
   */
  private final static String DEFAULT_LAGUANGE_FILE_NAME = "idiom";

  /**
   * Propriedade que define arquivos de idiomas adicionais
   */
  private final static String ADDITIONAL_LANGUAGE_FILE_PROPERTY =
    "additional.language.file";

  /**
   * Propriedades do servio carregadas pelo arquivo de propriedades.
   */
  private final ServerSideProperties serviceProperties;
  /**
   * Nome do servio
   */
  private String serviceName;
  /**
   * O estado do servio.
   */
  private ServiceState state;
  /**
   * Indica se o servio est habilitado.
   */
  private boolean enabled;

  /**
   * Bundles de internacionalizao (indexados por objetos locale)
   */
  private final Hashtable<Locale, HierachicalResourceBundle> bundles =
    new Hashtable<Locale, HierachicalResourceBundle>();

  /**
   * Dados locais da thread, utilizado exclusivamente para armazenar a sesso do
   * usurio que realizou a chamada remota.
   */
  private static ThreadLocal<Object> threadUserKey = new ThreadLocal<Object>();

  /**
   * Dados locais da thread, utilizado para armazenar a string que identifica o
   * sistema onde o usurio da sesso atual originou a chamada remota. O default
   *  ser nulo, e indica que o sistema  o prprio levantado pelo servidor.
   */
  private static ThreadLocal<String> threadSystemKey =
    new ThreadLocal<String>();

  /**
   * Atualiza a informao especfica do SLF4J que identifica o usurio da
   * thread corrente.
   */
  private static void updateMDCUser() {
    User user = getUser();
    if (user != null) {
      MDC.put("user", user.getLogin());
    }
    else {
      MDC.remove("user");
    }
  }

  /**
   * Guarda nos dados locais do thread corrente o identificador da sesso do
   * usurio. Este mtodo deve ser chamado pelos objetos remotos exportados pelo
   * servidor antes de repassar a chamada ao respectivo servio.
   * 
   * @param key a chave a ser guardada.
   */
  public static void setKey(final Object key) {
    threadUserKey.set(key);
    updateMDCUser();
  }

  /**
   * Guarda o usurio nos dados locais do thread corrente. Este mtodo  chamado
   * quando a chamada de um mtodo do servio no  feito pelo proxy de um
   * cliente, mas quando  feito, por exemplo, por outros servios, ou pelo WIO.
   * 
   * @param userId o identificador do usurio.
   */
  public static void setUserId(final Object userId) {
    threadUserKey.set(userId);
    updateMDCUser();
  }

  /**
   * Indica se a thread que origino a chamada deste mtodo veio internamente do
   * servidor.
   * 
   * @return true quando a chamda a este mtodo veio de uma thread interna do
   *         servidor. False quando a chamada  feita pelo cliente ou por outro
   *         servidor, ou seja, atravs dos proxies para os servios.
   */
  public static boolean isInternalServerRequest() {
    final Object localObject = threadUserKey.get();
    if (localObject instanceof SecureKey) {
      return false;
    }
    return true;
  }

  /**
   * Este mtodo retorna o usurio que realizou a chamada a um mtodo de um
   * servio. Para esse mecanismo funcionar, quando este mtodo  chamdo durante
   * a execuo de uma chamada remota,  preciso que cada objeto remoto
   * exportado pelo servidor chame o setKey antes de repassar a chamada para o
   * respectivo servio. Quando este mtodo  chamado a partir de outro servio,
   * para funcionar, o mtodo setUser deve ser chamado antes de fazer chamadas
   * para o servio.
   * 
   * @return o usurio registrado.
   */
  public static final User getUser() {
    final Object localObject = threadUserKey.get();
    if (localObject == null) {
      return null;
    }
    if (localObject instanceof SecureKey) {
      return LoginService.getInstance().getUserByKey(localObject);
    }
    try {
      return User.getUser(localObject);
    }
    catch (Exception e) {
      throw new ServiceFailureException(
        "Falha ao obter usurio " + localObject, e);
    }
  }

  /**
   * Esse mtodo s pode ser chamado durante a execuo de uma chamada remota.
   * Nessa situao, este mtodo retorna o usurio que realizou a chamada. Para
   * esse mecanismo funcionar,  preciso que cada objeto remoto exportado pelo
   * servidor chame o setKey antes de repassar a chamada para o respectivo
   * servio.
   * 
   * @return a chave solicitada.
   */
  public static final Object getKey() {
    return threadUserKey.get();
  }

  /**
   * Carregamento de bundles de internacionalizao do servio. Este mtodo
   * carrega o arquivo padro do servio e todos os arquivos adicionais
   * definidos pelas propriedades:
   * 
   * 1. {@link Service#DEFAULT_LAGUANGE_FILE_NAME}<br/>
   * 2. {@link Service#ADDITIONAL_LANGUAGE_FILE_PROPERTY}<br/>
   * 
   * Adicionalmente, o bundle do servidor tambm  carregado.
   * 
   * @param locale locale da traduo.
   * @return true se algum arquivo foi carregado, false caso contrrio.
   */
  private boolean loadLanguageBundle(Locale locale) {
    List<String> languageFiles = new ArrayList<String>();
    languageFiles.add(DEFAULT_LAGUANGE_FILE_NAME);
    languageFiles
      .addAll(getStringListProperty(ADDITIONAL_LANGUAGE_FILE_PROPERTY));

    boolean hasBundlesForService = false;
    HierachicalResourceBundle parent = null;

    for (String filePath : languageFiles) {
      String fileName = String.format("%s_%s.properties", filePath, locale);
      Class<?> thisClass = getClass();

      parent = Server.getInstance().getBundle(locale);
      if (parent != null) {
        hasBundlesForService = true;
        bundles.put(locale, parent);
      }

      /*
       * Carrega todos os resources bundle da classe corrente at a super-classe
       * Service. Retorna false se nenhum bundles for carregado.
       */
      while (thisClass != Service.class) {
        try (InputStream in = thisClass.getResourceAsStream(fileName)) {
          thisClass = thisClass.getSuperclass();
          if (in == null) {
            continue;
          }

          hasBundlesForService = true;

          HierachicalResourceBundle bundle = new HierachicalResourceBundle(in);
          bundles.put(locale, bundle);

          String logFormat = "Arquivo de bundle %s carregado.";
          Server.logInfoMessage(String.format(logFormat, fileName));

          if (parent != null) {
            bundle.setParent(parent);
          }
          parent = bundle;
        }
        catch (Exception e) {
          String logFormat = "Falha na leitura de: %s --> %s";
          String msg = e.getMessage();
          Server.logSevereMessage(String.format(logFormat, filePath, msg));

          logFormat = "Falha no bundle de: %s";
          Server.logSevereMessage(String.format(logFormat, locale));

          return hasBundlesForService;
        }
      }
    }
    return hasBundlesForService;
  }

  /**
   * Retorno do locale padro de internacionalizao.
   * 
   * @return o locale.
   */
  final public Locale getDefaultLocale() {
    final Server server = Server.getInstance();
    return server.getDefaultLocale();
  }

  /**
   * Consulta do locale da thread.
   * 
   * @return o locale ou <code>null</code>.
   */
  final protected Locale getThreadLocale() {
    final Object key = getKey();
    if (key == null) {
      return null;
    }
    final LoginService lgService = LoginService.getInstance();
    final Locale locale = lgService.getUserSessionLocale(key);
    return locale;
  }

  /**
   * Obtm uma determinada permisso para o usurio que realizou a chamada ao
   * mtodo no servio.
   * 
   * @param <T> Tipo de permisso.
   * 
   * @param permissionClass classe que representa a permisso.
   * @return uma instncia da permisso.
   */
  final protected <T extends Permission> T getUserPermission(
    Class<T> permissionClass) {
    return getUser().getPermission(permissionClass);
  }

  /**
   * Traduz uma determinada chave na sua string formatada com base no locale.
   * 
   * @param key chave do bundle.
   * @param locale o locale desejado para intercionalizao.
   * @param objects array de objetos de formatao.
   * @return string com o texto respectivo a chave.
   */

  final public String getFormattedString(final String key,
    final Object[] objects, final Locale locale) {
    final String txt = getString(key, locale);
    final String msg = MessageFormat.format(txt, objects);
    return msg;
  }

  /**
   * Traduz uma determinada chave na sua string formatada inferindo o locale do
   * usurio (thread) que originou a chamada (ser usado um locale padro se
   * houver falha nesta identificao).
   * 
   * @param key chave do bundle.
   * @param objects array de objetos de formatao.
   * @return string com o texto respectivo a chave.
   */
  final public String getFormattedString(final String key,
    final Object[] objects) {
    final String txt = getString(key);
    final String msg = MessageFormat.format(txt, objects);
    return msg;
  }

  /**
   * Traduz uma determinada chave na sua string inferindo o locale do usurio
   * (thread) que originou a chamada (ser usado um locale padro se houver
   * falha nesta identificao).
   * 
   * @param key chave do bundle.
   * @return String com o texto respectivo a chave.
   */
  final public String getString(final String key) {
    Locale locale = getThreadLocale();
    if (locale == null) {
      locale = getDefaultLocale();
    }
    return getString(key, locale);
  }

  /**
   * Traduz uma determinada chave na sua string inferindo o locale do usurio
   * (thread) que originou a chamada (ser usado um locale padro se houver
   * falha nesta identificao). Se a chave no estiver definida, retorna null.
   * Nos demais casos de erro, uma string de erro  retornada.
   * 
   * @param key chave do bundle.
   * @return String com o texto respectivo a chave.
   */
  final public String getOptionalString(final String key) {
    Locale locale = getThreadLocale();
    if (locale == null) {
      locale = getDefaultLocale();
    }
    return getOptionalString(key, locale);
  }

  /**
   * Traduz uma determinada chave na sua string correspondente de acordo com o
   * locale desejado.
   * 
   * @param key chave do bundle.
   * @param locale o locale desejado para intercionalizao.
   * @return String com o texto respectivo a chave.
   */
  final public String getString(final String key, final Locale locale) {
    String str = getOptionalString(key, locale);
    if (str != null) {
      return str;
    }
    return "<<<" + key + ":" + locale + ">>>";
  }

  /**
   * Traduz uma determinada chave na sua string correspondente de acordo com o
   * locale desejado. Se a chave no estiver definida, retorna null. Nos demais
   * casos de erro, uma string de erro  retornada.
   * 
   * @param key chave do bundle.
   * @param locale o locale desejado para intercionalizao.
   * @return String com o texto respectivo a chave.
   */
  final public String getOptionalString(final String key, final Locale locale) {
    if (key == null) {
      Server.logSevereMessage("Chave nula em consulta  internacionalizao");
      return "<<<null-key>>>";
    }
    if (locale == null) {
      Server
        .logSevereMessage("Locale nulo em consulta  internacionalizao da chave: "
          + key);
      return "<<<null-locale>>>";
    }
    if (!bundles.containsKey(locale)) {
      final boolean loaded = loadLanguageBundle(locale);
      if (!loaded) {
        return null;
      }
    }
    try {
      final ResourceBundle bnd = bundles.get(locale);
      return bnd.getString(key);
    }
    catch (MissingResourceException mre) {
      return null;
    }
    catch (Exception e) {
      final String txt = "Falha de internacionalizao";
      final String msg = e.getMessage();
      Server.logSevereMessage(txt + " [" + key + ":" + locale + "] - " + msg);
      return "<<<" + key + ":" + locale + ">>>";
    }
  }

  /**
   * Carregamento das propriedades do servio.
   * 
   * @param fileName o path para o arquivo de propriedades a ser carregado.
   * @return as propriedades do servio
   * @throws ServerException em caso de falha na carga do servio.
   */
  private ServerSideProperties loadProperties(final String fileName)
    throws ServerException {
    ServerSideProperties prop = new ServerSideProperties(fileName);
    prop.load();
    return prop;
  }

  /**
   * Retorna o nome prefixado da chave. Fora o uso de prefixo, mesmo que a
   * propriedade <code>"Server.service.prefixed.properties"</code> no esteja
   * ajustada para <code>true</code>.
   * 
   * @param key a chave
   * @return a chave com prefixo
   */
  final private String getPrefixedPropertyKey(final String key) {
    return getName() + "." + key;
  }

  /**
   * Faz checagem de propriedades corretamente carregadas.
   */
  private void checkPropertiesLoaded() {
    if (serviceProperties == null) {
      final String msg =
        "Falha interna de implementao do servio!\n\n"
          + "Propriedades do servio ainda no devidamente carregadas";
      throw new IllegalStateException(msg);
    }
  }

  /**
   * Verifica se uma propriedade foi definida como "no usada" (
   * <code>NULL</code>).
   * 
   * @param key a identificao da propriedade, sem o prefixo da classe
   * 
   * @return indicativo
   */
  final public boolean isPropertyNull(final String key) {
    checkPropertiesLoaded();
    final String prefixedKey = getPrefixedPropertyKey(key);

    // Verificando se a propriedade foi redefinida pelo servidor (Server).
    final Server server = Server.getInstance();
    if (server.overridesServiceProperty(prefixedKey)) {
      // Se no existir no servidor, ser levantada exceo!
      final String value = server.getStringServiceProperty(prefixedKey);
      return ServerSideProperties.isPropertyValueNull(value);
    }

    // Se no existir no arquivo do servio, ser levantada exceo!
    return serviceProperties.isPropertyNull(prefixedKey);
  }

  /**
   * Obtm um conjunto de propriedades com base em um arquivo definido pela
   * chave.
   * 
   * @param key o nome da propriedade, sem o prefixo da classe
   * @return o o conjunto da propriedade (que pode ser vazia, caso o arquivo no
   *         exista ou no esteja especificado).
   */
  protected final Properties getExternalPropertyFile(final String key) {
    final String filePath = getStringProperty(key);
    if (ServerSideProperties.isPropertyValueNull(filePath)) {
      return new Properties();
    }
    try {
      return PropertiesUtils.loadProperties(filePath);
    }
    catch (Exception e) {
      throw new ServiceFailureException("Erro na carga de propriedades de "
        + filePath, e);
    }
  }

  /**
   * Recupera uma propriedade da aplicao (lista). Caso a propriedade tenha
   * sido redefinida no arquivo de configuao do servidor, este valor ser
   * retornado.
   * 
   * @param key o nome da propriedade, sem o prefixo da classe e sem o sufixo
   *        ".1", ".2" etc.
   * @return os valores em forma de lista de strings.
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  public final List<String> getStringListProperty(final String key) {
    checkPropertiesLoaded();
    final Server server = Server.getInstance();
    final String prefixedKey = getPrefixedPropertyKey(key);
    final boolean overriden =
      server.overridesServiceProperty(prefixedKey.concat(".1"));
    if (overriden) {
      return server.getStringListServiceProperty(prefixedKey);
    }
    return serviceProperties.getStringListProperty(prefixedKey);
  }

  /**
   * Obtm uma propriedade do servio, como uma string. Caso a propriedade tenha
   * sido redefinida no arquivo de configuao do servidor, este valor ser
   * retornado.
   * 
   * @param key o nome da propriedade, sem o prefixo da classe
   * @return o valor da propriedade como uma string
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  public final String getStringProperty(final String key) {
    checkPropertiesLoaded();
    final Server server = Server.getInstance();
    final String prefixedKey = getPrefixedPropertyKey(key);
    final boolean overriden = server.overridesServiceProperty(prefixedKey);
    if (overriden) {
      return server.getStringServiceProperty(prefixedKey);
    }
    return serviceProperties.getStringProperty(prefixedKey);
  }

  /**
   * Recupera uma propriedade do servio, como um double. Caso a propriedade
   * tenha sido redefinida no arquivo de configuao do servidor, este valor
   * ser retornado.
   * 
   * @param key o nome da propriedade, sem o prefixo da classe
   * @return o valor como um double
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  public final double getDoubleProperty(final String key) {
    final Server server = Server.getInstance();
    final String prefixedKey = getPrefixedPropertyKey(key);
    final boolean overriden = server.overridesServiceProperty(prefixedKey);
    if (overriden) {
      return server.getDoubleServiceProperty(prefixedKey);
    }
    return serviceProperties.getDoubleProperty(prefixedKey);
  }

  /**
   * Recupera uma propriedade do servio, como um int. Caso a propriedade tenha
   * sido redefinida no arquivo de configuao do servidor, este valor ser
   * retornado.
   * 
   * @param key o nome da propriedade, sem o prefixo da classe
   * @return o valor como um inteiro
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final public int getIntProperty(final String key) {
    final Server server = Server.getInstance();
    final String prefixedKey = getPrefixedPropertyKey(key);
    final boolean overriden = server.overridesServiceProperty(prefixedKey);
    if (overriden) {
      return server.getIntServiceProperty(prefixedKey);
    }
    return serviceProperties.getIntProperty(prefixedKey);
  }

  /**
   * Recupera uma propriedade do servio, como um long. Caso a propriedade tenha
   * sido redefinida no arquivo de configuao do servidor, este valor ser
   * retornado.
   * 
   * @param key o nome da propriedade, sem o prefixo da classe
   * @return o valor como um long
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final public long getLongProperty(final String key) {
    final Server server = Server.getInstance();
    final String prefixedKey = getPrefixedPropertyKey(key);
    final boolean overriden = server.overridesServiceProperty(prefixedKey);
    if (overriden) {
      return server.getLongServiceProperty(prefixedKey);
    }
    return serviceProperties.getLongProperty(prefixedKey);
  }

  /**
   * Recupera uma propriedade do servio, como um boolean. Caso a propriedade
   * tenha sido redefinida no arquivo de configuao do servidor, este valor
   * ser retornado.
   * 
   * @param key o nome da propriedade, sem o prefixo da classe
   * @return o valor como um boolean
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final public boolean getBooleanProperty(final String key) {
    final Server server = Server.getInstance();
    final String prefixedKey = getPrefixedPropertyKey(key);
    final boolean overriden = server.overridesServiceProperty(prefixedKey);
    if (overriden) {
      return server.getBooleanServiceProperty(prefixedKey);
    }
    return serviceProperties.getBooleanProperty(prefixedKey);
  }

  /**
   * Retorna o valor default de uma propriedade, conforme definido no respectivo
   * arquivo de propriedades (no verifica sobrescrita).
   * 
   * @param key nome da propriedade, <b>incluindo o prefixo</b>
   * @return valor default da propriedade, ou <code>null</code> caso a chave no
   *         possua valor
   */
  final String getDefaultPropertyValue(String key) {
    return serviceProperties.hasProperty(key) ? serviceProperties
      .getStringProperty(key) : null;
  }

  /**
   * Obtm o formato de caminho adequado para o sistema operacional em uso, a
   * partir de 3 tipos possveis de formatos de propriedades indicando caminhos
   * para arquivos:
   * <ol>
   * <li>Formato Unix relativo: os separadores "/" sero substitudos por
   * <code>File.separator</code> (recomendado)</li>
   * <li>Formato Unix absoluto: o caminho ser retornado sem alteraes e
   * s funcionar em sistemas <i>UNIX-LIKE</i></li>
   * <li>Outros formatos: o caminho ser retornado sem alteraes e s 
   * garantido de funcionar no sistema para o qual foi criado</li>
   * </ol>
   * 
   * @param propertyPath caminho para um arquivo.
   * 
   * @return caminho no formato do S.O. corrente (usando
   *         <code>File.separator</code>) para caminhos Unix relativos; retorna
   *         o caminho original caso este esteja em outros formatos.
   */
  final protected String getOSPropertyPath(final String propertyPath) {
    final String UNIX_SEPARATOR = "/";
    if (propertyPath.indexOf(UNIX_SEPARATOR) < 1) {
      return propertyPath;
    }
    String[] splittedPath = FileUtils.splitPath(propertyPath, UNIX_SEPARATOR);
    String osPath = FileUtils.joinPath(splittedPath);
    if (propertyPath.lastIndexOf(UNIX_SEPARATOR) == propertyPath.length() - 1) {
      return osPath + File.separator;
    }
    return osPath;
  }

  /**
   * Informa o nome do servio.
   * 
   * @return o nome do servio.
   */
  @Override
  public final String getName() {
    return serviceName;
  }

  /**
   * Informa o nome de servio para ser usado como remetente no envio de
   * notificaes aos usurios.
   * 
   * @return O nome do servio a ser usado como remetente.
   */
  public String getSenderName() {
    String serviceName = getName();
    String text = getOptionalString(serviceName);
    if (text == null) {
      return serviceName;
    }
    Server server = Server.getInstance();
    String serverName = server.getSystemName();
    text = text.replaceAll("\\$SERVICE_ID", serviceName);
    text = text.replaceAll("\\$SERVER_NAME", serverName);
    return text;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isLoggingNotifications() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void logNotification(String msg) {
    Server.logInfoMessage(getName() + " " + msg);
  }

  /**
   * Mtodo para obteno de todos os servios.
   * 
   * @return Hashtable com as referncias para todos os servios.
   */
  public static Map<String, Service> getServices() {
    return ServiceManager.getInstance().getServices();
  }

  /**
   * Verifica se o servio est ativo.
   * 
   * @return um flag booleano indicativo.
   */
  @Override
  public boolean isActive() {
    return (this.isEnabled() && state.equals(ServiceState.INITED));
  }

  /**
   * Verifica se o servio est habilitado.
   * 
   * @return {@code true} caso o servio esteja habilitado, ou {@code false},
   *         caso contrrio.
   */
  public final boolean isEnabled() {
    return this.enabled;
  }

  /**
   * Define se o servio est habilitado.
   * 
   * @param enabled Indica se o servio deve ser habilitado.
   */
  protected final void setEnabled(boolean enabled) {
    this.enabled = enabled;
  }

  /**
   * Obtm a instncia do servio.
   * 
   * @param serviceName o nome do servio desejado.
   * 
   * @return a instncia.
   */
  public static Service getInstance(final String serviceName) {
    return ServiceManager.getInstance().getService(serviceName);
  }

  /**
   * Construtor protegido da classe <code>Servico</code> que se encarrega de
   * inicializar todas as funcionalidades de um servio (log, propriedades etc).
   * 
   * @param srvName nome no servio
   * 
   * @throws ServerException se houver erro na instanciao.
   */
  protected Service(final String srvName) throws ServerException {
    super(Server.getInstance().getDefaultLocale());
    serviceName = srvName;
    this.enabled = true;

    final String sep = File.separator;

    /* Carga de propriedades do servio */
    final String proDirName = Server.getPropertiesRootDirectoryName();
    final String proFileName = srvName + ".properties";
    final String servicePropertiesFileName = proDirName + sep + proFileName;
    serviceProperties = loadProperties(servicePropertiesFileName);

    this.state = ServiceState.CREATED;
    /* Adio automtica do servio ao Service Manager */
    final ServiceManager serviceManager = ServiceManager.getInstance();
    serviceManager.addService(this);
  }

  /**
   * Guarda nos dados locais da thread corrente, a string que identifica o
   * sistema onde o usurio da sesso atual originou a chamada remota. Este
   * mtodo  chamado quando a chamada de um mtodo do servio no  feito pelo
   * proxy de um cliente, mas quando  feito, por exemplo, por outros servios.
   * 
   * Um exemplo de atribuio a essa varivel, por exemplo, temos quando h uma
   * execuo de um comando do sistema atual, feita via o servio OpenDreams.
   * Nesse caso, o servio OpenDreams deve garantir que a atribuio do sistema
   * em que a chamada remota foi originada seja feita na thread do servio, para
   * futura checagem de permisso para realizar essa operao.
   * 
   * 
   * @param systemId o identificador (string) do sistema
   */
  public static void setSystemId(final String systemId) {
    threadSystemKey.set(systemId);
  }

  /**
   * Obtm a string que identifica o sistema onde o usurio da sesso atual
   * originou a chamada remota. Se a varivel de sistema no foi inicializada,
   * ento o valor retornado  null.
   * 
   * Para esse mecanismo funcionar, quando este mtodo  chamado a partir de
   * outro servio, o mtodo setSystemId deve ter sido chamado antes de fazer
   * chamadas para o servio.
   * 
   * @return o identificado (string) do sistema que originou a chamada remota,
   *         ou retorna null caso a varivel que indica o sistema no tenha sido
   *         atualizada
   */
  public static final String getSystemId() {
    return threadSystemKey.get();
  }

  /**
   * Obtm todos os servios dos quais este servio depende para a sua
   * inicializao.
   * 
   * @return Os servios dos quais este servio depende.
   */
  protected Service[] getInitializationDependencies() {
    return new Service[] {};
  }

  /**
   * Tenta inicializar o servio. Um servio no ser inicializado caso alguma
   * de suas dependncias de inicializao no esteja inicializada.
   * 
   * @return {@code true} caso o servio seja inicializado, ou {@code false},
   *         caso contrrio.
   * 
   * @throws ServerException Caso ocorra algum erro ao inicializar o servio.
   * 
   * @see #getInitializationDependencies()
   */
  final boolean tryInit() throws ServerException {
    Server.logFineMessage(String.format("Tentando inicializar o servio %s.",
      this.getName()));
    if (!this.isEnabled()) {
      throw new ServerException(String.format(
        "O servio %s no pode ser inicializado porque est desabilitado.",
        this.getName()));
    }
    Service[] dependencies = this.getInitializationDependencies();
    for (int i = 0; i < dependencies.length; i++) {
      if (dependencies[i].isEnabled()) {
        if (dependencies[i].getState() != ServiceState.INITED) {
          return false;
        }
      }
      else {
        Server
          .logWarningMessage(String
            .format(
              "O servio %s, que  uma dependncia do servio %s, no est habilitado",
              dependencies[i].getName(), this.getName()));
      }
    }
    this.init();
    return true;
  }

  /**
   * Inicializa o servio.
   * 
   * @throws ServerException Caso ocorra algum erro na inicializao do servio.
   */
  private final void init() throws ServerException {
    Server.logInfoMessage(String.format("Inicializando o servio %s",
      this.getName()));
    this.initService();
    this.state = ServiceState.INITED;
    Server.logInfoMessage(String.format(
      "O servio %s foi iniciado com sucesso", this.getName()));
  }

  /**
   * Mtodo abstrato de inicializao do servio.
   * 
   * @throws ServerException em caso de falha
   */
  protected abstract void initService() throws ServerException;

  /**
   * Termina o servio.
   * 
   * @throws ServerException Caso ocorra algum erro no trmino do servio.
   */
  final void shutdown() throws ServerException {
    Server.logInfoMessage(String.format("Terminando o servio %s",
      this.getName()));
    if (!this.isEnabled()) {
      throw new ServerException(String.format(
        "O servio %s no pode ser terminado porque est desabilitado.",
        this.getName()));
    }
    this.shutdownService();
    this.state = ServiceState.TERMINATED;
    Server.logInfoMessage(String.format(
      "O servio %s foi terminado com sucesso", this.getName()));
  }

  /**
   * Mtodo abstrato de trmino do servio
   * 
   * @throws ServerException em caso de falha.
   */
  protected abstract void shutdownService() throws ServerException;

  /**
   * Obtm o estado do servio.
   * 
   * @return O estado do servio.
   */
  final ServiceState getState() {
    return this.state;
  }

  /**
   * Retorna um conjunto ordenado com os nomes das propriedades definidas para o
   * servio.
   * 
   * @return conjunto ordenado com os nomes das propriedades definidas para o
   *         servio
   */
  SortedSet<Object> getPropertiesKeys() {
    return serviceProperties.getPropertiesKeys();
  }

  /**
   * Obtm as propriedades do servio sob a forma de um mapa.
   * 
   * @return mapa com as propriedades do servio
   */
  Map<String, String> getPropertiesMap() {
    return serviceProperties.getPropertiesMap();
  }

  /**
   * Incrementa um contador de um mapa que associa chaves a contadores.
   * 
   * @param <T> tipo das chaves do mapa
   * @param counter mapa de contadores
   * @param key chave a ter seu contador incrementado
   */
  protected <T> void incrCounter(Map<T, Integer> counter, T key) {
    Integer count = counter.get(key);
    if (count == null) {
      counter.put(key, 1);
    }
    else {
      counter.put(key, count.intValue() + 1);
    }
  }

  /**
   * Verifica se existe uma propriedade no servio com o nome passado como
   * parmetro.
   * 
   * @param propertyKey a chave da propriedade a ser verificada
   * @return {@code true} se existe uma propriedade no servio com a chave
   *         indicada ou {@code false} se no for encontrada uma propriedade com
   *         essa chave.
   */
  public boolean hasProperty(String propertyKey) {
    return serviceProperties.hasProperty(propertyKey);
  }
}
