package csbase.server.services.projectservice;

import java.util.Arrays;

import csbase.logic.ProjectFileInfo;

/**
 * Define um template de projeto a partir de um diretrio-base. O template define restries impostas a partir do
 * diretrio-base especificado.
 */
public class DefaultProjectTemplate extends BasicProjectTemplate {

  /**
   * Construtor.
   *
   * @param baseDir diretrio-base do template.
   * @param params parmetros para configurao do template.
   */
  public DefaultProjectTemplate(ProjectFileInfo baseDir, String... params) {
    super(baseDir, params);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canRename(ServerProjectFile file, String newName) {
    // No se pode renomear os diretrios-base do template
    return !isRestricted(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canDelete(ServerProjectFile file) {
    // No se pode apagar os diretrios-base do template
    return !isRestricted(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canCreate(ServerProjectFile parentDir, String name, String type) {
    // No h restrio para criao de arquivos.
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canChangeType(ServerProjectFile file, String newType) {
    // No se pode trocar o tipo dos diretrios-base do template
    return !isRestricted(file);
  }

  /**
   * Determina se o acesso ao arquivo deve ser restrito.
   *
   * @param file o arquivo.
   * @return verdadeiro caso o arquivo seja restrito ou falso, caso contrrio.
   */
  private boolean isRestricted(ServerProjectFile file) {
    boolean isTemplateDir = Arrays.equals(file.getPath(), getBaseDir().getPath());
    boolean isAncestor = isAncestor(file.getPath(), getBaseDir().getPath());
    return isTemplateDir || isAncestor;
  }


  /**
   * Indica se um caminho  ancestral do outro.
   *
   * @param ancestorPath   O caminho potencialmente ancestral.
   * @param descendantPath O caminho potencialmente descendente.
   * @return Verdadeiro se ancestorPath for ancestral de descendantPath, falso
   * caso contrrio.
   */
  public static boolean isAncestor(String[] ancestorPath, String[] descendantPath) {
    if (ancestorPath == null || descendantPath == null) {
      return false;
    }
    if (ancestorPath.length >= descendantPath.length) {
      return false;
    }
    for (int i = 0; i < ancestorPath.length; i++) {
      if (!ancestorPath[i].equals(descendantPath[i])) {
        return false;
      }
    }
    return true;
  }

}
