/*
 * $Id$
 */

package csbase.server.services.notificationservice;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;

import csbase.logic.Notification;

/**
 * Classe interna de armazenamento de notificaes no Servio de Notificaes.
 * 
 * @see NotificationService
 * @author Tecgraf/PUC-Rio
 */
class NotificationContainer implements Serializable {

  /**
   * Tabela de mensagens de usurios
   */
  final private Hashtable<Object, ArrayList<Notification>> userHash =
    new Hashtable<Object, ArrayList<Notification>>();

  /**
   * Dump do objeto.
   * 
   * @param delta o intervalo de tempo em (ms).
   * 
   * @return um indicativo de sucesso.
   */
  final synchronized boolean clean(final long delta) {
    return cleanNotifications(delta);
  }

  /**
   * Limpeza em uma lista de notificaes.
   * 
   * @param delta um lag de time para limpeza.
   * 
   * @return um indicativo de sucesso na limpeza..
   */
  private boolean cleanNotifications(final long delta) {
    for (final Enumeration<ArrayList<Notification>> e = userHash.elements(); e
      .hasMoreElements();) {
      final ArrayList<Notification> v = e.nextElement();
      if (!cleanVector(v, delta)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Limpeza em uma lista de notificaes.
   * 
   * @param vector vetor de notificaes.
   * @param delta tempo aps o qual as mensagens sero apagadas.
   * @return um flag indicativo da correta limpeza.
   */
  private boolean cleanVector(final ArrayList<Notification> vector,
    final long delta) {
    final long now = (new Date().getTime());
    int i = 0;
    while (i < vector.size()) {
      final Notification notif = vector.get(i);
      final long ntime = notif.getCreationDate();
      if (notif.isVolatile() || Math.abs(now - ntime) > delta) {
        vector.remove(i);
      }
      else {
        i++;
      }
    }
    return true;
  }

  /**
   * Dump do objeto.
   * 
   * @param file arquivo de sada.
   * 
   * @return um indicativo de sucesso.
   */
  final synchronized boolean dumpContainer(final File file) {
    try {
      final FileOutputStream fStream = new FileOutputStream(file);
      final BufferedOutputStream bStream = new BufferedOutputStream(fStream);
      final ObjectOutputStream outStream = new ObjectOutputStream(bStream);
      outStream.writeObject(this);
      outStream.flush();
      outStream.close();
      return true;
    }
    catch (final Exception e) {
      return false;
    }
  }

  /**
   * Insero em uma lista de notificaes.
   * 
   * @param key a chave de busca.
   * @param notif a notificao a ser inserida.
   * 
   * @return um indicativo de sucesso.
   */
  private boolean insertNotification(final Object key, final Notification notif) {
    ArrayList<Notification> vector = userHash.get(key);
    if (vector == null) {
      final ArrayList<Notification> newVector = new ArrayList<Notification>();
      userHash.put(key, newVector);
      vector = newVector;
    }
    return vector.add(notif);
  }

  /**
   * Insero de notificao de usurio.
   * 
   * @param userId o identificador do usurio.
   * @param notif a notificao.
   * 
   * @return um indicativo de sucesso.
   */
  final synchronized boolean insertUserNotification(final Object userId,
    final Notification notif) {
    return insertNotification(userId, notif);
  }

  /**
   * Busca em uma lista de notificaes.
   * 
   * @param key a chave de busca.
   * 
   * @return um vetor de notificaes.
   */
  private ArrayList<Notification> retrieveNotifications(final Object key) {
    return userHash.remove(key);
  }

  /**
   * Remoo de notificao de usurio.
   * 
   * @param userId o identificador do usurio.
   * 
   * @return um vetor de notificaes.
   */
  final synchronized ArrayList<Notification> retrieveUserNotifications(
    final Object userId) {
    return retrieveNotifications(userId);
  }

  /**
   * Verifica se existem notificaes para um usurio.
   * 
   * @param userId identificador do usurio
   * @return <code>true</code> se existem notificaes para o usurio
   */
  final synchronized boolean userHasNotifications(Object userId) {
    ArrayList<Notification> notifications = userHash.get(userId);
    return notifications != null && !notifications.isEmpty();
  }

  /**
   * Construtor simples
   */
  NotificationContainer() {
  }
}
