/**
 * $Id$
 */
package csbase.server.services.projectservice;

import java.lang.ref.WeakReference;

import csbase.logic.SecureKey;
import csbase.server.services.loginservice.LoginService;

/**
 * Locks exclusivos. Apenas um lock exclusivo  permitido para cada arquivo.
 */
class ExclusiveFileLock implements FileLockInterface {
  /**
   * Identificador do lock.  usado para identificar que o desbloqueio  feito
   * por quem pediu o lock.
   */
  Object id;

  /**
   * Indica se o lock foi solicitado por um servio, a partir de uma
   * <code>Thread</code> do servidor. Quando o lock  obtido por uma
   * <code>Thread</code> que no vem diretamente do cliente, no h sesso
   * associada.
   */
  private boolean serverRequested;

  /** Referncia para a sesso do usurio que obteve o lock. */
  private WeakReference<SecureKey> sessionKeyRef;

  /**
   * Cria o lock, associando uma sesso a ele.
   * 
   * @param skey sesso associada ao lock. Permite <code>null</code> quando no
   *        houver sesso associada
   * @param id identificador do lock
   */
  ExclusiveFileLock(SecureKey skey, Object id) {
    this.id = id;
    if (skey == null) {
      this.serverRequested = true;
    }
    else {
      this.sessionKeyRef = new WeakReference<SecureKey>(skey);
      this.serverRequested = false;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getLockRefCount(SecureKey sessionKey) {
    if (serverRequested) {
      if (sessionKey == null) {
        return 1;
      }
    }
    else {
      if (sessionKeyRef != null) {
        SecureKey sKey = sessionKeyRef.get();
        if ((sKey != null) && (sKey.equals(sessionKey))) {
          return 1;
        }
      }
    }
    return 0;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getLockRefCount() {
    if (hasExpired()) {
      return 0;
    }
    return 1;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasExpired() {
    if (serverRequested) {
      return false;
    }
    if (sessionKeyRef == null) {
      return true;
    }
    Object sessionKey = sessionKeyRef.get();
    return sessionKey == null
      || LoginService.getInstance().getUserByKey(sessionKey) == null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public LockStatus checkLockStatus(SecureKey sessionKey) {
    return checkLockStatus(sessionKey.toString());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean newLocker(SecureKey skey, Object lockId) {
    if (!hasExpired()) {
      return false;
    }
    this.id = lockId;
    if (skey == null) {
      serverRequested = true;
    }
    else {
      sessionKeyRef = new WeakReference<SecureKey>(skey);
      serverRequested = false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeLocker(Object lockId) {
    if (!id.equals(lockId)) {
      return false;
    }
    serverRequested = false;
    sessionKeyRef = null;
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isShared() {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public LockStatus checkLockStatus(String sessionKey) {
    if (serverRequested) {
      if (sessionKey == null) {
        return LockStatus.LOCKED_BY_USER;
      }
    }
    else {
      if (sessionKeyRef == null) {
        return LockStatus.UNLOCKED;
      }
      SecureKey ref = sessionKeyRef.get();
      if (ref == null) {
        return LockStatus.EXPIRED;
      }
      if (ref.toString().equals(sessionKey)) {
        return LockStatus.LOCKED_BY_USER;
      }
    }
    return LockStatus.LOCKED_BY_OTHERS;
  }

  /**
   * Obtm o identificador do lock
   * 
   * @return o identificador do lock
   */
  Object getId() {
    return id;
  }

}
