/*
 * ProjectAdministrator.java
 * 
 * $Id$
 */
package csbase.server.services.projectservice;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import csbase.exception.ServiceFailureException;
import csbase.logic.CommonProjectInfo;
import csbase.logic.ProjectAdminInfo;
import csbase.logic.ProjectAllocationState;
import csbase.server.Server;

/**
 * Modela a administrao dos projetos dos usurios. Os projetos podem estar nos
 * seguintes estados: - Desbloqueado - Bloqueado - Removido Os projetos podem
 * ter ou no pedido de reserva de rea de alocao de disco (mount). O
 * administrador pode ou no liberar a rea solicitada e, liberando, deve
 * informar quanto de rea foi alocada. Quando um projeto  removido, no caso
 * dele possuir uma rea alocada (mount), o administrador deve indicar que a
 * rea foi liberada.
 * 
 * @author Tecgraf/PUC-Rio
 */
class ProjectAdministrator {

  /**
   * Nome do arquivo de controle de projetos com alocao.
   */
  static final String LOCKED_PROJECTS_FILE_NAME = "locked_projects.csbase";

  /**
   * Caractere que ser utilizado para separar os campos de cada entrada do
   * arquivo locked_projects.csbase (que persiste as informaes dos objetos
   * ProjectAdminInfo).
   */
  private static final String SEPARATOR = ";";

  /**
   * Guarda a instncia do servio de projetos.
   */
  private static ProjectService service;

  /**
   * Guarda as informaes de projetos bloqueados. Mapeia userId (Object) em
   * informao de um projeto (ProjectAdminInfo).
   * 
   * Todos os mtodos que incluem / removem / alteram entradas desta hashtable
   * devem ser synchronized e persistir o resultado, atravs do mtodo
   * {@link ProjectAdministrator#saveLockedProjects(boolean)}.
   */
  private Hashtable<Object, Hashtable<Object, ProjectAdminInfo>> lockedProjects =
    new Hashtable<Object, Hashtable<Object, ProjectAdminInfo>>();

  /**
   * Finaliza o administrador.
   */
  void finish() {
    saveLockedProjects(true);
  }

  /**
   * Obtm as informaes de todos os projetos dos usurios.
   * 
   * @return um array de <code>ProjectAdminInfo</code>.
   */
  ProjectAdminInfo[] getAllProjectAdminInfo() {
    final List<ProjectAdminInfo> projs = new ArrayList<ProjectAdminInfo>();
    for (Map<Object, ProjectAdminInfo> map : lockedProjects.values()) {
      Collection<ProjectAdminInfo> projects = map.values();
      for (ProjectAdminInfo projectAdminInfo : projects) {
        /*
         * Verifica se o projeto ainda existe, pois podem haver entradas
         * invlidas na lista de projetos.
         */
        if (isValidProject(projectAdminInfo)) {
          projs.add(projectAdminInfo);
        }
      }
    }
    return (projs.toArray(new ProjectAdminInfo[0]));
  }

  /**
   * Obtm a informao de um projeto de um usurio.
   * 
   * @param projectId O identificador do projeto.
   * 
   * @return Um <code>ProjectAdminInfo</code> do projeto de um usurio ou nulo
   *         caso o projeto no esteja bloqueado.
   */
  ProjectAdminInfo getProjectAdminInfo(Object projectId) {
    Object ownerId = ServerProject.getOwnerId(projectId);
    Hashtable<Object, ProjectAdminInfo> userProjects =
      lockedProjects.get(ownerId);
    if (userProjects == null) {
      return null;
    }
    ProjectAdminInfo info = userProjects.get(projectId);
    return info;
  }

  /**
   * Remove a informao de um projeto de um usurio.
   * 
   * @param projectId O identificador do projeto.
   * 
   * @return um <code>ProjectAdminInfo</code> do projeto de um usurio.
   */
  synchronized ProjectAdminInfo removeProjectAdminInfo(Object projectId) {
    Object ownerId = ServerProject.getOwnerId(projectId);
    final Hashtable<Object, ProjectAdminInfo> userProjects =
      lockedProjects.get(ownerId);
    if (userProjects == null) {
      return null;
    }
    final ProjectAdminInfo info = userProjects.remove(projectId);
    saveLockedProjects(false);
    return info;
  }

  /**
   * Bloqueia um projeto por reserva de rea.
   * 
   * @param projectId O identificador do projeto.
   * @param size a rea requisitada
   * 
   * @return A informao de administrao criada para o novo projeto.
   */
  synchronized ProjectAdminInfo lockProject(Object projectId, final long size) {
    String projectName = ServerProject.getProjectName(projectId);
    Object ownerId = ServerProject.getOwnerId(projectId);
    final ProjectAdminInfo info =
      new ProjectAdminInfo(projectId, projectName, ownerId, service.getRoot(
        projectId).getAbsolutePath(),
        ProjectAllocationState.WAITING_AREA_ALLOCATION);
    info.setAreaRequestSize(size);
    info.setAreaRequestDate(Calendar.getInstance().getTime());
    Hashtable<Object, ProjectAdminInfo> userProjects =
      lockedProjects.get(ownerId);
    if (userProjects == null) {
      userProjects = new Hashtable<Object, ProjectAdminInfo>();
      lockedProjects.put(ownerId, userProjects);
    }
    userProjects.put(projectId, info);
    saveLockedProjects(false);
    return info;
  }

  /**
   * Desbloqueia um projeto devido a alocao de uma rea.
   * 
   * @param projectId O identificador do projeto.
   * @param size Tamanho da rea reservada.
   * @return A informao de administrao de projeto.
   */
  synchronized ProjectAdminInfo unlockProject(Object projectId, long size) {
    final ProjectAdminInfo info = getProjectAdminInfo(projectId);
    if (info == null) {
      throw new ServiceFailureException("ProjectAdministrator:unlockProject: "
        + "no existe informao administrativa do projeto " + projectId);
    }
    info.setState(ProjectAllocationState.UNLOCKED_WITH_AREA_ALLOCATED);
    info.setAreaLockedSize(size);
    saveLockedProjects(false);
    return info;
  }

  /**
   * Retira a informao de administrao de um projeto removido que teve sua
   * rea liberada.
   * 
   * @param projectId O identificador do projeto.
   * @return A informao de administrao de projeto.
   */
  ProjectAdminInfo freeProjectArea(Object projectId) {
    final ProjectAdminInfo info = removeProjectAdminInfo(projectId);
    if (info == null) {
      throw new ServiceFailureException("ProjectAdministrator:freeProject: "
        + "no existe informao administrativa do projeto " + projectId);
    }
    saveLockedProjects(false);
    return info;
  }

  /**
   * Informa que um projeto foi removido e aguarda liberao da rea reservada.
   * 
   * @param projectId O identificador do projeto
   * @return O objeto ProjectAdminInfo modificado.
   */
  synchronized ProjectAdminInfo setAllocatedProjectRemoved(Object projectId) {
    final ProjectAdminInfo info = getProjectAdminInfo(projectId);
    if (info == null) {
      throw new ServiceFailureException("ProjectAdministrator:removedProject: "
        + "no existe informao administrativa do projeto " + projectId);
    }
    info.setState(ProjectAllocationState.WAITING_AREA_FREE);
    saveLockedProjects(false);
    return info;
  }

  /**
   * Verifica se um projeto est bloqueado.
   * 
   * @param projectId O identificador do projeto.
   * 
   * @return Verdadeiro, se o projeto est bloqueado ou falso, caso contrrio.
   */
  boolean isLocked(final Object projectId) {
    final ProjectAdminInfo info = getProjectAdminInfo(projectId);
    if (info == null) {
      return false;
    }
    return info.isLocked();
  }

  /**
   * Verifica se um projeto est desbloqueado utilizando rea reservada.
   * 
   * @param projectId O identificador do projeto.
   * @return verdadeiro, se o projeto est desbloqueado utilizando rea
   *         reservada ou falso, caso contrrio.
   */
  boolean isUnlockedWithAreaAllocated(Object projectId) {
    ProjectAdminInfo info = getProjectAdminInfo(projectId);
    if (info == null) {
      return false;
    }
    return info.isUnlockedWithAreaAllocated();
  }

  /**
   * Inicializa as informaes sobre projetos bloqueados.
   */
  private synchronized void loadLockedProjects() {
    String lockFileName = getLockFileName();
    try (BufferedReader in = new BufferedReader(new FileReader(lockFileName))) {
      String nextLine = null;
      String[] tokens = null;
      while ((nextLine = in.readLine()) != null) {
        tokens = nextLine.split(SEPARATOR);
        Object projectId = tokens[0];
        int stateInt = Integer.parseInt(tokens[1]);
        final ProjectAllocationState state =
          ProjectAllocationState.getProjectAllocationStateFromCode(stateInt);

        long reqSize = Long.parseLong(tokens[2]);
        long reqDate = Long.parseLong(tokens[3]);
        long lockSize = Long.parseLong(tokens[4]);

        String absolutePath;
        if (state != ProjectAllocationState.WAITING_AREA_FREE) {
          /*
           * Verifica se o arquivo de configurao do projeto  consistente.
           *
           * TODO - Ver issue [CSBASE-2741].
           */
          try {
            final String path = ServerProject.getAbsolutePath(projectId);
            final File pathFile = new File(path);
            final File configFile = ServerProject.getConfigFile(pathFile);
            final Object ownerId = ServerProject.getOwnerId(projectId);
            final CommonProjectInfo info =
              ServerProject.readProjectInfoFromConfigFile(configFile, ownerId);
            if (info == null) {
              continue;
            }
            final File root = service.getRoot(projectId);
            absolutePath = root.getAbsolutePath();
          }
          catch (Exception e) {
            /*
             * Trata possveis excees para permitir que a leitura do arquivo
             * de projetos bloqueados continue mesmo quando h uma entrada
             * invlida.
             */
            Server.logWarningMessage(
              "ProjectAdministrator: no foi possvel encontrar o projeto " +
                projectId);
            absolutePath = null;
          }
        }
        else {
          absolutePath = null;
        }
        String projectName = ServerProject.getProjectName(projectId);
        Object userId = ServerProject.getOwnerId(projectId);
        ProjectAdminInfo info =
          new ProjectAdminInfo(projectId, projectName, userId, absolutePath,
            state);
        info.setAreaRequestSize(reqSize);
        info.setAreaRequestDate(new Date(reqDate));
        info.setAreaLockedSize(lockSize);
        Hashtable<Object, ProjectAdminInfo> userProjects =
          lockedProjects.get(userId);
        if (userProjects == null) {
          userProjects = new Hashtable<Object, ProjectAdminInfo>();
          lockedProjects.put(userId, userProjects);
        }
        userProjects.put(projectId, info);
      }
    }
    catch (FileNotFoundException e) {
      Server.logWarningMessage(
        "ProjectAdministrator: arquivo " + lockFileName + " no existe");
    }
    catch (IOException e) {
      Server.logSevereMessage(
        "ProjectAdministrator: erro de leitura no " + "arquivo " + lockFileName,
        e);
    }
  }

  /**
   * Guarda em arquivo todas as informaes de projetos bloqueados.
   * 
   * @param cleanInvalidProjects verdadeiro se devemos apagar as entradas
   *        invlidas (que referenciam projetos que no existem mais no
   *        repositrio).
   */
  private synchronized void saveLockedProjects(boolean cleanInvalidProjects) {
    try (BufferedWriter out = new BufferedWriter(
      new FileWriter(getLockFileName()))) {
      for (Enumeration<Object> keys = lockedProjects.keys(); keys
        .hasMoreElements(); ) {
        Object userId = keys.nextElement();
        Hashtable<Object, ProjectAdminInfo> projects =
          lockedProjects.get(userId);
        if (projects == null) {
          continue;
        }
        for (Enumeration<Object> ids = projects.keys(); ids
          .hasMoreElements(); ) {
          Object projectId = ids.nextElement();
          ProjectAdminInfo info = projects.get(projectId);
          if (info == null) {
            continue;
          }
          if (cleanInvalidProjects && !isValidProject(info)) {
            // No salva a entrada com projeto invlido
            continue;
          }
          Date date = info.getAreaRequestDate();
          String dateText = (date == null) ? "" : String.valueOf(date.getTime());
          out.write(projectId.toString());
          out.write(SEPARATOR);
          out.write(String.valueOf(info.getState().getCode()));
          out.write(SEPARATOR);
          out.write(String.valueOf(info.getAreaRequestSize()));
          out.write(SEPARATOR);
          out.write(dateText);
          out.write(SEPARATOR);
          out.write(String.valueOf(info.getAreaLockedSize()));
          out.write(SEPARATOR);
          out.newLine();
        }
      }
    }
    catch (IOException e) {
      Server
        .logSevereMessage("Erro ao gravar dados de administrao dos projetos.",
          e);
    }
  }

  /**
   * Verifica se um projeto  vlido ou no, a partir do seu id.
   * 
   * OBS. Um projeto que tinha rea alocada e foi removido no ser retornado
   * pelo mtodo existsProject() do ProjectService, mas tem uma entrada vlida
   * referente a alocao de rea (est aguardando desalocao de rea).
   * 
   * @param info Informaes administrativas do projeto.
   * @return Boolean indicando se um projeto  vlido ou no.
   */
  private static boolean isValidProject(final ProjectAdminInfo info) {
    return service.existsProject(info.getProjectId())
      || info.isWaitingAreaFree();
  }

  /**
   * Retorna o nome do arquivo que guarda as informaes de projetos bloqueados.
   * 
   * @return o nome do arquivo.
   */
  private String getLockFileName() {
    return ProjectService.getInstance().getProjectRepositoryPath()
      + File.separator + LOCKED_PROJECTS_FILE_NAME;
  }

  /**
   * Construtor.
   * 
   * @param service O servio de projetos.
   */
  ProjectAdministrator(ProjectService service) {
    ProjectAdministrator.service = service;
    loadLockedProjects();
  }
}
