package csbase.server.services.messageservice;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InvalidClassException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;

import csbase.server.Server;
import csbase.util.messages.MessageStore;
import csbase.util.messages.dao.IMessageStoreDAO;

/**
 * Classe responsvel por persistir os tpicos.
 * 
 * @author Tecgraf
 */
public class MessageStoreDAO implements IMessageStoreDAO {

  /**
   * Arquivo de persistncia dos tpicos.
   */
  private File backup;

  /**
   * Construtor.
   * 
   * @param file Arquivo de persistncia dos tpicos.
   */
  public MessageStoreDAO(File file) {
    this.backup = file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized Collection<MessageStore> getAllMessageStores() {
    final String backupFilePath = backup.getAbsolutePath();
    Server.logInfoMessage("Iniciando recuperao de backup de: "
      + backupFilePath);

    if (!backup.exists()) {
      final String msg = "No foi encontrado arquivo de backup de mensagens: ";
      Server.logWarningMessage(msg + backupFilePath);
      return Collections.emptyList();
    }

    ObjectInputStream in = null;
    try {
      in =
        new ObjectInputStream(new BufferedInputStream(new FileInputStream(
          backup)));
      @SuppressWarnings("unchecked")
      Collection<MessageStore> stores =
        (Collection<MessageStore>) in.readObject();
      Server.logInfoMessage("Mensagens recuperadas de: " + backupFilePath);
      return stores;
    }
    catch (InvalidClassException ce) {
      String err = "No foi possvel fazer recuperao de backup. ";
      String cause = "Verso do tpico no compatvel com arquivo: ";
      Server.logWarningMessage(err + cause + backupFilePath);
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha de recuperao de backup: "
        + backupFilePath, e);
    }
    finally {
      if (in != null) {
        closeCloseable(in);
      }
    }
    return Collections.emptyList();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized <T extends Collection<MessageStore> & Serializable> void saveAllMessageStores(
    T stores) {
    if (stores == null) {
      throw new NullPointerException("stores == null");
    }

    FileOutputStream fStream = null;
    ObjectOutputStream outStream = null;
    try {
      fStream = new FileOutputStream(backup);
      outStream = new ObjectOutputStream(fStream);
      outStream.writeObject(stores);
      outStream.flush();
    }
    catch (IOException e) {
      Server.logSevereMessage("Falha ao escrever no arquivo de backup: "
        + backup.getAbsolutePath(), e);
    }
    finally {
      closeCloseable(outStream);
      closeCloseable(fStream);
    }
  }

  /**
   * Mtodo para facilitar o fechamento de um {@link Closeable};
   * 
   * @param closeable Objecto a ser fechado.
   */
  private void closeCloseable(Closeable closeable) {
    if (closeable != null) {
      try {
        closeable.close();
      }
      catch (IOException e) {
        Server.logWarningMessage(e.getMessage());
      }
    }
  }
}
