/*
 * $Id$
 */
package csbase.server.services.projectservice;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import csbase.exception.ServiceFailureException;
import csbase.logic.FileUpdateNotification;
import csbase.logic.FileUpdateNotification.Result;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.loginservice.LoginService;
import csbase.server.services.loginservice.LoginServiceListener;
import csbase.server.services.notificationservice.NotificationService;

/**
 * Responsvel por agendar atualizaes de arquivos de projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
final class ProjectFileUpdateScheduler {
  /**
   * Os responsveis pela atualizao propriamente dita (atualizadores).
   */
  private Map<Class<? extends ProjectFileUpdater>, ProjectFileUpdater> updaters;
  /**
   * Responsvel por agendar as tarefas de atualizaco de arquivos.
   */
  private Timer scheduler;
  /**
   * As tarefas de atualizaco de arquivos.
   */
  private Map<UpdatableFileLocation, TimerTask> tasks;
  /**
   * As atualizaes de arquivo que esto agendadas.
   */
  private Map<UpdatableFileLocation, FileUpdateInfo> updates;
  /**
   * Trata notificaes de <i>logout</i> dos usurios para poder finalizar as
   * suas atualizaes.
   */
  private LoginServiceListenerImpl loginServiceListener;

  /**
   * Cria o agendador de atualizaes.
   * 
   * @param updaters A lista de nomes de classes de atualizadores.
   */
  ProjectFileUpdateScheduler(List<String> updaters) {
    this.updaters =
      new HashMap<Class<? extends ProjectFileUpdater>, ProjectFileUpdater>();
    for (String updaterClassName : updaters) {
      try {
        Class<? extends ProjectFileUpdater> updaterClass =
          (Class<? extends ProjectFileUpdater>) Class.forName(updaterClassName);
        try {
          this.updaters.put(updaterClass, updaterClass.newInstance());
        }
        catch (InstantiationException e) {
          Server.logSevereMessage(String.format(
            "Falha ao instanciar o atualizador de arquivos da classe {0}",
            new Object[] { updaterClassName }), e);
        }
        catch (IllegalAccessException e) {
          Server.logSevereMessage(String.format(
            "Falha ao instanciar o atualizador de arquivos da classe {0}",
            new Object[] { updaterClassName }), e);
        }
      }
      catch (ClassNotFoundException e) {
        Server.logSevereMessage(String.format(
          "No foi possvel carregar o atualizador de arquivos da classe {0}",
          new Object[] { updaterClassName }), e);
      }
    }
    this.scheduler = new Timer(true);
    this.tasks = new HashMap<UpdatableFileLocation, TimerTask>();
    this.updates = new HashMap<UpdatableFileLocation, FileUpdateInfo>();
    this.loginServiceListener = new LoginServiceListenerImpl();
  }

  /**
   * Inicializa o agendador de atualizaes.
   */
  public void init() {
    LoginService.getInstance().addListener(this.loginServiceListener);
  }

  /**
   * Finaliza o agendador de atualizaes.
   */
  public void shutdown() {
    LoginService.getInstance().removeListener(this.loginServiceListener);
    this.scheduler.cancel();
  }

  /**
   * Obtm a localizao de um arquivo atualizvel.
   * 
   * @param spf O arquivo atualizvel.
   * 
   * @return A localizao do arquivo atualizvel.
   */
  public UpdatableFileLocation getFileLocation(ServerProjectFile spf) {
    return new UpdatableFileLocation(spf.getProjectId(), spf.getPath());
  }

  /**
   * Obtm informaes sobre a atualizao de um arquivo.
   * 
   * @param fileLocation A localizao do arquivo.
   * 
   * @return Informaes sobre a atualizao do arquivo.
   */
  public FileUpdateInfo getFileUpdateInfo(UpdatableFileLocation fileLocation) {
    return this.updates.get(fileLocation);
  }

  /**
   * Solicita a atualizao de um arquivo.<br>
   * OBS: este procedimento  assncrono.
   * 
   * @param spf O arquivo.
   */
  void singleUpdate(ServerProjectFile spf) {
    this.startUpdate(spf, -1, false);
  }

  /**
   * Faz o agendamento da atualizao de um arquivo.
   * 
   * @param spf O arquivo.
   * @param interval O intervalo (em segundos) de atualizaes sucessivas do
   *        arquivo.
   * @param notification Indica se uma notificao deve ser enviada ao usurio
   *        ao trmino da atualizao.
   */
  void startUpdate(ServerProjectFile spf, long interval, boolean notification) {
    UpdatableFileInfo info = spf.getUpdatableFileInfo();
    Class<? extends ProjectFileUpdater> updaterClass;
    try {
      updaterClass = info.getServiceClass();

      // Verificar se a classe responsvel pela atualizao do arquivo est configurada na propriedade ProjectService.updater.class
      if (this.updaters.get(updaterClass) == null) {
        Server
          .logSevereMessage(MessageFormat
            .format(
              "A atualizao do arquivo {0} no pode ser agendada pois a classe do atualizador, {1}, no est definida na propriedade ProjectService.updater.class.",
              new Object[] { spf.getName(), updaterClass.getName() }));
        return;
      }
    }
    catch (ClassNotFoundException e) {
      throw new ServiceFailureException(
        "A classe de um atualizador de arquivo no foi encontrada.", e);
    }
    UpdatableFileLocation fileLocation =
      new UpdatableFileLocation(spf.getProjectId(), spf.getPath());
    TimerTask task;
    synchronized (this.tasks) {
      task = this.tasks.get(fileLocation);
      if (task != null) {
        task.cancel();
        this.scheduler.purge();
      }
      task =
        new UpdaterTimerTask(spf.getProjectId(), fileLocation, this.updaters
          .get(updaterClass), info, notification);
      this.tasks.put(fileLocation, task);
      this.updates.put(fileLocation, new FileUpdateInfo(Service.getUser()
        .getLogin(), interval));
      this.loginServiceListener.addFileLocation(fileLocation);
    }
    if (interval > 0) {
      this.scheduler.schedule(task, interval * 1000, interval * 1000);
    }
    else {
      this.scheduler.schedule(task, 0);
    }
  }

  /**
   * ra a atualizao de um arquivo.
   * 
   * @param spf O arquivo.
   */
  void stopUpdate(ServerProjectFile spf) {
    synchronized (this.tasks) {
      this.stopUpdate(new UpdatableFileLocation(spf.getProjectId(), spf
        .getPath()));
    }
  }

  /**
   * Pra a atualizao de um arquivo.
   * 
   * @param path O caminho para o arquivo cuja atualizao foi solicitada.
   */
  void stopUpdate(UpdatableFileLocation path) {
    synchronized (this.tasks) {
      this.loginServiceListener.removeFileLocation(path);
      this.updates.remove(path);
      TimerTask task = this.tasks.remove(path);
      if (task == null) {
        return;
      }
      task.cancel();
      this.scheduler.purge();
    }
  }

  /**
   * /** Pra a atualizao de um conjunto de arquivos.
   * 
   * @param fileLocationSet O conjunto de caminhos dos arquivos cuja atualizao
   *        foi solicitada.
   */
  void stopUpdate(Set<UpdatableFileLocation> fileLocationSet) {
    synchronized (this.tasks) {
      for (UpdatableFileLocation filePath : fileLocationSet) {
        this.stopUpdate(filePath);
      }
    }
  }

  /**
   * Tarefa responsvel pela atualizao de um arquivo.
   * 
   * @author Tecgraf/PUC-Rio
   * 
   */
  private class UpdaterTimerTask extends TimerTask {
    /**
     * A chave do usurio que solicitou a atualizao.
     */
    private Object key;
    /**
     * O login do usurio que solicitou a atualizao.
     */
    private String userLogin;
    /**
     * O identificador do projeto ao qual o arquivo pertence.
     */
    private Object projectId;
    /**
     * A localizao do arquivo.
     */
    private UpdatableFileLocation fileLocation;
    /**
     * A classe responsvel por atualizar efetivamente o arquivo.
     */
    private ProjectFileUpdater updater;
    /**
     * Informaes sobre o arquivo a ser atualizado.
     */
    private UpdatableFileInfo fileInfo;
    /**
     * O path para o arquivo.
     */
    private String fileLocationPath;
    /**
     * Indica se uma notificao deve ser enviada ao usurio ao trmino da
     * atualizao.
     */
    private boolean notification;

    /**
     * Cria a tarefa para atualizar um arquivo.
     * 
     * @param projectId O identificador do projeto ao qual o arquivo pertence.
     * @param fileLocation A localizao do arquivo.
     * @param updater A classe responsvel por atualizar efetivamente o arquivo.
     * @param fileInfo Informaes sobre o arquivo a ser atualizado.
     * @param notification Indica se uma notificao deve ser enviada ao usurio
     *        ao trmino da atualizao.
     */
    UpdaterTimerTask(Object projectId, UpdatableFileLocation fileLocation,
      ProjectFileUpdater updater, UpdatableFileInfo fileInfo,
      boolean notification) {
      this.key = Service.getKey();
      this.userLogin = Service.getUser().getLogin();
      this.projectId = projectId;
      this.fileLocation = fileLocation;
      this.updater = updater;
      this.fileInfo = fileInfo;
      this.notification = notification;

      StringBuilder filePathBuilder = new StringBuilder();
      String[] filePath = this.fileLocation.getPath();
      filePathBuilder.append(filePath[0]);
      for (int i = 1; i < filePath.length; i++) {
        filePathBuilder.append("/");
        filePathBuilder.append(filePath[i]);
      }
      this.fileLocationPath = filePathBuilder.toString();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      try {
        Service.setKey(this.key);
        if (ProjectService.getInstance().existsFile(this.projectId,
          this.fileLocation.getPath())) {
          Result result =
            this.updater.update(fileLocation, fileInfo.getUserData(), fileInfo
              .getExtraUserData());
          if (this.notification) {
            Object[] ids = new Object[] { this.userLogin };
            NotificationService.getInstance().notifyTo(
              ids,
              new FileUpdateNotification(ProjectService.getInstance()
                .getSenderName(), this.fileLocationPath, result));
          }
        }
        else {
          this.cancel();
        }
      }
      finally {
        Service.setKey(null);
      }
    }
  }

  /**
   * Tratador de notificaes do Servio de Login.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private class LoginServiceListenerImpl implements LoginServiceListener {
    /**
     * Informa os arquivos utilizados por um determinado usurio em uma sesso.
     */
    private Map<Object, Set<UpdatableFileLocation>> usersToFileLocationSet;
    /**
     * Informa o usurio responsvel pela atualizao de um arquivo.
     */
    private Map<UpdatableFileLocation, Object> fileLocationToUser;

    /**
     * Cria um tratador de notificaes.
     */
    LoginServiceListenerImpl() {
      this.usersToFileLocationSet =
        new HashMap<Object, Set<UpdatableFileLocation>>();
      this.fileLocationToUser = new HashMap<UpdatableFileLocation, Object>();
    }

    /**
     * Adiciona um arquivo  lista do usurio corrente.
     * 
     * @param fileLocation O caminho do arquivo.
     */
    void addFileLocation(UpdatableFileLocation fileLocation) {
      Object sessionKey = Service.getKey();
      if (sessionKey == null) {
        return;
      }
      Set<UpdatableFileLocation> fileLocationSet =
        this.usersToFileLocationSet.get(sessionKey);
      if (fileLocationSet == null) {
        fileLocationSet = new HashSet<UpdatableFileLocation>();
      }
      fileLocationSet.add(fileLocation);
      this.usersToFileLocationSet.put(sessionKey, fileLocationSet);
      this.fileLocationToUser.put(fileLocation, sessionKey);
    }

    /**
     * Remove um arquivo da lista do usurio corrente.
     * 
     * @param fileLocation O caminho do arquivo.
     */
    void removeFileLocation(UpdatableFileLocation fileLocation) {
      Object sessionKey = this.fileLocationToUser.remove(fileLocation);
      if (sessionKey == null) {
        return;
      }
      Set<UpdatableFileLocation> fileLocationSet =
        this.usersToFileLocationSet.get(sessionKey);
      fileLocationSet.remove(fileLocation);
      if (fileLocationSet.size() == 0) {
        this.usersToFileLocationSet.remove(sessionKey);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void userLoggingOut(String login, Object sessionKey,
      String systemName, long time) {
      Set<UpdatableFileLocation> fileLocationSet =
        this.usersToFileLocationSet.get(sessionKey);
      if (fileLocationSet != null) {
        stopUpdate(new HashSet<UpdatableFileLocation>(fileLocationSet));
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void systemNameSet(String login, Object sessionKey,
      String systemName, long time) {
      // Nada a ser feito.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void sessionCreated(String login, Object sessionKey, long time) {
      // Nada a ser feito.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void connectionLost(String login, Object sessionKey,
      String systemName, long time) {
      // Nada a ser feito.
    }
  }
}
