/**
 * $Id: LoginService.java 181929 2018-10-01 20:15:07Z parthur $
 */

package csbase.server.services.loginservice;

import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.security.DigestException;
import java.security.KeyPair;
import java.security.PublicKey;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.Timer;
import java.util.TimerTask;
import java.util.TreeMap;
import java.util.Vector;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

import com.novell.ldap.LDAPConnection;
import com.novell.ldap.LDAPConstraints;
import com.novell.ldap.LDAPException;

import csbase.exception.CSBaseException;
import csbase.exception.OperationFailureException;
import csbase.exception.ParseException;
import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.logic.AdministrationEvent;
import csbase.logic.EncryptedPassword;
import csbase.logic.LoginAsPermission;
import csbase.logic.LoginPasswordCipher;
import csbase.logic.MDigest;
import csbase.logic.Permission;
import csbase.logic.PreLoginData;
import csbase.logic.SecureKey;
import csbase.logic.ServerURI;
import csbase.logic.Session;
import csbase.logic.User;
import csbase.logic.UserOutline;
import csbase.logic.openbus.OpenBusLoginToken;
import csbase.logic.server.ServerInfo;
import csbase.remote.ServerEntryPoint;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.keystore.CSKeyStore;
import csbase.server.services.administrationservice.AdministrationService;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.server.services.restservice.RestService;
import csbase.server.services.serverservice.ServerService;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;
import csbase.util.messages.filters.BodyTypeFilter;

/**
 * A classe <code>LoginService</code> implementa o servio de login de um
 * usurio.  responsvel pela autenticao do usurio no sistema e
 * gerenciamento da sesso.
 * <p>
 * Esse servio pode ser observado para eventos de logout.
 * </p>
 *
 * @author Tecgraf/PUC-Rio
 *
 * @see csbase.server.services.loginservice.LogoutEvent
 */
public class LoginService extends Service {
  /** Nome do servio para o <code>ServiceManager</code> */
  public static final String SERVICE_NAME = "LoginService";

  /** Nome da propriedade que indica o mtodo de autenticao. */
  public static final String LOGIN_METHOD = "loginMethod";
  /** Valor que indica que o mtodo de autenticao usar senhas prprias. */
  public static final String LOCAL_LOGIN_METHOD = "LOCAL";
  /** Valor que indica que o mtodo de autenticao usar servidores LDAP. */
  public static final String LDAP_LOGIN_METHOD = "LDAP";
  /**
   * Nome da propriedade que indica o tempo para expirao de uma solicitao de
   * pr-login.
   */
  public static final String PRE_LOGIN_DELAY = "preLoginDelay";
  /** Propriedade que armazena o timeout para a conexo com o servidor. */
  private static final String LDAP_TIMEOUT = "LDAPConnectionTimeout";
  /** Porta padro para acesso a servidores LDAP. */
  public static final int LDAP_DEFAULT_PORT = 389;
  /** Marca a ser substituda no padro LDAP pelo usurio. */
  public static final String LDAP_PATTERN_USER = "[%U]";
  /**
   * Propriedade que armazena o nmero mximo de sessoes simultneas para um
   * mesmo usurio (0=infinito)
   */
  public static final String MAX_SIMULTANEOUS_SESSIONS =
    "maxUserSimultaneousSessions";

  /**
   * Nome do atributo de sesso que representa dados de um usurio que foi
   * logado no sistema por um sistema automador.
   */
  private static final String USER_DATA = "USER_DATA";

  /** 1 Minuto em millisegundos. */
  private static final long MINUTE_IN_MS = 1000 * 60;

  /**
   * Guarda a sesso dos usurios que esto logados no servidor. A chave  da
   * classe <code>SecureKey</code>. A estrutura  a seguinte: chave - hash de
   * propriedades da sesso. Ex: { (chave1 - { (USER - user1), (LOCALE -
   * locale) } ) }
   */
  protected Hashtable<SecureKey, ServerSession> loggedUsers =
    new Hashtable<SecureKey, ServerSession>();

  /**
   * Atributo que indica o usurio real que est fazendo o login para um outro
   * usurio (sudo)
   */
  public static final String REAL_USER_ATTRIBUTE = "realUser";

  /**
   * Bean para agir como chave para contar o nmero de sesses abertas pelo
   * mesmo login, considerando a delegao.
   */
  private static final class SessionLimiterKey {
    /** Login do super-usurio */
    private final String superUser;
    /** Login do usurio */
    private final String user;

    /**
     * Construtor.
     *
     * @param user o usurio
     */
    public SessionLimiterKey(User user) {
      if (user == null) {
        throw new IllegalArgumentException("login no pode ser nulo!");
      }
      this.superUser = user.getSuperUserLogin();
      this.user = user.getLogin();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result =
        prime * result + ((superUser == null) ? 0 : superUser.hashCode());
      result = prime * result + ((user == null) ? 0 : user.hashCode());
      return result;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj) {
      if (this == obj) {
        return true;
      }
      if (obj == null) {
        return false;
      }
      if (getClass() != obj.getClass()) {
        return false;
      }
      SessionLimiterKey other = (SessionLimiterKey) obj;
      if (superUser == null) {
        if (other.superUser != null) {
          return false;
        }
      }
      else if (!superUser.equals(other.superUser)) {
        return false;
      }
      if (user == null) {
        if (other.user != null) {
          return false;
        }
      }
      else if (!user.equals(other.user)) {
        return false;
      }
      return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
      if (superUser != null) {
        return superUser + ">" + user;
      }
      else {
        return user;
      }
    }
  }

  /**
   * Contador de sesses abertas por usurio, levando em conta o caso de
   * delegao de login. Em
   * {@link LoginService#createSession(SecureKey, User, Locale, Map, Map, Remote)}
   * os contadores so incrementados e confrontados com a propriedade definida
   * por {@link LoginService#MAX_SIMULTANEOUS_SESSIONS}. Se o valor no for zero
   * (infinito), lana exceo caso o nmero mximo de sesses tenha sido
   * atingido.
   */
  private final Map<SessionLimiterKey, AtomicInteger> sessionCounter =
    new ConcurrentHashMap<SessionLimiterKey, AtomicInteger>();

  /**
   * Nmero mximo de sesses simultneas abertas pelo mesmo usurio,
   * considerando delegao. Este valor  obtido da propriedade
   * {@link #MAX_SIMULTANEOUS_SESSIONS}. Se for zero, no impe nenhum limite.
   */
  private int maxSimultaneousSessions;

  /**
   * Essa tabela armazena as referncia para os usurios que realizaram uma
   * operao de preLogin para posterior login.
   *
   * @see #preLogin(String, String, Locale)
   * @see #login(String)
   */
  protected Hashtable<String, PreLoginInfo> preLoggedUsers =
    new Hashtable<String, PreLoginInfo>();

  /** Indica se o mtodo de autencao  local ou via LDAP. */
  private boolean localLogin;

  /** Lista dos servidores LDAP disponveis para autenticao. */
  private List<String> LDAPServers;

  /**
   * Porta do servio LDAP em cada servidor disponvel para autenticao.
   */
  private Vector<Integer> LDAPPorts;

  /** Lista de padres disponveis para autenticao dos usurios. */
  private List<String> LDAPPatternList;

  /** CharSet utilizado na converso da senha em um byte[]. */
  private String LDAPCharSet;

  /**
   * Tempo, em millisegundos, para verificao se um preLogin foi confirmado.
   */
  private long preLoginDelay;

  /** Timer para programao de tasks de limpeza. Roda como daemon. */
  private Timer timer;

  /** Conjunto de listeners do servio. */
  private final Set<LoginServiceListener> listeners;

  /**
   * Contador de logins bem-sucedidos por usurio.
   */
  private final Map<String, Integer> loginCounter =
    new TreeMap<String, Integer>();

  /**
   * Contador de falhas de autenticao por usurio.
   */
  private final Map<String, Integer> failedLoginCounter =
    new TreeMap<String, Integer>();

  /**
   * Identificador do servio no MessageService.
   */
  private Serializable consumerId;

  /**
   * Par de chaves pblica e privada para criptografar e descriptografar a senha
   * do login.
   */
  private KeyPair loginPasswordKeyPair;

  /**
   * Constri a instncia do servio.
   *
   * @throws ServerException caso ocorra um problema na instanciao.
   */
  public static void createService() throws ServerException {
    new LoginService();
  }

  /**
   * Obtm a instncia <i>Singleton</i> do servio de login.
   *
   * @return instncia <i>Singleton</i> do servio de login.
   */
  public static LoginService getInstance() {
    return (LoginService) getInstance(SERVICE_NAME);
  }

  /**
   * Constri a instncia do servio. Construtor protegido para garantir a
   * unicidade do <i>Singleton</i>.
   *
   * @throws ServerException em caso de erro na construo.
   */
  protected LoginService() throws ServerException {
    super(SERVICE_NAME);
    this.listeners = new HashSet<LoginServiceListener>();
    try {
      this.loginPasswordKeyPair = LoginPasswordCipher.generateKeyPair();
    }
    catch (Exception e) {
      throw new ServerException(
        "Erro ao gerar o par de chaves pblica e privada para criptografar e descriptografar a senha.",
        e);
    }
  }

  /**
   * Retorna a chave pblica para criptografar a senha do login.
   *
   * @return chave pblica para criptografia da senha do login.
   */
  public PublicKey getPublicKey() {
    return this.loginPasswordKeyPair.getPublic();
  }

  /**
   * Adiciona um listener ao servio.
   *
   * @param listener O listener a ser adicionado.
   * @throws IllegalArgumentException Caso o listener esteja nulo.
   */
  public synchronized void addListener(LoginServiceListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O listener no pode ser nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * Remove um listener do servio.
   *
   * @param listener O listener a ser removido.
   * @return true, caso o listener tenha sido removido, ou false, caso o
   * listener no esteja cadastrado.
   * @throws IllegalArgumentException Caso o listener esteja nulo.
   */
  public synchronized boolean removeListener(LoginServiceListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O listener no pode ser nulo.");
    }
    return this.listeners.remove(listener);
  }

  /**
   * Notifica todos listeners que uma sesso foi criada.
   *
   * @param login Login do usurio
   * @param sessionKey Chave da sesso do usurio
   * @param time Hora em que ocorreu o evento representado em milisegundos
   */
  protected void fireSessionCreated(String login, Object sessionKey,
    long time) {
    for (LoginServiceListener listener : listeners) {
      listener.sessionCreated(login, sessionKey, time);
    }
  }

  /**
   * Notifica todos listeners que um usurio est efetando um logout.
   *
   * @param login Login do usurio
   * @param sessionKey Chave da sesso do usurio
   * @param systemName Nome do sistema definido pelo usurio.
   * @param time Hora em que ocorreu o evento representado em milisegundos
   */
  protected void fireUserLoggingOut(String login, Object sessionKey,
    String systemName, long time) {
    for (LoginServiceListener listener : listeners) {
      listener.userLoggingOut(login, sessionKey, systemName, time);
    }
  }

  /**
   * Notifica todos listeners que a conexo com um usurio foi perdida
   *
   * @param login Login do usurio que perdeu a conexo
   * @param sessionKey Chave da sesso do usurio
   * @param systemName Nome do sistema definido pelo usurio ou
   * <code>NULL</code> se no tiver sido definido.
   * @param time Hora em que ocorreu o evento representado em milisegundos
   */
  protected void fireConnectionLost(String login, Object sessionKey,
    String systemName, long time) {
    for (LoginServiceListener listener : listeners) {
      listener.connectionLost(login, sessionKey, systemName, time);
    }
  }

  /**
   * Notifica todos listeners que o nome do sistema foi definido
   *
   * @param login Login do usurio
   * @param sessionKey Chave da sesso do usurio
   * @param systemName Nome do sistema definido pelo usurio
   * @param time Hora em que ocorreu o evento representado em milisegundos
   */
  protected void fireSystemNameSet(String login, Object sessionKey,
    String systemName, long time) {
    for (LoginServiceListener listener : listeners) {
      listener.systemNameSet(login, sessionKey, systemName, time);
    }
  }

  /**
   * Inicializao do servio.
   *
   * @throws ServerException se ocorrer um erro na inicializao
   */
  @Override
  public void initService() throws ServerException {
    timer = new Timer(true);
    loadAuthenticationProperties();
    try {
      install();
    }
    catch (RemoteException re) {
      throw new ServerException(re);
    }
    maxSimultaneousSessions = getIntProperty(MAX_SIMULTANEOUS_SESSIONS);
  }

  /**
   * Carrega as propriedades que configuram o login dos usurios. O login pode
   * ser feito atravs de um servio LDAP ou atravs de senhas prprias do
   * sistema, cujo digest  guardado no arquivo de definio do usurio.
   *
   * @throws ServerException
   */
  private void loadAuthenticationProperties() throws ServerException {
    String loginMethod = getStringProperty(LOGIN_METHOD);
    if (!loginMethod.equals(LOCAL_LOGIN_METHOD) && !loginMethod
      .equals(LDAP_LOGIN_METHOD)) {
      throw new ServerException(
        String.format("Mtodo de autenticao %s desconhecido!", loginMethod));
    }
    preLoginDelay = getIntProperty(PRE_LOGIN_DELAY) * MINUTE_IN_MS;
    if (loginMethod.equals(LOCAL_LOGIN_METHOD)) {
      localLogin = true;
      Server
        .logWarningMessage("Mtodo de autenticao configurado para LOCAL.");
    }
    else {
      localLogin = false;
      LDAPServers = getStringListProperty("LDAPServer");
      int size = LDAPServers.size();
      if (size == 0) {
        throw new ServerException(
          "Nenhum servidor LDAP definido para autenticao!");
      }
      LDAPPorts = new Vector<Integer>();
      for (int i = 1; i <= LDAPServers.size(); i++) {
        LDAPPorts.add(getIntProperty("LDAPPort." + i));
      }
      this.LDAPPatternList = getStringListProperty("LDAPPattern");
      LDAPCharSet = getStringProperty("LDAPCharSet");
      Server.logInfoMessage("Mtodo de autenticao configurado para LDAP.");
    }
  }

  /**
   * Registra no servidor um observador.
   *
   * @throws RemoteException Em caso de falha.
   */
  public void install() throws RemoteException {
    IMessageListener listener = new IMessageListener() {
      @Override
      public void onMessagesReceived(Message... messages) throws Exception {
        for (Message message : messages) {
          AdministrationEvent event = (AdministrationEvent) message.getBody();
          if (!(event.item instanceof User)) {
            return;
          }
          User targetUser = (User) event.item;
          Vector<SecureKey> logoutUsers = new Vector<SecureKey>();
          for (Enumeration<SecureKey> e = loggedUsers.keys(); e
            .hasMoreElements(); ) {
            SecureKey key = e.nextElement();
            ServerSession session = loggedUsers.get(key);
            if (session == null) {
              return;
            }
            User user = session.getUser();
            if (!user.getId().equals(targetUser.getId())) {
              continue;
            }
            switch (event.type) {
              case AdministrationEvent.MODIFY:
                session.setUser(targetUser);
                break;
              case AdministrationEvent.DELETE:
                logoutUsers.add(key);
                break;
            }
          }
          for (int i = 0; i < logoutUsers.size(); i++) {
            logout(logoutUsers.get(i));
          }
        }
      }

    };
    consumerId = MessageService.getInstance().setServerMessageListener(listener,
      new BodyTypeFilter(AdministrationEvent.class));
  }

  ;

  /**
   * Trmino do servio.
   *
   * @throws ServerException caso ocorra um problema na finalizao.
   */
  @Override
  public void shutdownService() throws ServerException {
    try {
      ReferedServerCache.getInstance().clear();
      uninstall();
    }
    catch (RemoteException re) {
      throw new ServerException(re);
    }
    if (timer != null) {
      timer.cancel();
    }
  }

  /**
   * Remove do servidor o registro desse observador.
   *
   * @throws RemoteException Em caso de falha.
   */
  public void uninstall() throws RemoteException {
    MessageService.getInstance().clearServerMessageListener(consumerId);
  }

  /**
   * Todos observadores sero sempre notificados.
   *
   * @param arg Argumento do Observador
   * @param event Evento gerado para notificar o observador
   * @return true
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return true;
  }

  /**
   * Informa se o servidor aceite troca de senhas. A troca pode no ser aceita
   * quando o mtodo de autenticao utilizado pelo servidor acesse um servidor
   * externo. Atualmente, este  o caso da autenticao via LDAP.
   *
   * @return verdadeiro caso o servidor aceite trocar senhas, falso caso
   *         contrrio.
   */
  /**
   * Indica se o servidor permite a troca de senhas.
   *
   * @return <code>true</code> se a alterao de senhas for permitida,
   * <code>false</code> caso contrrio.
   */
  public boolean canChangePasswords() {
    return localLogin;
  }

  /**
   * Indica se o mtodo de autenticao do servidor  local ou no.
   *
   * @return <code>true</code> se os usurios forem autenticados localmente,
   * <code>false</code> caso contrrio.
   */
  public boolean isLocalLogin() {
    return localLogin;
  }

  /**
   * Autentica um usurio no sistema e cria um token para ser usado como
   * identificao em um posterior login onde no h a informao de usurio e
   * senha. Esse token pode ser utilizado como parmetro em endereos urls.
   *
   * @param login O identificador do usurio que deseja fazer seu login.
   * @param password A senha do respectivo usurio.
   * @param locale A Locale que o usurio deseja utilizar no posterior login.
   * @return Uma string que  o token para identificao de um prelogin vlido.
   * <code>null</code> se o login e/ou senha forem invlidos.
   * @see #login(String)
   */
  public PreLoginData preLogin(String login, String password, Locale locale) {
    return this
      .preLogin(login, password, locale, (Map<String, Serializable>) null);
  }

  /**
   * Autentica um usurio no sistema e cria um token para ser usado como
   * identificao em um posterior login onde no h a informao de usurio e
   * senha. Esse token pode ser utilizado como parmetro em endereos urls.
   *
   * @param login O identificador do usurio que deseja fazer seu login.
   * @param password A senha do respectivo usurio.
   * @param locale A Locale que o usurio deseja utilizar no posterior login.
   * @param attributes Os atributos da sesso do usurio.
   * @return Uma string que  o token para identificao de um prelogin vlido.
   * <code>null</code> se o login e/ou senha forem invlidos.
   * @throws IllegalArgumentException se <code>login</code> for nulo ou
   * <code>password</code> for nulo ou <code>locale</code> for nulo.
   * @see #login(String)
   */
  public PreLoginData preLogin(String login, String password, Locale locale,
    Map<String, Serializable> attributes) {
    if (login == null || password == null || locale == null) {
      throw new IllegalArgumentException(
        "(login || password || locale) == null");
    }
    User user = checkLogin(login, password, attributes);
    if (user == null) {
      return null;
    }
    return doPreLogin(user, locale, attributes, null, null);
  }

  /**
   * Autentica um usurio no sistema e cria um token para ser usado como
   * identificao em um posterior login onde no h a informao de usurio e
   * senha. Esse token pode ser utilizado como parmetro em endereos urls.
   *
   * @param restToken O token de autenticao REST.
   * @param locale A Locale que o usurio deseja utilizar no posterior login.
   * @param attributes Os atributos da sesso do usurio.
   * @return Uma string que  o token para identificao de um prelogin vlido.
   * <code>null</code> se o login e/ou senha forem invlidos.
   * @throws IllegalArgumentException se <code>restToken</code> for nulo ou
   * <code>locale</code> for nulo.
   * @see #login(String)
   */
  public PreLoginData preLogin(String restToken, Locale locale,
    Map<String, Serializable> attributes) {
    if (restToken == null || locale == null) {
      throw new IllegalArgumentException("(restToken || locale) == null");
    }
    // Loga via token REST.
    String userId;
    try {
      userId = RestService.getInstance().parserToken(restToken, null);
    }
    catch (ParseException e) {
      return null;
    }
    if (userId == null) {
      return null;
    }
    AdministrationService adminService = AdministrationService.getInstance();
    User user = adminService.getUser(userId);
    if (user == null) {
      return null;
    }
    // Gera novo token, usado para identificar o cliente
    return doPreLogin(user, locale, attributes, null, null);
  }

  /**
   * <p>
   * Autentica um usurio no sistema e cria um token para ser usado como
   * identificao em um posterior login onde no h a informao de usurio e
   * senha.
   * </p>
   *
   * <p>
   * O usurio  autenticado por um super-usurio, que  um usurio com
   * permisso de super-usurio({@link csbase.logic.SuperUserPermission}).
   * Autentica e cria no servidor uma sesso do usurio delegado.
   * </p>
   *
   * <p>
   * OBS: O Administrador(admin) no precisa de permisso de super-usurio para
   * delegar um outro usurio e, por questes de segurana, no pode ser um
   * usurio delegado por nenhum super-usurio.
   * </p>
   *
   * @param login o identificador do super-usurio.
   * @param password a senha do super-usurio.
   * @param locale A Locale que ser usado no login do usurio delegado.
   * @param delegatedLogin login do usurio que ser delegado pelo
   * super-usurio
   * sem a necessidade de senha.
   * @param control Objeto de controle.
   * @param userData Identificao do usurio.
   * @return Dados sobe o <i>preLogin</i> realizado.
   */
  public PreLoginData preLogin(String login, String password, Locale locale,
    String delegatedLogin, Remote control, Serializable userData) {
    User user = checkLogin(login, password);
    if (user == null) {
      return null;
    }
    if (delegatedLogin != null) {
      // Faz a delegao.
      user =
        AdministrationService.getInstance().changeUser(user, delegatedLogin);
      user.setSuperUserLogin(login);
      Server.logInfoMessage(String.format(
        "Usurio %s foi delegado pelo usurio %s.", delegatedLogin, login));
    }
    Map<String, Serializable> serverAttributes =
      new HashMap<String, Serializable>();
    serverAttributes.put(USER_DATA, userData);
    return doPreLogin(user, locale, null, serverAttributes, control);
  }

  /**
   * <p>
   * Autentica um usurio no sistema e cria um token para ser usado como
   * identificao em um posterior login onde no h a informao de usurio e
   * senha.
   * </p>
   *
   * <p>
   * O usurio  autenticado por um super-usurio, que  um usurio com
   * permisso de super-usurio({@link csbase.logic.SuperUserPermission}).
   * Autentica e cria no servidor uma sesso do usurio delegado.
   * </p>
   *
   * <p>
   * OBS: O Administrador(admin) no precisa de permisso de super-usurio para
   * delegar um outro usurio e, por questes de segurana, no pode ser um
   * usurio delegado por nenhum super-usurio.
   * </p>
   *
   * @param login o identificador do super-usurio.
   * @param password a senha do super-usurio.
   * @param locale A Locale que ser usado no login do usurio delegado.
   * @param delegatedLogin login do usurio que ser delegado pelo
   * super-usurio
   * sem a necessidade de senha.
   * @return Dados sobe o <i>preLogin</i> realizado.
   */
  public PreLoginData preLogin(String login, String password, Locale locale,
    String delegatedLogin) {
    User user = checkLogin(login, password);
    if (user == null) {
      return null;
    }
    User delegatedUser =
      AdministrationService.getInstance().changeUser(user, delegatedLogin);
    if (delegatedUser != null) {
      user = delegatedUser;
      user.setSuperUserLogin(login);
    }
    Map<String, Serializable> serverAttributes =
      new HashMap<String, Serializable>();
    return doPreLogin(user, locale, null, serverAttributes, null);
  }

  /**
   * Efetua um {@code preLogin}.
   *
   * @param user O usurio que est realizando o {@code preLogin}.
   * @param locale A <i>locale</i> do usurio.
   * @param attributes Atributos-extra que sero armazenados na sesso do
   * usurio.
   * @param serverAttributes Atributos que sero armazenados na sesso do
   * usurio, mas que no so enviadas ao cliente.
   * @param control Objeto de controle
   * @return Dados sobe o <i>preLogin</i> realizado.
   */
  private PreLoginData doPreLogin(User user, Locale locale,
    Map<String, Serializable> attributes,
    Map<String, Serializable> serverAttributes, Remote control) {
    String token;
    try {
      token = new SecureKey().digest();
    }
    catch (DigestException e) {
      Server.logSevereMessage(String
        .format("Erro ao gerar digest para criao do token do usurio %s.",
          user.getLogin()), e);
      throw new ServiceFailureException("Erro ao gerar token para pr-login.");
    }
    SecureKey sessionKey = new SecureKey();
    CheckPreLoginTask task = new CheckPreLoginTask(token);
    PreLoginInfo info =
      new PreLoginInfo(sessionKey, user, locale, attributes, serverAttributes,
        control);
    preLoggedUsers.put(token, info);
    timer.schedule(task, preLoginDelay);
    Server.logInfoMessage(String
      .format("Prelogin de %s para o locale %s. Token: %s",
        user.getLogin() + getRealUserForLog(attributes), locale, token));
    return new PreLoginData(sessionKey, token);
  }

  /**
   * Executa o login de um usurio via login e senha. Autentica e cria no
   * servidor uma sesso do usurio logado. Pode ser utilizado para se logar a
   * primeira vez ou aps uma queda do servidor.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param login o identificador do usurio que deseja fazer seu login.
   * @param encryptedPassword a senha do respectivo usurio.
   * @param locale A Locale que o usurio escolheu no login.
   * @return A sesso do usurio. <code>null</code> se chave e/ou senha foram
   * invlidos
   * @see #logout(Object)
   */
  public Session login(String login, EncryptedPassword encryptedPassword,
    Locale locale) {
    return login(login, encryptedPassword, locale, null, null, null);
  }

  /**
   * Executa o login de um usurio via login e senha. Autentica e cria no
   * servidor uma sesso do usurio logado. Pode ser utilizado para se logar a
   * primeira vez ou aps uma queda do servidor. Armazena na sesso o
   * {@link TimeZone} do usurio.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param login o identificador do usurio que deseja fazer seu login.
   * @param encryptedPassword a senha do respectivo usurio.
   * @param locale A Locale que o usurio escolheu no login.
   * @param timeZone TimeZone do cliente pelo qual o usurio se logou.
   * @return A sesso do usurio. <code>null</code> se chave e/ou senha foram
   * invlidos
   * @see #logout(Object)
   */
  public Session login(String login, EncryptedPassword encryptedPassword,
    Locale locale, TimeZone timeZone) {
    return login(login, encryptedPassword, locale, null, timeZone, null);
  }

  /**
   * Executa o login de um usurio via login e senha com a possibilidade de
   * delegar o login a um outro usurio, para isso o usurio dever ser um
   * super-usurio do sistema que significa ter permisso de super-usurio(
   * {@link csbase.logic.SuperUserPermission}). Autentica e cria no servidor uma
   * sesso do usurio delegado. Pode ser utilizado para se logar a primeira vez
   * ou aps uma queda do servidor. OBS: O Administrador(admin) no precisa de
   * permisso de super-usurio para delegar um outro usurio e, por questes de
   * segurana, no pode ser um usurio delegado por nenhum super-usurio.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param login o identificador do super-usurio.
   * @param encryptedPassword a senha do super-usurio.
   * @param locale A Locale que ser usado no login do usurio delegado.
   * @param delegatedLogin login do usurio que ser delegado pelo
   * super-usurio
   * sem a necessidade de senha.
   * @return A sesso do usurio. <code>null</code> se chave e/ou senha foram
   * invlidos
   * @throws IllegalArgumentException se <code>login</code> for nulo ou
   * <code>password</code> for nulo ou <code>locale</code> for nulo.
   * @see csbase.logic.SuperUserPermission
   * @see #logout(Object)
   */
  public Session login(String login, EncryptedPassword encryptedPassword,
    Locale locale, String delegatedLogin) {
    return login(login, encryptedPassword, locale, delegatedLogin, null, null);
  }

  /**
   * Executa o login de um usurio via login e senha com a possibilidade de
   * delegar o login a um outro usurio, para isso o usurio dever ser um
   * super-usurio do sistema que significa ter permisso de super-usurio(
   * {@link csbase.logic.SuperUserPermission}). Autentica e cria no servidor uma
   * sesso do usurio delegado. Pode ser utilizado para se logar a primeira vez
   * ou aps uma queda do servidor. OBS: O Administrador(admin) no precisa de
   * permisso de super-usurio para delegar um outro usurio e, por questes de
   * segurana, no pode ser um usurio delegado por nenhum super-usurio.
   * Armazena na sesso o {@link TimeZone} do usurio.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param login o identificador do super-usurio.
   * @param encryptedPassword a senha do super-usurio.
   * @param locale A Locale que ser usado no login do usurio delegado.
   * @param delegatedLogin login do usurio que ser delegado pelo
   * super-usurio
   * sem a necessidade de senha.
   * @param timeZone TimeZone do cliente pelo qual o usurio se logou.
   * @param params Parmetros que sero armazenados na sesso do usurio
   * @return A sesso do usurio. <code>null</code> se chave e/ou senha foram
   * invlidos
   * @throws IllegalArgumentException se <code>login</code> for nulo ou
   * <code>password</code> for nulo ou <code>locale</code> for nulo.
   * @see csbase.logic.SuperUserPermission
   * @see #logout(Object)
   */
  public Session login(String login, EncryptedPassword encryptedPassword,
    Locale locale, String delegatedLogin, TimeZone timeZone,
    Map<String, Serializable> params) {
    if (login == null) {
      throw new IllegalArgumentException("login == null");
    }
    if (encryptedPassword == null) {
      throw new IllegalArgumentException("password == null");
    }
    if (locale == null) {
      throw new IllegalArgumentException("locale == null");
    }

    String plainPassoword = LoginPasswordCipher
      .decrypt(encryptedPassword, this.loginPasswordKeyPair.getPrivate());
    User user = checkLogin(login, plainPassoword);
    if (user == null) {
      /*
       * autenticao falhou
       */
      incrCounter(failedLoginCounter, login.toLowerCase());
      return null;
    }
    if (delegatedLogin != null) {
      // Faz a delegao.
      user =
        AdministrationService.getInstance().changeUser(user, delegatedLogin);
      user.setSuperUserLogin(login);
      Server.logInfoMessage(String
        .format("O usurio %s foi delegado pelo usurio %s.", delegatedLogin,
          login));
    }
    /*
     * incrementamos as estatsticas de login para o usurio em questo
     */
    incrCounter(loginCounter, user.getLogin());

    Session session =
      this.createSession(new SecureKey(), user, locale, params, null, null);
    saveTimeZone(session.getKey(), timeZone);

    Server.logInfoMessage(String
      .format("Login de %s com locale %s. (%d sesso(es) simultnea(s))",
        getLoginStr(user) + getRealUserForLog(params), locale,
        sessionCounter.get(new SessionLimiterKey(user)).get()));
    return session;
  }

  /**
   * Verifica se o nome do servidor passado por parmetro resolve para o mesmo
   * endereo IP deste servidor.
   *
   * @param hostName O nome do servidor
   * @return true se hostName resolve para o mesmo endereo deste servidor ou
   * resolve para a interface de loop back, false caso contrrio
   */
  private boolean isSameAddress(final String hostName) {
    if (hostName == null) {
      throw new IllegalArgumentException("hostName == null");
    }
    try {
      if (InetAddress.getByName(hostName).isLoopbackAddress() || InetAddress
        .getByName(hostName)
        .equals(InetAddress.getByName(Server.getInstance().getHostName()))) {
        return true;
      }
      return false;
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro resolvendo endereo IP.", e);
    }
    return false;
  }

  /**
   * Executa o login de um usurio que  validado atravs de uma referncia de
   * outro servidor onde ele deve ter uma sesso vlida. O servidor deve possuir
   * o certificado do servidor referenciado para garantir a relao de
   * confiana. Se o servidor referenciado for o mesmo (mesmo host e porta deste
   * servidor) ento basta verificar se a sesso  valida.
   *
   * @param referedServerURI A URI do servidor de referncia para validao
   * @param attr Os atributos para serem armazenados na sesso do usurio
   * @param sessionkey A chave da sesso a ser validada no servidor de
   * referncia
   * @param copyServerSessionAttrs se true copia os atributos de sesso do
   * servidor de referncia para a sesso criada nesse servidor
   * @param login O login do usurio
   * @param delegatedLogin O login do usurio para delegao ou null para o
   * caso
   * sem delegao
   * @param locale O Locale
   * @param tz O Timezone
   * @return A sesso do usurio ou null caso o usurio no seja validado no
   * servidor de referncia ou ocorra algum erro durante o login.
   */
  public Session login(ServerURI referedServerURI,
    Map<String, Serializable> attr, boolean copyServerSessionAttrs,
    Object sessionkey, String login, String delegatedLogin, Locale locale,
    TimeZone tz) {

    try {
      User user = User.getUserByLogin(login);
      String realUserLog = getRealUserForLog(attr);
      if (user == null) {
        Server.logSevereMessage(String
          .format("Login por referncia negado. Usurio %s inexistente.",
            login + realUserLog));
        return null;
      }

      Map<String, Serializable> referedServerSessionAttrs = null;

      if ((referedServerURI.getPort() != Server.getInstance()
        .getRegistryPort()) || (!this
        .isSameAddress(referedServerURI.getHost()))) {

        CSKeyStore keyStore = CSKeyStore.getInstance();

        if (keyStore == null) {
          Server.logWarningMessage(String.format(
            "Login por referncia negado. Usurio %s. O repositrio de " +
              "chaves/certificados no existe.",
            login + realUserLog));
          return null;
        }

        if (!keyStore.containsAlias(Server.getInstance().getSystemName())) {
          Server.logWarningMessage(String.format(
            "Login por referncia negado. Usurio %s. O servidor no possui " +
              "certificado cadastrado.",
            login + realUserLog));
          return null;
        }

        byte[] signedServerName;

        String privateKeyPassword =
          Server.getInstance().getPrivateKeyPassword();

        if (privateKeyPassword == null) {
          Server.logSevereMessage(
            "A senha da chave privada do servidor no foi informada.");
          return null;
        }

        signedServerName = keyStore
          .sign(Server.getInstance().getSystemName(), privateKeyPassword,
            Server.getInstance().getSystemName());

        if (signedServerName == null) {
          Server.logSevereMessage(
            "Login por referncia negado. Usurio " + login + ". O servidor " +
              "no possui chave gerada no repositrio de chaves/certificados.");
          return null;
        }

        ServerEntryPoint server =
          ReferedServerCache.getInstance().getServer(referedServerURI);

        if (server == null) {
          Server.logSevereMessage(
            "Login por referncia negado. Servidor " + referedServerURI + " " +
              "fora do ar");
          return null;
        }

        if (!server.getVersionName()
          .equals(Server.getInstance().getVersion())) {
          Server.logSevereMessage(
            "Login por referncia negado. Servidor " + referedServerURI + " " +
              "possui verso diferente deste.");
          return null;
        }

        referedServerSessionAttrs = server
          .isValidSession(sessionkey, Server.getInstance().getSystemName(),
            signedServerName);

        if (referedServerSessionAttrs == null) {
          Server.logSevereMessage(String.format(
            "Login por referncia negado. Usurio %s no foi validado no " +
              "servidor %s.",
            login + realUserLog, referedServerURI));
          return null;
        }
      }
      else {
        if (!this.isValidSession(sessionkey)) {
          Server.logSevereMessage(String.format(
            "Login por referncia negado. Usurio %s no foi validado no " +
              "servidor %s.",
            login + realUserLog, referedServerURI));
          return null;
        }
        else {
          referedServerSessionAttrs =
            this.loggedUsers.get(sessionkey).getPropertes();
        }

      }

      if (delegatedLogin != null) {
        user =
          AdministrationService.getInstance().changeUser(user, delegatedLogin);
        user.setSuperUserLogin(login);
        Server.logInfoMessage(String
          .format("Usurio %s foi delegado pelo usurio %s.", delegatedLogin,
            login + realUserLog));
      }

      Session session = null;

      if (copyServerSessionAttrs) {
        session = this.createSession(new SecureKey(), user, locale, attr,
          referedServerSessionAttrs, null);
      }
      else {
        session =
          this.createSession(new SecureKey(), user, locale, attr, null, null);
      }

      saveTimeZone(session.getKey(), tz);

      Server.logInfoMessage(String.format(
        "Login de %s com locale %s. Validado por %s. (%d sesso(es) " +
          "simultnea(s))",
        getLoginStr(user) + realUserLog, locale, referedServerURI,
        sessionCounter.get(new SessionLimiterKey(user)).get()));

      return session;
    }
    catch (Exception e) {
      Server.logSevereMessage(String
        .format("Erro no login por referncia para usurio %s do servidor %s.",
          login, referedServerURI), e);
    }

    return null;
  }

  /**
   * Verifica o login de um usurio.  utilizado pelos outros servios. Este
   * mtodo no est na interface remota. Se o usurio for um administrador, a
   * validao da sua senha ser sempre feita localmente.  tambm possvel para
   * o administrador configurar alguns usurios para que estes sempre sejam
   * validados localmente, independentemente do mtodo de validao em uso no
   * servidor. Se o usurio tentando o login tiver este atributo, sua senha ser
   * validada localmente.
   *
   * @param login o identificador do usurio que deseja fazer seu login.
   * @param password a senha digitada pelo usurio.
   * @return O usurio que se logou ou null se o login e/ou a senha forem
   * invlidos.
   */
  public User checkLogin(String login, String password) {
    return checkLogin(login, password, null);
  }

  /**
   * Verifica o login de um usurio.  utilizado pelos outros servios. Este
   * mtodo no est na interface remota. Se o usurio for um administrador, a
   * validao da sua senha ser sempre feita localmente.  tambm possvel para
   * o administrador configurar alguns usurios para que estes sempre sejam
   * validados localmente, independentemente do mtodo de validao em uso no
   * servidor. Se o usurio tentando o login tiver este atributo, sua senha ser
   * validada localmente.
   *
   * Leva em conta os atributos que podem conter informao de um usurio para o
   * qual se deseja logar a partir de um usurio j logado (espcie de sudo)
   *
   * @param login o identificador do usurio que deseja fazer seu login.
   * @param password a senha digitada pelo usurio.
   * @param attributes os atributos.
   * @return O usurio que se logou ou null se o login e/ou a senha forem
   * invlidos.
   */
  public User checkLogin(String login, String password,
    Map<String, Serializable> attributes) {
    User user = null;
    try {
      user = User.getUserByLogin(login);
      if (user == null) {
        Server.logWarningMessage(String
          .format("Tentativa de login de usurio inexistente: %s.", login));
        return null;
      }
      // Checa o login no caso de sudo: senha vazia e atributo realUser escrito
      if (attributes != null && password.equals("") && attributes
        .containsKey(REAL_USER_ATTRIBUTE)) {
        String realUser = (String) attributes.get(REAL_USER_ATTRIBUTE);
        return (canLoginAs(realUser, login) ? user : null);
      }
      if (user.getId().equals(User.getAdminId()) || localLogin) {
        return localCheckLogin(user, password);
      }
      Object forceLocalLogin = user.getAttribute(User.FORCE_LOCAL_LOGIN);
      if (forceLocalLogin != null && forceLocalLogin.equals(Boolean.TRUE)) {
        return localCheckLogin(user, password);
      }
      return authenticateUser(user, password);
    }
    catch (Exception e) {
      Server.logSevereMessage(String.format("Erro no login de: %s.", login), e);
      return null;
    }
  }

  /**
   * Devolve true se realUSerLogin puder logar como newUserLogin
   *
   * @param realUserLogin usurio real j logado
   * @param newUserLogin usurio para o qual se deseja logar
   * @return true se realUSerLogin puder logar como newUserLogin
   */
  private boolean canLoginAs(String realUserLogin, String newUserLogin) {
    boolean ret = false;
    try {
      List<Permission> perms = Permission.getAllPermissions();
      for (Permission permission : perms) {
        if (permission instanceof LoginAsPermission) {
          LoginAsPermission loginPerm = (LoginAsPermission) permission;
          if (loginPerm.canLoginAs(realUserLogin, newUserLogin)) {
            ret = true;
            break;
          }
        }
      }
    }
    catch (Exception e) {
      Server.logSevereMessage(String
        .format("Erro ao verificar permisses para usurio %s logar como: %s.",
          realUserLogin, newUserLogin), e);
    }
    return ret;
  }

  /**
   * Mtodo que efetivamente faz a autenticao do usurio, consultando os
   * servidores. A implementao default faz autenticao via LDAP, mas
   * subclasses podem redefinir este mtodo para usar outros esquemas para
   * autenticao.
   *
   * @param user - usurio
   * @param password - senha
   * @return O usurio que se logou ou null se o login e/ou a senha forem
   * invlidos.
   */
  protected User authenticateUser(User user, String password) {
    return LDAPCheckLogin(user, password);
  }

  /**
   * Verifica o login de um usurio utilizando o mtodo de autenticao local.
   * Nesse caso, o digest da senha fornecida ser comparado ao da senha local,
   * que se encontra gravado no arquivo de definio do usurio.
   *
   * @param user objeto contendo todas as informaes do usurio.
   * @param password a senha digitada pelo usurio.
   * @return O usurio que se logou ou null se o login e/ou a senha forem
   * invlidos.
   * @throws DigestException caso ocorra algum problema no processamento da
   * senha.
   */
  private User localCheckLogin(User user,
    String password) throws DigestException {
    String digest = MDigest.getDigest(password);
    if (!user.getPasswordDigest().equals(digest)) {
      Server.logInfoMessage(
        String.format("Login invlido de %s.", user.getLogin()));
      return null;
    }
    return user;
  }

  /**
   * Verifica o login de um usurio utilizando o mtodo de autenticao LDAP.
   * Nesse caso, o login e concatenado ao sufixo definido na configurao do
   * servio e enviado, junto com a senha fornecida, aos servidores LDAP que
   * foram fornecidos tambm na configurao do servio. A lista de servidores 
   * seguida em ordem e, na primeira validao com sucesso, o processo 
   * interrompido e a validao  aceita. Caso nenhum servidor valide as
   * informaes fornecidas, a validao  rejeitada.
   *
   * @param user objeto contendo informaes sobre o usurio.
   * @param password a senha digitada pelo usurio.
   * @return O usurio que se logou ou null se o login e/ou a senha no forem
   * forem aceitos por nenhum dos servidores LDAP disponveis.
   */
  private User LDAPCheckLogin(User user, String password) {
    byte[] passwd = null;
    try {
      passwd = password.getBytes(LDAPCharSet);
    }
    catch (UnsupportedEncodingException e) {
      Server.logSevereMessage(
        String.format("LDAPCharSet no suportado: %s.", LDAPCharSet), e);
      return null;
    }
    List<String> distinguishedNameList =
      new ArrayList<String>(LDAPPatternList.size());
    for (String pattern : LDAPPatternList) {
      pattern = pattern.replace(LDAP_PATTERN_USER, user.getLogin());
      distinguishedNameList.add(pattern);
    }
    for (int i = 0; i < LDAPServers.size(); i++) {
      String server = LDAPServers.get(i);
      int port = (LDAPPorts.get(i)).intValue();
      final int connectionTimeout = getIntProperty(LDAP_TIMEOUT);
      // multiplicamos o timeout por 1000 pois este  definido em segundos
      LDAPConnection conn = new LDAPConnection(connectionTimeout * 1000);
      /*
       * definimos o timeLimit nas restries para o mesmo timeout. O valor
       * default neste caso  0, que significa "ilimitado"
       */
      final LDAPConstraints constraints = conn.getConstraints();
      constraints.setTimeLimit(connectionTimeout * 1000);
      conn.setConstraints(constraints);
      try {
        conn.connect(server, port);
      }
      catch (LDAPException e) {
        Server.logInfoMessage(String
          .format("Conexo rejeitada pelo servidor %s:%d [%s].", server, port,
            e.resultCodeToString()));
        continue;
      }
      try {
        for (String distinguishedName : distinguishedNameList) {
          try {
            conn.bind(LDAPConnection.LDAP_V3, distinguishedName, passwd);
            if (conn.getAuthenticationDN() != null) {
              return user;
            }
          }
          catch (LDAPException e) {
            Server.logInfoMessage(String
              .format("%s rejeitado pelo servidor %s [%s].", distinguishedName,
                server, e.resultCodeToString()));
          }
        }
      } finally {
        try {
          conn.disconnect();
        }
        catch (LDAPException e) {
          Server.logInfoMessage(String
            .format("Erro ao desconectar do servidor %s [%s]", server,
              e.resultCodeToString()));
        }
      }
    }
    return null;
  }

  /**
   * Executa o login de um usurio que fez preLogin. Verifica a validade do
   * token de identificao do preLogin e cria no servidor uma sesso do usurio
   * logado.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param token Identificao de um pr-login.
   * @return A sesso do usurio. <code>null</code> se chave e/ou senha foram
   * invlidos
   * @throws IllegalArgumentException se <code>token</code> for null
   * @see #preLogin(String, String, Locale)
   * @see #logout(Object)
   */
  public Session login(String token) {
    return login(token, null);
  }

  /**
   * Executa o login de um usurio que fez preLogin. Verifica a validade do
   * token de identificao do preLogin e cria no servidor uma sesso do usurio
   * logado. Armazena na sesso o {@link TimeZone} do usurio.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param token Identificao de um pr-login.
   * @param timeZone TimeZone do cliente pelo qual o usurio se logou.
   * @return A sesso do usurio. <code>null</code> se chave e/ou senha foram
   * invlidos
   * @throws IllegalArgumentException se <code>token</code> for null
   * @see #preLogin(String, String, Locale)
   * @see #logout(Object)
   */
  public Session login(String token, TimeZone timeZone) {
    if (token == null) {
      throw new IllegalArgumentException("token == null");
    }
    PreLoginInfo info = preLoggedUsers.remove(token);
    if (info == null) {
      Server.logWarningMessage(String
        .format("Tentativa de login a partir de um token no vlido: %s.",
          token));
      return null;
    }
    User user = info.getUser();
    Session session = createSession(info);
    saveTimeZone(session.getKey(), timeZone);
    Server.logInfoMessage(String
      .format("login a partir de preLogin de %s para locale %s.",
        user.getLogin() + getRealUserForLog(session.getAttributes()),
        info.getLocale()));
    return session;
  }

  /**
   * Executa o login de um servidor local. Autentica e cria no servidor uma
   * sesso do usurio logado.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param serverName O nome do servidor local
   * @param signedServerName O nome do servidor local, assinado.
   * @param locale O locale do servidor local.
   * @return A sesso do usurio logado. <code>null</code> se chave e/ou senha
   * foram invlidos
   * @throws IllegalArgumentException em caso de nulidade nos argumentos.
   * @see #logout(Object)
   */
  public Session login(String serverName, byte[] signedServerName,
    Locale locale) {
    if (serverName == null || signedServerName == null || locale == null) {
      throw new IllegalArgumentException(
        "(localServerName || signedLocalServerName || locale) == null");
    }
    User user = checkLogin(serverName, signedServerName);
    if (user == null) {
      return null;
    }
    Session session = createSession(user, locale);
    Server.logInfoMessage(String
      .format("Login (servidor local) de %s com locale %s.", user.getLogin(),
        locale));
    return session;
  }

  /**
   * Executa o login de um usurio atravs de uma credencial do OpenBus.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param token O token de login no Openbus.
   * @param locale A Locale que o usurio escolheu no login.
   * @param timeZone TimeZone do cliente pelo qual o usurio se logou.
   * @return A sesso do usurio. <code>null</code> se chave e/ou senha foram
   * invlidos
   * @see #logout(Object)
   */
  public Session login(OpenBusLoginToken token, Locale locale,
    TimeZone timeZone) {
    if (token == null) {
      throw new IllegalArgumentException("token == null");
    }
    if (locale == null) {
      throw new IllegalArgumentException("locale == null");
    }

    String login = token.user;

    if (!OpenBusService.getInstance().isEnabled()) {
      Server.logWarningMessage(String.format(
        "No foi possvel logar o usurio %s porque o OpenBus est " +
          "desabilitado.",
        login));
      return null;
    }

    User user;
    try {
      user = User.getUserByLogin(login);
    }
    catch (Exception e) {
      Server
        .logSevereMessage(String.format("Erro ao obter o usurio %s.", login),
          e);
      return null;
    }
    if (user == null) {
      Server.logWarningMessage(
        String.format("Tentativa de login de usurio inexistente: %s.", login));
      return null;
    }

    OpenBusLoginToken clientToken =
      OpenBusService.getInstance().doTokenLogin(token);
    if (clientToken == null) {
      Server.logSevereMessage(
        String.format("Erro ao validar o usurio %s no barramento.", login));
      return null;
    }

    //Coloca na sesso o token para o cliente criar uma conexo com o barramento
    Map<String, Serializable> attributes = new HashMap<String, Serializable>();
    attributes.put("CLIENT_TOKEN", clientToken);

    Session session =
      this.createSession(new SecureKey(), user, locale, attributes, null, null);
    saveTimeZone(session.getKey(), timeZone);

    Server.logInfoMessage(String
      .format("Login de %s com locale %s (%d sesso(es) simultnea(s))",
        getLoginStr(user), locale,
        sessionCounter.get(new SessionLimiterKey(user)).get()));

    return session;
  }

  /**
   * Verifica o login de um servidor local.
   *
   * @param serverName O nome do servidor que est requisitando o login.
   * @param signedServerName O nome do servidor assinado, para verificar se o
   * servidor tem permisso para se conectar.
   * @return O usurio administrador do servidor central (admin) ou null caso o
   * servidor no tenha permisso de acesso, no seja local ou se a
   * operao falhar.
   */
  private User checkLogin(String serverName, byte[] signedServerName) {
    ServerInfo serverInfo;
    try {
      serverInfo = ServerService.getInstance().getServerInfo(serverName);
    }
    catch (OperationFailureException e) {
      Server.logSevereMessage(
        String.format("Falha ao obter o servidor local %s.", serverName), e);
      return null;
    }
    if (serverInfo == null) {
      Server.logWarningMessage(
        String.format("O servidor local %s no foi encontrado.", serverName));
      return null;
    }
    if (!serverInfo.isLocal()) {
      Server.logWarningMessage(
        String.format("O servidor %s no  um servidor local.", serverName));
      return null;
    }
    if (serverInfo.isSuspended()) {
      Server.logWarningMessage(
        String.format("O servidor local %s est suspenso.", serverName));
      return null;
    }
    CSKeyStore keyStore = CSKeyStore.getInstance();
    if (keyStore == null) {
      Server
        .logWarningMessage("O repositrio de chaves/certificados no existe.");
      return null;
    }
    try {
      if (!keyStore.containsAlias(serverName)) {
        Server.logWarningMessage(String
          .format("O servidor local %s no possui certificado cadastrado.",
            serverName));
        return null;
      }
    }
    catch (CSBaseException e) {
      Server.logSevereMessage(String.format(
        "Falha ao verificar se o servidor local %s existe no repositrio.",
        serverName), e);
      return null;
    }
    try {
      if (!keyStore.verify(serverName, serverName, signedServerName)) {
        Server.logWarningMessage(String
          .format("A assinatura do servidor local %s est invlida.",
            serverName));
        return null;
      }
    }
    catch (CSBaseException e) {
      Server.logSevereMessage(String
        .format("Erro ao verificar a assinatura do servidor local %s.",
          serverName), e);
      return null;
    }
    try {
      return User.getUserByLogin((String) User.getAdminId());
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro ao obter o usurio administrador.", e);
      return null;
    }
  }

  /**
   * Cria a sesso do usurio para a camada do cliente e inicia uma sesso do
   * usurio no servidor.
   *
   * @param user Usurio da sesso corrente.
   * @param locale Locale da sesso corrente.
   * @return Sesso para a camada do cliente.
   */
  protected Session createSession(User user, Locale locale) {
    return this.createSession(new SecureKey(), user, locale, null, null, null);
  }

  /**
   * Cria uma sesso para um usurio no servidor.
   *
   * @param info Informaes de um pr-login executado pelo usurio
   * anteriormente.
   * @return Uma sesso de usurio.
   */
  private Session createSession(PreLoginInfo info) {
    return this
      .createSession(info.getSessionKey(), info.getUser(), info.getLocale(),
        info.getAttributes(), info.getServerAttributes(), info.getControl());
  }

  /**
   * Cria uma sesso para um usurio no servidor.
   *
   * @param sessionKey A chave de sesso do usurio.
   * @param user O usurio.
   * @param locale O idioma utilizado pelo usurio.
   * @param attributes Atributos da sesso.
   * @param serverAttributes Atributos que sero armazenados na sesso do
   * usurio, mas que no so enviadas ao cliente.
   * @param control Objeto de controle
   * @return Uma sesso de usurio.
   * @throws PermissionException Caso o nmero de logins simultneos definidos
   * por {@link #MAX_SIMULTANEOUS_SESSIONS} seja excedido.
   */
  private Session createSession(SecureKey sessionKey, User user, Locale locale,
    Map<String, Serializable> attributes,
    Map<String, Serializable> serverAttributes, Remote control) {
    Remote spy = null;

    SessionLimiterKey sessionLimiterKey = new SessionLimiterKey(user);
    AtomicInteger counter = sessionCounter.get(sessionLimiterKey);
    if (counter == null) {
      counter = new AtomicInteger(0);
      sessionCounter.put(sessionLimiterKey, counter);
    }
    if (maxSimultaneousSessions > 0 && counter
      .get() >= maxSimultaneousSessions) {
      Server.logWarningMessage(
        "Usurio " + getLoginStr(user) + getRealUserForLog(
          attributes) + " excedeu o nmero " + "de sesses simultneas (" +
          maxSimultaneousSessions + ")");
      throw new PermissionException(
        "Nmero mximo de sesses excedido (" + maxSimultaneousSessions + ")." +
          " Faa logout em outras sesses para " + "poder realizar este login" +
          ".");
    }
    counter.incrementAndGet();

    try {
      spy = new ClientConnectionSpy(sessionKey);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao criar espio de conexo.", e);
      throw new ServiceFailureException("Erro ao criar sesso do usurio.");
    }
    Session clientSession = new Session(user, sessionKey, spy, attributes);

    ServerSession serverSession = new ServerSession(user, locale);
    serverSession.setControl(control);
    if (attributes != null) {
      serverSession.putAllProperties(attributes);
    }
    if (serverAttributes != null) {
      serverSession.putAllProperties(serverAttributes);
    }
    loggedUsers.put(sessionKey, serverSession);

    String login = (user != null ? user.getLogin() : null);
    fireSessionCreated(login, sessionKey, System.currentTimeMillis());
    return clientSession;
  }

  /**
   * Informa se uma dada sesso  vlida.
   *
   * @param sessionKey Chave da sesso.
   * @return Verdadeiro se a sesso for vlida, falso caso contrrio.
   */
  public boolean isValidSession(Object sessionKey) {
    return loggedUsers.containsKey(sessionKey);
  }

  /**
   * Mtodo invocado sempre que a conexo com o cliente  perdida. Fora um
   * logout se o mesmo no foi feito da forma normal.
   *
   * @param sessionKey Chave da sesso.
   */
  void warnConnectionLost(Object sessionKey) {
    ServerSession session = loggedUsers.get(sessionKey);
    if (session != null) {
      User user = session.getUser();
      String name = session.getApp();
      String login = (user != null ? user.getLogin() : null);
      fireConnectionLost(login, sessionKey, name, System.currentTimeMillis());
      logout(sessionKey);
    }
  }

  /**
   * Faz o logout de um usurio do sistema. Finaliza a sesso e avisa os
   * observadores.
   *
   * @param sessionKey O usurio que est efetuando o logout.
   * @throws IllegalArgumentException em caso de nulidade nos argumentos.
   * @see LogoutEvent
   */
  public void logout(Object sessionKey) {
    if (sessionKey == null) {
      throw new IllegalArgumentException("sessionKey == null");
    }
    ServerSession session = loggedUsers.get(sessionKey);
    if (session != null) {
      MessageService.getInstance().clearMessageListener(session);
      User user = session.getUser();
      String login = (user != null ? user.getLogin() : null);
      String name = session.getApp();
      fireUserLoggingOut(login, sessionKey, name, System.currentTimeMillis());
      loggedUsers.remove(sessionKey);
      notifyObservers(new LogoutEvent(sessionKey));
      SessionLimiterKey sessionLimiterKey = new SessionLimiterKey(user);
      if (sessionCounter.get(sessionLimiterKey).decrementAndGet() == 0) {
        sessionCounter.remove(sessionLimiterKey);
      }
      Server.logInfoMessage(String.format("Logout de %s.",
        user.getLogin() + getRealUserForLog(session.getPropertes())));
    }
    else {
      Server.logWarningMessage(String
        .format("Tentativa de logout para uma sesso invlida: %s.",
          sessionKey));
    }
  }

  /**
   * Informa o Locale de uma sesso.
   *
   * @param key A chave da sesso.
   * @return O Locale da sesso.
   */
  public Locale getUserSessionLocale(Object key) {
    ServerSession session = this.loggedUsers.get(key);
    if (session == null) {
      return null;
    }
    return session.getLocale();
  }

  /**
   * Obtm os usurios atualmente conectados ao sistema.
   *
   * @return usurios atualmente conectados ao sistema.
   */
  public UserOutline[] getLoggedUsers() {
    Enumeration<ServerSession> userEnumeration = this.loggedUsers.elements();
    List<UserOutline> userList = new ArrayList<UserOutline>();
    while (userEnumeration.hasMoreElements()) {
      ServerSession session = userEnumeration.nextElement();
      User user = session.getUser();
      try {
        userList.add(user.getOutline());
      }
      catch (Exception e) {
        e.printStackTrace();
      }
    }
    return userList.toArray(new UserOutline[userList.size()]);
  }

  /**
   * Obtm referncia para o usurio especificado.
   *
   * @param key chave para a sesso do usurio.
   * @return <code>User</code> representando o usurio. <code>null</code> se
   * chave no representa uma sesso vlida.
   */
  public User getUserByKey(Object key) {
    ServerSession session = loggedUsers.get(key);
    if (session == null) {
      return null;
    }
    return session.getUser();
  }

  /**
   * Obtm o valor de uma propriedade da sesso especificada.
   *
   * @param <T> tipo da propriedade.
   * @param key chave para a sesso do usurio.
   * @param propertyName nome da propriedade a ser buscada.
   * @return valor da propriedade na sesso especificada.
   */
  @SuppressWarnings("unchecked")
  public <T> T getSessionProperty(Object key, String propertyName) {
    ServerSession session = loggedUsers.get(key);
    if (session == null) {
      return null;
    }
    return (T) session.getProperty(propertyName);
  }

  /**
   * Atribui um valor a uma propriedade da sesso especificada.
   *
   * @param key chave para a sesso do usurio.
   * @param propertyName nome da propriedade.
   * @param propertyValue valor da propriedade.
   */
  public void setSessionProperty(Object key, String propertyName,
    Serializable propertyValue) {
    ServerSession session = loggedUsers.get(key);
    if (session != null) {
      session.setProperty(propertyName, propertyValue);
    }
  }

  /**
   * Remove uma propriedade da sesso especificada.
   *
   * @param key chave para a sesso do usurio.
   * @param propertyName nome da propriedade.
   */
  public void removeSessionProperty(Object key, String propertyName) {
    ServerSession session = loggedUsers.get(key);
    if (session != null) {
      session.removeProperty(propertyName);
    }
  }

  /**
   * Define o nome do sistema onde uma determinada sesso est sendo usada.
   *
   * @param sessionKey A chave da sesso.
   * @param systemName O nome do sistema.
   */
  public void setSystemName(Object sessionKey, String systemName) {
    ServerSession session = this.loggedUsers.get(sessionKey);
    if (session != null) {
      session.setApp(systemName);
      User user = session.getUser();
      String login = (user != null ? user.getLogin() : null);
      fireSystemNameSet(login, sessionKey, systemName,
        System.currentTimeMillis());
    }
  }

  /**
   * Estrutura para armazenar as informaes de um prelogin que sero usadas
   * posteriormente para completar um login.
   */
  private class PreLoginInfo {
    /**
     * A chave da sesso do usurio.
     */
    private final SecureKey sessionKey;
    /**
     * O usurio que efetuou o {@code preLogin}.
     */
    private final User user;
    /**
     * A <i>locale</i> a ser usada no login.
     */
    private final Locale locale;

    /**
     * Esses atributos sero includos na sesso do usurio quando ocorrer o
     * login efetivo.
     */
    private final Map<String, Serializable> attributes;

    /**
     * Esses atributos sero includos na sesso do usurio quando ocorrer o
     * login efetivo, mas apenas na sesso que fica armazenada no servidor.
     */
    private final Map<String, Serializable> serverAttributes;

    /**
     * O objeto que controla uma automao. Pode ser nulo.
     */
    private final Remote control;

    /**
     * Controi um <code>PreLoginInfo</code> com os seguintes parmetros:
     *
     * @param sessionKey A chave da sesso do usurio.
     * @param user Referncia para o usurio que realizou prelogin.
     * @param locale Locale a ser usado no login.
     * @param attributes Os atributos do registro de pr-login.
     * @param serverAttributes Os atributos do registro de pr-login, que sero
     * armazenados na sesso do servidor .
     * @param control Objeto de controle de automao. Pode ser nulo.
     * @throws IllegalArgumentException Caso algum dos parmetros esteja nulo.
     */
    public PreLoginInfo(SecureKey sessionKey, User user, Locale locale,
      Map<String, Serializable> attributes,
      Map<String, Serializable> serverAttributes, Remote control) {
      if (sessionKey == null) {
        throw new IllegalArgumentException(
          "Chave de sesso no pode ser nula.");
      }
      if (user == null) {
        throw new IllegalArgumentException("Usurio no pode ser nulo.");
      }
      if (locale == null) {
        throw new IllegalArgumentException(
          "O locale do usurio no pode ser nulo.");
      }
      this.sessionKey = sessionKey;
      this.user = user;
      this.locale = locale;
      this.control = control;
      this.attributes = new HashMap<String, Serializable>();
      if (attributes != null) {
        this.attributes.putAll(attributes);
      }
      this.serverAttributes = new HashMap<String, Serializable>();
      if (serverAttributes != null) {
        this.serverAttributes.putAll(serverAttributes);
      }
    }

    /**
     * Obtm a chave da sesso do usurio.
     *
     * @return A chave da sesso do usurio.
     */
    public SecureKey getSessionKey() {
      return this.sessionKey;
    }

    /**
     * Obtm o usurio que efetuou o {@code preLogin}.
     *
     * @return O usurio que efetuou o {@code preLogin}.
     */
    public User getUser() {
      return this.user;
    }

    /**
     * Obtm a <i>locale</i> a ser usada no login.
     *
     * @return A <i>locale</i> a ser usada no login.
     */
    public Locale getLocale() {
      return this.locale;
    }

    /**
     * Obtm os atributos de um registro de pr-login.
     *
     * @return Os atributos de um registro de pr-login.
     */
    public Map<String, Serializable> getAttributes() {
      return Collections.unmodifiableMap(this.attributes);
    }

    /**
     * Obtm os atributos de um registro de pr-login, que devem ser armazenados
     * apenas na sesso no servidor.
     *
     * @return Os atributos de um registro de pr-login.
     */
    public Map<String, Serializable> getServerAttributes() {
      return Collections.unmodifiableMap(this.serverAttributes);
    }

    /**
     * Informa o objeto que controla uma operao de automao.
     *
     * @return O objeto que controla a automao. Pode ser nulo.
     */
    public Remote getControl() {
      return this.control;
    }
  }

  /**
   * Task de verificao se um preLogin foi confirmado. De acordo com a poltica
   * de tempo definida, essa task invlida um token de identificao de um
   * pr-login.
   */
  private class CheckPreLoginTask extends TimerTask {
    /**
     * Token de identificao do pre-login.
     */
    private final Object token;

    /**
     * Constri uma <code>CheckLoginTask</code> com os seguintes parmetros:
     *
     * @param token Token de identificao de um pre-login realizado.
     * @throws IllegalArgumentException Caso o token recebido esteja nulo.
     */
    public CheckPreLoginTask(Object token) {
      if (token == null) {
        throw new IllegalArgumentException("token == null");
      }
      this.token = token;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      if (preLoggedUsers.remove(token) != null) {
        Server.logInfoMessage(
          String.format("Prelogin no confirmado para token %s", token));
      }
    }
  }

  /**
   * Grava na sesso do usurio o TimeZone do cliente pelo qual ele se logou.
   *
   * @param key Identificador da sesso do usurio.
   * @param timeZone {@link TimeZone} onde roda o cliente.
   */
  private void saveTimeZone(Object key, TimeZone timeZone) {
    if (timeZone != null) {
      ServerSession session = loggedUsers.get(key);
      if (session == null) {
        throw new IllegalStateException("Sesso invlida.");
      }
      session.setTimeZone(timeZone);
    }
  }

  /**
   * Obtm da sesso do usurio o {@link TimeZone} do cliente pelo qual ele se
   * logou. Caso no exista {@link TimeZone} na sesso do usurio, retornar o
   * {@link TimeZone} default do servidor.
   *
   * @return {@link TimeZone}
   */
  public TimeZone getTimeZone() {
    ServerSession session = this.loggedUsers.get(getKey());
    if (session == null) {
      return TimeZone.getDefault();
    }
    return session.getTimeZone();
  }

  /**
   * @param key Identificador da sesso do usurio
   * @return Os atributos do servidor da sesso de um usurio logado
   */
  public Map<String, Serializable> getSessionAttributes(Object key) {
    if (this.loggedUsers.get(key) != null) {
      return this.loggedUsers.get(key).getPropertes();
    }
    return null;
  }

  /**
   * Obtm estatsticas sobre os logins.
   *
   * @param succeeded <code>true</code> para estatsticas dos logins
   * bem-sucedidos, <code>false</code> para as falhas de autenticao
   * @return estatsticas sobre os logins
   */
  public Map<String, Integer> getLoginStats(boolean succeeded) {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
    if (succeeded) {
      return Collections.unmodifiableMap(loginCounter);
    }
    else {
      return Collections.unmodifiableMap(failedLoginCounter);
    }
  }

  /**
   * Mtodo interno de convenincia para escrever o nome do login de um usurio,
   * considerando que ele pode ter sido delegado de um superusurio.
   *
   * @param user Usurio cujo login formatado ser recuperado.
   * @return Login formatado para o caso de delegao.
   */
  private static final String getLoginStr(User user) {
    if (user.getSuperUserLogin() == null) {
      return user.getLogin();
    }
    else {
      return user.getSuperUserLogin() + ">>" + user.getLogin();
    }
  }

  /**
   * Dados um conjunto de atributos da sesso, devolve a string a ser logada com
   * o realUser, ou string vazia se no houver
   *
   * @param attributes conjunto de atributos da sesso
   * @return a string a ser logada com o realUser, ou string vazia se no houver
   */
  private String getRealUserForLog(Map<String, Serializable> attributes) {
    String realUserLog = "";
    if (attributes != null && attributes.containsKey(REAL_USER_ATTRIBUTE)) {
      realUserLog =
        "/" + REAL_USER_ATTRIBUTE + ":" + attributes.get(REAL_USER_ATTRIBUTE);
    }
    return realUserLog;
  }

  /**
   * Retorna uma string para logar o usurio real (se houver) ou string vazia,
   * caso contrrio
   *
   * @param key - a chave da sesso
   * @return uma string para logar o usurio real (se houver) ou string vazia,
   * caso contrrio
   */
  public String getRealUserForLog(Object key) {
    return getRealUserForLog(getSessionAttributes(key));
  }
}
