/**
 * $Id: AlgorithmService.java 181607 2018-06-18 22:23:37Z cviana $
 */

package csbase.server.services.algorithmservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.StringReader;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.SortedSet;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import com.fasterxml.jackson.databind.ObjectMapper;
import csbase.exception.BugException;
import csbase.exception.ConfigurationException;
import csbase.exception.HttpServiceException;
import csbase.exception.InfoException;
import csbase.exception.OperationFailureException;
import csbase.exception.ParseException;
import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.exception.algorithms.AlgorithmNotFoundException;
import csbase.exception.algorithms.CategoriesFileNotSavedException;
import csbase.logic.AdminPermission;
import csbase.logic.AlgoEvent;
import csbase.logic.AlgorithmAdminPermission;
import csbase.logic.AlgorithmExecutionPermission;
import csbase.logic.AlgorithmsReloadNotification;
import csbase.logic.CategoryAlgorithmsExecutionPermission;
import csbase.logic.FileInfo;
import csbase.logic.IPathFactory;
import csbase.logic.Permission;
import csbase.logic.User;
import csbase.logic.UserNotification;
import csbase.logic.Utilities;
import csbase.logic.algorithms.*;
import csbase.logic.algorithms.ExecutionLocation.ExecutionLocationConverter;
import csbase.logic.algorithms.flows.Flow;
import csbase.logic.algorithms.flows.FlowAlgorithmParser;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.ParameterRegistry;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parsers.ParameterFactory;
import csbase.logic.algorithms.parsers.SimpleAlgorithmParser;
import csbase.logic.algorithms.xml.algorithmspack.XmlAlgorithmsPackReader;
import csbase.logic.algorithms.xml.category.XmlCategoriesReader;
import csbase.logic.algorithms.xml.category.XmlCategoriesWriter;
import csbase.remote.AlgorithmServiceInterface;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.TransactionCallbackInterface;
import csbase.server.FileSystem;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.TransactionManager;
import csbase.server.services.algorithmservice.Algorithm.CopyDirectoryRecord;
import csbase.server.services.algorithmservice.Algorithm.CopyFileRecord;
import csbase.server.services.algorithmservice.Algorithm.ICopyRecord;
import csbase.server.services.ftcservice.FTCRequester;
import csbase.server.services.ftcservice.FTCService;
import csbase.server.services.httpservice.HttpService;
import csbase.server.services.httpservice.UploadHandler;
import csbase.server.services.mailservice.MailService;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.repositoryservice.IRepositoryFile;
import csbase.util.FileSystemUtils;
import csbase.util.Unzip;
import csbase.util.ZipUtils;
import csbase.util.messages.Message;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.timestamp.TStamp64;
import tecgraf.javautils.xml.exception.XMLParseException;

/**
 * A classe <code>AlgoService</code> implementa o servio de gerncia de
 * algoritmos instalados no SSI. Observa o HttpService para saber quando o
 * upload dos executaveis terminou.
 *
 * @author Tecgraf/PUC-Rio
 */
public class AlgorithmService extends Service implements
  AlgorithmServiceInterface {

  /**
   * Caminho para o repositrio de algoritmos.
   */
  private final String algorithmRepositoryPath;

  /**
   * Chave para identificao do histrico de um diretrio dentro da tabela de
   * histrico.
   */
  private static final String DIR_HISTORY_KEY = "_DIR_HISTORY_KEY";

  /**
   * Mensagem de erro para parmetro nulo
   */
  private static final String ERROR_PARAMETER_NULL =
    "AlgoService.error.parameter.null";

  /**
   * Mensagem de erro para parmetro invlido
   */
  private static final String ERROR_PARAMETER_INVALID =
    "AlgoService.error.parameter.invalid";

  /**
   * Nome do arquivo de categorias xml usado para persistncia
   */
  private static final String CATEGORIES_FILE_NAME = "categories.xml";

  /**
   * Nome do arquivo xml (DTD) usado para validao do pacote de algoritmos
   */
  private static final String DTD_ALGORITHMS_PACK_FILE_NAME =
    "DTD_PacoteAlgoritmos.dtd";

  /**
   * Mensagem de erro para usurio que no tem permisso para administrar um
   * algoritmo
   */
  private String ERROR_ALGO_ADMIN_PERMISSION =
    "AlgoService.error.algo.admin.no_permission";

  /**
   * Mensagem de erro para usurio que no tem permisso para executar um
   * algoritmo
   */
  private String ERROR_ALGO_EXECUTE_PERMISSION =
    "AlgoService.error.algo.execute.no_permission";

  /**
   * Mensagem de erro para usurio que no tem permisso para administrar um
   * algoritmo
   */
  private String ERROR_IMPORT_PA_USER_PERMISSION =
    "AlgoService.error.import.pa.user.no_permission";

  /**
   * Mensagem de erro para usurio que no tem permisso para administrar um
   * algoritmo
   */
  private String ERROR_ALGO_ADMIN_SERVER_PERMISSION =
    "AlgoService.error.algo.admin.server.no_permission";

  /**
   * Mensagem de erro para algoritmo no encontrado
   */
  private static final String ERROR_ALGO_NOT_FOUND =
    "AlgoService.error.algo.not_found";

  /**
   * Mensagem de erro para arquivo de configurao no encontrado
   */
  private static final String ERROR_CONF_VERSION_NOT_FOUND =
    "server.algoservice.error.conf_version_not_found";

  /**
   * Mensagem de erro para problema durante a renomeao de arquivos
   */
  private static final String ERROR_RENAME = "server.algoservice.error.rename";

  /**
   * Mensagem de erro para problema durante a cpia de arquivos
   */
  private static final String ERROR_COPY = "server.algoservice.error.copy";

  //TODO A remover quando todos sistemas CSBase forem assinados {
  /**
   * Mensagem de erro para arquivo de documentao no encontrado
   */
  private static final String ERROR_DOC_VERSION_NOT_FOUND =
    "server.algoservice.error.doc_version_not_found";

  /**
   * Mensagem de erro para arquivo de release notes no encontrado
   */
  private static final String ERROR_RELEASE_NOTES_VERSION_NOT_FOUND =
    "server.algoservice.error.release_notes_version_not_found";

  /**
   * Mensagem de erro para plataforma no encontrada
   */
  private static final String ERROR_PLAT_VERSION_NOT_FOUND =
    "server.algoservice.error.plat_version_not_found";
  //TODO A remover quando todos sistemas CSBase forem assinados

  /**
   * Mensagem de erro para verso no encontrada
   */
  private static final String ERROR_VERSION_NOT_FOUND =
    "AlgoService.error.version.not_found";

  /**
   * Registro de histrico para arquivo copiado
   */
  private static final String HISTORY_COPY_FILE =
    "server.algoservice.history.copy_file";

  /**
   * Registro de histrico para diretrio copiado
   */
  private static final String HISTORY_COPY_DIRECTORY =
    "server.algoservice.history.copy_directory";

  /**
   * Registro de histrico para algoritmo criado
   */
  private static final String HISTORY_CREATE_ALGORITHM =
    "server.algoservice.history.create_algorithm";

  /**
   * Registro de histrico para verso criada
   */
  private static final String HISTORY_CREATE_VERSION =
    "server.algoservice.history.create_version";

  /**
   * Registro de histrico para verso duplicada
   */
  private static final String HISTORY_DUPLICATE_VERSION =
    "server.algoservice.history.duplicate_version";

  /**
   * Registro de histrico para plataforma includa
   */
  private static final String HISTORY_INCLUDE_PLATFORM =
    "server.algoservice.history.include_platform";

  /**
   * Registro de histrico para algoritmo removido
   */
  private static final String HISTORY_REMOVE_ALGORITHM =
    "server.algoservice.history.remove_algorithm";

  /**
   * Registro de histrico para algoritmo renomeado
   */
  private static final String HISTORY_RENAME_ALGORITHM =
    "server.algoservice.history.rename_algorithm";

  /**
   * Registro de histrico para arquivo removido
   */
  private static final String HISTORY_REMOVE_FILE =
    "server.algoservice.history.remove_file";

  /**
   * Registro de histrico para plataforma removida
   */
  private static final String HISTORY_REMOVE_PLATFORM =
    "server.algoservice.history.remove_platform";
  /**
   * Registro de histrico para verso removida
   */
  private static final String HISTORY_REMOVE_VERSION =
    "server.algoservice.history.remove_version";
  /**
   * Nome do arquivo serializado que contm a tabela de histrico
   */
  private static final String HISTORY_TABLE_FILE_NAME =
    ".algorithms.csbase_history";

  /**
   * Registro de histrico para arquivo de configurao carregado
   */
  private static final String HISTORY_UPDATE_CONFIGURATOR =
    "server.algoservice.history.update_configurator";

  /**
   * Registro de histrico para arquivo de documentao carregado
   */
  private static final String HISTORY_UPDATE_DOC =
    "server.algoservice.history.update_doc";

  /**
   * Registro de histrico para arquivo de release notes carregado
   */
  private static final String HISTORY_UPDATE_RELEASE_NOTES =
    "server.algoservice.history.update_release_notes";

  /**
   * Registro de histrico para executvel carregado
   */
  private static final String HISTORY_UPDATE_EXECUTABLE =
    "server.algoservice.history.update_executable";

  /**
   * Extenso para arquivos ZIP
   */
  private static final String ZIP_EXTENSION = "zip";

  /**
   * Chave para um identificador de algoritmo.
   */
  private static final String ALGO_ID_KEY = "algoId";

  /**
   * Chave para um identificador de usurio.
   */
  private static final String USER_ID_KEY = "userId";

  /**
   * Chave para um tipo de arquivo da rvore de algoritmos.
   */
  private static final String ALGO_FILE_TYPE_KEY = "type";

  /**
   * Chave para um identificador de verso.
   */
  private static final String VERSION_ID_KEY = "versionId";

  /**
   * Chave para o nome de uma plataforma.
   */
  private static final String PLATFORM_NAME_KEY = "platformName";

  /**
   * Propriedade que define o intervalo de recarga dos algoritmos em minutos.
   * Caso essa propriedade no seja definida, o servio no faz a recarga.
   */
  private static final String PROP_RELOAD_ALGS_INTERVAL_IN_MIN =
    "reloadAlgsIntervalInMinutes";

  /**
   * Propriedade que define o intervalo de recarga dos algoritmos em minutos.
   * Caso essa propriedade no seja definida, o servio no faz a recarga.
   */
  private static final String PROP_CLEANUP_TEMP_DIR_INTERVAL_IN_SEC =
    "temporary.dir.cleanupIntervalInSeconds";

  /**
   * Propriedade que define o diretrio temporrio parapara
   * importao/exportao de algoritmos a partir de um PA.
   */
  private static final String PROP_PA_TEMP_DIR = "algorithms.temporary.dir";

  /**
   * Propriedade que define a classe de registro dos parmetros.
   */
  private static final String PROP_PARAMETER_REGISTRY_CLASS =
    "parameters.registry.class";

  /**
   * Chave para um nome de arquivo.
   */
  private static final String FILE_NAME_KEY = "fileName";

  /**
   * Chave para o caminho absoluto no sistema de arquivos.
   */
  private static final String FILE_PATH_KEY = "filePath";

  /**
   * Chave para o indicador que indica se o arquivo deve ser expandido ao final
   * do upload, caso este seja um arquivo zip.
   */
  private static final String EXPAND_IF_ZIP = "expandIfZip";

  /**
   * Chave para indicar se o arquivo zip, referente ao diretrio requisitado
   * para transferncia para o cliente, ser deve excluido.
   */
  private static final String EXCLUDE_TMP_ZIP = "excludeTmpZip";

  /**
   * Nome do arquivo do pacote de algoritmos que vai ser copiado temporariamente
   * no servidor
   */
  private static final String ALGO_PACK_FILE_NAME = "algoritmos.pa";

  /**
   * Nome do arquivo de metadados do pacote de algoritmos
   */
  private static final String ALGO_PACK_METADATA_FILE_NAME =
    "pacote_algoritmos.xml";

  /**
   * Mensagem retornada quando o pacote de algoritmos  vlido
   */
  private static final String VALID_ALGORITHMS_PACK_MSG =
    ">> Pacote de Algoritmo vlido.\n";

  /**
   * Mensagem retornada quando o pacote de algoritmos  invlido
   */
  private static final String INVALID_ALGORITHMS_PACK_MSG =
    ">> Pacote de Algoritmo invlido.\n";

  private final ParameterRegistry parameterRegistry;

  /**
   * Tipos de arquivos na rvore de algoritmos
   */
  private enum AlgoFileType {
    /**
     * Arquivo executvel (ou binrio)
     */
    EXECUTABLE,
    /**
     * Arquivo de configurao
     */
    CONFIGURATION,
    /**
     * Arquivo de documentao
     */
    DOCUMENTATION,
    /**
     * Arquivo de release notes
     */
    RELEASE_NOTES,
    /**
     * Pacote de verso
     */
    VERSION_PACK,
    /**
     * Pacote de verso
     */
    ALGORITHMS_PACK
  }

  /**
   * Flag usado para desativar a thread de reinicio do servio
   */
  private boolean shutdown;

  /**
   * Hashtable que contm os algoritmos cadastrados.
   */
  private Hashtable<String, Algorithm> registeredAlgorithms;

  /**
   * Hashtable que contm as configuraes dos parmetros cadastrados.
   */
  private Properties parametersConfiguration;

  /**
   * Transao que garante a execuo da gerncia do repositrio em modo
   * exclusivo.
   */
  private TransactionManager transaction;

  /**
   * Thread que recarrega os algoritmos.
   */
  private ReloadAlgsThread reloadAlgsThread;

  /**
   * Caminho para o diretrio temporrio utilizado pelo pacote de algorimtos
   * para descompactar os arquivos do Pacote de Algoritmos
   */
  private final String tempDirAlgorithmPath;

  /**
   * Flag indicativo de sada da thread de limpeza do diretrio temporrio usado
   * para operaes de importao e exportao sobre um PA
   */
  private boolean exitCleanupThread = false;

  /**
   * Thread que faz a limpeza do diretrio temporrio usado para operaes de
   * importao e exportao sobre um PA.
   */
  private Thread cleanupTempDirThread = null;

  /**
   * As categorias.
   */
  private CategorySet categorySet;

  /**
   * Indica se est sendo realizada a primeira carga de categorias, para que se
   * possa gerar inicialmente os ids das categorias
   */
  private boolean isFirstCategoryLoad;

  /**
   * Local padro de execuo de comandos.
   */
  private ExecutionLocation defaultAlgorithmExecutionLocation;

  /**
   * Mapa que relaciona o token que identifica univocamente um dado de PA, com
   * as informaes necessrias para realizar as operaes de importao ou
   * exportao
   */
  private Hashtable<String, ImportAlgorithmsPackDataInfo> algoPackTokenMap;

  /**
   * Operaes possveis sobre arquivos
   */
  private enum FileChannelOp {
    /**
     * Envio
     */
    UPLOAD,
    /**
     * Recebimento
     */
    DOWNLOAD,
    /**
     * Leitura
     */
    READ,
    /**
     * Escrita
     */
    WRITE
  }

  /**
   * Modela uma requisio FTC.
   *
   * @author Tecgraf
   */
  private class Requester implements FTCRequester {
    /**
     * Mapa de argumentos para notificao aps o trmino da transferncia.
     */
    private Map<String, Object> notifyArgs = null;
    /**
     * Indica se o acesso  s de leitura ou leitura e escrita.
     */
    private boolean readOnly;

    /**
     * Constri uma requisio.
     *
     * @param notifyArgs Os argumentos para notificao aps o trmino.
     * @param readOnly Indicativo de leitura ou leitura e escrita.
     */
    Requester(Map<String, Object> notifyArgs, boolean readOnly) {
      this.notifyArgs = notifyArgs;
      this.readOnly = readOnly;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public FileChannel createFileChannel(IRepositoryFile file, boolean readOnly)
      throws Exception {
      if (this.readOnly && !readOnly) {
        String msg = "Tentativa de abrir arquivo RO para escrita: " + file
          .getPath();
        Server.logSevereMessage(msg);
        throw new Exception(msg);
      }
      String mode;
      if (readOnly) {
        mode = "r";
      }
      else {
        mode = "rw";
      }
      try {
        return file.getFileChannel(mode);
      }
      catch (FileNotFoundException e) {
        Server.logSevereMessage("Arquivo no encontrado: " + file.getPath(), e);
        return null;
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void fileChannelClosed(IRepositoryFile file) throws Exception {
      try {
        file.close();
      }
      catch (IOException e) {
        Server.logSevereMessage(
          "Erro ao tentar fechar arquivo utilizado pelo servidor de "
            + "arquivos: " + file.getName(), e);
      }
      if (notifyArgs == null) {
        return;
      }
      AlgoFileType type = (AlgoFileType) notifyArgs.get(ALGO_FILE_TYPE_KEY);
      Object userId = notifyArgs.get(USER_ID_KEY);
      Object algoId = notifyArgs.get(ALGO_ID_KEY);

      Algorithm algo = null;
      if (algoId != null && type != AlgoFileType.ALGORITHMS_PACK) {
        algo = registeredAlgorithms.get(algoId);
        if (algo == null) {
          String detailedMessage = String.format(
            "O algoritmo %s no foi encontrado.\n", algoId);
          Server.logSevereMessage(detailedMessage);
          String userMsg = getString(
            "server.algoservice.error.algo_dir_not_found");
          String[] ids = new String[] { (String) userId };
          String login = User.getUser(userId).getLogin();
          MessageService.getInstance().send(new Message(new UserNotification(
            login, userMsg, false, false)), ids);
          return;
        }
      }
      switch (type) {
        case CONFIGURATION:
          configurationUploaded(notifyArgs);
          break;
        case DOCUMENTATION:
          documentationUploaded(notifyArgs);
          break;
        case RELEASE_NOTES:
          releaseNotesUploaded(notifyArgs);
          break;
        case EXECUTABLE:
          if (notifyArgs.containsKey(EXPAND_IF_ZIP)) {
            executableUploaded(notifyArgs);
          }
          else {
            if (notifyArgs.containsKey(EXCLUDE_TMP_ZIP)) {
              executableDownloaded(notifyArgs);
            }
          }
          break;
        case VERSION_PACK:
          boolean toExpand = (Boolean) notifyArgs.get(EXPAND_IF_ZIP);
          if (toExpand) {
            versionPackUploaded(notifyArgs);
          }
          else {
            versionPackDownloaded(notifyArgs);
          }
          break;

        default:
          throw new BugException(getMessageFormatted(
            "AlgoService.error.algo.file.type_unknown", type));
      }
      AlgorithmInfo algoInfo = algo.getInfo();
      fireModifyEvent(algoInfo);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isLocked(IRepositoryFile file) throws Exception {
      return false;
    }
  }

  /**
   * Cria o servio de algoritmos. Dispara uma thread para aguardar solicitaes
   * de transferncia de arquivos de/para a rvore de algoritmos.
   *
   * @throws ServerException em caso de falha ou erro na costruo do servio
   */
  protected AlgorithmService() throws ServerException {
    super(SERVICE_NAME);
    algorithmRepositoryPath = this.getStringProperty("base.algorithm.dir");
    Server.logInfoMessage("Diretrio base de algoritmos: "
      + algorithmRepositoryPath);
    if (getAndCheckAlgoRootDir() == null) {
      File algorithDir = new File(algorithmRepositoryPath);
      try {
        System.out.println(MessageFormat.format(
          "\nVerifique a propriedade AlgorithmService.base.algorithm.dir que "
            + "define o diretrio base de algoritmos, pois o diretrio {0} no "
            + "existe.\n", algorithDir.getCanonicalPath()));
      }
      catch (IOException e) {
      }
      throw new ServerException(getMessageFormatted(
        "AlgoService.error.algo.dir_not_found", algorithmRepositoryPath));
    }

    tempDirAlgorithmPath = this.getStringProperty(PROP_PA_TEMP_DIR);
    Server.logInfoMessage(
      "Diretrio temporrio para importao/exportao de algoritmos a partir "
        + "de um PA: " + tempDirAlgorithmPath);
    String registryClass = getStringProperty(PROP_PARAMETER_REGISTRY_CLASS);
    try {
      parameterRegistry = createParameterRegistry(registryClass);
    }
    catch (OperationFailureException e) {
      throw new ServerException(e);
    }

    categorySet = new CategorySet();
    algoPackTokenMap = new Hashtable<String, ImportAlgorithmsPackDataInfo>();

    // XXX - Temporrio. Aguardando a soluo do Bug #5847.
    ClientRemoteLocator.algorithmService = this;
  }

  /**
   * Cria uma instncia, por reflexo, do registro de parmetros do sistema.
   *
   * @param registryClassName O nome completo da classe que implementa o
   *        registro.
   * @return Uma instncia do registro.
   * @throws OperationFailureException Caso no seja possvel criar o registro.
   */
  private ParameterRegistry createParameterRegistry(String registryClassName)
    throws OperationFailureException {
    Class<?> registryClass;
    try {
      registryClass = Class.forName(registryClassName);
    }
    catch (ClassNotFoundException e) {
      throw new OperationFailureException("A classe {0} no foi encontrada.",
        registryClassName);
    }
    if (!ParameterRegistry.class.isAssignableFrom(registryClass)) {
      throw new OperationFailureException(
        "A classe {0} no implementa a interface necessria {1}.",
        registryClassName, ParameterFactory.class.getName());
    }
    Constructor<?> registryConstructor;
    try {
      registryConstructor = registryClass.getConstructor();
    }
    catch (NoSuchMethodException e) {
      throw new OperationFailureException(
        "No existe um construtor vazio em {0}.", registryClassName);
    }
    ParameterRegistry registry;
    try {
      registry = (ParameterRegistry) registryConstructor.newInstance();
    }
    catch (InstantiationException e) {
      throw new OperationFailureException(MessageFormat.format(
        "No foi possvel construir o registro {0}.", registryClassName), e);
    }
    catch (IllegalAccessException e) {
      throw new OperationFailureException(MessageFormat.format(
        "Construtor do registro {0} no aceita os parmetros usados.",
        registryClassName), e);
    }
    catch (InvocationTargetException e) {
      throw new OperationFailureException(MessageFormat.format(
        "Erro ao construir o registro de parmetros {0}.", registryClassName), e
          .getTargetException());
    }
    return registry;
  }

  /**
   * Constri a instncia do servio de gerncia de algoritmos
   *
   * @throws ServerException caso ocorra algum problema durante a instanciao.
   */
  public static void createService() throws ServerException {
    new AlgorithmService();
  }

  /**
   * Obtm uma referncia para a instncia nica (singleton) do servio de
   * algoritmos.
   *
   * @return referncia para a instncia nica do servio de algoritmos.
   */
  public static AlgorithmService getInstance() {
    return (AlgorithmService) getInstance(SERVICE_NAME);
  }

  /**
   * Verifica se existe um algoritmo com o identificador especificado.
   *
   * @param id identificador do algoritmo
   * @return <code>true</code> se existe o algoritmo, <code>false</code> caso
   *         contrrio
   */
  public boolean algorithmIdRegistered(Object id) {
    return (registeredAlgorithms.get(id) != null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ParameterRegistry getParameterRegistry() {
    return parameterRegistry;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized AlgorithmInfo changeAlgorithmProperties(String algoName,
    Hashtable<String, String> newValues) throws AlgorithmNotFoundException {
    User user = Service.getUser();
    if (!user.isAdmin() && !isAlgorithmOwnerUser(algoName)) {
      checkAlgorithmAdminPermission(algoName);
    }
    Algorithm algorithm = getAlgorithm(algoName);
    algorithm.setAlgorithmFieldValuesList(newValues);
    AlgorithmInfo algoInfo = algorithm.getInfo();
    Server.logInfoMessage("Algoritmo " + algoName + " modificado em " + algoInfo
      .getDirectory() + " (apenas propriedades)");
    fireModifyEvent(algoInfo);
    return algoInfo;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized AlgorithmInfo renameAlgorithm(Object algoId, String name)
    throws RemoteException {
    checkAlgorithmAdminPermission(algoId);

    if (name == null || name.trim().length() == 0) {
      String msg = getMessageFormatted("AlgoService.error.algo.invalid_name");
      throw new ServiceFailureException(msg);
    }

    String newName = name.trim();
    for (Algorithm algorithm : registeredAlgorithms.values()) {
      AlgorithmInfo info = algorithm.getInfo();
      String algoName = info.getName().trim();
      if (algoName.equals(newName)) {
        String msg = getMessageFormatted("AlgoService.error.algo.existing_name",
          newName);
        throw new ServiceFailureException(msg);
      }
    }

    Algorithm algorithm = registeredAlgorithms.get(algoId);
    String currentName = algorithm.getInfo().getName();
    try {
      algorithm.rename(newName);
    }
    catch (ServerException e) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.algo.rename_failed", currentName, newName), e);
      return null;
    }

    AlgorithmInfo info = algorithm.getInfo();
    Map historyTable = readHistoryTable();
    if (historyTable != null) {
      // Reindexa a tabela do histrico pelo novo nome
      Object algoHistory = historyTable.remove(currentName);
      historyTable.put(newName, algoHistory);
      writeHistoryTable(historyTable);

      String[] path = new String[] { getSymbolicRootName(), newName };
      String description = getMessageFormatted(HISTORY_RENAME_ALGORITHM,
        currentName, newName);
      appendHistory(path, description);
    }

    Server.logInfoMessage("Algoritmo " + currentName + " renomeado para "
      + newName);
    fireModifyEvent(info);
    return info;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized AlgorithmInfo changeVersionProperties(String algoName,
    AlgorithmVersionId versionId, Hashtable<String, String> newValues)
    throws AlgorithmNotFoundException {
    User user = Service.getUser();
    if (!user.isAdmin() && !isAlgorithmOwnerUser(algoName)) {
      checkAlgorithmAdminPermission(algoName);
    }
    Algorithm algorithm = getAlgorithm(algoName);
    algorithm.setVersionFieldValuesList(versionId, newValues);
    AlgorithmInfo algoInfo = getInfo(algoName);
    Server.logInfoMessage("Verso " + versionId + " do algoritmo " + algoName
      + " modificada " + "(apenas propriedades)");
    fireModifyEvent(algoInfo);
    return algoInfo;
  }

  /**
   * Retorna o nome que representa o repositrio de algoritmos no histrico. O
   * repositrio  um caminho do sistema de arquivos, ento esse nome simblico
   *  utilizado como uma abstrao.
   *
   * @return O nome simblico que representa o reposittio de algoritmos.
   */
  private String getSymbolicRootName() {
    return "algorithms";
  }

  /**
   * <p>
   * Copia arquivos de uma pasta para a outra com a possibilidade de dar
   * permisso de execuo s cpias.
   * </p>
   * <p>
   * Para definir o caminho do arquivo de destino, ser construido um FileInfo
   * usando o targetDir como path se este for diferente de <tt>null</tt> e
   * mantendo o nome do arquivo original. Este novo FileInfo ser passado como
   * parmetro para o {@link IPathFactory#getPath(FileInfo)
   * targetPathFactory.getPath(FileInfo)} que retornar o caminho fsico do novo
   * arquivo.
   * </p>
   * <p>
   * Ao final da execuo os algoritmos sero recarregados.
   * </p>
   *
   * @param sourceVersion Informaes da verso de origem.
   * @param files Arquivos a serem transferidos.
   * @param sourcePathFactory Fbrica para criar o caminho dos arquivos a serem
   *        transferidos.
   * @param targetVersion Informaes da verso de destino.
   * @param targetDir diretrio de destino.
   * @param targetPathFactory Fbrica para criar o caminho dos arquivos de
   *        destino. Este caminho  relativo ao valor do parmetro targetDir.
   * @param setExecutables <tt>true</tt> se as cpias devero ter permisso de
   *        execuo ou no.
   */
  @Override
  public synchronized void copyFiles(final AlgorithmVersionInfo sourceVersion,
    final List<FileInfo> files, final IPathFactory sourcePathFactory,
    final AlgorithmVersionInfo targetVersion, final FileInfo targetDir,
    final IPathFactory targetPathFactory, final boolean setExecutables) {

    if (null == sourceVersion) {
      throw new IllegalArgumentException(getParameterNullMessage(
        "sourceVersion"));
    }
    if (files == null || files.size() == 0) {
      throw new IllegalArgumentException(getParameterNullMessage("files"));
    }
    if (null == sourcePathFactory) {
      throw new IllegalArgumentException(getParameterNullMessage(
        "sourcePathFactory"));
    }
    if (null == targetVersion) {
      throw new IllegalArgumentException(getParameterNullMessage(
        "targetVersion"));
    }
    if (null == targetPathFactory) {
      throw new IllegalArgumentException(getParameterNullMessage(
        "targetPathFactory"));
    }

    String sourceAlgorithmId = sourceVersion.getInfo().getId();
    Algorithm sourceAlgorithm = registeredAlgorithms.get(sourceAlgorithmId);
    if (null == sourceAlgorithm) {
      throw new ServiceFailureException(getAlgoNotFoundMessage(
        sourceAlgorithmId));
    }

    Object targetAlgorithmId = targetVersion.getInfo().getId();
    checkAlgorithmAdminPermission(targetAlgorithmId);

    Algorithm targetAlgorithm = registeredAlgorithms.get(targetAlgorithmId);
    List<ICopyRecord> records = new ArrayList<ICopyRecord>();
    try {
      targetAlgorithm.copyFiles(files, sourcePathFactory, targetDir,
        targetPathFactory, setExecutables, records);
    }
    catch (InfoException ex) {
      throw ex;
    }
    catch (Exception e) {
      StringBuilder fileName = new StringBuilder();
      for (int inx = 0; inx < files.size(); inx++) {
        if (inx > 0) {
          fileName.append(", ");
        }
        fileName.append(files.get(inx).getPath());
      }
      throw new ServiceFailureException(getMessageFormatted(ERROR_COPY,
        fileName, targetVersion.getId(), targetAlgorithmId), e);
    }

    /*
     * Cria entradas no histrico a reapeito dos arquivos que foram copiados.
     */
    for (ICopyRecord record : records) {
      if (record.isDirectory()) {
        CopyDirectoryRecord directoryRecord = (CopyDirectoryRecord) record;
        String path = getHistoryPath(sourceVersion, targetPathFactory,
          directoryRecord.getDirectory());

        String description = getMessageFormatted(HISTORY_COPY_DIRECTORY,
          directoryRecord.getDirectory().getPath());
        appendHistory(Utilities.splitProjectPath(path), description);
      }
      else {
        CopyFileRecord fileRecord = (CopyFileRecord) record;
        String sourcePath = getHistoryPath(sourceVersion, sourcePathFactory,
          fileRecord.getSourceFile());
        String targetPath = getHistoryPath(targetVersion, targetPathFactory,
          fileRecord.getTargetFile());

        String description = getMessageFormatted(HISTORY_COPY_FILE, fileRecord
          .getTargetFile().getPath(), sourcePath);
        appendHistory(Utilities.splitProjectPath(targetPath), description);
      }
    }

    /*
     * Se algum arquivo de destino tiver sido criado, notifica mudana no
     * algoritmo.
     */
    for (ICopyRecord record : records) {
      if (record.wasCreated()) {
        fireModifyEvent(targetAlgorithm.getInfo());
        break;
      }
    }
  }

  /**
   * Cria no repositrio a estrutura para um novo algoritmo.
   *
   * @param algoName nome do novo algoritmo
   * @param atributeValues hashtable com os atributos do algoritmo
   * @return informaes do novo algoritmo
   */
  @Override
  public synchronized AlgorithmInfo createAlgorithm(String algoName,
    String algoId, Hashtable<String, String> atributeValues)
    throws PermissionException {
    User user = Service.getUser();
    if (!user.isAdmin()) {
      checkAlgorithmAdminPermission(algoName);
    }
    Algorithm algo = null;
    try {
      algo = Algorithm.createAlgorithm(algoId, algoName, atributeValues);
    }
    catch (Exception e) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.algo.create_failed", algoName), e);
      return null;
    }
    AlgorithmInfo algoInfo = algo.getInfo();
    registeredAlgorithms.put(algoInfo.getId(), algo);
    Server.logInfoMessage("Algoritmo " + algoName + " criado em " + algoInfo
      .getDirectory());
    String[] path = new String[] { getSymbolicRootName() };
    String description = getMessageFormatted(HISTORY_CREATE_ALGORITHM,
      algoName);
    appendHistory(path, description);
    AlgoEvent action = new AlgoEvent(AlgoEvent.CREATE, algoInfo);
    sendEvent(action);
    return algoInfo;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized AlgorithmConfigurator createAlgorithmConfigurator(
    String algorithmName, AlgorithmVersionId algorithmVersionId)
    throws AlgorithmNotFoundException {
    if (algorithmName == null) {
      throw new IllegalArgumentException(getParameterNullMessage(
        "algorithmName"));
    }
    if (algorithmVersionId == null) {
      throw new IllegalArgumentException(getParameterNullMessage(
        "algorithmVersionId"));
    }
    Algorithm algorithm = getAlgorithm(algorithmName);
    AlgorithmInfo algorithmInfo = algorithm.getInfo();
    AlgorithmVersionInfo version = algorithmInfo.getVersionInfo(
      algorithmVersionId);
    if (version == null) {
      throw new AlgorithmNotFoundException(getVersionNotFoundMessage(
        algorithmInfo.getId(), algorithmVersionId));
    }

    try {
      SimpleAlgorithmConfigurator simpleConfigurator =
        createSimpleAlgorithmConfigurator(version);
      if (simpleConfigurator != null) {
        return simpleConfigurator;
      }
      FlowAlgorithmConfigurator flowConfigurator =
        createFlowAlgorithmConfigurator(version);
      if (flowConfigurator != null) {
        return flowConfigurator;
      }
    }
    catch (ParseException e) {
      throw new ServiceFailureException(e.getLocalizedMessage(), e);
    }
    catch (OperationFailureException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
    throw new ServiceFailureException(getMessageFormatted("error_no_component",
      version.getInfo(), version, AlgorithmVersionInfo.CONFIG_FILE,
      AlgorithmVersionInfo.FLOW_CONFIG_FILE));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized AlgorithmInfo createVersion(Object algoId, int major,
    int minor, int patch, Hashtable<String, String> properties) {
    checkAlgorithmAdminPermission(algoId);
    Server.logInfoMessage("Tentando criar uma verso para o algoritmo: "
      + algoId);

    Object versionId;
    Algorithm algo = registeredAlgorithms.get(algoId);
    try {
      versionId = algo.createVersion(major, minor, patch, properties);
    }
    catch (ServerException se) {
      String versionStr = String.format("%d.%d.%d", major, minor, patch);
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.algo.version.create_failed", versionStr, algo
          .getInfo().getName()), se);
      return null;
    }
    String[] path = new String[] { getSymbolicRootName(), algo.getInfo()
      .getName() };
    String description = getMessageFormatted(HISTORY_CREATE_VERSION, versionId);
    appendHistory(path, description);
    Server.logInfoMessage("Verso " + versionId + " criada para algoritmo "
      + algo.getInfo().getName());
    fireModifyEvent(algo.getInfo());
    return algo.getInfo();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void duplicateVersion(Object algorithmId,
    Object sourceVersionId, int majorTo, int minorTo, int patchTo) {
    checkAlgorithmAdminPermission(algorithmId);

    // Verifica a existncia do algoritmo
    Server.logInfoMessage("Tentando duplicar uma verso para o algoritmo: "
      + algorithmId);

    // Verifica a existncia da verso de origem
    Algorithm algorithm = registeredAlgorithms.get(algorithmId);
    AlgorithmInfo algorithmInfo = algorithm.getInfo();
    AlgorithmVersionInfo sourceVersion = algorithmInfo.getVersionInfo(
      sourceVersionId);
    if (sourceVersion == null) {
      throw new ServiceFailureException(getVersionNotFoundMessage(algorithmId,
        sourceVersion));
    }
    Map<String, String> properties = sourceVersion.getPropertyValues();
    try {
      // Pega referncia para a verso de destino
      Object versionIdTo = algorithm.createVersion(majorTo, minorTo, patchTo,
        properties);
      AlgorithmVersionInfo targetVersion = algorithmInfo.getVersionInfo(
        versionIdTo);

      FileInfo releaseNotes = sourceVersion.getReleaseNotes();
      if (releaseNotes != null) {
        IPathFactory sourcePathFactory = new ReleaseNotesPathFactory(
          sourceVersion);
        IPathFactory targetPathFactory = new ReleaseNotesPathFactory(
          targetVersion);
        FileInfo targetDir = null; // Grava tudo a partir da raiz do diretrio
        // de documentao.
        List<ICopyRecord> records = null; // No tenho interesse em saber o
        // resultado de cada cpia.
        List<FileInfo> releaseNotesFiles = new ArrayList<FileInfo>();
        releaseNotesFiles.add(releaseNotes);
        algorithm.copyFilesWithoutReloading(releaseNotesFiles,
          sourcePathFactory, targetDir, targetPathFactory, false, records);
      }

      // Copia os arquivos de documentao
      List<FileInfo> docs = sourceVersion.getDocumentation();
      if (docs != null) {
        IPathFactory sourcePathFactory = new DocumentationPathFactory(
          sourceVersion);
        IPathFactory targetPathFactory = new DocumentationPathFactory(
          targetVersion);
        FileInfo targetDir = null; // Grava tudo a partir da raiz do diretrio
        // de documentao.
        List<ICopyRecord> records = null; // No tenho interesse em saber o
        // resultado de cada cpia.
        algorithm.copyFilesWithoutReloading(docs, sourcePathFactory, targetDir,
          targetPathFactory, false, records);
      }

      // Copia os arquivos de configurao
      List<FileInfo> configs = sourceVersion.getConfigurators();
      if (configs != null) {
        IPathFactory sourcePathFactory = new ConfigurationPathFactory(
          sourceVersion);
        IPathFactory targetPathFactory = new ConfigurationPathFactory(
          targetVersion);
        FileInfo targetDir = null; // Grava tudo a partir da raiz do diretrio
        // dos configuradores.
        List<ICopyRecord> records = null; // No tenho interesse em saber o
        // resultado de cada cpia.
        algorithm.copyFilesWithoutReloading(configs, sourcePathFactory,
          targetDir, targetPathFactory, false, records);
      }

      // Copia os executveis
      Map<String, List<FileInfo>> platforms = sourceVersion.getPlatforms();
      if (platforms != null) {
        // Para cada plataforma, insere a plataforma e os executaveis
        for (String platform : platforms.keySet()) {
          algorithm.includePlatform(targetVersion.getId(), platform);

          List<FileInfo> execs = platforms.get(platform);
          IPathFactory sourcePathFactory = new ExecutablePathFactory(
            sourceVersion, platform);
          IPathFactory targetPathFactory = new ExecutablePathFactory(
            targetVersion, platform);
          FileInfo targetDir = null; // Grava tudo a partir da raiz do diretrio
          // dos configuradores.
          List<ICopyRecord> records = null; // No tenho interesse em saber o
          // resultado de cada cpia.
          algorithm.copyFilesWithoutReloading(execs, sourcePathFactory,
            targetDir, targetPathFactory, true, records);
        }
      }
      algorithm.reload();

      String[] path = new String[] { getSymbolicRootName(), algorithm.getInfo()
        .getName() };
      String description = getMessageFormatted(HISTORY_DUPLICATE_VERSION,
        sourceVersion, versionIdTo);
      appendHistory(path, description);

      // Notifica mudana no algoritmo
      fireModifyEvent(algorithmInfo);
    }
    catch (Exception e) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.algo.version.duplicate_failed", algorithmInfo
          .getName()), e);
      throw new ServiceFailureException(getMessageFormatted(
        "AlgoService.error.algo.version.create_failed", sourceVersionId,
        algorithm.getInfo().getName()), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<AlgorithmProperty> getAlgorithmProperties() {
    return Algorithm.getAlgorithmProperties();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CategorySet getAllCategories() {
    return categorySet;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object[] getAllIds() {
    List<Object> sendList = new Vector<Object>();
    for (String id : registeredAlgorithms.keySet()) {
      sendList.add(id);
    }
    return sendList.toArray(new Object[sendList.size()]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AlgorithmInfo[] getAllInfo() {
    Enumeration<Algorithm> elements = registeredAlgorithms.elements();
    List<AlgorithmInfo> sendList = new Vector<AlgorithmInfo>();
    while (elements.hasMoreElements()) {
      Algorithm algo = elements.nextElement();
      sendList.add(algo.getInfo());
    }
    return sendList.toArray(new AlgorithmInfo[sendList.size()]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AlgorithmOutline[] getAllOutlines() {
    if (registeredAlgorithms == null) {
      return null;
    }
    Enumeration<Algorithm> elements = registeredAlgorithms.elements();
    List<AlgorithmOutline> sendList = new Vector<AlgorithmOutline>();

    while (elements.hasMoreElements()) {
      Algorithm algo = elements.nextElement();
      sendList.add(algo.getOutline());
    }
    return sendList.toArray(new AlgorithmOutline[sendList.size()]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AlgorithmInfo getInfo(Object id) throws PermissionException {
    if (id == null) {
      return null;
    }
    Algorithm algo = registeredAlgorithms.get(id);
    if (algo == null) {
      return null;
      // TODO Essa exceo precisa passar a ser tratada na RemoteTask
      // throw new ServiceFailureException(getAlgoNotFoundMessage(id));
    }
    return algo.getInfo();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AlgorithmInfo getInfo(String name) throws PermissionException {
    Collection<Algorithm> algorithmCollection = registeredAlgorithms.values();
    Iterator<Algorithm> algorithmIterator = algorithmCollection.iterator();
    while (algorithmIterator.hasNext()) {
      Algorithm algo = algorithmIterator.next();
      AlgorithmInfo info = algo.getInfo();
      if (info.getName().equals(name)) {
        return info;
      }
    }
    return null;
    // TODO Essa exceo precisa passar a ser tratada na RemoteTask
    // throw new ServiceFailureException(getAlgoNotFoundMessage(name));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<AlgorithmProperty> getVersionProperties() {
    return Algorithm.getVersionProperties();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized AlgorithmInfo includePlatform(Object algoId,
    Object versionId, String platform) {
    checkAlgorithmAdminPermission(algoId);

    Algorithm algo = registeredAlgorithms.get(algoId);
    try {
      algo.includePlatform(versionId, platform);
    }
    catch (ServerException se) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.algo.platform.create_failed", platform, versionId,
        algo.getInfo().getName()), se);
      return null;
    }
    String[] path = new String[] { getSymbolicRootName(), algo.getInfo()
      .getName(), versionId.toString(), AlgorithmVersionInfo.BIN_DIR };
    String description = getMessageFormatted(HISTORY_INCLUDE_PLATFORM,
      platform);
    appendHistory(path, description);
    Server.logInfoMessage("Plataforma " + platform + " inserida para verso "
      + versionId + " do " + "algoritmo " + algo.getInfo().getName());
    fireModifyEvent(algo.getInfo());
    return algo.getInfo();
  }

  /**
   * Inicializao do servio.
   *
   * @throws ServerException se ocorrer um erro durante a inicializao.
   */
  @Override
  public void initService() throws ServerException {
    transaction = new TransactionManager();
    int reloadAlgsIntervalInMinutes = getIntProperty(
      PROP_RELOAD_ALGS_INTERVAL_IN_MIN);
    Server.logInfoMessage("Intervalo de recarga de algoritmos (em min): "
      + reloadAlgsIntervalInMinutes);
    if (reloadAlgsIntervalInMinutes != 0) {
      reloadAlgsThread = new ReloadAlgsThread(reloadAlgsIntervalInMinutes);
      Server.logFineMessage("Thread de recarga de algoritmos criada.");
    }

    loadInvalidsAlgoPacksTokens();
    /** Criao da thread de execuo do servio */
    createCleanupTempDirThread();

    readExecutionLocationConfiguration();
    loadParameters();
    loadAlgorithms();

    isFirstCategoryLoad = true;
    loadCategories();
    initHistoryTable();
  }

  /**
   * Carrega tokens invlidos, que no foram removidos na limpeza realizada
   * enquanto o ltimo servidor estava no ar.
   */
  private void loadInvalidsAlgoPacksTokens() {
    try {
      File tempDir = new File(tempDirAlgorithmPath);
      Server.checkDirectory(tempDirAlgorithmPath);
      algoPackTokenMap = new Hashtable<String, ImportAlgorithmsPackDataInfo>();
      File[] tokensDirs = tempDir.listFiles();
      for (File tokenDir : tokensDirs) {
        if (tokenDir.isDirectory() && !tokenDir.isHidden()) {
          String tokenStr = tokenDir.getName().trim();
          algoPackTokenMap.put(tokenStr, new ImportAlgorithmsPackDataInfo());
          Server.logFineMessage("Carregando token invlido:" + tokenStr);
        }
      }
    }
    catch (Exception e) {
      throw new ServiceFailureException(getMessageFormatted(
        "AlgoService.error.algorithm_pack.tokens.load", tempDirAlgorithmPath),
        e);
    }
  }

  /**
   * Criao da thread que faz a limpeza do diretrio temporrio usado para
   * operaes de importao e exportao sobre um PA.
   *
   * @see #initService()
   */
  private void createCleanupTempDirThread() {
    /*
     * Busca das propriedades que identifica o intervalo de tempo para realizar
     * a limpeza do diretrio temporrio
     */
    int cleanupIntervalInSec = getIntProperty(
      PROP_CLEANUP_TEMP_DIR_INTERVAL_IN_SEC);

    //Tempo em milisegundos
    final long sleepTimeMs = cleanupIntervalInSec * 1000;
    Server.logInfoMessage(
      "Intervalo de limpeza de um diretrio que representa um token de um PA: "
        + "" + cleanupIntervalInSec + " segundos.");
    cleanupTempDirThread = new Thread(new Runnable() {
      @Override
      public void run() {
        try {
          while (!exitCleanupThread) {
            try {
              //Realiza a limpeza
              cleanupPATempDir();

              Thread.sleep(sleepTimeMs);
            }
            catch (InterruptedException ie) {
              Server.logWarningMessage(
                "Gravao de dados do SGA interrompido!");
            }
          }
          Server.logWarningMessage(
            "Finalizando thread de gravao de dados...");
        }
        catch (Throwable t) {
          final String msg = (new Date())
            + " - Erro na limpeza do diretrio temporrio: "
            + tempDirAlgorithmPath;
          System.err.println(msg);
          t.printStackTrace(System.err);
          Server.logSevereMessage(msg, t);
        }
      }
    });
    exitCleanupThread = false;
    cleanupTempDirThread.setName(this.getClass().getSimpleName() + "::"
      + "CleanupTempDirThread");
    cleanupTempDirThread.start();
    Server.logFineMessage(
      "Criao da thread para limpeza do diretrio temporrio usado para "
        + "operaes de importao e exportao sobre um PA.");
  }

  /**
   * Realiza a limpeza do diretrio temporrio usado para operaes de
   * importao e exportao sobre um PA.
   *
   * @throws RemoteException erro de rmi
   */
  protected void cleanupPATempDir() throws RemoteException {
    Enumeration<String> tokens = algoPackTokenMap.keys();
    while (tokens.hasMoreElements()) {
      String token = tokens.nextElement();
      removeAlgoPackTokensDirs(token);
    }
    algoPackTokenMap.clear();
  }

  /**
   * L a propriedade que define a configurao padro para o local de execuo
   * de algoritmos do sistema.
   *
   * @throws ServerException em caso de erro na configurao.
   */
  private void readExecutionLocationConfiguration() throws ServerException {
    String algorithmExecutionLocationString = getStringProperty(
      "algorithmExecutionLocation");
    ExecutionLocationConverter converter = new ExecutionLocationConverter();
    try {
      defaultAlgorithmExecutionLocation = converter.valueOf(
        algorithmExecutionLocationString);
    }
    catch (ParseException e) {
      String acceptedValues = "";
      for (ExecutionLocation value : ExecutionLocation.values()) {
        String[] possibleMatches = converter.getPossibleMatches(value);
        for (int i = 0; i < possibleMatches.length; i++) {
          String execValue = possibleMatches[i];
          acceptedValues += execValue;
          acceptedValues += (i == possibleMatches.length - 1) ? "\n" : ", ";
        }

      }
      acceptedValues = acceptedValues.substring(0, acceptedValues.length() - 1);
      throw new ServerException(getMessageFormatted(
        "AlgoService.error.algo.execution_location",
        "algorithmExecutionLocation", getName(),
        algorithmExecutionLocationString, acceptedValues));
    }
  }

  /**
   * Verifica se h transao ativa de gerncia do repositrio.
   *
   * @return true caso haja transao ativa, false caso contrrio.
   */
  @Override
  public boolean isLocked() {
    return transaction.isLocked();
  }

  /**
   * Inicia uma transao de gerncia do repositrio de algoritmos (equivale 
   * obteno de um "lock").
   *
   * @param cb Callback que  utilizada pelo servio para saber se quem efetuou
   *        o lock ainda est respondendo.
   * @return true caso o lock tenha sido feito com sucesso, false caso
   *         contrrio.
   */
  @Override
  public boolean lock(TransactionCallbackInterface cb) {
    checkAlgorithmServerAdminPermission();
    return transaction.lock(cb);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized boolean reloadAlgorithms() {
    return reloadAlgorithms(true);
  }

  /**
   * Recarrega os algoritmos.
   *
   * @param notifyAllUsers true indica que todos os usurios devem ser
   *        notificados, false apenas o usurio que solicitou a recarga
   * @return true caso a recarga tenha sido efetuada com sucesso
   */
  public synchronized boolean reloadAlgorithms(boolean notifyAllUsers) {
    checkAlgorithmServerAdminPermission();
    TransactionCallbackInterface cb = new TransactionCallbackInterface() {
      @Override
      public boolean isAlive() {
        return true;
      }
    };
    final String uName = getUser().getName();
    Server.logInfoMessage("Pedido de recarga de algoritmos vinda de " + uName);
    try {
      if (!lock(cb)) {
        Server.logSevereMessage(getMessageFormatted(
          "AlgoService.error.algo.block_reload_failed"));
        return false;
      }
      loadParameters();
      loadAlgorithms();
      loadCategories();
      unlock(cb);
      Server.logInfoMessage("Recarga de algoritmos efetuada!");

      final AlgorithmsReloadNotification data =
        new AlgorithmsReloadNotification(getSenderName());
      if (notifyAllUsers) {
        MessageService.getInstance().sendToAll(new Message(data));
      }
      else {
        String[] ids = new String[] { (String) getUser().getId() };
        try {
          MessageService.getInstance().send(new Message(data), ids);
        }
        catch (RemoteException e) {
          Server.logSevereMessage(
            "Erro ao enviar evento de recarregar lista de algoritmos.", e);
        }
      }
      AlgoEvent action = new AlgoEvent(AlgoEvent.RELOAD, null);
      sendEvent(action);
      return true;
    }
    catch (ServerException se) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.algo.reload_failed"), se);
      return false;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized boolean removeAlgorithm(Object algoId) {
    checkAlgorithmAdminPermission(algoId);

    Server.logInfoMessage("Tentando remover o algoritmo: " + algoId);
    Algorithm algo = registeredAlgorithms.get(algoId);
    try {
      algo.remove();
    }
    catch (ServerException se) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.algo.remove_failed", algoId), se);
      return false;
    }
    registeredAlgorithms.remove(algoId);
    AlgorithmInfo info = algo.getInfo();
    String[] path = new String[] { getSymbolicRootName() };
    String description = getMessageFormatted(HISTORY_REMOVE_ALGORITHM, algoId);
    appendHistory(path, description);
    Server.logInfoMessage("Algoritmo " + algoId + " removido");
    AlgoEvent action = new AlgoEvent(AlgoEvent.DELETE, info);
    sendEvent(action);
    return true;
  }

  /**
   * Remove arquivos executveis de uma verso de algoritmo.
   *
   * @param version Verso que detm os arquivos.
   * @param platform Plataforma dos executveis
   * @param files Representao dos arquivos a serem removidos.
   * @throws ServiceFailureException caso no tenha sido possvel remover o
   *         arquivo.
   */
  @Override
  public synchronized void removeExecutableFiles(AlgorithmVersionInfo version,
    String platform, FileInfo[] files) {

    if (null == version) {
      throw new IllegalArgumentException(getParameterNullMessage("version"));
    }
    if (null == platform) {
      throw new IllegalArgumentException(getParameterNullMessage("platform"));
    }
    if (null == files) {
      throw new IllegalArgumentException(getParameterNullMessage("files"));
    }
    if (0 == files.length) {
      throw new IllegalArgumentException(getParameterInvalidMessage(
        "files.length == 0"));
    }

    Object algorithmId = version.getInfo().getId();
    checkAlgorithmAdminPermission(algorithmId);

    Algorithm algorithm = registeredAlgorithms.get(algorithmId);
    try {
      algorithm.removeExecutableFiles(version, platform, files);

      String[] basePath = new String[5];
      basePath[0] = getSymbolicRootName();
      basePath[1] = algorithm.getInfo().getName();
      basePath[2] = version.getId().toString();
      basePath[3] = version.getBinDirName();
      basePath[4] = platform;
      for (FileInfo file : files) {
        String description = getMessageFormatted(HISTORY_REMOVE_FILE, file
          .getPath());
        appendHistory(basePath, description);
      }

      fireModifyEvent(algorithm.getInfo());
    }
    catch (Exception e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * Remove arquivos de configurao de uma verso de algoritmo.
   *
   * @param version Verso que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * @throws ServiceFailureException caso no tenha sido possvel remover o
   *         arquivo.
   */
  @Override
  public synchronized void removeConfigurationFiles(
    AlgorithmVersionInfo version, FileInfo[] files) {

    if (null == version) {
      throw new IllegalArgumentException(getParameterNullMessage("version"));
    }
    if (null == files) {
      throw new IllegalArgumentException(getParameterNullMessage("files"));
    }
    if (0 == files.length) {
      throw new IllegalArgumentException(getParameterInvalidMessage(
        "files.length == 0"));
    }

    Object algorithmId = version.getInfo().getId();
    checkAlgorithmAdminPermission(algorithmId);

    Algorithm algorithm = registeredAlgorithms.get(algorithmId);
    try {
      algorithm.removeConfigurationFiles(version, files);

      String[] basePath = new String[4];
      basePath[0] = getSymbolicRootName();
      basePath[1] = algorithm.getInfo().getName();
      basePath[2] = version.getId().toString();
      basePath[3] = version.getConfiguratorDirName();
      for (FileInfo file : files) {
        String description = getMessageFormatted(HISTORY_REMOVE_FILE, file
          .getPath());

        appendHistory(basePath, description);
      }

      fireModifyEvent(algorithm.getInfo());
    }
    catch (Exception e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * Remove arquivos de documentao de uma verso de algoritmo.
   *
   * @param version Verso que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * @throws ServiceFailureException caso no tenha sido possvel remover o
   *         arquivo.
   */
  @Override
  public synchronized void removeDocumentationFiles(
    AlgorithmVersionInfo version, FileInfo[] files) {

    if (null == version) {
      throw new IllegalArgumentException(getParameterNullMessage("version"));
    }
    if (null == files) {
      throw new IllegalArgumentException(getParameterNullMessage("files"));
    }
    if (0 == files.length) {
      throw new IllegalArgumentException(getParameterInvalidMessage(
        "files.length == 0"));
    }

    Object algorithmId = version.getInfo().getId();
    checkAlgorithmAdminPermission(algorithmId);

    Algorithm algorithm = registeredAlgorithms.get(algorithmId);
    try {
      algorithm.removeDocumentationFiles(version, files);

      String[] basePath = new String[4];
      basePath[0] = getSymbolicRootName();
      basePath[1] = algorithm.getInfo().getName();
      basePath[2] = version.getId().toString();
      basePath[3] = version.getDocumentationDirName();
      for (FileInfo file : files) {
        String description = getMessageFormatted(HISTORY_REMOVE_FILE, file
          .getPath());
        appendHistory(basePath, description);
      }

      fireModifyEvent(algorithm.getInfo());
    }
    catch (Exception e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void removeReleaseNotesFiles(AlgorithmVersionInfo version,
    FileInfo[] files) {

    if (null == version) {
      throw new IllegalArgumentException(getParameterNullMessage("version"));
    }
    if (null == files) {
      throw new IllegalArgumentException(getParameterNullMessage("files"));
    }
    if (0 == files.length) {
      throw new IllegalArgumentException(getParameterInvalidMessage(
        "files.length == 0"));
    }

    Object algorithmId = version.getInfo().getId();
    checkAlgorithmAdminPermission(algorithmId);

    Algorithm algorithm = registeredAlgorithms.get(algorithmId);
    try {
      algorithm.removeReleaseNotesFiles(version, files);

      String[] basePath = new String[4];
      basePath[0] = getSymbolicRootName();
      basePath[1] = algorithm.getInfo().getName();
      basePath[2] = version.getId().toString();
      basePath[3] = version.getDocumentationDirName();
      for (FileInfo file : files) {
        String description = getMessageFormatted(HISTORY_REMOVE_FILE, file
          .getPath());
        appendHistory(basePath, description);
      }

      fireModifyEvent(algorithm.getInfo());
    }
    catch (Exception e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized AlgorithmInfo removePlatform(Object algoId,
    Object versionId, String platform) {
    checkAlgorithmAdminPermission(algoId);

    Algorithm algo = registeredAlgorithms.get(algoId);
    try {
      algo.removePlatform(versionId, platform);
    }
    catch (ServerException se) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.algo.platform.remove_failed", platform, versionId,
        algo.getInfo().getName()), se);
      return null;
    }
    String[] path = new String[] { getSymbolicRootName(), algo.getInfo()
      .getName(), versionId.toString(), AlgorithmVersionInfo.BIN_DIR };
    String description = getMessageFormatted(HISTORY_REMOVE_PLATFORM, platform);
    appendHistory(path, description);
    Server.logInfoMessage("Plataforma " + platform + " removida da verso "
      + versionId + " do " + "algoritmo " + algo.getInfo().getName());
    fireModifyEvent(algo.getInfo());
    return algo.getInfo();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized AlgorithmInfo removeVersion(Object algoId,
    Object versionId) {
    boolean toNotify = true;
    return removeVersion(algoId, versionId, toNotify);
  }

  /**
   * Remove uma verso do algoritmo, indicando se deve ou no ser disparada uma
   * notificao desse evento de modificao do algoritmo.
   *
   * @param algoId identificador do algoritmo
   * @param versionId identificador da verso do algoritmo
   * @param toNotify se true, os observadores desse servio sero notificados da
   *        alterao do algoritmo, caso contrrio, no ser enviada a
   *        notificao
   * @return o objeto com a informao do algoritmo modificado
   */
  private AlgorithmInfo removeVersion(Object algoId, Object versionId,
    boolean toNotify) {
    checkAlgorithmAdminPermission(algoId);

    Server.logInfoMessage("Tentando remover verso " + versionId
      + " do algoritmo: " + algoId);
    Algorithm algo = registeredAlgorithms.get(algoId);
    try {
      algo.removeVersion(versionId);
    }
    catch (ServerException se) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.algo.version.remove_failed", versionId, algo
          .getInfo().getName()), se);
      return null;
    }
    String[] path = new String[] { getSymbolicRootName(), algo.getInfo()
      .getName() };
    String description = getMessageFormatted(HISTORY_REMOVE_VERSION, versionId);
    appendHistory(path, description);
    Server.logInfoMessage("Verso " + versionId + " removida do algoritmo "
      + algo.getInfo().getName());

    if (toNotify) {
      fireModifyEvent(algo.getInfo());
    }
    return algo.getInfo();
  }

  //TODO A remover quando todos sistemas CSBase forem assinados {

  /**
   * {@inheritDoc}
   */
  @Override
  public String retrieveConfigUploadURL(Object algoId, Object versionId) {
    checkAlgorithmAdminPermission(algoId);
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmInfo info = algo.getInfo();
    AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);
    String algoPath = versionInfo.getConfiguratorDirPath();
    try {
      File file = new File(algoPath);
      if (!file.exists()) {
        throw new ServiceFailureException(getMessageFormatted(
          ERROR_CONF_VERSION_NOT_FOUND, algoId, versionId));
      }
      String canonicalPath = file.getCanonicalPath();
      AlgorithmObserver obs = new AlgorithmObserver(algoId, versionId,
        AlgorithmObserver.UPLOAD_CONFIG, Service.getUser().getId());
      String uploadPresentation = getStringProperty("uploadConfigPresentation");
      String uploadResult = getStringProperty("uploadResult");
      HttpService httpService = getHttpService();
      return httpService.getUploadURL(canonicalPath, uploadPresentation,
        uploadResult, obs);
    }
    catch (IOException ex) {
      throw new ServiceFailureException(ex.getMessage(), ex);
    }
    catch (OperationFailureException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String retrieveDocUploadURL(Object algoId, Object versionId) {
    checkAlgorithmAdminPermission(algoId);
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmInfo info = algo.getInfo();
    AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);
    String docPath = versionInfo.getDocDirPath();
    try {
      File file = new File(docPath);
      if (!file.exists()) {
        throw new ServiceFailureException(getMessageFormatted(
          ERROR_DOC_VERSION_NOT_FOUND, algoId, versionId));
      }
      String canonicalPath = file.getCanonicalPath();
      AlgorithmObserver obs = new AlgorithmObserver(algoId, versionId,
        AlgorithmObserver.UPLOAD_HTML, Service.getUser().getId());
      String uploadPresentation = getStringProperty("uploadHtmlPresentation");
      String uploadResult = getStringProperty("uploadResult");
      HttpService httpService = getHttpService();
      return httpService.getUploadURL(canonicalPath, uploadPresentation,
        uploadResult, obs);
    }
    catch (IOException ex) {
      throw new ServiceFailureException(ex.getMessage(), ex);
    }
    catch (OperationFailureException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String retrieveReleaseNotesUploadURL(Object algoId, Object versionId) {
    checkAlgorithmAdminPermission(algoId);
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmInfo info = algo.getInfo();
    AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);
    String docPath = versionInfo.getDocDirPath();
    try {
      File file = new File(docPath);
      if (!file.exists()) {
        throw new ServiceFailureException(getMessageFormatted(
          ERROR_RELEASE_NOTES_VERSION_NOT_FOUND, algoId, versionId));
      }
      String canonicalPath = file.getCanonicalPath();
      AlgorithmObserver obs = new AlgorithmObserver(algoId, versionId,
        AlgorithmObserver.UPLOAD_RELEASE_NOTES, Service.getUser().getId());
      String uploadPresentation = getStringProperty("uploadReleaseNotes");
      String uploadResult = getStringProperty("uploadResult");
      HttpService httpService = getHttpService();
      return httpService.getUploadURL(canonicalPath, uploadPresentation,
        uploadResult, obs);
    }
    catch (IOException ex) {
      throw new ServiceFailureException(ex.getMessage(), ex);
    }
    catch (OperationFailureException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }
  //TODO A remover quando todos sistemas CSBase forem assinados

  /**
   * {@inheritDoc}
   */
  @Override
  public String retrieveDownloadURL(Object algoId, String[] filePath) {
    //   checkAlgorithmAdminPermission(algoId);
    checkAlgorithmExecutePermission(algoId);
    Algorithm algo = registeredAlgorithms.get(algoId);
    String filePathText = FileUtils.joinPath(filePath);
    String baseDir = FileUtils.joinPath(algo.getInfo().getDirectory(),
      File.separatorChar, algorithmRepositoryPath);
    File file = new File(baseDir, filePathText);
    String absFilePath = file.getAbsolutePath();
    try {
      HttpService httpService = getHttpService();
      return httpService.getFileDownloadURL(absFilePath);
    }
    catch (IOException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
    catch (OperationFailureException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  //TODO A remover quando todos sistemas CSBase forem assinados

  /**
   * {@inheritDoc}
   */
  @Override
  public String retrieveExecUploadURL(Object algoId, Object versionId,
    String platform) {
    checkAlgorithmAdminPermission(algoId);
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmInfo info = algo.getInfo();
    AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);
    String platPath = versionInfo.getPlatformPath(platform);
    try {
      File file = new File(platPath);
      if (!file.exists()) {
        throw new ServiceFailureException(getMessageFormatted(
          ERROR_PLAT_VERSION_NOT_FOUND, algoId, versionId, platform));
      }
      String canonicalPath = file.getCanonicalPath();
      AlgorithmObserver obs = new AlgorithmObserver(algoId, versionId, platform,
        Service.getUser().getId());
      String uploadPresentation = getStringProperty("uploadExecPresentation");
      String uploadResult = getStringProperty("uploadResult");
      HttpService httpService = getHttpService();
      return httpService.getUploadURL(canonicalPath, uploadPresentation,
        uploadResult, obs);
    }
    catch (IOException ex) {
      throw new ServiceFailureException(ex.getMessage(), ex);
    }
    catch (OperationFailureException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String retrieveVersionUploadURL(final Object algoId) {
    checkAlgorithmAdminPermission(algoId);
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmInfo info = algo.getInfo();
    File repositoryDirectory = new File(algorithmRepositoryPath);
    File algorithmDirectory = new File(repositoryDirectory, info
      .getDirectory());
    File file = new File(algorithmDirectory, AlgorithmInfo.VERSIONS_DIR);
    if (!file.exists()) {
      String userMessage = getMessageFormatted(
        "AlgoService.error.algo.version_upload.dir.invalid", info.getName());
      String logDetails = getMessageFormatted(
        "AlgoService.error.detailed.algo.version_upload.dir.invalid", file
          .getAbsolutePath());
      Server.logSevereMessage(userMessage + logDetails);
      throw new ServiceFailureException(userMessage);
    }
    final String directoryPath;
    try {
      directoryPath = file.getCanonicalPath();
    }
    catch (IOException e) {
      String userMessage = getMessageFormatted(
        "AlgoService.error.algo.version_upload.io_error", info.getName());
      Server.logSevereMessage(userMessage, e);
      throw new ServiceFailureException(userMessage);
    }
    User user = Service.getUser();
    final Object userId = user.getId();
    String uploadPresentation = getStringProperty("uploadVersionPresentation");
    String uploadResult = getStringProperty("uploadResult");
    try {
      return getHttpService().getUploadURL(directoryPath, uploadPresentation,
        uploadResult, new UploadHandler() {

          @Override
          public void execute(String filePath) throws HttpServiceException {
            try {
              Algorithm algorithm = registeredAlgorithms.get(algoId);
              if (algorithm == null) {
                String detailedMessage = getMessageFormatted(
                  "AlgoService.error.detailed.algo.version_upload.algo" + ""
                    + ".not_found", algoId, filePath);
                Server.logSevereMessage(detailedMessage);
                throw new HttpServiceException(getMessageFormatted(
                  "AlgoService.error.algo.version_upload.algo.not_found",
                  algoId));
              }
              Unzip unzip = new Unzip(new File(filePath));
              List<ZipEntry> zipEntries = unzip.listZipEntries();
              if (zipEntries.isEmpty()) {
                String userMessage = getMessageFormatted(
                  "AlgoService.error.algo.version_upload.invalid_zip");
                String detailedMessage = getMessageFormatted(
                  "AlgoService.error.detailed.algo.version_upload.empty_zip",
                  filePath, algoId, userId);
                Server.logSevereMessage(userMessage + detailedMessage);
                throw new HttpServiceException(userMessage);
              }

              ZipEntry rootZipEntry = null;
              String rootDirectoryName = null;
              for (ZipEntry zipEntry : zipEntries) {
                String[] path = FileUtils.splitPath(zipEntry.getName(), "/");
                if (path.length == 1) {
                  if (rootZipEntry != null) {
                    String userMessage = getMessageFormatted(
                      "AlgoService.error.algo.version_upload.invalid_zip");
                    String detailedMessage = getMessageFormatted(
                      "AlgoService.error.detailed.algo.version_upload" + ""
                        + ".invalid_root_zip", filePath, algoId, userId);
                    Server.logSevereMessage(userMessage + "\n"
                      + detailedMessage);
                    throw new HttpServiceException(userMessage);
                  }
                  rootZipEntry = zipEntry;
                  rootDirectoryName = zipEntry.getName();
                  rootDirectoryName = rootDirectoryName.replace("/", "");
                }
              }

              if (rootZipEntry == null) {
                String userMessage = getMessageFormatted(
                  "AlgoService.error.algo.version_upload.invalid_zip");
                String detailedMessage = getMessageFormatted(
                  "AlgoService.error.detailed.algo.version_upload" + ""
                    + ".no_root_name_zip", filePath, algoId, userId);
                Server.logSevereMessage(userMessage + "\n" + detailedMessage);
                throw new HttpServiceException(userMessage);
              }

              if (!rootZipEntry.isDirectory()) {
                String userMessage = getMessageFormatted(
                  "AlgoService.error.algo.version_upload.invalid_zip");
                String detailedMessage = getMessageFormatted(
                  "AlgoService.error.detailed.algo.version_upload.no_root_zip",
                  filePath, algoId, userId, rootZipEntry);
                Server.logSevereMessage(userMessage + "\n" + detailedMessage);
                throw new HttpServiceException(userMessage);
              }

              String regex = "v_([0-9]{3})_([0-9]{3})_([0-9]{3})";
              Pattern versionPattern = Pattern.compile(regex);
              Matcher versionMatcher = versionPattern.matcher(
                rootDirectoryName);
              if (!versionMatcher.matches()) {
                String userMessage = getMessageFormatted(
                  "AlgoService.error.algo.version_upload.invalid_zip");
                String detailedMessage = getMessageFormatted(
                  "AlgoService.error.detailed.algo.version_upload" + ""
                    + ".invalid_root_zip_format", filePath, algoId, userId,
                  rootDirectoryName, regex);
                Server.logSevereMessage(userMessage + "\n" + detailedMessage);
                throw new HttpServiceException(userMessage);
              }

              int major = Integer.parseInt(versionMatcher.group(1));
              int minor = Integer.parseInt(versionMatcher.group(2));
              int patch = Integer.parseInt(versionMatcher.group(3));

              AlgorithmVersionId algorithmVersionId = new AlgorithmVersionId(
                major, minor, patch);
              AlgorithmInfo algorithmInfo = algorithm.getInfo();
              String algorithmName = algorithmInfo.getName();
              if (algorithmInfo.getVersionInfo(algorithmVersionId) != null) {
                String versionStr = String.format("%d.%d.%d", major, minor,
                  patch);
                String userMessage = getMessageFormatted(
                  "AlgoService.error.algo.version.already_exists", versionStr,
                  algorithmName);
                String detailedMessage = getMessageFormatted(
                  "AlgoService.error.detailed.algo.version_upload.more_info",
                  filePath, algoId, userId);
                Server.logSevereMessage(userMessage + "\n" + detailedMessage);
                throw new HttpServiceException(userMessage);
              }

              File repositoryDirectory = new File(algorithmRepositoryPath);
              File algorithmDirectory = new File(repositoryDirectory,
                algorithmInfo.getDirectory());
              File versionsDirectory = new File(algorithmDirectory,
                AlgorithmInfo.VERSIONS_DIR);

              unzip.decompress(versionsDirectory);

              if (!new File(filePath).delete()) {
                String detailedMessage = String.format(
                  "Erro ao tentar excluir o arquivo %s.\n" + "Por favor, "
                    + "tente exclu-lo manualmente.\n", filePath);
                Server.logWarningMessage(detailedMessage);
              }

              File versionDirectory = new File(versionsDirectory,
                rootDirectoryName);
              File binaryDirectory = new File(versionDirectory,
                AlgorithmVersionInfo.BIN_DIR);

              if (!binaryDirectory.isDirectory()) {
                String userMessage = getMessageFormatted(
                  "AlgoService.error.algo.version_upload.invalid_zip");
                String detailedMessage = getMessageFormatted(
                  "AlgoService.error.detailed.algo.version_upload.no_bin_dir",
                  filePath, algoId, userId);
                Server.logSevereMessage(userMessage + "\n" + detailedMessage);
                if (!FileUtils.delete(versionDirectory)) {
                  detailedMessage = getMessageFormatted(
                    "AlgoService.error.version_upload.version_dir" + ""
                      + ".delete_failed", versionDirectory.getAbsolutePath());
                  Server.logWarningMessage(detailedMessage);
                }
                throw new HttpServiceException(userMessage);
              }

              File[] platformDirectories = binaryDirectory.listFiles();

              for (File platformDirectory : platformDirectories) {
                if (!platformDirectory.isDirectory()) {
                  String userMessage = getMessageFormatted(
                    "AlgoService.error.algo.version_upload.invalid_zip");
                  String detailedMessage = getMessageFormatted(
                    "AlgoService.error.detailed.algo.version_upload.file"
                      + ".not_dir", userMessage, platformDirectory, filePath,
                    algoId, userId);
                  Server.logSevereMessage(userMessage + "\n" + detailedMessage);
                  if (!FileUtils.delete(versionDirectory)) {
                    detailedMessage = getMessageFormatted(
                      "AlgoService.error.detailed.algo.version.delete_failed",
                      versionDirectory.getAbsolutePath());
                    Server.logWarningMessage(detailedMessage);
                  }
                  throw new HttpServiceException(userMessage);
                }

                File[] binaryFiles = platformDirectory.listFiles();
                for (File binaryFile : binaryFiles) {
                  FileSystem.enableExecutionPermission(binaryFile
                    .getAbsolutePath());
                }
              }

              algorithm.readVersion(versionsDirectory.getAbsolutePath(),
                versionDirectory);

              fireModifyEvent(algorithmInfo);

            }
            catch (IOException e) {
              String userMessage = getMessageFormatted(
                "AlgoService.error.algo.version_upload.io_failed");
              String detailedMessage = getMessageFormatted(
                "AlgoService.error.detailed.algo.version_upload.more_info",
                filePath, algoId, userId);
              Server.logSevereMessage(userMessage + "\n" + detailedMessage, e);
              throw new HttpServiceException(userMessage);
            }
            catch (ServerException e) {
              String userMessage = getMessageFormatted(
                "AlgoService.error.algo.version.read_failed");
              String detailedMessage = getMessageFormatted(
                "AlgoService.error.detailed.algo.version_upload.more_info",
                filePath, algoId, userId);
              Server.logSevereMessage(userMessage + "\n" + detailedMessage, e);
              throw new HttpServiceException(userMessage);
            }
          }
        });
    }
    catch (OperationFailureException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  //TODO A remover quando todos sistemas CSBase forem assinados

  /**
   * {@inheritDoc}
   */
  @Override
  public List<HistoryRecord> retrieveHistory(String[] viewPath) {
    if (viewPath == null) {
      throw new IllegalArgumentException(getParameterNullMessage("viewPath"));
    }
    Map historyTable = readHistoryTable();
    if (historyTable == null) {
      historyTable = resetHistoryTable();
    }
    Map dirTable = getDirTable(viewPath, historyTable);
    List rawList = getDirHistory(dirTable);
    List<HistoryRecord> recordList = getRecordList(viewPath, rawList);
    return recordList;
  }

  /**
   * Trmino do servio de gerncia de algoritmos.
   */
  @Override
  public void shutdownService() {
    if (reloadAlgsThread != null) {
      reloadAlgsThread.interrupt();
    }
    exitCleanupThread = true;
    if (cleanupTempDirThread != null) {
      cleanupTempDirThread.interrupt();
    }

    if (categorySet != null && !categorySet.isCategorySetSaved()) {
      saveCategoriesBeforeShutdown();
    }
  }

  /**
   * Tenta salvar o conjunto de categorias antes do trmino do servio. O
   * resultado dessa operao  registrado no log do servio.
   */
  private void saveCategoriesBeforeShutdown() {
    Server.logInfoMessage("Tentando salvar o arquivo de categorias "
      + CATEGORIES_FILE_NAME + "com" + " problema na ltima persistncia...");
    try {
      if (writeCategoriesFile(categorySet)) {
        Server.logInfoMessage("Shutdown do servio - o arquivo de categorias "
          + CATEGORIES_FILE_NAME + " foi salvo com sucesso.");
      }
    }
    catch (CategoriesFileNotSavedException ex) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.category.file.not_saved", CATEGORIES_FILE_NAME));
    }
  }

  /**
   * Termina a transao de gerncia do repositrio de algoritmos.
   *
   * @param cb Callback que  utilizada pelo servio para saber se quem efetuou
   *        o lock ainda est respondendo. Deve ser passado o mesmo objeto
   *        inicialmente passado para o mtodo lock.
   */
  @Override
  public void unlock(TransactionCallbackInterface cb) {
    checkAlgorithmServerAdminPermission();
    transaction.unlock(cb);
  }

  /**
   * Dispara uma notificao para os observadores da classe de que determinado
   * algoritmo mudou.
   *
   * @param info informaes sobre o algoritmo que sofreu mudanas.
   */
  protected void fireModifyEvent(AlgorithmInfo info) {
    AlgoEvent event = new AlgoEvent(AlgoEvent.MODIFY, info);
    sendEvent(event);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return true;
  }

  //TODO A remover quando todos sistemas CSBase forem assinados

  /**
   * Este mtodo  chamado pelo observador quando o upload do configurador
   * termina para notificar os clientes.
   *
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso do algoritmo.
   * @param filePath caminho do arquivo recm-carregado.
   * @param userId identificador do usurio.
   */
  void updateConfigurator(Object algoId, Object versionId, String filePath,
    Object userId) {
    Algorithm algo = registeredAlgorithms.get(algoId);
    if (algo == null) {
      return;
    }
    AlgorithmInfo info = algo.getInfo();
    AlgorithmVersionInfo vInfo = info.getVersionInfo(versionId);
    FileInfo from = new FileInfo(new File(filePath));
    vInfo.addConfigurator(from);
    String[] path = new String[] { getSymbolicRootName(), algo.getInfo()
      .getName(), versionId.toString(), AlgorithmVersionInfo.CONFIGURATOR_DIR };
    String description = MessageFormat.format(getString(
      HISTORY_UPDATE_CONFIGURATOR), new Object[] { from.getName() });
    appendHistory(userId, path, description);
    fireModifyEvent(info);
  }

  /**
   * Este mtodo  chamado pelo observador quando o upload do arquivo de
   * documentao termina para notificar os clientes.
   *
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso do algoritmo.
   * @param filePath caminho do arquivo recm-carregado.
   * @param userId identificador do usurio que carregou o arquivo.
   */
  void updateDoc(Object algoId, Object versionId, String filePath,
    Object userId) {
    Algorithm algo = registeredAlgorithms.get(algoId);
    if (algo == null) {
      return;
    }
    AlgorithmInfo info = algo.getInfo();
    AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);
    FileInfo from = new FileInfo(new File(filePath));
    versionInfo.addDocumentation(from);
    String[] path = new String[] { getSymbolicRootName(), algo.getInfo()
      .getName(), versionId.toString(),
        AlgorithmVersionInfo.DOCUMENTATION_DIR };
    String description = MessageFormat.format(getString(HISTORY_UPDATE_DOC),
      new Object[] { from.getName() });
    appendHistory(userId, path, description);
    fireModifyEvent(info);
  }

  /**
   * <p>
   * Este mtodo  chamado pelo observador quando o upload de um executvel
   * termina para notificar os clientes. Caso o arquivo carregado seja um zip,
   * seu contedo  extrado. Como essa operao  possivelmente demorada, ela 
   * executada por uma thread.
   * </p>
   * OBS: durante o upload, eventuais permisses de execuo so perdidas pela
   * api que realiza a carga. Para compensar essa perda, este mtodo concede
   * permisso de execuo para cada arquivo carregado. Isso gera um efeito
   * colateral indesejado: arquivos no-executveis carregados tambm tero
   * permisso de execuo. Devido  dificuldade de identificar arquivos
   * realmente executveis dentre os carregados no servidor, e principalmente
   * pelo fato de uma permisso no-intencional de execuo no afetar em nada o
   * fluxo de funcionamento normal do sistema, optou-se por no tratar essa
   * inconsistncia.
   *
   * @param algoId Identificador do algoritmo.
   * @param versionId Identificador da verso do algoritmo.
   * @param platform Nome da plataforma.
   * @param filePath caminho do executvel recm-carregado.
   * @param userId identificador do usurio que carregou o executvel.
   * @see UpdateExecutableThread
   */
  void updateExecutable(Object algoId, Object versionId, String platform,
    String filePath, Object userId) {
    Thread t = new UpdateExecutableThread(algoId, versionId, platform, filePath,
      userId);
    t.start();
  }

  //TODO A remover quando todos sistemas CSBase forem assinados

  /**
   * Adiciona uma entrada ao histrico do servio de algoritmos.
   *
   * @param userId identificador do usurio que efetuou a ao.
   * @param path caminho no qual ocorreu uma ao que deve ser registrada no
   *        histrico.
   * @param description descrio da ao ocorrida.
   */
  private void appendHistory(Object userId, String[] path, String description) {
    User user = null;
    try {
      user = User.getUser(userId);
    }
    catch (Exception e) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.user.not_found", userId));
    }
    if (user == null) {
      return;
    }
    appendHistory(user, path, description);
  }

  /**
   * Adiciona uma entrada ao histrico do servio de algoritmos.
   *
   * @param path caminho no qual ocorreu uma ao que deve ser registrada no
   *        histrico.
   * @param description descrio da ao ocorrida.
   */
  private void appendHistory(String[] path, String description) {
    User user = Service.getUser();
    if (user == null) {
      return;
    }
    appendHistory(user, path, description);
  }

  /**
   * Adiciona uma entrada ao histrico do servio de algoritmos.
   *
   * @param user usurio que efetuou a ao.
   * @param path caminho no qual ocorreu uma ao que deve ser registrada no
   *        histrico.
   * @param description descrio da ao ocorrida.
   */
  private void appendHistory(User user, String[] path, String description) {
    List record = new ArrayList();
    record.add(HistoryRecord.DATE_COLUMN_INDEX, new Date());
    record.add(HistoryRecord.PATH_COLUMN_INDEX, path);
    record.add(HistoryRecord.LOGIN_COLUMN_INDEX, user.getLogin());
    record.add(HistoryRecord.NAME_COLUMN_INDEX, user.getName());
    record.add(HistoryRecord.DESCRIPTION_COLUMN_INDEX, description);
    insertRecord(path, record);
  }

  /**
   * Verifica se  o administrador ou, caso contrrio, se o usurio tem
   * permisso de administrao sobre o servio de algoritmos do servidor
   * corrente.
   */
  private void checkAlgorithmServerAdminPermission() {
    User user = Service.getUser();
    if (user.isAdmin()) {
      return;
    }
    Server server = Server.getInstance();
    String serverName = AdminPermission.LOCAL + server.getSystemName();
    Permission p = null;
    try {
      p = user.getMatchAttributesPermission(AlgorithmAdminPermission.class,
        serverName);

    }
    catch (Exception e) {
      String errorMsg = getMessageFormatted(
        "AlgoService.error.algo.admin.permission_failed", user.getLogin());
      Server.logSevereMessage(errorMsg, e);
    }
    if (p == null) {
      throw new PermissionException(getAlgoAdminServerPermissionErrorMessage(
        serverName));
    }
  }

  /**
   * Verifica se  o administrador ou, caso contrrio, se o usurio tem
   * permisso para gerenciar o algoritmo com o identificador especificado, no
   * servidor corrente.
   *
   * @param algoId identificador do algoritmo
   * @throws PermissionException usurio sem permisso para administrar o
   *         algoritmo
   */
  private void checkAlgorithmAdminPermission(Object algoId)
    throws PermissionException {
    if (algoId == null) {
      throw new IllegalArgumentException(getParameterNullMessage("algoId"));
    }
    Algorithm algo = registeredAlgorithms.get(algoId);
    if (algo == null) {
      throw new ServiceFailureException(getAlgoNotFoundMessage(algoId));
    }
    String algoName = algo.getInfo().getName();
    User user = Service.getUser();
    if (!user.isAdmin() && !isAlgorithmOwnerUser(algoName)) {
      checkAlgorithmAdminPermission(algoName);
    }
  }

  /**
   * Verifica se o usurio tem permisso para execuo do algoritmo (se for um
   * algoritmo simples), ou verifica se o usurio possui permisso para executar
   * todos os algoritmos que fazem parte do configurador (no caso de ser um
   * fluxo).
   *
   * Se o usurio for administrador, ele tem permisso sempre.
   *
   * @param algoId identificador do algoritmo
   * @throws PermissionException erro de permisso para execuo do algoritmo
   */
  private void checkAlgorithmExecutePermission(Object algoId)
    throws PermissionException {
    if (algoId == null) {
      throw new IllegalArgumentException(getParameterNullMessage("algoId"));
    }
    Algorithm algo = registeredAlgorithms.get(algoId);
    if (algo == null) {
      throw new ServiceFailureException(getAlgoNotFoundMessage(algoId));
    }
    String algoName = algo.getInfo().getName();
    User user = Service.getUser();
    if (user.isAdmin() || isAlgorithmOwnerUser(algoName)) {
      return;
    }

    List<String> categoriesFullNames = categorySet
      .getAlgorithmCategoriesFullNames(algo.getInfo());

    String systemId = Service.getSystemId();

    try {
      if (AlgorithmExecutionPermission.checkSystemAndAlgorithmExecPermission(
        user, systemId, algoName)) {
        return;
      }

      // Verifica se tem alguma permisso para as categorias em que o
      // algoritmo faz parte
      if (CategoryAlgorithmsExecutionPermission
        .checkSystemAndCategoriesExecPermission(user, systemId,
          categoriesFullNames)) {
        return;
      }
    }
    catch (Exception e) {
      String errorMsg = getMessageFormatted(
        "AlgoService.error.algo.admin.permission_failed", user.getLogin());
      Server.logSevereMessage(errorMsg, e);
    }
    throw new PermissionException(getAlgoExecutePermissionErrorMessage(
      algoName));
  }

  /**
   * Obtm mensagem de erro para o caso do usurio no ter permisso para
   * executar o algoritmo com o nome especificado.
   *
   * @param algoName parmetro invlido
   * @return a mensagem de erro correspondente
   */
  private String getAlgoExecutePermissionErrorMessage(String algoName) {
    String msg = getString(ERROR_ALGO_EXECUTE_PERMISSION);
    Object[] args = new Object[] { algoName };
    return MessageFormat.format(msg, args);
  }

  /**
   * Verifica se  o administrador ou, caso contrrio, se o usurio tem
   * permisso para gerenciar o algoritmo com o nome especificado, no servidor
   * corrente.
   *
   * @param algoName nome do algoritmo
   * @throws PermissionException usurio sem permisso para administrar o
   *         algoritmo
   */
  private void checkAlgorithmAdminPermission(String algoName)
    throws PermissionException {
    User user = Service.getUser();
    String algoAttribute = AlgorithmAdminPermission.ALGORITHM_UPDATE + algoName;
    AlgorithmAdminPermission ap = null;
    try {
      ap = (AlgorithmAdminPermission) user.getMatchAttributesPermission(
        AlgorithmAdminPermission.class, algoAttribute);
    }
    catch (Exception e) {
      String errorMsg = getMessageFormatted(
        "AlgoService.error.algo.execute.permission_failed", user.getLogin());
      Server.logSevereMessage(errorMsg, e);
    }
    if (ap == null) {
      throw new PermissionException(getAlgoAdminPermissionErrorMessage(
        algoName));
    }
    String serverName = Server.getInstance().getSystemName();
    String name = AdminPermission.LOCAL + serverName;
    if (ap.getMatchAttribute(name) == null) {
      throw new PermissionException(getAlgoAdminServerPermissionErrorMessage(
        serverName));
    }
  }

  /**
   * Verifica se o usurio que solicitante do servio  o mesmo que criou o
   * algoritmo especificado.
   *
   * @param algoName nome do algoritmo
   * @return retorna true, se o usurio que solicitou o servio  o mesmo que
   *         criou o algoritmo especificado, caso contrrio, retorna false
   */
  private boolean isAlgorithmOwnerUser(String algoName) {
    AlgorithmInfo algoInfo = getInfo(algoName);
    if (algoInfo == null) {
      throw new ServiceFailureException(getAlgoNotFoundMessage(algoName));
    }
    String algorithmOwner = algoInfo.getOwner();
    User user = Service.getUser();
    if (algorithmOwner != null && algorithmOwner.equals(user.getLogin())) {
      return true;
    }
    return false;
  }

  /**
   * Cria o configurador lgico do fluxo de algoritmos.
   *
   * @param version A verso do algoritmo (No aceita {@code null}).
   * @return O configurador.
   * @throws OperationFailureException Se houver um erro.
   */
  private FlowAlgorithmConfigurator createFlowAlgorithmConfigurator(
    AlgorithmVersionInfo version) throws OperationFailureException {
    if (version == null) {
      throw new IllegalArgumentException(getParameterNullMessage("version"));
    }
    String configText = readConfiguratorFile(version,
      AlgorithmVersionInfo.FLOW_CONFIG_FILE);
    if (configText == null) {
      return null;
    }
    Flow flow;
    try {
      flow = new FlowAlgorithmParser().decode(configText);
    }
    catch (ConfigurationException e) {
      throw new OperationFailureException(e.getLocalizedMessage(), e);
    }
    catch (ParseException e) {
      throw new OperationFailureException(e.getLocalizedMessage(), e);
    }
    return new FlowAlgorithmConfigurator(version, flow);
  }

  /**
   * Consulta o pattern default para montagem de linhas de comando.
   *
   * @return pattern
   */
  private String getServiceCommandLinePattern() {
    return getStringProperty("default.command.line.pattern");
  }

  /**
   * Criar configurador.
   *
   * @param version A verso do algoritmo.
   * @return O configurador de algoritmo.
   * @throws ParseException
   * @throws IllegalArgumentException Se o parmetro {@code versionInfo} for
   *         nulo.
   */
  private SimpleAlgorithmConfigurator createSimpleAlgorithmConfigurator(
    AlgorithmVersionInfo version) throws ParseException {
    if (version == null) {
      throw new IllegalArgumentException(getParameterNullMessage("version"));
    }
    String configText = readConfiguratorFile(version,
      AlgorithmVersionInfo.CONFIG_FILE);
    String configPropertiesText = readConfiguratorFile(version,
      AlgorithmVersionInfo.CONFIG_PROPERTIES_FILE);
    String paramPropertiesText = readConfiguratorFile(version,
      AlgorithmVersionInfo.PARAMETERS_PROPERTIES_FILE);

    if (configText != null) {
      StringReader configXmlReader = new StringReader(configText);

      Map<String, String> config = new HashMap<String, String>();
      Map<String, String> params = new HashMap<String, String>();

      Properties configProperties = getProperties(configPropertiesText);
      Properties paramProperties = getProperties(paramPropertiesText);

      /*
       * Carrega as propriedades-padro de todos os algoritmos + propriedades
       * especficas do algoritmo em questo. As propriedades do algoritmo devem
       * ser carregadas por ltimo, para poderem sobrescrever as
       * propriedades-padro.
       */
      addPropertiesToMap(this.parametersConfiguration, config);
      addPropertiesToMap(configProperties, config);
      addPropertiesToMap(paramProperties, params);

      final SimpleAlgorithmParser parser = new SimpleAlgorithmParser();
      final String srvCmdPattern = getServiceCommandLinePattern();
      return parser.load(configXmlReader, config, getParameterRegistry(),
        version, defaultAlgorithmExecutionLocation, srvCmdPattern);
    }
    return null;
  }

  /**
   * Obtm a lista de arquivos monitorados configurado pelo algoritmo. O arquivo
   *  definido em {@link AlgorithmVersionInfo#MONITOR_FILE}.
   *
   * @param algorithmName nome do algoritmo
   * @param algorithmVersionId verso do algoritmo
   *
   * @return A lista de arquivos monitorados
   *
   * @throws AlgorithmNotFoundException
   */
  public List<MonitoredFile> getMonitoredFiles(String algorithmName, AlgorithmVersionId algorithmVersionId)
    throws AlgorithmNotFoundException {
    if (algorithmName == null) {
      throw new IllegalArgumentException(getParameterNullMessage(
        "algorithmName"));
    }
    if (algorithmVersionId == null) {
      throw new IllegalArgumentException(getParameterNullMessage(
        "algorithmVersionId"));
    }
    Algorithm algorithm = getAlgorithm(algorithmName);
    AlgorithmInfo algorithmInfo = algorithm.getInfo();
    AlgorithmVersionInfo version = algorithmInfo.getVersionInfo(
      algorithmVersionId);
    if (version == null) {
      throw new AlgorithmNotFoundException(getVersionNotFoundMessage(
        algorithmInfo.getId(), algorithmVersionId));
    }

    List<MonitoredFile> files = java.util.Collections.EMPTY_LIST;
    String monitorText =
      readConfiguratorFile(version, AlgorithmVersionInfo.MONITOR_FILE);
    try {
      if (monitorText != null) {
        ObjectMapper mapper = new ObjectMapper();
        MonitoredFile[] monitoredFiles = mapper.readValue(monitorText, MonitoredFile[].class);
        files = Arrays.asList(monitoredFiles);
      }
    } catch (Exception e) {
      Server.logWarningMessage(MessageFormat.format(
        "Erro ao ler configurao de arquivos monitorados do algoritmo {0}:{1}",
        algorithmName, algorithmVersionId.toString()));
    }

    return files;
  }


  /**
   * Mtodo utilitrio que adiciona todas as propriedades {@link Properties} a
   * um mapa {@link Map}.
   *
   * @param <K> Tipo da chave do mapa.
   * @param <V> Tipo do valor do mapa.
   * @param properties Propriedades a serem adicionadas.
   * @param map Mapa que receber as propriedades.
   */
  @SuppressWarnings("unchecked")
  private <K, V> void addPropertiesToMap(final Properties properties,
    final Map<K, V> map) {
    for (Entry<Object, Object> entry : properties.entrySet()) {
      map.put((K) entry.getKey(), (V) entry.getValue());
    }
  }

  /**
   * Retorna as propriedades {@link Properties} a partir do contedo de um
   * arquivo de propriedades.
   *
   * @param configPropertiesText Contedo do arquivo de propriedades.
   * @return Propriedades contidas no arquivo.
   */
  private Properties getProperties(String configPropertiesText) {
    if (configPropertiesText == null) {
      return new Properties();
    }
    ByteArrayInputStream configPropertiesInputStream = new ByteArrayInputStream(
      configPropertiesText.getBytes());
    Properties properties = new Properties();
    try {
      properties.load(configPropertiesInputStream);
    }
    catch (IOException e) {
      IllegalStateException re = new IllegalStateException(e
        .getLocalizedMessage());
      re.initCause(e);
      throw re;
    }
    return properties;
  }

  /**
   * Obtm mensagem de erro internacionalizada e que pode ter ou no argumentos.
   *
   * @param key chave da mensagem no arquivo de internacionalizao
   * @param args argumentos da mensagem, caso ela possua
   * @return a mensagem de erro formatada com os argumentos
   */
  private String getMessageFormatted(String key, Object... args) {
    String msg = getString(key);
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro para o caso do algoritmo especificado no ter sido
   * encontrado.
   *
   * @param algoId identificador do algoritmo no encontrado.
   * @return mensagem de erro.
   */
  private String getAlgoNotFoundMessage(Object algoId) {
    String msg = getString(ERROR_ALGO_NOT_FOUND);
    Object[] args = new Object[] { algoId };
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro para o caso do parmetro de um mtodo ser nulo.
   *
   * @param parameter parmetro nulo
   * @return a mensagem de erro correspondente
   */
  private String getParameterNullMessage(String parameter) {
    String msg = getString(ERROR_PARAMETER_NULL);
    Object[] args = new Object[] { parameter };
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro para o caso do parmetro de um mtodo ser invlido.
   *
   * @param parameter parmetro invlido
   * @return a mensagem de erro correspondente
   */
  private String getParameterInvalidMessage(String parameter) {
    String msg = getString(ERROR_PARAMETER_INVALID);
    Object[] args = new Object[] { parameter };
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro para o caso do usurio no ter permisso para
   * administrar o algoritmo com o nome especificado.
   *
   * @param algoName parmetro invlido
   * @return a mensagem de erro correspondente
   */
  private String getAlgoAdminPermissionErrorMessage(String algoName) {
    String msg = getString(ERROR_ALGO_ADMIN_PERMISSION);
    Object[] args = new Object[] { algoName };
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro para o caso do usurio no ter permisso para
   * administrar algoritmos no servidor com o nome especificado.
   *
   * @param serverName parmetro invlido
   * @return a mensagem de erro correspondente
   */
  private String getAlgoAdminServerPermissionErrorMessage(String serverName) {
    String msg = getString(ERROR_ALGO_ADMIN_SERVER_PERMISSION);
    Object[] args = new Object[] { serverName };
    return MessageFormat.format(msg, args);
  }

  /**
   * Retorna um algoritmo.
   *
   * @param name O nome do algoritmo
   * @return o algoritmo solicitado
   * @throws AlgorithmNotFoundException quando o algoritmo no  encontrado
   */
  private Algorithm getAlgorithm(String name)
    throws AlgorithmNotFoundException {
    Collection<Algorithm> algorithmCollection = registeredAlgorithms.values();
    Iterator<Algorithm> algorithmIterator = algorithmCollection.iterator();
    while (algorithmIterator.hasNext()) {
      Algorithm algo = algorithmIterator.next();
      AlgorithmInfo info = algo.getInfo();
      if (info.getName().equals(name) || info.getId().equals(name)) {
        return algo;
      }
    }
    Server.logWarningMessage("Algoritmo inexistente: " + name);
    throw new AlgorithmNotFoundException(getAlgoNotFoundMessage(name));
  }

  /**
   * <p>
   * Obtm o histrico do diretrio e de todos os seus subdiretrios.
   * </p>
   * NOTA DE IMPLEMENTAO: Para evitar problemas de versionamento com objetos
   * serializados, s so serializados objetos bsicos do Java, como Integers,
   * Strings e Collections. O histrico retornado se compe de uma lista em que
   * cada item aponta para outras listas, estas ltimas contendo os dados de
   * entradas de histrico.
   *
   * @param table tabela que guarda todas as entradas de histrico para o
   *        diretrio e seus subdiretrios.
   * @return histrico do diretrio e de todos os seus subdiretrios.
   */
  private List getDirHistory(Map table) {
    List allHistory = new ArrayList();
    for (Iterator iter = table.keySet().iterator(); iter.hasNext();) {
      Object key = iter.next();
      if (DIR_HISTORY_KEY.equals(key)) {
        List dirHistory = (List) table.get(DIR_HISTORY_KEY);
        if (dirHistory != null) {
          allHistory.addAll(dirHistory);
        }
      }
      else {
        Map subTable = (Map) table.get(key);
        if (subTable == null) {
          Server.logSevereMessage(getMessageFormatted(
            "AlgoService.error.history.dir_failed"), new IllegalStateException(
              "subTable == null"));
          continue;
        }
        List subDirsHistory = getDirHistory(subTable);
        if (subDirsHistory != null) {
          allHistory.addAll(subDirsHistory);
        }
      }
    }
    return allHistory;
  }

  /**
   * Obtm a tabela que guarda todas as entradas de histrico para o diretrio
   * especificado e seus subdiretrios.
   *
   * @param viewPath caminho do diretrio (na viso do cliente).
   * @param historyTable referncia para a tabela de histrico.
   * @return tabela que guarda todas as entradas de histrico para o diretrio
   *         especificado e seus subdiretrios. Se no havia nenhuma entrada de
   *         histrico para o diretrio em questo, uma tabela vazia 
   *         retornada.
   */
  private Map getDirTable(String[] viewPath, Map historyTable) {
    Map table = historyTable;

    //TODO Esse mtodo  praticamente idntico ao insertRecord: refactoring
    //  necessrio.
    // O path[0] equivale ao diretrio "algorithms", que no figura na tabela
    for (int i = 1; i < viewPath.length; i++) {
      String key = viewPath[i];
      Map swap = (Map) table.get(key);
      if (swap == null) {
        swap = new HashMap();
        table.put(key, swap);
      }
      table = swap;
    }
    return table;
  }

  /**
   * Obtm o caminho para o arquivo que armazena a tabela de histrico.
   *
   * @return caminho para o arquivo que armazena a tabela de histrico.
   */
  private String getHistoryTablePath() {
    return algorithmRepositoryPath + File.separator + HISTORY_TABLE_FILE_NAME;
  }

  /**
   * Obtm uma lista de <code>HistoryRecord</code>, a partir de uma lista de
   * listas (<code>List</code> de <code>List</code>). A justificativa para isso
   *  que somente so serializados objetos bsicos do Java, para evitar
   * problemas de compatibilidade com verses futuras.
   *
   * @param viewPath caminho do diretrio (na viso do cliente).
   * @param rawList lista em estado bruto (lista de listas).
   * @return lista de <code>HistoryRecord</code>.
   */
  private List<HistoryRecord> getRecordList(String[] viewPath, List rawList) {
    List<HistoryRecord> recordList = new ArrayList<HistoryRecord>();
    for (Iterator iter = rawList.iterator(); iter.hasNext();) {
      List l = (List) iter.next();
      Date date = (Date) l.get(HistoryRecord.DATE_COLUMN_INDEX);
      String[] absLogPath = (String[]) l.get(HistoryRecord.PATH_COLUMN_INDEX);
      String login = (String) l.get(HistoryRecord.LOGIN_COLUMN_INDEX);
      String name = (String) l.get(HistoryRecord.NAME_COLUMN_INDEX);
      String description = (String) l.get(
        HistoryRecord.DESCRIPTION_COLUMN_INDEX);
      String[] relLogPath;
      if (viewPath.length == absLogPath.length) {
        relLogPath = new String[] { "." };
      }
      else {
        int len = absLogPath.length - viewPath.length;
        relLogPath = new String[len];
        System.arraycopy(absLogPath, viewPath.length, relLogPath, 0, len);
      }
      HistoryRecord record = new HistoryRecord(date, relLogPath, login, name,
        description);
      recordList.add(record);
    }
    return recordList;
  }

  /**
   * Obtm mensagem de erro para o caso da verso especificada no ter sido
   * encontrada.
   *
   * @param algoId identificador do algoritmo ao qual deveria pertencer a verso
   *        no encontrada.
   * @param versionId identificador da verso no encontrada.
   * @return mensagem de erro.
   */
  private String getVersionNotFoundMessage(Object algoId, Object versionId) {
    String msg = getString(ERROR_VERSION_NOT_FOUND);
    Object[] args = new Object[] { versionId, algoId };
    return MessageFormat.format(msg, args);
  }

  /**
   * Verifica a existncia da tabela de histrico; caso no encontre, constri e
   * serializa uma tabela vazia.
   */
  private void initHistoryTable() {
    File historyFile = new File(getHistoryTablePath());
    if (!historyFile.exists()) {
      Server.logInfoMessage("Tabela de histrico no encontrada. Criando...");
      writeHistoryTable(new HashMap());
    }
    Map<?, ?> table = readHistoryTable();
    if (table == null) {
      resetHistoryTable();
    }
  }

  /**
   * Reinicializa a tabela de histrico de algoritmos, avisando ao administrador
   * por e-mail.
   *
   * @return um mapa vazio.
   */
  public Map resetHistoryTable() {
    Server.logSevereMessage("Tabela de histrico corrompida. Recriando...");
    writeHistoryTable(new HashMap());
    StringBuilder content = new StringBuilder();
    content.append("# Servio de Algoritmos");
    content.append("\n\n");
    content.append("A tabela de histrico de algoritmos foi corrompida. ");
    content.append("Uma nova tabela foi recriada em ");
    content.append(getHistoryTablePath());
    notifyAdminByMail(content.toString());
    return new HashMap();
  }

  /**
   * Avisa o administrador do sistema sobre erros ocorridos no servio.
   *
   * @param message a mensagem a ser enviada ao administrador.
   */
  private void notifyAdminByMail(String message) {
    MailService mailService = MailService.getInstance();

    Vector<Object> ids = new Vector<Object>();
    try {
      ids.addAll(User.getAdminIds());
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro ao obter a lista de usurios adminstradores.", e);
    }

    Object[] address = ids.toArray(new Object[0]);

    mailService.mailSomeUsersFromService(this, address, message);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Service[] getInitializationDependencies() {
    return new Service[] { MailService.getInstance() };
  }

  /**
   * Insere uma nova entrada na tabela de histrico.
   *
   * @param path caminho no qual ocorreu uma ao que deve ser registrada no
   *        histrico.
   * @param record registro a ser inserido na tabela.
   */
  private void insertRecord(String[] path, List record) {
    Map historyTable = readHistoryTable();
    if (historyTable == null) {
      historyTable = resetHistoryTable();
    }
    Map table = getDirTable(path, historyTable);
    List dirHistory = (List) table.get(DIR_HISTORY_KEY);
    if (dirHistory == null) {
      dirHistory = new ArrayList();
      table.put(DIR_HISTORY_KEY, dirHistory);
    }
    dirHistory.add(record);
    writeHistoryTable(historyTable);
  }

  /**
   * Faz a (re)carga dos algoritmos no servidor.
   *
   * @throws ServerException caso ocorra algum problema durante a (re)carga.
   */
  private void loadAlgorithms() throws ServerException {
    searchIds(algorithmRepositoryPath);
  }

  /**
   * Carregas as categorias do sistema (novo leitor xml - ainda em teste).
   *
   * @throws ServerException caso ocorra algum problema durante a carga
   */
  private void loadCategories() throws ServerException {
    categorySet.removeAllCategories();
    if (!readCategoriesFile(categorySet)) {
      return;
    }
    Server.logInfoMessage(
      "As categorias de algoritmos foram carregadas com sucesso.");
    isFirstCategoryLoad = false;
    // System.out.println("AlgorithmService: nmero de categorias (raiz) lidas: "
    // + categorySet.getSize());
  }

  /**
   * L um arquivo do configurador de uma verso de algoritmo.
   *
   * @param version A verso do algoritmo (No aceita {@code null}).
   * @param configFileName O nome do arquivo (No aceita {@code null}).
   * @return O contedo do arquivo ou {@code null} se houver algum erro.
   */
  private String readConfiguratorFile(AlgorithmVersionInfo version,
    String configFileName) {
    String configuratorPath = version.getConfiguratorDirPath() + "/"
      + configFileName;
    configuratorPath = configuratorPath.replace('/', File.separatorChar);
    File configuratorFile = new File(configuratorPath);
    if (!configuratorFile.exists()) {
      String algorithmName = version.getInfo().getName();
      AlgorithmVersionId versionId = version.getId();
      String message = MessageFormat.format(
        "Foi solicitado o arquivo {0} do algoritmo {1} verso {2}, porm "
          + "o arquivo no existe.", configuratorPath, algorithmName,
        versionId);
      Server.logFineMessage(message);
      return null;
    }
    BufferedReader bufferedReader = null;
    try {
      bufferedReader = new BufferedReader(new FileReader(configuratorFile));
      StringBuffer buffer = new StringBuffer();
      String line;
      while ((line = bufferedReader.readLine()) != null) {
        buffer.append(line + "\n");
      }
      return buffer.toString();
    }
    catch (IOException e) {
      String algorithmName = version.getInfo().getName();
      AlgorithmVersionId versionId = version.getId();
      String errorMessage = MessageFormat.format(
        "Foi solicitado o '{2}' do algoritmo {0} verso {1}, " + "porm "
          + "ocorreu uma exceo de IO.", algorithmName, versionId,
        configFileName);
      Server.logSevereMessage(errorMessage, e);
      return null;
    }
    finally {
      if (bufferedReader != null) {
        try {
          bufferedReader.close();
        }
        catch (IOException e) {
          String algorithmName = version.getInfo().getName();
          AlgorithmVersionId versionId = version.getId();
          String errorMessage = MessageFormat.format(
            "Foi solicitado o '{2}' do algoritmo {0} verso {1}, " + "porm "
              + "ocorreu uma exceo de IO, " + "quando o stream estava sendo "
              + "fechado.", algorithmName, versionId, configFileName);
          Server.logSevereMessage(errorMessage, e);
        }
        bufferedReader = null;
      }
    }
  }

  /**
   * L o arquivo serializado que representa a tabela de histricos e retorna a
   * tabela lida na forma de um objeto.
   *
   * @return tabela de histrico ou <code>null</code> caso tenha ocorrido algum
   *         problema durante a leitura da tabela.
   */
  private Map readHistoryTable() {
    Map table = null;
    try (ObjectInputStream input = new ObjectInputStream(
      new BufferedInputStream(new FileInputStream(getHistoryTablePath())))) {
      table = (Map) input.readObject();
      return table;
    }
    catch (Exception ex) {
      Server.logSevereMessage("Erro ao ler a tabela de histrico.", ex);
      return null;
    }
  }

  /**
   * Percorre o repositrio de algoritmos, obtendo as informaes dos algoritmos
   * correntemente cadastrados.
   *
   * @param path caminho para o repositrio de algoritmos.
   * @throws ServerException caso ocorra algum problema durante a busca.
   */
  private void searchIds(String path) throws ServerException {
    try {
      File repDir = new File(path);
      Server.checkDirectory(path);
      registeredAlgorithms = new Hashtable<String, Algorithm>();
      File[] algoDirs = repDir.listFiles();
      for (int i = 0; i < algoDirs.length; i++) {
        File dir = algoDirs[i];
        if (dir.isDirectory() && !dir.isHidden()) {
          String dirName = dir.getName().trim();
          if (dirName.equalsIgnoreCase(FileSystemUtils.VERSION_CONTROL_DIR)) {
            continue;
          }
          Server.logFineMessage("Identificando algoritmo em :" + dirName);
          try {
            Algorithm algo = Algorithm.includeAlgorithm(dirName);
            String newId = algo.getInfo().getId();
            registeredAlgorithms.put(newId, algo);
            // LOG
            Server.logFineMessage("Algoritmo registrado: " + newId);
            Server.logFineMessage("Propriedades:" + algo.getInfo()
              .getPropertyValues());
          }
          catch (ServerException se) {
            Server.logSevereMessage("Falha na obteno de um algoritmo no "
              + "repositrio do SSI - " + "Exceo: " + se + "\n", se);
          }
        }
      }
    }
    catch (Exception e) {
      throw new ServerException(getMessageFormatted(
        "AlgoService.error.algo.search", path), e);
    }
  }

  /**
   * Grava (serializa) a tabela de histrico em arquivo.
   *
   * @param table tabela de histrico.
   */
  private void writeHistoryTable(Map table) {
    try (ObjectOutputStream output = new ObjectOutputStream(
      new BufferedOutputStream(new FileOutputStream(getHistoryTablePath())))) {
      output.writeObject(table);
    }
    catch (IOException ex) {
      Server.logSevereMessage("Erro ao gravar a tabela de histrico.", ex);
    }
  }

  /**
   * Obtm o {@link HttpService servio de HTTP}.
   *
   * @return O servio.
   * @throws OperationFailureException Se o servio no existir.
   */
  private HttpService getHttpService() throws OperationFailureException {
    HttpService service = HttpService.getInstance();
    if (service == null) {
      throw new OperationFailureException(getMessageFormatted(
        "AlgoService.error.http_service.not_found"));
    }
    return service;
  }

  /**
   * Cria a informao de canal para o caminho indicado.
   *
   * @param targetPath Caminho do arquivo a ser acessado.
   * @param notifyArgs Informaes de notificao. Pode ser nulo.
   * @param op Operao a ser realizada: upload ou download.
   * @return A informao para construo do canal.
   */
  private RemoteFileChannelInfo createChannel(String targetPath,
    Map<String, Object> notifyArgs, FileChannelOp op) {
    try {
      boolean ro = (op == FileChannelOp.DOWNLOAD || op == FileChannelOp.READ);
      FTCRequester r = new Requester(notifyArgs, ro);
      File f = new File(targetPath);
      return FTCService.getInstance().createFileChannelInfo(r, f, ro);
    }
    catch (Exception e) {
      String clientMsg = "";
      switch (op) {
        case UPLOAD:
          clientMsg = getString("server.algoservice.error.upload_fatal");
          break;
        case DOWNLOAD:
          clientMsg = getString("server.algoservice.error.download_fatal");
          break;
        case READ:
          clientMsg = getString("server.algoservice.error.read_fatal");
          break;
        case WRITE:
          clientMsg = getString("server.algoservice.error.write_fatal");
          break;
      }
      throw new ServiceFailureException(clientMsg, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized RemoteFileChannelInfo prepareUploadVersionPack(
    Object algoId, String fileName) {
    checkAlgorithmAdminPermission(algoId);
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmInfo algoInfo = algo.getInfo();
    File repositoryDirectory = new File(algorithmRepositoryPath);
    File algorithmDirectory = new File(repositoryDirectory, algoInfo
      .getDirectory());
    File versionsDir = new File(algorithmDirectory, AlgorithmInfo.VERSIONS_DIR);
    if (!versionsDir.exists()) {
      String userMessage = getMessageFormatted(
        "server.algoservice.error.illegal_algo_dir_state", algoInfo.getName());
      String logDetails = String.format(
        "\nDetalhes: o diretrio %s no foi encontrado.\n", versionsDir
          .getAbsolutePath());
      Server.logSevereMessage(userMessage + logDetails);
      throw new ServiceFailureException(userMessage);
    }
    String targetPath;
    try {
      targetPath = versionsDir.getCanonicalPath() + File.separator + fileName;
    }
    catch (IOException e) {
      String userMessage = getMessageFormatted(
        "AlgoService.error.upload_algo_pack.create_file", algoInfo.getName());
      Server.logSevereMessage(userMessage, e);
      throw new ServiceFailureException(userMessage);
    }
    Map<String, Object> notifyArgs = new HashMap<String, Object>();
    notifyArgs.put(ALGO_FILE_TYPE_KEY, AlgoFileType.VERSION_PACK);
    notifyArgs.put(USER_ID_KEY, Service.getUser().getId());
    notifyArgs.put(ALGO_ID_KEY, algoId);
    notifyArgs.put(FILE_PATH_KEY, targetPath);
    notifyArgs.put(EXPAND_IF_ZIP, Boolean.TRUE);
    return createChannel(targetPath, notifyArgs, FileChannelOp.UPLOAD);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized RemoteFileChannelInfo prepareUploadExecFile(Object algoId,
    Object versionId, String platformName, String fileName,
    boolean expandIfZip) {
    checkAlgorithmAdminPermission(algoId);
    String platformPath = getPlatformPath(algoId, versionId, platformName);
    String targetPath = platformPath + File.separator + fileName;
    Map<String, Object> notifyArgs = new HashMap<String, Object>();
    notifyArgs.put(ALGO_FILE_TYPE_KEY, AlgoFileType.EXECUTABLE);
    notifyArgs.put(USER_ID_KEY, Service.getUser().getId());
    notifyArgs.put(ALGO_ID_KEY, algoId);
    notifyArgs.put(VERSION_ID_KEY, versionId);
    notifyArgs.put(PLATFORM_NAME_KEY, platformName);
    notifyArgs.put(FILE_NAME_KEY, fileName);
    notifyArgs.put(EXPAND_IF_ZIP, expandIfZip);
    return createChannel(targetPath, notifyArgs, FileChannelOp.UPLOAD);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized RemoteFileChannelInfo prepareUploadConfigFile(
    Object algoId, Object versionId, String fileName, boolean expandIfZip) {
    checkAlgorithmAdminPermission(algoId);
    String configDirPath = getConfiguratorDirPath(algoId, versionId);
    String targetPath = configDirPath + File.separator + fileName;
    Map<String, Object> notifyArgs = new HashMap<String, Object>();
    notifyArgs.put(ALGO_FILE_TYPE_KEY, AlgoFileType.CONFIGURATION);
    notifyArgs.put(USER_ID_KEY, Service.getUser().getId());
    notifyArgs.put(ALGO_ID_KEY, algoId);
    notifyArgs.put(VERSION_ID_KEY, versionId);
    notifyArgs.put(FILE_NAME_KEY, fileName);
    notifyArgs.put(EXPAND_IF_ZIP, expandIfZip);
    return createChannel(targetPath, notifyArgs, FileChannelOp.UPLOAD);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized RemoteFileChannelInfo prepareUploadDocFile(Object algoId,
    Object versionId, String fileName, boolean expandIfZip) {
    checkAlgorithmAdminPermission(algoId);
    String docDirPath = getDocumentationDirPath(algoId, versionId);
    String targetPath = docDirPath + File.separator + fileName;
    Map<String, Object> notifyArgs = new HashMap<String, Object>();
    notifyArgs.put(ALGO_FILE_TYPE_KEY, AlgoFileType.DOCUMENTATION);
    notifyArgs.put(USER_ID_KEY, Service.getUser().getId());
    notifyArgs.put(ALGO_ID_KEY, algoId);
    notifyArgs.put(VERSION_ID_KEY, versionId);
    notifyArgs.put(FILE_NAME_KEY, fileName);
    notifyArgs.put(EXPAND_IF_ZIP, expandIfZip);
    return createChannel(targetPath, notifyArgs, FileChannelOp.UPLOAD);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized RemoteFileChannelInfo prepareUploadReleaseNotesFile(
    Object algoId, Object versionId, String fileName, boolean expandIfZip) {
    checkAlgorithmAdminPermission(algoId);
    String docDirPath = getReleaseNotesDirPath(algoId, versionId);

    Path releaseFolder = Paths.get(docDirPath);
    if (!Files.exists(releaseFolder)) {
      new File(docDirPath).mkdirs();
    }

    String targetPath = docDirPath + File.separator + "releasenotes.txt";
    Map<String, Object> notifyArgs = new HashMap<String, Object>();
    notifyArgs.put(ALGO_FILE_TYPE_KEY, AlgoFileType.RELEASE_NOTES);
    notifyArgs.put(USER_ID_KEY, Service.getUser().getId());
    notifyArgs.put(ALGO_ID_KEY, algoId);
    notifyArgs.put(VERSION_ID_KEY, versionId);
    notifyArgs.put(FILE_NAME_KEY, "releasenotes.txt");
    notifyArgs.put(EXPAND_IF_ZIP, expandIfZip);
    return createChannel(targetPath, notifyArgs, FileChannelOp.UPLOAD);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean configFileExists(Object algoId, Object versionId,
    String fileName) {
    checkAlgorithmAdminPermission(algoId);
    String configDirPath = getConfiguratorDirPath(algoId, versionId);
    String targetPath = configDirPath + File.separator + fileName;
    return new File(targetPath).exists();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean docFileExists(Object algoId, Object versionId,
    String fileName) {
    checkAlgorithmAdminPermission(algoId);
    String docDirPath = getDocumentationDirPath(algoId, versionId);
    String targetPath = docDirPath + File.separator + fileName;
    return new File(targetPath).exists();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean releaseNotesFileExists(Object algoId, Object versionId,
    String fileName) {
    checkAlgorithmAdminPermission(algoId);
    String docDirPath = getReleaseNotesDirPath(algoId, versionId);
    String targetPath = docDirPath + File.separator + fileName;
    return new File(targetPath).exists();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean execFileExists(Object algoId, Object versionId,
    String platformName, String fileName) {
    checkAlgorithmAdminPermission(algoId);
    String platformPath = getPlatformPath(algoId, versionId, platformName);
    String targetPath = platformPath + File.separator + fileName;
    return new File(targetPath).exists();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean renameConfigFile(Object algoId, Object versionId,
    String[] filePath, String fileName) throws RemoteException {
    checkAlgorithmAdminPermission(algoId);
    String configDirPath = getConfiguratorDirPath(algoId, versionId);

    String joinedSourceFilePath = FileUtils.joinPath(filePath);
    String sourcePath = configDirPath + File.separator + joinedSourceFilePath;
    final File file = new File(sourcePath);

    String[] targetFilePath = filePath;
    String targetPath = configDirPath + File.separator;
    if (targetFilePath.length > 1) {
      targetFilePath[targetFilePath.length - 1] = fileName;
      String joinedTargetFilePath = FileUtils.joinPath(targetFilePath);
      targetPath += joinedTargetFilePath;
    }
    else {
      targetPath += fileName;
    }
    final File newFile = new File(targetPath);
    if (newFile.exists()) {
      String infoMsg = getMessageFormatted(
        "AlgoService.error.file.rename.existing_name", fileName);
      throw new InfoException(infoMsg);
    }
    // Tenta mover o arquivo.
    boolean renamed = file.renameTo(newFile);
    if (!renamed) {
      renamed = FileSystem.move(file, newFile);
    }
    Algorithm algo = registeredAlgorithms.get(algoId);
    try {
      algo.reload();
      fireModifyEvent(algo.getInfo());
    }
    catch (ServerException e) {
      e.printStackTrace();
    }
    return renamed;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean renameDocFile(Object algoId, Object versionId,
    String[] filePath, String fileName) throws RemoteException {
    checkAlgorithmAdminPermission(algoId);
    String docDirPath = getDocumentationDirPath(algoId, versionId);

    String joinedSourceFilePath = FileUtils.joinPath(filePath);
    String sourcePath = docDirPath + File.separator + joinedSourceFilePath;
    final File file = new File(sourcePath);

    String[] targetFilePath = filePath;
    String targetPath = docDirPath + File.separator;
    if (targetFilePath.length > 1) {
      targetFilePath[targetFilePath.length - 1] = fileName;
      String joinedTargetFilePath = FileUtils.joinPath(targetFilePath);
      targetPath += joinedTargetFilePath;
    }
    else {
      targetPath += fileName;
    }
    final File newFile = new File(targetPath);
    if (newFile.exists()) {
      String infoMsg = getMessageFormatted(
        "AlgoService.error.file.rename.existing_name", fileName);
      throw new InfoException(infoMsg);
    }
    // Tenta mover o arquivo.
    boolean renamed = file.renameTo(newFile);
    if (!renamed) {
      renamed = FileSystem.move(file, newFile);
    }
    Algorithm algo = registeredAlgorithms.get(algoId);
    try {
      algo.reload();
      fireModifyEvent(algo.getInfo());
    }
    catch (ServerException e) {
      e.printStackTrace();
    }
    return renamed;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean renameExecFile(Object algoId, Object versionId,
    String[] filePath, String platformName, String fileName)
    throws RemoteException {
    checkAlgorithmAdminPermission(algoId);
    String execDirPath = getPlatformPath(algoId, versionId, platformName);

    String joinedSourceFilePath = FileUtils.joinPath(filePath);
    String sourcePath = execDirPath + File.separator + joinedSourceFilePath;
    final File file = new File(sourcePath);

    String[] targetFilePath = filePath;
    String targetPath = execDirPath + File.separator;
    if (targetFilePath.length > 1) {
      targetFilePath[targetFilePath.length - 1] = fileName;
      String joinedTargetFilePath = FileUtils.joinPath(targetFilePath);
      targetPath += joinedTargetFilePath;
    }
    else {
      targetPath += fileName;
    }
    final File newFile = new File(targetPath);
    if (newFile.exists()) {
      String infoMsg = getMessageFormatted(
        "AlgoService.error.file.rename.existing_name", fileName);
      throw new InfoException(infoMsg);
    }
    // Tenta mover o arquivo.
    boolean renamed = file.renameTo(newFile);
    if (!renamed) {
      renamed = FileSystem.move(file, newFile);
    }
    Algorithm algo = registeredAlgorithms.get(algoId);
    try {
      algo.reload();
      fireModifyEvent(algo.getInfo());
    }
    catch (ServerException e) {
      e.printStackTrace();
    }
    return renamed;
  }

  /**
   * Carrega os parmetros cadastrados no sistema e suas configuraes.
   *
   * @throws ServerException em caso de erro na configurao dos parmetros.
   */
  private void loadParameters() throws ServerException {
    parametersConfiguration = new Properties();

    loadParameterPropertiesFromFilePath(parametersConfiguration,
      "AlgorithmParametersConfiguration.properties");

    loadParameterPropertiesFromTag(parametersConfiguration,
      "parameters.configuration.property.file");
  }

  /**
   * Carrega propriedades {@link Properties} a partir de um caminho relativo
   * para um arquivo de propriedades.
   *
   * @param properties Instncia de Properties que receber as propriedades.
   * @param path Caminho relativo para o arquivo de propriedades.
   */
  private void loadParameterPropertiesFromFilePath(Properties properties,
    String path) {
    InputStream inStream;
    try {
      inStream = getClass().getResourceAsStream(path);
      loadPropertiesFromStream(properties, inStream);
    }
    catch (FileNotFoundException e) {
      Server.logWarningMessage(
        "No foi possvel encontrar arquivo de propriedades: " + path + ".");
    }
    catch (IOException e) {
      Server.logWarningMessage(
        "No foi possvel ler o arquivo de propriedades: " + path + ".");
    }
  }

  /**
   * Carrega propriedades {@link Properties} a partir de uma propriedade do
   * servio que aponta para um outro arquivo de propriedades.
   *
   * @param properties Instncia de Properties que receber as propriedades.
   * @param tag Nome da propriedade do servio que aponta pro arquivo de
   *        propriedades a ser carregado.
   * @throws ServerException em caso de erro na configurao dos parmetros.
   */
  private void loadParameterPropertiesFromTag(Properties properties, String tag)
    throws ServerException {
    if (!isPropertyNull(tag)) {
      String propertiesFile = getStringProperty(tag);
      String path = getOSPropertyPath(propertiesFile);
      InputStream inStream;
      try {
        inStream = new FileInputStream(path);
        loadPropertiesFromStream(properties, inStream);
      }
      catch (FileNotFoundException e) {
        throw new ServerException(getMessageFormatted(
          "AlgoService.error.property.file.not_found", path, tag), e);
      }
      catch (IOException e) {
        throw new ServerException(getMessageFormatted(
          "AlgoService.error.property.file.io_error", path, tag), e);
      }
    }
  }

  /**
   * Mtodo utilitrio que carrega propriedades {@link Properties} a partir de
   * um {@link InputStream} de um arquivo de propriedades.
   *
   * @param properties Instncia de Properties que receber as propriedades.
   * @param inStream Stream do arquivo de propriedades a ser carregado.
   * @throws IOException Caso no seja possvel ler as propriedades do arquivo.
   */
  private void loadPropertiesFromStream(Properties properties,
    InputStream inStream) throws IOException {
    try {
      if (inStream != null) {
        properties.load(inStream);
      }
    }
    finally {
      if (inStream != null) {
        inStream.close();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized RemoteFileChannelInfo prepareDownloadExecFile(
    Object algoId, Object versionId, String platformName, String fileName,
    boolean isDirectory) {
    checkAlgorithmAdminPermission(algoId);
    String platformPath = getPlatformPath(algoId, versionId, platformName);
    String sourcePath = platformPath + File.separator + fileName;
    File newFile = createTmpZipFile(sourcePath, isDirectory);
    String newFilePath = newFile.getAbsolutePath();
    Map<String, Object> notifyArgs = new HashMap<String, Object>();
    notifyArgs.put(ALGO_FILE_TYPE_KEY, AlgoFileType.EXECUTABLE);
    notifyArgs.put(USER_ID_KEY, Service.getUser().getId());
    notifyArgs.put(ALGO_ID_KEY, algoId);
    notifyArgs.put(FILE_PATH_KEY, newFilePath);
    notifyArgs.put(EXCLUDE_TMP_ZIP, isDirectory);
    return createChannel(newFilePath, notifyArgs, FileChannelOp.DOWNLOAD);
  }

  /**
   * Cria um arquivo zip temporrio do diretrio informado.
   *
   * @param file Diretrio que ser compactado.
   * @param isDirectory indica se o nome do diretrio  diretrio ou no.
   * @return Um arquivo zip do diretrio passado ou apenas um arquivo quando no
   *         for passado um diretrio.
   */
  private synchronized File createTmpZipFile(String file, boolean isDirectory) {
    if (!isDirectory) {
      return new File(file);
    }
    String zipFileName = String.format("%s.zip", file);
    File sourceDirectoryFile = new File(file);
    File directoryTmpZipFile = new File(zipFileName);
    try {
      zipFiles(sourceDirectoryFile, directoryTmpZipFile);
    }
    catch (IOException e) {
      // Notificar quando tivermos erro na hora de criar o zip
      String msg = String.format(getString(
        "server.algoservice.error.invalid_zip_file"), zipFileName);
      Server.logSevereMessage(msg, e);
      try {
        String[] ids = new String[] { zipFileName };
        MessageService.getInstance().send(new Message(new UserNotification(
          getUser().getLogin(), msg, false, false)), ids);
      }
      catch (RemoteException re) {
        Server.logSevereMessage("Erro ao enviar evento de arquivo invlido.",
          re);
      }
    }
    return directoryTmpZipFile;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized RemoteFileChannelInfo prepareDownloadConfigFile(
    Object algoId, Object versionId, String fileName) {
    checkAlgorithmAdminPermission(algoId);
    String configDirPath = getConfiguratorDirPath(algoId, versionId);
    String sourcePath = configDirPath + File.separator + fileName;
    return createChannel(sourcePath, null, FileChannelOp.DOWNLOAD);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized RemoteFileChannelInfo prepareDownloadDocFile(
    Object algoId, Object versionId, String fileName) {
    //checkAlgorithmAdminPermission(algoId);
    checkAlgorithmExecutePermission(algoId);

    String docDirPath = getDocumentationDirPath(algoId, versionId);
    String sourcePath = docDirPath + File.separator + fileName;
    return createChannel(sourcePath, null, FileChannelOp.DOWNLOAD);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public RemoteFileChannelInfo prepareDownloadNotesFile(Object algoId,
    Object versionId, String fileName) throws RemoteException {
    checkAlgorithmExecutePermission(algoId);

    String notesDirPath = getReleaseNotesDirPath(algoId, versionId);
    String sourcePath = notesDirPath + File.separator + fileName;
    return createChannel(sourcePath, null, FileChannelOp.DOWNLOAD);
  }

  /**
   * Obtem o diretorio completo para a versao (repositorio+algoritmo+versao)
   *
   * @param algoId Identificador do algoritmo
   * @return caminho para a versao
   */
  public String getAlgorithmDirPath(Object algoId) {
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmInfo algoInfo = algo.getInfo();
    String algoDirPath = algoInfo.getAlgorithmRepositoryPath() + "/" + algoInfo
      .getDirectory();
    return algoDirPath;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized RemoteFileChannelInfo prepareDownloadVersionPackFile(
    Object algoId, AlgorithmVersionId versionId, String versionPackFileName) {
    checkAlgorithmAdminPermission(algoId);
    String algorithmDirPath = getAlgorithmDirPath(algoId);
    String sourcePath = algorithmDirPath + File.separator + versionPackFileName;
    createVersionPack(algoId, versionId, versionPackFileName);
    Map<String, Object> notifyArgs = new HashMap<String, Object>();
    notifyArgs.put(ALGO_FILE_TYPE_KEY, AlgoFileType.VERSION_PACK);
    notifyArgs.put(USER_ID_KEY, Service.getUser().getId());
    notifyArgs.put(ALGO_ID_KEY, algoId);
    notifyArgs.put(FILE_PATH_KEY, sourcePath);
    notifyArgs.put(EXPAND_IF_ZIP, Boolean.FALSE);
    return createChannel(sourcePath, notifyArgs, FileChannelOp.DOWNLOAD);
  }

  /**
   * Cria um pacote de verso de algoritmo (PVA).
   *
   * @param algoId identificador do algoritmo
   * @param algoVersionId identificador da verso do algoritmo
   * @param versionPackFileName nome do pacote de verso do algoritmo
   * @return o arquivo correspondente ao PVA
   */
  private synchronized File createVersionPack(Object algoId,
    AlgorithmVersionId algoVersionId, String versionPackFileName) {
    checkAlgorithmAdminPermission(algoId);
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmInfo algoInfo = algo.getInfo();
    File repositoryDirectory = new File(algorithmRepositoryPath);
    File algorithmDirectory = new File(repositoryDirectory, algoInfo
      .getDirectory());

    File versionsDir = new File(algorithmDirectory, AlgorithmInfo.VERSIONS_DIR);
    if (!versionsDir.exists()) {
      String userMessage = getMessageFormatted(
        "server.algoservice.error.illegal_algo_dir_state", algoInfo.getName());
      String logDetails = String.format(
        "\nDetalhes: o diretrio %s no foi encontrado.\n", versionsDir
          .getAbsolutePath());
      Server.logSevereMessage(userMessage + logDetails);
      throw new ServiceFailureException(userMessage);
    }
    AlgorithmVersionInfo versionInfo = algo.getInfo().getVersionInfo(
      algoVersionId);
    String algoVersionPath = versionInfo.getDirPath();
    File algoVersionDir = new File(algoVersionPath);

    File versionPackFile = new File(algorithmDirectory, versionPackFileName);
    try {
      zipFiles(algoVersionDir, versionPackFile);
    }
    catch (IOException e) {
      Server.logSevereMessage("Erro ao criar o pacote de verso do algoritmo "
        + algoInfo.getName() + " verso " + algoVersionId + ".", e);
    }
    return versionPackFile;
  }

  /**
   * Compacta os arquivos do diretrio especificado para a verso de um
   * algoritmo, e cria o arquivo de PVA (Pacote de Verso de Algoritmo)
   * especificado.
   *
   * @param algoVersionDir diretrio da verso do algoritmo
   * @param versionPackFile arquivo de PVA
   * @return retorna true se a compactao foi realizada com sucesso, caso
   *         contrrio, retorna false
   * @throws IOException erro de IO
   */
  public boolean zipFiles(File algoVersionDir, File versionPackFile)
    throws IOException {
    if (algoVersionDir == null || versionPackFile == null) {
      return false;
    }

    FileOutputStream zipFile = new FileOutputStream(versionPackFile);
    ZipOutputStream zos = new ZipOutputStream(new BufferedOutputStream(
      zipFile));
    ZipUtils.addFile(zos, algoVersionDir, 0);
    zos.flush();
    zipFile.flush();
    zos.close();
    zipFile.close();

    return true;
  }

  /**
   * Obtm o caminho no sistema de arquivos de uma determinada plataforma
   * (relativo  raiz do sistema).
   *
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @param platformName nome da plataforma.
   * @return caminho da plataforma no sistema de arquivos.
   */
  private String getPlatformPath(Object algoId, Object versionId,
    String platformName) {
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmVersionInfo versionInfo = algo.getInfo().getVersionInfo(versionId);
    return versionInfo.getPlatformPath(platformName);
  }

  /**
   * Obtm o caminho no sistema de arquivos de uma determinado diretrio de
   * arquivos de configurao (relativo  raiz do sistema).
   *
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @return caminho do diretrio de arquivos de configurao no sistema de
   *         arquivos.
   */
  private String getConfiguratorDirPath(Object algoId, Object versionId) {
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmVersionInfo versionInfo = algo.getInfo().getVersionInfo(versionId);
    return versionInfo.getConfiguratorDirPath();
  }

  /**
   * Obtm o caminho no sistema de arquivos de uma determinado diretrio de
   * arquivos de documentao (relativo  raiz do sistema).
   *
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @return caminho do diretrio de arquivos de documentao no sistema de
   *         arquivos.
   */
  private String getDocumentationDirPath(Object algoId, Object versionId) {
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmVersionInfo versionInfo = algo.getInfo().getVersionInfo(versionId);
    return versionInfo.getDocDirPath();
  }

  /**
   * Obtm o caminho no sistema de arquivos de uma determinado diretrio de
   * arquivos de release notes (relativo  raiz do sistema).
   *
   * @param algoId identificador do algoritmo.
   * @param versionId identificador da verso.
   * @return caminho do diretrio de arquivos de release notes no sistema de
   *         arquivos.
   */
  private String getReleaseNotesDirPath(Object algoId, Object versionId) {
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmVersionInfo versionInfo = algo.getInfo().getVersionInfo(versionId);
    return versionInfo.getReleaseNotesDirPath();
  }

  /**
   * Indica se o arquivo possui extenso ZIP.
   *
   * @param path caminho para o arquivo.
   * @return true caso o arquivo possua extenso ZIP.
   */
  private boolean isZip(String path) {
    return (ZIP_EXTENSION.equalsIgnoreCase(FileUtils.getFileExtension(path)));
  }

  /**
   * Extrai o contedo de um arquivo zip na raiz de uma plataforma.
   *
   * @param pathFactory
   * @param file
   * @param userId
   * @return {@code true} se a extrao for bem-sucedida.
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  private boolean extractZipFile(IPathFactory pathFactory, FileInfo file,
    Object userId) throws RemoteException {
    String zipPath = pathFactory.getPath(file);
    File zipFile = new File(zipPath);
    String login = User.getUser(userId).getLogin();
    try {
      Unzip unzip = new Unzip(zipFile);
      if (unzip.listZipEntries().isEmpty()) {
        String msg = String.format(getString(
          "server.algoservice.error.invalid_zip_file"), zipFile.getName());
        Server.logSevereMessage(msg);
        String[] ids = new String[] { (String) userId };

        MessageService.getInstance().send(new Message(new UserNotification(
          login, msg, false, false)), ids);
        return false;
      }
      unzip.decompress(zipFile.getParentFile(), true);
    }
    catch (IOException e) {
      String msg = String.format(getString("server.algoservice.error.unzip_io"),
        zipFile.getName());
      Server.logSevereMessage(msg, e);
      String[] ids = new String[] { (String) userId };
      try {
        MessageService.getInstance().send(new Message(new UserNotification(
          login, msg, false, false)), ids);
      }
      catch (RemoteException e1) {
        Server.logSevereMessage(
          "Erro ao enviar evento de erro na extrao do arquivo de algoritmo.",
          e1);
      }
      return false;
    }
    finally {
      FileUtils.delete(zipFile);
    }
    return true;
  }

  /**
   * Callback indicando que um arquivo de configurao acabou de ser importado.
   *
   * @param args informaes sobre a importao recm-realizada.
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  private void configurationUploaded(Map args) throws RemoteException {
    Object userId = args.get(USER_ID_KEY);
    String login = User.getUser(userId).getLogin();
    final Object algoId = args.get(ALGO_ID_KEY);
    Algorithm algo = registeredAlgorithms.get(algoId);
    final Object versionId = args.get(VERSION_ID_KEY);
    String configName = (String) args.get(FILE_NAME_KEY);

    // Ser sempre um arquivo. Diretrios so encapsulados em arquivos zip.
    FileInfo file = new FileInfo(configName, false);
    if ((Boolean) args.get(EXPAND_IF_ZIP) && isZip(file.getName())) {
      IPathFactory pathFactory = new IPathFactory() {
        @Override
        public String getPath(FileInfo fileInfo) {
          return getConfiguratorDirPath(algoId, versionId) + File.separator
            + fileInfo.getPath();
        }
      };
      if (!extractZipFile(pathFactory, file, userId)) {
        return;
      }
    }

    try {
      algo.updateConfigurations(versionId);
    }
    catch (OperationFailureException e) {
      String clientMsg = getString("server.algoservice.error.refresh");
      Server.logSevereMessage(
        "No foi possvel atualizar os arquivos de configurao da verso "
          + versionId + " do algoritmo " + algoId, e);
      String[] ids = new String[] { (String) userId };

      try {
        MessageService.getInstance().send(new Message(new UserNotification(
          login, clientMsg, false, false)), ids);
      }
      catch (RemoteException e1) {
        Server.logSevereMessage(
          "Erro ao enviar evento de erro na carga do arquivo de configorao "
            + "de algoritmo.", e1);
      }
      return;
    }

    String[] path = new String[] { getSymbolicRootName(), algo.getInfo()
      .getName(), versionId.toString(), AlgorithmVersionInfo.CONFIGURATOR_DIR };
    String description = MessageFormat.format(getString(
      HISTORY_UPDATE_CONFIGURATOR), configName);
    appendHistory(userId, path, description);
  }

  /**
   * Callback indicando que um arquivo de documentao acabou de ser importado.
   *
   * @param args informaes sobre a importao recm-realizada.
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  private void documentationUploaded(Map args) throws RemoteException {
    Object userId = args.get(USER_ID_KEY);
    String login = User.getUser(userId).getLogin();
    final Object algoId = args.get(ALGO_ID_KEY);
    Algorithm algo = registeredAlgorithms.get(algoId);
    final Object versionId = args.get(VERSION_ID_KEY);
    String docName = (String) args.get(FILE_NAME_KEY);

    // Ser sempre um arquivo. Diretrios so encapsulados em arquivos zip.
    FileInfo file = new FileInfo(docName, false);
    if ((Boolean) args.get(EXPAND_IF_ZIP) && isZip(file.getName())) {
      IPathFactory pathFactory = new IPathFactory() {
        @Override
        public String getPath(FileInfo fileInfo) {
          return getDocumentationDirPath(algoId, versionId) + File.separator
            + fileInfo.getPath();
        }
      };
      if (!extractZipFile(pathFactory, file, userId)) {
        return;
      }
    }

    try {
      algo.updateDocumentations(versionId);
    }
    catch (OperationFailureException e) {
      String clientMsg = getString("server.algoservice.error.refresh");
      Server.logSevereMessage(
        "No foi possvel atualizar o arquivo de release notes da verso "
          + versionId + " do algoritmo " + algoId, e);
      String[] ids = new String[] { (String) userId };
      try {
        MessageService.getInstance().send(new Message(new UserNotification(
          login, clientMsg, false, false)), ids);
      }
      catch (RemoteException e1) {
        Server.logSevereMessage(
          "Erro ao enviar evento de erro na atualizao de arquivo de "
            + "release notes de algoritmo.", e);
      }
      return;
    }

    String[] path = new String[] { getSymbolicRootName(), algo.getInfo()
      .getName(), versionId.toString(),
        AlgorithmVersionInfo.DOCUMENTATION_DIR };
    String description = MessageFormat.format(getString(HISTORY_UPDATE_DOC),
      docName);
    appendHistory(userId, path, description);
  }

  /**
   * Callback indicando que um arquivo de documentao acabou de ser importado.
   *
   * @param args informaes sobre a importao recm-realizada.
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  private void releaseNotesUploaded(Map args) throws RemoteException {
    Object userId = args.get(USER_ID_KEY);
    String login = User.getUser(userId).getLogin();
    final Object algoId = args.get(ALGO_ID_KEY);
    Algorithm algo = registeredAlgorithms.get(algoId);
    final Object versionId = args.get(VERSION_ID_KEY);
    String fileName = (String) args.get(FILE_NAME_KEY);
    String targetName = (String) args.get(FILE_NAME_KEY);

    // Ser sempre um arquivo. Diretrios so encapsulados em arquivos zip.
    FileInfo file = new FileInfo(fileName, false);
    if ((Boolean) args.get(EXPAND_IF_ZIP) && isZip(file.getName())) {
      IPathFactory pathFactory = new IPathFactory() {
        @Override
        public String getPath(FileInfo fileInfo) {
          return getReleaseNotesDirPath(algoId, versionId) + File.separator
            + fileInfo.getPath();
        }
      };
      if (!extractZipFile(pathFactory, file, userId)) {
        return;
      }
    }

    try {
      algo.updateReleaseNotes(versionId, fileName);
    }
    catch (OperationFailureException e) {
      String clientMsg = getString("server.algoservice.error.refresh");
      Server.logSevereMessage(
        "No foi possvel atualizar os arquivos de documentao da verso "
          + versionId + " do algoritmo " + algoId, e);
      String[] ids = new String[] { (String) userId };
      try {
        MessageService.getInstance().send(new Message(new UserNotification(
          login, clientMsg, false, false)), ids);
      }
      catch (RemoteException e1) {
        Server.logSevereMessage(
          "Erro ao enviar evento de erro na atualizao de arquivo de "
            + "documentao de algoritmo.", e);
      }
      return;
    }

    String[] path = new String[] { getSymbolicRootName(), algo.getInfo()
      .getName(), versionId.toString() };
    String description = MessageFormat.format(getString(
      HISTORY_UPDATE_RELEASE_NOTES), fileName);
    appendHistory(userId, path, description);
  }

  /**
   * Callback indicando que um arquivo executvel acabou de ser importado. Se
   * for um zip, este ser descompactado.
   *
   * @param args informaes sobre a importao recm-realizada.
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  private void executableUploaded(Map args) throws RemoteException {
    Object userId = args.get(USER_ID_KEY);
    String login = User.getUser(userId).getLogin();
    final Object algoId = args.get(ALGO_ID_KEY);
    Algorithm algo = registeredAlgorithms.get(algoId);
    final Object versionId = args.get(VERSION_ID_KEY);
    final String platform = (String) args.get(PLATFORM_NAME_KEY);
    String exeName = (String) args.get(FILE_NAME_KEY);

    // Ser sempre um arquivo. Diretrios so encapsulados em arquivos zip.
    FileInfo file = new FileInfo(exeName, false);
    if ((args.containsKey(EXPAND_IF_ZIP) && (Boolean) args.get(EXPAND_IF_ZIP))
      && isZip(file.getName())) {
      IPathFactory pathFactory = new IPathFactory() {
        @Override
        public String getPath(FileInfo fileInfo) {
          return getPlatformPath(algoId, versionId, platform) + File.separator
            + fileInfo.getPath();
        }
      };
      if (!extractZipFile(pathFactory, file, userId)) {
        return;
      }
    }

    try {
      algo.updateExecutables(versionId, platform);
    }
    catch (OperationFailureException e) {
      String clientMsg = getString("server.algoservice.error.refresh");
      Server.logSevereMessage("No foi possvel atualizar a plataforma "
        + platform + " da verso " + versionId + " do algoritmo " + algoId, e);
      String[] ids = new String[] { (String) userId };
      try {
        MessageService.getInstance().send(new Message(new UserNotification(
          login, clientMsg, false, false)), ids);
      }
      catch (RemoteException e1) {
        Server.logSevereMessage(
          "Erro ao enviar evento de erro na atualizao da plataforma de "
            + "algoritmo.", e1);
      }
      return;
    }
    String[] path = new String[] { getSymbolicRootName(), algo.getInfo()
      .getName(), versionId.toString(), AlgorithmVersionInfo.BIN_DIR,
        platform };
    String description = MessageFormat.format(getString(
      HISTORY_UPDATE_EXECUTABLE), exeName);
    appendHistory(userId, path, description);
  }

  /**
   * Callback indicando que um diretorio da plataforma de verso acabou de ser
   * exportado. Se for um zip no formato apropriado, este ser descompactado no
   * cliente.
   *
   * @param args informaes sobre a importao recm-realizada.
   */
  private void executableDownloaded(Map args) {
    String filePath = (String) args.get(FILE_PATH_KEY);
    File file = new File(filePath);
    if ((args.containsKey(EXCLUDE_TMP_ZIP) && (Boolean) args.get(
      EXCLUDE_TMP_ZIP)) && isZip(file.getName())) {
      FileUtils.delete(file);
    }
  }

  /**
   * Callback indicando que um pacote de verso acabou de ser importado. Se for
   * um zip no formato apropriado, este ser descompactado.
   *
   * @param args informaes sobre a importao recm-realizada.
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  private void versionPackUploaded(Map args) throws RemoteException {
    Object userId = args.get(USER_ID_KEY);
    String login = User.getUser(userId).getLogin();
    Object algoId = args.get(ALGO_ID_KEY);
    Algorithm algo = registeredAlgorithms.get(algoId);
    AlgorithmInfo algoInfo = algo.getInfo();
    String filePath = (String) args.get(FILE_PATH_KEY);
    try {
      Unzip unzip = new Unzip(new File(filePath));
      List<ZipEntry> zipEntries = unzip.listZipEntries();
      if (zipEntries.isEmpty()) {
        String userMessage = getString(
          "server.algoservice.error.invalid_version_pack");
        String detailedMessage = String.format(
          "%sDetalhes: o arquivo zip est vazio.\n" + "Arquivo utilizado: %s"
            + ".\nIdentificador do algoritmo: %s.\n" + "Identificador do "
            + "usurio: %s.\n", userMessage, filePath, algoId, userId);
        Server.logSevereMessage(detailedMessage);
        String[] ids = new String[] { (String) userId };
        MessageService.getInstance().send(new Message(new UserNotification(
          login, userMessage, false, false)), ids);

        return;
      }
      ZipEntry rootZipEntry = null;
      String rootDirectoryName = null;
      for (ZipEntry zipEntry : zipEntries) {
        String[] splittedPath = FileUtils.splitPath(zipEntry.getName(), "/");
        if (splittedPath.length == 1) {
          if (rootZipEntry != null) {
            String userMessage = getString(
              "server.algoservice.error.invalid_version_pack");
            String detailedMessage = String.format(
              "%sDetalhes: o arquivo possui 2 ou mais arquivos/diretrios na "
                + "raz.\n" + "Arquivo utilizado: %s.\nIdentificador do "
                + "algoritmo: %s.\n" + "Identificador do usurio: %s.\n",
              userMessage, filePath, algoId, userId);
            Server.logSevereMessage(detailedMessage);
            String[] ids = new String[] { (String) userId };

            MessageService.getInstance().send(new Message(new UserNotification(
              login, userMessage, false, false)), ids);
            return;
          }
          rootZipEntry = zipEntry;
          rootDirectoryName = zipEntry.getName();
          rootDirectoryName = rootDirectoryName.replace("/", "");
        }
      }
      if (rootZipEntry == null) {
        String userMessage = getString(
          "server.algoservice.error.invalid_version_pack");
        String detailedMessage = String.format(
          "%sDetalhes: o arquivo no possui o nome do diretrio raz da verso"
            + ".\n" + "Arquivo utilizado: %s.\nIdentificador do algoritmo: %s"
            + ".\n" + "Identificador do usurio: %s.\n", userMessage, filePath,
          algoId, userId);
        Server.logSevereMessage(detailedMessage);
        String[] ids = new String[] { (String) userId };

        MessageService.getInstance().send(new Message(new UserNotification(
          login, userMessage, false, false)), ids);
        return;
      }
      if (!rootZipEntry.isDirectory()) {
        String userMessage = getString(
          "server.algoservice.error.invalid_version_pack");
        String detailedMessage = String.format(
          "%sDetalhes: o arquivo no possui um diretrio raz da verso.\n"
            + "Arquivo utilizado: %s.\nIdentificador do algoritmo: %s.\n"
            + "Identificador do usurio: %s.\nArquivo na raz: %s.\n",
          userMessage, filePath, algoId, userId, rootZipEntry);
        Server.logSevereMessage(detailedMessage);
        String[] ids = new String[] { (String) userId };

        MessageService.getInstance().send(new Message(new UserNotification(
          login, userMessage, false, false)), ids);
        return;
      }
      String regex = "v_([0-9]{3})_([0-9]{3})_([0-9]{3})";
      Pattern versionPattern = Pattern.compile(regex);
      Matcher versionMatcher = versionPattern.matcher(rootDirectoryName);
      if (!versionMatcher.matches()) {
        String userMessage = getString(
          "server.algoservice.error.invalid_version_pack");
        String detailedMessage = String.format(
          "%sDetalhes: o nome do diretrio raz no est no formato vlido.\n"
            + "Arquivo utilizado: %s.\nIdentificador do algoritmo: %s.\n"
            + "Identificador do usurio: %s.\nNome do arquivo: %s.\n"
            + "Expresso regular do padro: %s.\n", userMessage, filePath,
          algoId, userId, rootDirectoryName, regex);
        Server.logSevereMessage(detailedMessage);
        String[] ids = new String[] { (String) userId };

        MessageService.getInstance().send(new Message(new UserNotification(
          login, userMessage, false, false)), ids);
        return;
      }
      int major = Integer.parseInt(versionMatcher.group(1));
      int minor = Integer.parseInt(versionMatcher.group(2));
      int patch = Integer.parseInt(versionMatcher.group(3));
      AlgorithmVersionId algorithmVersionId = new AlgorithmVersionId(major,
        minor, patch);
      String algorithmName = algoInfo.getName();
      if (algoInfo.getVersionInfo(algorithmVersionId) != null) {
        String userMessage = String.format(getString(
          "server.algoservice.error.version_already_exists"), algorithmName,
          major, minor, patch);
        String detailedMessage = String.format(
          "%sIdentificador do algoritmo: %s\nArquivo utilizado: %s.\n"
            + "Identificador do usurio: %s.\n", userMessage, filePath, algoId,
          userId);
        Server.logSevereMessage(detailedMessage);
        String[] ids = new String[] { (String) userId };

        MessageService.getInstance().send(new Message(new UserNotification(
          login, userMessage, false, false)), ids);
        return;
      }
      File repositoryDirectory = new File(algorithmRepositoryPath);
      File algorithmDirectory = new File(repositoryDirectory, algoInfo
        .getDirectory());
      File versionsDirectory = new File(algorithmDirectory,
        AlgorithmInfo.VERSIONS_DIR);
      unzip.decompress(versionsDirectory);
      File versionDirectory = new File(versionsDirectory, rootDirectoryName);
      File binaryDirectory = new File(versionDirectory,
        AlgorithmVersionInfo.BIN_DIR);
      if (!binaryDirectory.isDirectory()) {
        String userMessage =
          "O arquivo informado no  um pacote de verso de algoritmo.\n";
        String detailedMessage = String.format(
          "%s\nDiretrio de binrios ausente.\n" + "Identificador do "
            + "algoritmo: %s\nArquivo utilizado: %s.\n" + "Identificador do "
            + "usurio: %s.\n", userMessage, filePath, algoId, userId);
        Server.logSevereMessage(detailedMessage);
        if (!FileUtils.delete(versionDirectory)) {
          detailedMessage = String.format(
            "Erro ao tentar excluir o arquivo %s.\n" + "Por favor, tente "
              + "exclu-lo manualmente.\n", versionDirectory.getAbsolutePath());
          Server.logWarningMessage(detailedMessage);
        }
        String[] ids = new String[] { (String) userId };

        MessageService.getInstance().send(new Message(new UserNotification(
          login, userMessage, false, false)), ids);
        return;
      }
      File[] platformDirectories = binaryDirectory.listFiles();
      for (File platformDirectory : platformDirectories) {
        if (!platformDirectory.isDirectory()) {
          String userMessage =
            "O arquivo informado no  um pacote de verso de algoritmo.\n";
          String detailedMessage = String.format(
            "%s\nO arquivo %s deveria ser um diretrio.\n" + "Identificador do"
              + " algoritmo: %s\nArquivo utilizado: %s.\n" + "Identificador do "
              + "usurio: %s.\n", userMessage, platformDirectory, filePath,
            algoId, userId);
          Server.logSevereMessage(detailedMessage);
          if (!FileUtils.delete(versionDirectory)) {
            detailedMessage = String.format(
              "Erro ao tentar excluir o arquivo %s.\n" + "Por favor, tente "
                + "exclu-lo manualmente.\n", versionDirectory
                  .getAbsolutePath());
            Server.logWarningMessage(detailedMessage);
          }
          String[] ids = new String[] { (String) userId };

          MessageService.getInstance().send(new Message(new UserNotification(
            login, userMessage, false, false)), ids);
          return;
        }
        File[] binaryFiles = platformDirectory.listFiles();
        for (File binaryFile : binaryFiles) {
          FileSystem.enableExecutionPermission(binaryFile.getAbsolutePath());
        }
      }
      algo.readVersion(versionsDirectory.getAbsolutePath(), versionDirectory);
      AlgorithmVersionInfo versionInfo = algoInfo.getVersionInfo(
        algorithmVersionId);
      String[] path = new String[] { getSymbolicRootName(), algo.getInfo()
        .getName() };
      String description = String.format(getString(
        "server.algoservice.history.import_version"), versionInfo);
      appendHistory(userId, path, description);
    }
    catch (IOException e) {
      String userMessage = getString(
        "server.algoservice.error.upload_version_server_io");
      String detailedMessage = String.format(
        "%sIdentificador do algoritmo: %s\n" + "Arquivo utilizado: %s"
          + ".\nIdentificador do usurio: %s.\n", userMessage, filePath, algoId,
        userId);
      Server.logSevereMessage(detailedMessage, e);
      String[] ids = new String[] { (String) userId };

      try {
        MessageService.getInstance().send(new Message(new UserNotification(
          login, userMessage, false, false)), ids);
      }
      catch (RemoteException e1) {
        Server.logSevereMessage(
          "Erro ao enviar evento de erro na atualizao de algoritmo.", e1);
      }
    }
    catch (ServerException e) {
      String userMessage = getString(
        "server.algoservice.error.upload_version_server");
      String detailedMessage = String.format(
        "%sIdentificador do algoritmo: %s\n" + "Arquivo utilizado: %s"
          + ".\nIdentificador do usurio: %s.\n", userMessage, filePath, algoId,
        userId);
      Server.logSevereMessage(detailedMessage, e);
      String[] ids = new String[] { (String) userId };
      try {
        MessageService.getInstance().send(new Message(new UserNotification(
          login, userMessage, false, false)), ids);
      }
      catch (RemoteException e1) {
        Server.logSevereMessage(
          "Erro ao enviar evento de erro na atualizao de algoritmo.", e1);
      }
    }
    finally {
      if (!new File(filePath).delete()) {
        String msg = String.format("Erro ao tentar excluir o arquivo %s.\n"
          + "Por favor, tente " + "exclu-lo manualmente.\n", filePath);
        Server.logWarningMessage(msg);
      }
    }
  }

  /**
   * Callback indicando que um pacote de verso acabou de ser importado. Se for
   * um zip no formato apropriado, este ser descompactado.
   *
   * @param args informaes sobre a importao recm-realizada.
   */
  private void versionPackDownloaded(Map args) {
    String filePath = (String) args.get(FILE_PATH_KEY);
    File zipFile = new File(filePath);
    FileUtils.delete(zipFile);
  }

  /**
   * Obtm o caminho a ser utilizado para criar entradas no histrico.<br>
   * Este caminho estar no formato:
   * algotihms/&lt;major&gt;.&lt;minor&gt;.&lt;patch
   * &gt;/[(bin/&lt;plataforma&gt;)|config|html]/...<br>
   * Ex.: .../aeepec3D/1.0.0/bin/Windows
   *
   * @param version Informaes da verso do arquivo.
   * @param pathFactory Fbrica para criar caminhos relativos ao sistema de
   *        arquivos. <br>
   *        Esta fbrica deve retornar um caminho no formato:
   *        .../&lt;algoritmo&gt
   *        ;/versions/v_&lt;major&gt;_&lt;minor&gt;_&lt;patch
   *        &gt;/[(bin/&lt;plataforma
   *        &gt;)|config|html]/.../[arquivo|(diretrio/)]<br>
   *        Ex.: .../aeepec3D/versions/v_001_000_000/bin/Windows/aeepec3D.lua
   * @param file Informaes do arquivo.
   * @return o caminho a ser utilizado para criar entradas no histrico.
   */
  private String getHistoryPath(final AlgorithmVersionInfo version,
    IPathFactory pathFactory, FileInfo file) {

    String fileSystemPath = pathFactory.getPath(file);
    String regex = "^.*/" + version.getInfo().getName()
      + "/[^/]+/[^/]+/(.+)/[^/]+/?$";
    String replacement = getSymbolicRootName() + "/" + version.getInfo()
      .getName() + "/" + version.getId() + "/$1";
    return fileSystemPath.replaceAll(regex, replacement);
  }

  //TODO A remover quando todos sistemas CSBase forem assinados

  /**
   * Thread responsvel por descompactar os executveis carregados (caso estejam
   * em um arquivo zip) e atualizar os objetos da lgica com os novos
   * executveis.
   */
  private class UpdateExecutableThread extends Thread {
    /**
     * Tamanho do buffer de leitura para arquivos zip
     *
     * @see #extractZipEntry(ZipEntry, ZipInputStream, String)
     */
    private static final int BUFFER_SIZE = 65536;

    /**
     * Extenso para arquivos ZIP
     */
    private static final String ZIP_EXTENSION = "zip";

    /**
     * Identificador do algoritmo no qual o arquivo foi carregado
     */
    private final Object algoId;

    /**
     * Caminho para o arquivo recm-carregado
     */
    private final String filePath;

    /**
     * Nome da plataforma na qual o arquivo foi carregado
     */
    private final String platform;

    /**
     * Identificador do usurio que realizou o carregamento do arquivo
     */
    private final Object userId;

    /**
     * Identificador da verso na qual o arquivo foi carregado
     */
    private final Object versionId;

    /**
     * Cria a thread.
     *
     * @param algoId Identificador do algoritmo.
     * @param versionId Identificador da verso do algoritmo.
     * @param platform Nome da plataforma.
     * @param filePath caminho do executvel recm-carregado.
     * @param userId identificador do usurio que carregou o executvel.
     */
    public UpdateExecutableThread(Object algoId, Object versionId,
      String platform, String filePath, Object userId) {
      this.algoId = algoId;
      this.versionId = versionId;
      this.platform = platform;
      this.filePath = filePath;
      this.userId = userId;
    }

    /**
     * Atualiza a representao lgica com o arquivo carregado. Se este for um
     * arquivo zip, seu contedo  extrado e o mesmo  removido, sendo que
     * somente os arquivos extrados sero includos na lgica. Por fim, dispara
     * uma notificao para os listeners avisando que a estrutura lgica mudou.
     */
    @Override
    public void run() {
      Server.logInfoMessage("Thread iniciada: updateExecutable.");
      Algorithm algo = registeredAlgorithms.get(algoId);
      if (algo == null) {
        return;
      }
      Collection<String> fileNames = new HashSet<String>();
      fileNames.add(FileUtils.getFileName(filePath));
      if (isZip(filePath)) {
        fileNames = getZippedFileNames(filePath);
        if (fileNames == null) {
          return;
        }
        removeZip(filePath);
      }
      try {
        algo.updateExecutables(versionId, platform);
      }
      catch (OperationFailureException oe) {
        logAndNotifyError(userId, oe);
      }
      AlgorithmInfo info = algo.getInfo();
      fireModifyEvent(info);
      String[] path = new String[] { getSymbolicRootName(), info.getName(),
          versionId.toString(), AlgorithmVersionInfo.BIN_DIR, platform };
      String description;
      for (String fileName : fileNames) {
        description = MessageFormat.format(getString(HISTORY_UPDATE_EXECUTABLE),
          new Object[] { fileName });
        appendHistory(userId, path, description);
      }
      Server.logInfoMessage("Terminando Thread updateExecutable.");
    }

    /**
     * Extrai o arquivo especificado (<code>ZipEntry</code>) de uma stream de
     * leitura de um arquivo zip (<code>ZipInputStream</code>), gravando o
     * arquivo extrado no diretrio especificado.
     *
     * @param entry arquivo compactado em um zip.
     * @param in <code>ZipInputStream</code> onde se encontra o arquivo.
     * @param outputPath caminho absoluto do diretrio de destino para o arquivo
     *        extrado.
     * @throws IOException se ocorrer algum problema durante a extrao.
     */
    private void extractZipEntry(ZipEntry entry, ZipInputStream in,
      String outputPath) throws IOException {
      BufferedOutputStream out = null;
      try {
        String outputFilePath = outputPath + File.separator + entry.getName();
        out = new BufferedOutputStream(new FileOutputStream(outputFilePath),
          BUFFER_SIZE);
        int bytesRead;
        byte[] buffer = new byte[BUFFER_SIZE];
        while ((bytesRead = in.read(buffer, 0, BUFFER_SIZE)) != -1) {
          out.write(buffer, 0, bytesRead);
        }
        out.flush();
      }
      catch (IOException e) {
        throw e;
      }
      finally {
        if (out != null) {
          try {
            out.close();
          }
          catch (IOException e) {
            Server.logSevereMessage(
              "Falha ao fechar stream de gravao para o arquivo " + entry
                .getName(), e);
          }
        }
      }
    }

    /**
     * Obtm os nomes dos arquivos compactados, extrados do arquivo zip.
     *
     * @param filePath caminho para o arquivo compactado.
     * @return nomes dos arquivos compactados ou <code>null</code> caso no
     *         tenha sido possvel extrair os arquivos.
     */
    private Collection<String> getZippedFileNames(String filePath) {
      Collection<String> fileNames = null;
      String outputPath = FileUtils.getFilePath(filePath);

      try {
        fileNames = unzip(filePath, outputPath);
      }
      catch (OperationFailureException oe) {
        String serverMsg = "Falha em pedido de servio para usurio " + userId;
        String zipName = FileUtils.getFileName(filePath);
        String clientMsg = MessageFormat.format(getString(
          "server.algoservice.error.unzip"), new Object[] { zipName });
        Exception se = new ServiceFailureException(clientMsg, oe);
        Server.logSevereMessage(serverMsg, se);
        String[] ids = new String[] { (String) userId };

        try {
          MessageService.getInstance().send(new Message(new UserNotification(
            getUser().getLogin(), clientMsg, false, false)), ids);
        }
        catch (RemoteException e) {
          Server.logSevereMessage(
            "Erro ao enviar evento listando os arquivos de um pacote de "
              + "algoritmo.", e);
        }
      }
      return fileNames;
    }

    /**
     * Indica se o arquivo possui extenso ZIP.
     *
     * @param path caminho para o arquivo.
     * @return true caso o arquivo possua extenso ZIP.
     */
    private boolean isZip(String path) {
      return (ZIP_EXTENSION.equalsIgnoreCase(FileUtils.getFileExtension(path)));
    }

    /**
     * Mtodo de convenincia para registrar em log uma falha ocorrida, bem como
     * notificar o usurio do problema com uma mensagem amigvel.
     *
     * @param userId identificador do usurio a ser notificado.
     * @param oe exceo ocorrida.
     */
    private void logAndNotifyError(final Object userId,
      OperationFailureException oe) {
      String serverMsg = "Falha em pedido de servio para usurio " + userId;
      String clientMsg = getString("server.algoservice.error.update_tree");
      Exception se = new ServiceFailureException(clientMsg, oe);
      Server.logSevereMessage(serverMsg, se);
      String[] ids = new String[] { (String) userId };

      try {
        MessageService.getInstance().send(new Message(new UserNotification(
          getUser().getLogin(), clientMsg, false, false)), ids);
      }
      catch (RemoteException e) {
        Server.logSevereMessage("Erro ao enviar evento de notificao de erro.",
          e);
      }
    }

    /**
     * Remove o arquivo zip especificado.
     *
     * @param filePath caminho absoluto para o arquivo zip.
     */
    private void removeZip(final String filePath) {
      if (!(new File(filePath).delete())) {
        Server.logSevereMessage("No foi possvel remover o arquivo zip "
          + filePath);
      }
    }

    /**
     * Descompacta um arquivo zip no diretrio especificado, retornando o nome
     * dos arquivos extrados.
     *
     * @param inputFilePath caminho absoluto para o arquivo zip (incluindo o
     *        nome do arquivo).
     * @param outputPath caminho absoluto para o diretrio de destino dos
     *        arquivos extrados.
     * @return coleo com os nomes dos arquivos extrados. Se ocorrer algum
     *         problema, retorna uma lista vazia (no-nula).
     * @throws OperationFailureException em caso de falha ao abrir o arquivo
     *         zip.
     */
    private Collection<String> unzip(String inputFilePath, String outputPath)
      throws OperationFailureException {
      ZipInputStream in = null;
      Collection<String> extractedFileNames = new HashSet<String>();
      try {
        in = new ZipInputStream(new BufferedInputStream(new FileInputStream(
          inputFilePath)));
        ZipEntry entry;
        while ((entry = in.getNextEntry()) != null) {
          extractedFileNames.add(entry.getName());
          extractZipEntry(entry, in, outputPath);
        }
      }
      catch (IOException e) {
        String serverMsg = getMessageFormatted(
          "AlgoService.error.file.open_failed", inputFilePath);
        throw new OperationFailureException(serverMsg, e);
      }
      finally {
        if (in != null) {
          try {
            in.close();
          }
          catch (IOException e) {
            Server.logSevereMessage("Falha ao fechar arquivo " + inputFilePath,
              e);
          }
        }
      }
      return extractedFileNames;
    }
  }

  //TODO A remover quando todos sistemas CSBase forem assinados

  /**
   * Thread que recarrega os algoritmos no intervalo definido pela propriedade
   * <code>restartServiceIntervalInMinutes</code>. Caso esta propriedade no
   * seja definida a recarga no  feita.
   */
  private class ReloadAlgsThread extends Thread {
    /**
     * Intervalo de renicio do servio de algoritmos.
     */
    private final int reloadIntervalInMilis;

    /**
     * Construtor.
     *
     * @param reloadAlgIntervalInMinutes intervalo de recarga dos algoritmos.
     */
    ReloadAlgsThread(int reloadAlgIntervalInMinutes) {
      reloadIntervalInMilis = reloadAlgIntervalInMinutes * 60 * 1000;
      setDaemon(true);
      start();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      while (!shutdown) {
        try {
          sleep(reloadIntervalInMilis);
        }
        catch (InterruptedException e) {
        }
        if (shutdown) {
          break;
        }
        try {
          setUserId(User.getAdminId());
          reloadAlgorithms(false);
        }
        catch (PermissionException pe) {
          Server.logSevereMessage(
            "Usurio admin no pode fazer recarga dos algoritmos", pe);
        }
      }
    }
  }

  /**
   * Retorna o path do arquivo de configurao de nomes de verso.
   *
   * @return o path
   */
  final String getVersionPropertyNamesPath() {
    final String tag = "version.property.names.file";
    final String path = getStringProperty(tag);
    return path;
  }

  /**
   * Retorna o path do arquivo de configurao de nomes em algoritmos. Este
   * arquivo de propriedades que especifica quais sao os atributos estendidos,
   * alem de especifica a ordem que os atributos devem ser apresentados
   *
   * @return o path
   */
  final public String getAlgorithmPropertyNamesPath() {
    final String tag = "algorithm.property.names.file";
    final String path = getStringProperty(tag);
    return path;
  }

  /**
   * Retorna o caminho para o repositrio de algoritmos.
   *
   * @return algorithmRepositoryPath o caminho para o repositrio de algoritmos.
   */
  public String getAlgorithmRepositoryPath() {
    return algorithmRepositoryPath;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized Category createCategory(String parentCategoryId,
    String name) throws RemoteException, CategoriesFileNotSavedException {

    boolean savexml = true;
    boolean toNotify = true;
    return createCategory(parentCategoryId, name, savexml, toNotify);
  }

  /**
   * Associa os algoritmos especificados s categorias especificadas, indicando
   * se deve ou no ser disparada uma notificao desse evento de criao do
   * algoritmo.
   *
   * @param parentCategoryId identificador da categoria pai da categoria a ser
   *        criado, ou null, caso seja uma categoria no primeiro nvel
   * @param name nome do categoria a ser criada
   * @param savexml se true, o xml de categorias ser salvo, caso contrrio, no
   *        ser persistido no xml a nova categoria. Em geral, esse parmetro
   *        deve ser true, exceto para o caso de vrias categorias estarem sendo
   *        criadas de uma vez, como no caso do Pacote de Algoritmos.
   * @param toNotify se true, os observadores desse servio sero notificados da
   *        criao do algoritmo, caso contrrio, no ser enviada a notificao
   * @return a categoria criada
   */
  private Category createCategory(String parentCategoryId, String name,
    boolean savexml, boolean toNotify) {

    // Busca a categoria pai, que vai receber uma nova sub-categoria
    Category parentCategory = null;
    if (parentCategoryId != null) {
      parentCategory = categorySet.getCategory(parentCategoryId);
      if (parentCategory == null) {
        Server.logSevereMessage("Falha na criao da categoria [" + name
          + "]: categoria pai " + "inexistente.");
        return null;
      }
    }
    Category category = categorySet.createCategory(parentCategory, name);

    if (savexml) {
      String errorMsg = "Falha na criao da categoria [" + name
        + "]: arquivo xml no foi " + "salvo.";
      saveCategoriesFile(errorMsg);
    }

    if (toNotify) {
      AlgoEvent action = new AlgoEvent(AlgoEvent.CREATE, category);
      sendEvent(action);
    }

    return category;
  }

  /**
   * Estabelece no conjunto de categorias um flag indicando se o conjunto foi ou
   * no salvo com sucesso. A persistncia  feita no xml correspondente ao
   * conjunto de categorias.
   *
   * @param state se true, indica que o conjunto de categorias foi salvo, caso
   *        contrrio, ocorreu um erro durante a persistncia e o conjunto no
   *        foi salvo
   */
  private void setCategorySetSavedFlag(boolean state) {
    categorySet.setCategorySetSavedFlag(state);
  }

  /**
   * Escreve o arquivo de categorias no diretrio raiz do repositrio de
   * algoritmos, a partir do conjunto de categorias especificado.
   *
   * @param categorySet conjunto de categorias de algoritmos a ser gravado
   * @return retorna true, se as categorias foram gravadas com sucesso, caso
   *         contrrio, retorna false
   * @throws CategoriesFileNotSavedException erro no salvamento do arquivo de
   *         categorias, por falha no diretrio raiz de algoritmos
   */
  private boolean writeCategoriesFile(CategorySet categorySet)
    throws CategoriesFileNotSavedException {
    String algoRootDir = getAndCheckAlgoRootDir();
    if (algoRootDir == null) {
      String message = getMessageFormatted(
        "AlgoService.error.category.file.not_found", CATEGORIES_FILE_NAME)
        + getMessageFormatted("AlgoService.error.category.root_dir.invalid");
      throw new CategoriesFileNotSavedException(message);
    }
    return writeCategoriesFile(algoRootDir, categorySet);
  }

  /**
   * Escreve o arquivo de categorias no diretrio raiz do repositrio de
   * algoritmos, a partir do conjunto de categorias especificado.
   *
   * @param algoRootDir diretrio raiz do repositrio de algoritmos
   * @param categorySet conjunto de categorias de algoritmos a ser gravado
   * @return retorna true, se as categorias foram gravadas com sucesso, caso
   *         contrrio, retorna false
   * @throws CategoriesFileNotSavedException erro ao gravar o arquivo xml de
   *         categorias
   */
  private boolean writeCategoriesFile(String algoRootDir,
    CategorySet categorySet) throws CategoriesFileNotSavedException {
    File categoriesFile = new File(algoRootDir, "categories.xml");
    FileWriter catFileWriter = null;

    try {
      catFileWriter = new FileWriter(categoriesFile);
      XmlCategoriesWriter xmlWriter = new XmlCategoriesWriter(categorySet);
      xmlWriter.write(catFileWriter);
    }
    catch (IOException e) {
      String message = String.format(
        "Erro ao ler/escrever o arquivo de categorias (%s).", categoriesFile
          .getAbsolutePath());
      Server.logSevereMessage(message, e);
      throw new CategoriesFileNotSavedException(getMessageFormatted(
        "AlgoService.error.categories.file.not_saved"));
    }
    finally {
      if (catFileWriter != null) {
        try {
          catFileWriter.close();
        }
        catch (IOException e) {
          String message = String.format(
            "Erro ao tentar fechar o stream de escrita para " + "o arquivo de "
              + "categorias (%s).", categoriesFile.getAbsolutePath());
          Server.logSevereMessage(message, e);
          throw new CategoriesFileNotSavedException(getMessageFormatted(
            "AlgoService.error.categories.file.not_saved"));
        }
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized CategorySet bindAlgorithmsToCategories(
    List<Object> algoIds, List<String> categoryIds) throws RemoteException,
    CategoriesFileNotSavedException {
    boolean toNotify = true;
    return bindAlgorithmsToCategories(algoIds, categoryIds, toNotify);
  }

  /**
   * Associa os algoritmos especificados s categorias especificadas, indicando
   * se deve ou no ser disparada uma notificao desse evento de modificao do
   * algoritmo.
   *
   * @param algoIds identificadores dos algoritmos a serem associados s
   *        categorias
   * @param categoryIds identificadores das categorias
   * @param toNotify se true, os observadores desse servio sero notificados da
   *        alterao dos algoritmos, caso contrrio, no ser enviada a
   *        notificao
   * @return retorna o conjunto de categorias que foi modificado pelas novas
   *         associaes de algoritmos
   */
  private CategorySet bindAlgorithmsToCategories(List<Object> algoIds,
    List<String> categoryIds, boolean toNotify) {
    CategorySet modifiedCategories = getModifiedCategorySet(categoryIds);
    List<AlgorithmInfo> algoInfoList = getAlgorithmInfoList(algoIds);
    categorySet.addAlgorithmsToCategories(modifiedCategories.getCategories(),
      algoInfoList);

    String errorMsg =
      "Falha na associao de algoritmos s categorias: arquivo xml no foi "
        + "salvo.";
    saveCategoriesFile(errorMsg);

    if (toNotify) {
      AlgoEvent action = new AlgoEvent(AlgoEvent.MODIFY, modifiedCategories);
      sendEvent(action);
    }
    return modifiedCategories;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized CategorySet unbindAlgorithmsFromCategories(
    List<Object> algoIds, List<String> categoryIds) throws RemoteException,
    CategoriesFileNotSavedException {

    CategorySet modifiedCategories = getModifiedCategorySet(categoryIds);
    List<AlgorithmInfo> algoInfoList = getAlgorithmInfoList(algoIds);

    categorySet.removeAlgorithmsFromCategories(modifiedCategories
      .getCategories(), algoInfoList);

    try {
      if (writeCategoriesFile(categorySet)) {
        setCategorySetSavedFlag(true);
      }
    }
    catch (CategoriesFileNotSavedException ex) {
      Server.logSevereMessage(
        "Falha na desassociao de algoritmos s categorias: arquivo xml no "
          + "foi salvo.");
      setCategorySetSavedFlag(false);
      throw ex;
    }
    finally {
      AlgoEvent action = new AlgoEvent(AlgoEvent.MODIFY, modifiedCategories);
      sendEvent(action);
    }
    return modifiedCategories;
  }

  /**
   * Obtm um conjunto de categorias, contendo as categorias que foram
   * modificadas, a partir dos ids especificados.
   *
   * @param categoryIds identificadores das categorias procuradas
   * @return o conjunto com as categorias modificadas, ou um conjunto vazio,
   *         caso nenhum dos ids informados tenham sido encontrados
   */
  private CategorySet getModifiedCategorySet(List<String> categoryIds) {
    CategorySet modifiedCategorySet = new CategorySet();
    for (String categoryId : categoryIds) {
      Category category = categorySet.getCategory(categoryId);
      if (category == null) {
        Server.logSevereMessage(getMessageFormatted(
          "AlgoService.error.category.not_found", categoryId));
        continue;
      }
      modifiedCategorySet.addCategory(category);
    }
    return modifiedCategorySet;
  }

  /**
   * Obtm uma lista com as informaes dos algoritmos, a partir dos seus
   * identificadores.
   *
   * @param algoIds identificadores dos algoritmos procurados
   * @return uma lista com as informaes dos algoritmos
   */
  private List<AlgorithmInfo> getAlgorithmInfoList(List<Object> algoIds) {
    List<AlgorithmInfo> algoInfoList = new java.util.Vector<AlgorithmInfo>();
    for (Object algoId : algoIds) {
      try {
        checkAlgorithmAdminPermission(algoId);
      }
      catch (PermissionException pe) {
        Server.logSevereMessage(pe.getMessage());
        continue;
      }
      AlgorithmInfo algoInfo = getInfo(algoId);
      if (algoInfo == null) {
        Server.logSevereMessage(getAlgoNotFoundMessage(algoId));
      }
      algoInfoList.add(algoInfo);
    }
    return algoInfoList;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized Category removeCategory(String categoryId)
    throws RemoteException, CategoriesFileNotSavedException {

    Category removedCategory = categorySet.removeCategory(categoryId);
    if (removedCategory == null) {
      String catMsgError = getMessageFormatted(
        "AlgoService.error.category.remove_failed", categoryId) + "\n"
        + getMessageFormatted("AlgoService.error.category.not_found",
          categoryId);
      Server.logSevereMessage(catMsgError);
      throw new ServiceFailureException(catMsgError);
    }

    try {
      if (writeCategoriesFile(categorySet)) {
        setCategorySetSavedFlag(true);
      }
    }
    catch (CategoriesFileNotSavedException ex) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.category.remove_failed", categoryId) + "\n"
        + getMessageFormatted("AlgoService.error.category.file.not_saved",
          CATEGORIES_FILE_NAME));
      setCategorySetSavedFlag(true);
      throw ex;
    }
    finally {
      AlgoEvent action = new AlgoEvent(AlgoEvent.DELETE, removedCategory);
      sendEvent(action);
    }
    return removedCategory;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Category getCategory(String id) throws RemoteException {
    Category category = categorySet.getCategory(id);
    return category;
  }

  /**
   * Obtm o diretrio raiz do repositrio de algoritmos e verifica se  um
   * diretrio vlido.
   *
   * @return retorna o nome do diretrio raiz do repositrio de algoritmos, ou
   *         retorna null caso no seja um diretrio vlido.
   */
  private String getAndCheckAlgoRootDir() {
    File algoDir = new File(algorithmRepositoryPath);
    if (!algoDir.exists() || !algoDir.isDirectory()) {
      return null;
    }
    return algorithmRepositoryPath;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized CategorySet setAlgorithmsToCategories(
    List<Object> algoIds, List<String> categoryIds) throws RemoteException,
    CategoriesFileNotSavedException {
    CategorySet modifiedCategories = getModifiedCategorySet(categoryIds);
    List<AlgorithmInfo> algoInfoList = getAlgorithmInfoList(algoIds);
    categorySet.removeAllAlgorithmsFromCategories(modifiedCategories
      .getCategories());
    categorySet.addAlgorithmsToCategories(modifiedCategories.getCategories(),
      algoInfoList);

    try {
      if (writeCategoriesFile(categorySet)) {
        setCategorySetSavedFlag(true);
      }
    }
    catch (CategoriesFileNotSavedException ex) {
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.category.bind_algorithms_failed") + "\n"
        + getMessageFormatted("AlgoService.error.category.file.not_saved",
          CATEGORIES_FILE_NAME));
      setCategorySetSavedFlag(false);
      throw ex;
    }
    finally {
      AlgoEvent action = new AlgoEvent(AlgoEvent.MODIFY, modifiedCategories);
      sendEvent(action);
    }
    return modifiedCategories;
  }

  /**
   * L o arquivo de categorias do diretrio raiz do repositrio de algoritmos,
   * e preenche o conjunto de categorias especificado.
   *
   * @param categorySet conjunto de categorias de algoritmos a ser preenchido
   *        pelo leitor
   * @return retorna true, se as categorias foram lidas com sucesso, caso
   *         contrrio, retorna false
   */
  private boolean readCategoriesFile(CategorySet categorySet) {
    String algoRootDir = getAndCheckAlgoRootDir();
    if (algoRootDir == null) {
      String message = getMessageFormatted(
        "AlgoService.error.category.file.read_failed", CATEGORIES_FILE_NAME)
        + "\n" + getMessageFormatted(
          "AlgoService.error.category.root_dir.invalid");
      Server.logSevereMessage(message);
      return false;
    }
    return readCategoriesFile(algoRootDir, categorySet);
  }

  /**
   * L o arquivo de categorias do diretrio raiz do repositrio de algoritmos,
   * e preenche o conjunto de categorias especificado.
   *
   * @param algoRootDir diretrio raiz do repositrio de algoritmos
   * @param categorySet conjunto de categorias de algoritmos a ser preenchido
   *        pelo leitor
   * @return retorna true, se as categorias foram lidas com sucesso, caso
   *         contrrio, retorna false
   */
  private boolean readCategoriesFile(String algoRootDir,
    CategorySet categorySet) {
    File categoriesFile = new File(algoRootDir, CATEGORIES_FILE_NAME);
    if (!categoriesFile.exists()) {
      return false;
    }

    InputStream inputStream = null;
    try {
      inputStream = new FileInputStream(categoriesFile);

      final XmlCategoriesReader categoriesReader = new XmlCategoriesReader(
        categorySet, inputStream);
      categoriesReader.read();
    }
    catch (AlgorithmNotFoundException e) {
      String message = getMessageFormatted(
        "AlgoService.error.category.file.parser_failed", categoriesFile
          .getAbsolutePath());
      System.err.println(message);
      Server.logSevereMessage(message, e);
    }
    catch (IOException e) {
      String message = getMessageFormatted(
        "AlgoService.error.category.file.read_failed", categoriesFile
          .getAbsolutePath());
      System.err.println(message);
      Server.logSevereMessage(message, e);
      return false;
    }
    catch (Exception e) {
      String message = getMessageFormatted(
        "AlgoService.error.category.file.read_failed", categoriesFile
          .getAbsolutePath());
      System.err.println(e);
      Server.logSevereMessage(message, e);
    }
    finally {
      if (isFirstCategoryLoad) {
        categorySet.changeCategoryIds();
        logCategorySet(categorySet);
        writeCategoriesFile(categorySet);
      }
      if (inputStream != null) {
        try {
          inputStream.close();
          return true;
        }
        catch (IOException e) {
          String message = getMessageFormatted(
            "AlgoService.error.category.file.close_failed", categoriesFile
              .getAbsolutePath());
          Server.logSevereMessage(message, e);
        }
        inputStream = null;
      }
    }

    return true;
  }

  /**
   * Gera no log as informaes de todas as categorias (inclusive as
   * sub-categorias) que foram criadas, com o identificador da categoria pai, o
   * identificador da categoria sendo criada e o seu nome.
   *
   * @param categorySet conjunto de categorias
   */
  private void logCategorySet(CategorySet categorySet) {
    if (categorySet == null) {
      return;
    }

    SortedSet<Category> allCategories = categorySet.getAllCategories();
    for (Category category : allCategories) {
      String msg = "Id da categoria pai: " + category.getParentId()
        + " - Criao da categoria com id = " + category.getId() + " : "
        + category.getName();
      Server.logInfoMessage(msg);
    }
    Server.logInfoMessage("Nmero de categorias da raiz: " + categorySet
      .getSize());
    Server.logInfoMessage("Nmero total de categorias criadas: " + allCategories
      .size());
  }

  /**
   * Copia um arquivo origem para outro arquivo destino.
   *
   * @param sourceFile stream do arquivo de origem
   * @param targetFile stream do arquivo de sada
   * @throws IOException erro de IO
   */
  private void copyFile(InputStream sourceFile, FileOutputStream targetFile)
    throws IOException {
    byte[] buffer = new byte[1024];

    try {
      int bytesLidos = 0;
      while ((bytesLidos = sourceFile.read(buffer)) > 0) {
        targetFile.write(buffer, 0, bytesLidos);
      }
    }
    finally {
      if (sourceFile != null) {
        sourceFile.close();
      }
      if (targetFile != null) {
        targetFile.close();
      }
    }
  }

  /**
   * Obtm o caminho do diretrio temporrio que contm o arquivo referente ao
   * pacote de algoritmos. O diretrio para cada dado possui como nome o valor
   * do token nico gerado. O diretrio temporrio definido como propriedade do
   * sistema deve existir previamente, assim como  feito para o diretrio base
   * de algoritmos.
   *
   * @param timestamp token que corresponde ao nome do diretrio a ser criado
   *        para importao/exportao do PA correspondente
   * @return retorna o path do diretrio que mantm temporariamente os arquivos
   *         do lidos do pacote de algoritmos. Caso o diretrio no exista, ele
   *          criado.
   */
  private File getAndCheckTempPackRootDir(String timestamp) {
    String tempPackPath = tempDirAlgorithmPath + File.separator + timestamp;
    File tempPackDir = new File(tempPackPath);
    try {
      Server.checkDirectory(tempPackPath);
    }
    catch (ServerException e) {
      Server.logSevereMessage(e.getMessage());
      throw new ServiceFailureException(e.getMessage());
    }
    return tempPackDir;
  }

  /**
   * Obtm o path absoluto do arquivo de DTD a ser usado para validao do
   * arquivo xml de metadados do pacote de algoritmos.
   *
   * @return o path absoluto do arquivo de DTD a ser usado para validao do
   *         metadados do pacote de algoritmos
   */
  private String getAlgorithmsPackDTDPath() {
    File dtdFile = new File(DTD_ALGORITHMS_PACK_FILE_NAME);
    String dtdName = dtdFile.getName();
    String message =
      "Obtendo o caminho absoluto do arquivo de DTD do Pacote de Algoritmos: "
        + dtdName;
    Server.logInfoMessage(message);
    return dtdName;
  }

  /**
   * L o arquivo xml de metadados do pacote de algoritmos e cria um pacote a
   * partir dos algoritmos lidos desse metadados. Cada algoritmo vem
   * especificando suas propriedades, verses, propriedades de cada verso e
   * categorias em que o algoritmo participa.
   *
   * @param algorithmsPackFile arquivo do Pacote de Algoritmos
   * @return retorna um pacote de algoritmos criado pelas informaes lidas do
   *         arquivo xml de metadados, caso contrrio, retorna null
   */
  private AlgorithmsPack readXMLAlgoPackageFile(File algorithmsPackFile) {
    if (!algorithmsPackFile.exists()) {
      return null;
    }

    AlgorithmsPack algorithmsPack = new AlgorithmsPack();
    InputStream inputStream = null;
    try {
      inputStream = new FileInputStream(algorithmsPackFile);

      String dtdPath = getAlgorithmsPackDTDPath();
      final XmlAlgorithmsPackReader algorithmsPackReader =
        new XmlAlgorithmsPackReader(algorithmsPack, inputStream, dtdPath);
      algorithmsPackReader.read();
    }
    catch (IOException e) {
      String message = getMessageFormatted(
        "AlgoService.error.algorithms_pack.file.read_failed", algorithmsPackFile
          .getPath());
      System.err.println(message);
      Server.logSevereMessage(message, e);
      return null;
    }
    catch (XMLParseException e) {
      String message = getMessageFormatted(
        "AlgoService.error.algorithms_pack.file.parser_failed",
        algorithmsPackFile.getPath());
      System.err.println(e);
      Server.logSevereMessage(message, e);
      algorithmsPack = null;
    }
    catch (Exception e) {
      String message = getMessageFormatted(
        "AlgoService.error.algorithms_pack.file.failed", algorithmsPackFile
          .getAbsolutePath());
      System.err.println(e);
      Server.logSevereMessage(message, e);
      algorithmsPack = null;
    }
    finally {
      logPackAlgorithms(algorithmsPack);
      if (inputStream != null) {
        try {
          inputStream.close();
        }
        catch (IOException e) {
          String message = getMessageFormatted(
            "AlgoService.error.algorithms_pack.file.close_failed",
            algorithmsPackFile.getAbsolutePath());
          Server.logSevereMessage(message, e);
          return null;
        }
        inputStream = null;
      }
    }

    return algorithmsPack;
  }

  /**
   * Gera no log as informaes dos algoritmos lidos do pacote de algoritmos.
   *
   * @param algorithmsPack pacote de algoritmos lido
   */
  private void logPackAlgorithms(AlgorithmsPack algorithmsPack) {
    for (AlgorithmInfo info : algorithmsPack.getAlgorithms()) {
      String msg = "PA - Id do algoritmo: " + info.getId()
        + "\nPA- Nome do algoritmo" + info.getName();
      Server.logInfoMessage(msg);
    }
    Server.logInfoMessage("PA - Nmero de algoritmos lidos: " + algorithmsPack
      .getSize());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized ImportAlgorithmsPackTransferInfo prepareImportAlgorithmsPack()
    throws RemoteException {
    checkAlgorithmServerAdminPermission();

    //Manter o usurio que requisitou o servio de importao de PA para fazer
    //o controle de acesso
    User user = Service.getUser();

    //Associar um timestamp nico ao arquivo de destino
    TStamp64 timestamp = new TStamp64();

    ImportAlgorithmsPackDataInfo importAlgoPackDataInfo =
      new ImportAlgorithmsPackDataInfo(user, timestamp);
    algoPackTokenMap.put(timestamp.toString(), importAlgoPackDataInfo);

    String targetPath;
    try {
      targetPath = getAndCheckTempPackRootDir(timestamp.toString())
        .getCanonicalPath() + File.separator + ALGO_PACK_FILE_NAME;
      File f = new File(targetPath);
      f.createNewFile();
    }
    catch (IOException e) {
      String userMessage = getMessageFormatted(
        "AlgoService.error.upload_algo_pack.read_file", ALGO_PACK_FILE_NAME);
      Server.logSevereMessage(userMessage, e);
      throw new ServiceFailureException(userMessage);
    }
    Map<String, Object> notifyArgs = new HashMap<String, Object>();
    notifyArgs.put(ALGO_FILE_TYPE_KEY, AlgoFileType.ALGORITHMS_PACK);
    notifyArgs.put(USER_ID_KEY, Service.getUser().getId());
    notifyArgs.put(FILE_PATH_KEY, targetPath);
    notifyArgs.put(EXPAND_IF_ZIP, Boolean.TRUE);
    RemoteFileChannelInfo channel = createChannel(targetPath, notifyArgs,
      FileChannelOp.UPLOAD);
    ImportAlgorithmsPackTransferInfo importInfo =
      new ImportAlgorithmsPackTransferInfo(channel, timestamp.toString());
    return importInfo;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized AlgorithmsPack getAlgorithmsPackInfo(
    String importDataToken) {
    if (importDataToken == null) {
      throw new IllegalArgumentException(getParameterNullMessage(
        importDataToken));
    }
    ImportAlgorithmsPackDataInfo importAlgoPackDataInfo = algoPackTokenMap.get(
      importDataToken);

    if (importAlgoPackDataInfo == null) {
      throw new ServiceFailureException(getMessageFormatted(
        "AlgoService.error.algorithm_pack.invalid_token", importDataToken));
    }
    checkUserImportAlgorithmsPackPermission(importAlgoPackDataInfo);

    AlgorithmsPack algorithmsPack = importAlgoPackDataInfo.getAlgoPackage();
    if (algorithmsPack != null) {
      return algorithmsPack;
    }

    try {
      return readAlgorithmsPackMetadata(importAlgoPackDataInfo);
    }
    catch (IOException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized boolean importAlgorithmsPack(String importDataToken,
    PAImportOperation... operations) throws RemoteException {
    if (importDataToken == null) {
      throw new IllegalArgumentException(getParameterNullMessage(
        importDataToken));
    }
    ImportAlgorithmsPackDataInfo importAlgoPackDataInfo = algoPackTokenMap.get(
      importDataToken);
    if (importAlgoPackDataInfo == null) {
      throw new ServiceFailureException(getMessageFormatted(
        "AlgoService.error.algorithm_pack.invalid_token", importDataToken));
    }

    checkUserImportAlgorithmsPackPermission(importAlgoPackDataInfo);

    if (!isValidAlgorithmsPack(importDataToken)) {
      return false;
    }

    //S continua a importao se o PA for vlido
    int nOperations = operations.length;
    if (nOperations >= 1 || nOperations <= 3) {
      PAImportOperation paImportOperation = operations[0];
      boolean toReplace = false;
      switch (paImportOperation) {
        case REPLACE_ALGORITHMS:
          toReplace = true;
          return addAlgorithmsFromAlgoPackInRepository(importAlgoPackDataInfo,
            toReplace);

        case KEEP_ALGORITHMS:
          toReplace = false;
          return addAlgorithmsFromAlgoPackInRepository(importAlgoPackDataInfo,
            toReplace);

        case MERGE_ALGORITHMS:
          PAImportOperation versionsOp = operations[1];
          PAImportOperation categoriesOp = operations[2];
          return mergeAlgorithmsFromAlgoPackInRepository(importAlgoPackDataInfo,
            versionsOp, categoriesOp);

        default:
          break;
      }
    }

    return true;
  }

  /**
   * Adiciona os algoritmos de um PA no repositrio de algoritmos, com base na
   * operao de combinao ("merge") dos algoritmos do PA que j existem.
   *
   * @param importAlgoPackDataInfo informaes dos dados de algoritmos do PA.
   * @param versionsOp operao a ser realizada sobre as verses j existentes
   *        no repositrio, considerando os dados dos algoritmos oriundos do PA
   * @param categoriesOp operao a ser realizada sobre as categorias j
   *        existentes no repositrio, considerando os dados dos algoritmos
   *        oriundos do PA
   * @return retorna true se todos os algoritmos foram importados do PA com
   *         sucesso, ou retorna false se o PA for invlido
   * @throws RemoteException falha na criao do algoritmo no repositrio, a
   *         partir dos dados oriundos do PA
   */
  private boolean mergeAlgorithmsFromAlgoPackInRepository(
    ImportAlgorithmsPackDataInfo importAlgoPackDataInfo,
    PAImportOperation versionsOp, PAImportOperation categoriesOp)
    throws RemoteException {

    String token = importAlgoPackDataInfo.getToken().toString();
    if (!importAlgoPackDataInfo.isDescompressed()) {
      if (!isValidAlgorithmsPack(token)) {
        return false;
      }
    }

    System.out.println(String.format(
      "\nEfetuando a importao dos algoritmos do Pacote de Algoritmos."));
    System.out.println(String.format(
      "A operao solicitada requer a combinao (merge) dos algoritmos j "
        + "existentes no repositrio.\n"));

    //Le o algoritmo do PA e busca o mesmo no repositrio de algoritmos
    AlgorithmsPack algoPackage = importAlgoPackDataInfo.getAlgoPackage();
    List<AlgorithmInfo> paAlgorithms = algoPackage.getAlgorithms();
    for (AlgorithmInfo paAlgoInfo : paAlgorithms) {
      String paAlgoId = paAlgoInfo.getId();
      //Busca o algoritmo no repositrio
      Algorithm repAlgo = registeredAlgorithms.get(paAlgoId);
      if (repAlgo != null) {
        //Remover o algoritmo do repositrio para substituir pelo que est no PA
        try {
          importVersionsFromPackToMergeAlgorithms(token, versionsOp, paAlgoInfo,
            repAlgo);
          importCategoriesFromPackToMergeAlgorithms(token, categoriesOp,
            paAlgoInfo, repAlgo);
        }
        catch (Exception e) {
          String clientMsg = getMessageFormatted(
            "AlgoService.error.algo.create_failed", paAlgoInfo.getName());
          Server.logSevereMessage(clientMsg, e);
          throw new ServiceFailureException(clientMsg);
        }
      }
      else {
        //Cria um novo algoritmo no repositrio a partir do PA
        try {
          System.out.println(String.format(
            "Criando o novo algoritmo %s no repositrio.", paAlgoId));
          createAlgorithm(token, paAlgoInfo);
        }
        catch (Exception e) {
          String clientMsg = getMessageFormatted(
            "AlgoService.error.algo.create_failed", paAlgoInfo.getName());
          Server.logSevereMessage(clientMsg, e);
          throw new ServiceFailureException(clientMsg);
        }
      }
    }
    return true;
  }

  /**
   * Realiza a importao das categorias do Pacote de Algoritmos para a operao
   * de merge de algoritmos.
   *
   * @param token identificador nico do dado (PA)
   * @param categoriesOp operao a ser realizada sobre as categorias dos
   *        algoritmos
   * @param paAlgoInfo objeto com as informaes do algoritmo no PA
   * @param repAlgo algoritmo no repositrio
   * @return retorna true se a importao das categorias do algoritmo do PA foi
   *         realizada com sucesso para o algoritmo no repositrio, caso
   *         contrrio, retorna false
   */
  private boolean importCategoriesFromPackToMergeAlgorithms(String token,
    PAImportOperation categoriesOp, AlgorithmInfo paAlgoInfo,
    Algorithm repAlgo) {
    boolean toReplace;

    switch (categoriesOp) {
      case REPLACE_CATEGORIES:
        /*
         * Essa operao quando aplicada a categorias, substitui todas as
         * categorias que j existem para o algoritmo do repositrio pelas do
         * algoritmo do PA
         */
        toReplace = true;
        return addCategoriesFromAlgoPackInRepository(token, toReplace,
          paAlgoInfo, repAlgo);

      case KEEP_CATEGORIES:
        /*
         * Essa operao quando aplicada a categorias, mantm todas as
         * associaes com categorias que j existem para o algoritmo do
         * repositrio
         */
        System.out.println(String.format(
          "Mantendo todas as categorias %s do algoritmo.", paAlgoInfo.getId()));
        return true;

      case MERGE_CATEGORIES:
        /*
         * Essa operao quando aplicada a categorias, mantm as categorias que
         * existem e cria as que ainda no existem para o algoritmo do
         * repositrio
         */
        toReplace = false;
        return addCategoriesFromAlgoPackInRepository(token, toReplace,
          paAlgoInfo, repAlgo);

      default:
        break;
    }
    return false;
  }

  /**
   * Adicona as categorias de um algoritmo do PA, em funo da operao
   * especificada para lidar com categorias j existentes do algoritmo no
   * repositrio.
   *
   * @param token identificador nico do dado (PA)
   * @param toReplace se true, indica que se j existir uma categoria com mesmo
   *        id associada ao algoritmo do repositrio, essa categoria ser
   *        substituda pela que foi lida do PA. Se false, indica que se j
   *        existir uma categoriam com mesmo id no algoritmo do repositrio, ela
   *        ser mantida e a do PA ignorada.
   * @param paAlgoInfo informaes do algoritmo do PA
   * @param repAlgo algoritmo correspondente no repositrio
   * @return retorna true se as verses foram adicionadas com sucesso, de acordo
   *         com a operao especificada, caso contrrio, retorna false
   */
  private boolean addCategoriesFromAlgoPackInRepository(String token,
    boolean toReplace, AlgorithmInfo paAlgoInfo, Algorithm repAlgo) {
    String paAlgoId = paAlgoInfo.getId();
    if (repAlgo == null) {
      return false;
    }
    AlgorithmInfo repAlgoInfo = repAlgo.getInfo();
    System.out.println(String.format(
      "\nRealizando o merge do algoritmo %s do PA no repositrio.", paAlgoId));

    try {
      if (toReplace) {
        System.out.println(String.format(
          "Substituindo o conjunto de categorias associadas ao algoritmo %s.",
          repAlgoInfo.getId()));

        replaceCategoriesToAlgorithmFromPack(paAlgoInfo, repAlgoInfo);
      }
      else {
        //Mantm as categorias que j existem.
        //Cria todas as novas categorias do algoritmo no PA que ainda no
        // existem e
        //associa cada categoria do PA ao algoritmo do repositrio

        System.out.println(String.format(
          "Associando as categorias do algoritmo %s do PA, e mantendo as "
            + "categorias j existentes.", repAlgoInfo.getId()));

        createAndBindCategoriesToAlgorithmFromPack(paAlgoInfo);
      }
    }
    catch (Exception e) {
      String clientMsg = getMessageFormatted(
        "AlgoService.error.algo.create_failed", paAlgoInfo.getName());
      Server.logSevereMessage(clientMsg, e);
      throw new ServiceFailureException(clientMsg);
    }
    return true;
  }

  /**
   * Realiza a substituio das categorias associadas ao algoritmo no
   * repositrio, pelas categorias do algoritmo no PA.
   *
   * @param paAlgoInfo objeto com as informaes do algoritmo no PA
   * @param repAlgoInfo objeto com as informaes do algoritmo no repositrio
   * @throws RemoteException erro de rmi
   * @throws CategoriesFileNotSavedException erro ao salvar o xml de categorias
   */
  private void replaceCategoriesToAlgorithmFromPack(AlgorithmInfo paAlgoInfo,
    AlgorithmInfo repAlgoInfo) throws CategoriesFileNotSavedException,
    RemoteException {
    //Desassocia o algoritmo das categorias atuais do algoritmo no repositrio
    unbindAlgorithmsFromCategories(repAlgoInfo);

    //Cria todas as novas categorias do algoritmo no PA que ainda no existem e
    //associa cada categoria do PA ao algoritmo do repositrio
    createAndBindCategoriesToAlgorithmFromPack(paAlgoInfo);
  }

  /**
   * Desassocia todas as categorias correntes do algoritmo do repositrio
   * especificado.
   *
   * @param algoInfo objeto com as informaes do algoritmo
   */
  private void unbindAlgorithmsFromCategories(AlgorithmInfo algoInfo) {
    SortedSet<Category> allCategories = categorySet.getAllCategories();
    for (Category category : allCategories) {
      Set<AlgorithmInfo> categoryAlgorithms = category.getAlgorithms();
      if (categoryAlgorithms.contains(algoInfo)) {
        category.removeAlgorithm(algoInfo);
      }
    }
  }

  /**
   * Cria todas as novas categorias do algoritmo no PA que ainda no existem e
   * associa cada categoria do PA ao algoritmo do repositrio
   *
   * @param paAlgoInfo objeto com as informaes do algoritmo no PA
   * @throws RemoteException erro de rmi
   * @throws CategoriesFileNotSavedException
   */
  private void createAndBindCategoriesToAlgorithmFromPack(
    AlgorithmInfo paAlgoInfo) throws CategoriesFileNotSavedException,
    RemoteException {
    List<String> paCategoryFullNames = paAlgoInfo
      .getAlgoPackCategoryFullNames();
    List<Category> paCategoriesInRepository = categorySet
      .getCategoriesFromFullNames(paCategoryFullNames);

    List<String> paCategoryIds = null;
    if (paCategoryFullNames.size() != paCategoriesInRepository.size()) {
      List<String> paCategoriesToCreate = new Vector<String>(
        paCategoryFullNames);
      for (Category category : paCategoriesInRepository) {
        paCategoriesToCreate.remove(category.getFullName());
      }

      createCategoriesFromAlgoPack(paCategoriesInRepository,
        paCategoriesToCreate);
      paCategoryIds = getCategoryIdsListFromCategories(
        paCategoriesInRepository);
    }
    else {
      paCategoryIds = getCategoryIdsListFromNames(paCategoryFullNames);
    }

    //Associa o algoritmo s categorias do algoritmo no PA
    boolean toNotify = false;
    List<Object> algorithmIds = new Vector<Object>();
    algorithmIds.add(paAlgoInfo.getId());
    bindAlgorithmsToCategories(algorithmIds, paCategoryIds, toNotify);
  }

  /**
   * Cria as categorias a partir do Pacote de Algoritmos.
   *
   * @param paCategoriesInRepository categorias do algoritmo no PA que j
   *        existem no servidor
   * @param paCategoryNamesToCreate categorias do algoritmo no PA que tem que
   *        ser criadas no servidor
   * @throws RemoteException erro de rmi
   * @throws CategoriesFileNotSavedException
   */
  private void createCategoriesFromAlgoPack(
    List<Category> paCategoriesInRepository,
    List<String> paCategoryNamesToCreate)
    throws CategoriesFileNotSavedException, RemoteException {

    Category parentCategory = null;
    Category category = null;

    for (String newCatName : paCategoryNamesToCreate) {
      String[] catLevelNames = newCatName.split(":");
      String parentCategoryFullName = null;
      parentCategory = null;
      System.out.println("teste");

      for (int i = 0; i < catLevelNames.length; i++) {
        if (i == 0) {
          parentCategoryFullName = null;
        }
        else {
          parentCategoryFullName = catLevelNames[i - 1];
        }
        String catName = catLevelNames[i];
        category = categorySet.getCategory(parentCategoryFullName, catName);
        if (category == null) {
          boolean savexml = false;
          boolean toNotify = false;
          String parentCategoryId = (parentCategory != null) ? parentCategory
            .getId() : null;
          category = createCategory(parentCategoryId, catName, savexml,
            toNotify);

          System.out.println(String.format(
            "Criada a categoria %s de algoritmo do PA.", category
              .getFullName()));
        }
        parentCategory = category;
      }
      paCategoriesInRepository.add(category);
    }
    String errorMsg =
      "Falha na criao das categorias vindas do Pacote de Algoritmos. "
        + "Arquivo" + " xml no foi salvo.";
    saveCategoriesFile(errorMsg);
  }

  /**
   * Salva o arquivo xml de categorias, persistindo todos os valores correntes
   * no servidor.
   *
   * @param errorMsg mensagem de erro no caso de falha no salvamento
   */
  private void saveCategoriesFile(String errorMsg) {
    try {
      if (writeCategoriesFile(categorySet)) {
        setCategorySetSavedFlag(true);
      }
    }
    catch (CategoriesFileNotSavedException ex) {
      Server.logSevereMessage(errorMsg);
      setCategorySetSavedFlag(false);
      throw ex;
    }
  }

  /**
   * Obtm uma lista com os identificadores das categorias a partir da lista de
   * nomes completos das categorias no servidor.
   *
   * @param repCategoryFullNames nomes completos das categorias do algoritmo do
   *        repositrio
   * @return a lista com os identificadores das categorias do algoritmo
   */
  private List<String> getCategoryIdsListFromNames(
    List<String> repCategoryFullNames) {
    List<String> categoryIds = new Vector<String>();
    List<Category> categories = categorySet.getCategoriesFromFullNames(
      repCategoryFullNames);
    for (Category category : categories) {
      categoryIds.add(category.getId());
    }
    return categoryIds;
  }

  /**
   * Obtm uma lista com os identificadores das categorias a partir da lista de
   * categorias especificada.
   *
   * @param paCategoriesInRepository categorias do algoritmo no PA que j
   *        existem no servidor
   * @return a lista com os identificadores das categorias do PA que existem no
   *         servidor
   */
  private List<String> getCategoryIdsListFromCategories(
    List<Category> paCategoriesInRepository) {
    List<String> categoryIds = new Vector<String>();
    for (Category category : paCategoriesInRepository) {
      categoryIds.add(category.getId());
    }
    return categoryIds;
  }

  /**
   * Importa as verses de um algoritmo do PA, em funo da operao de "merge"
   * dos algoritmos do PA, que j existem no repositrio.
   *
   * @param token identificador nico do dado (PA)
   * @param versionsOp operao que define o que deve ser realizada sobre a
   *        verso do algoritmo no repositrio, caso ela j exista l.
   * @param paAlgoInfo informaes do algoritmo do PA
   * @param repAlgo algoritmo correspondente no repositrio
   * @return retorna true se todos as verses dos algoritmos foram importados do
   *         PA com sucesso, caso contrrio, retorna false
   */
  private boolean importVersionsFromPackToMergeAlgorithms(String token,
    PAImportOperation versionsOp, AlgorithmInfo paAlgoInfo, Algorithm repAlgo) {
    boolean toReplace;

    switch (versionsOp) {
      case REPLACE_VERSIONS:
        toReplace = true;
        return addVersionsFromAlgoPackInRepository(token, toReplace, paAlgoInfo,
          repAlgo);

      case KEEP_VERSIONS:
        toReplace = false;
        return addVersionsFromAlgoPackInRepository(token, toReplace, paAlgoInfo,
          repAlgo);

      default:
        break;
    }
    return false;
  }

  /**
   * Adicona as verses de um algoritmo do PA, em funo da operao
   * especificada para lidar com verses j existentes do algoritmo no
   * repositrio.
   *
   * @param token identificador nico do dado (PA)
   * @param toReplace se true, indica que se j existir uma verso com mesmo id
   *        no algoritmo do repositrio, essa verso ser substituda pela que
   *        foi lida do PA. Se false, indica que se j existir uma verso com
   *        mesmo id no algoritmo do repositrio, ela ser mantida e a do PA
   *        ignorada.
   * @param paAlgoInfo informaes do algoritmo do PA
   * @param repAlgo algoritmo correspondente no repositrio
   * @return retorna true se as verses foram adicionadas com sucesso, de acordo
   *         com a operao especificada, caso contrrio, retorna false
   */
  private boolean addVersionsFromAlgoPackInRepository(String token,
    boolean toReplace, AlgorithmInfo paAlgoInfo, Algorithm repAlgo) {
    String paAlgoId = paAlgoInfo.getId();
    if (repAlgo == null) {
      return false;
    }
    AlgorithmInfo repAlgoInfo = repAlgo.getInfo();
    System.out.println(String.format(
      "\nRealizando o merge do algoritmo %s do PA no repositrio.", paAlgoId));

    Vector<AlgorithmVersionInfo> versions = paAlgoInfo.getVersions();
    for (AlgorithmVersionInfo paVersionInfo : versions) {
      //Remover o algoritmo do repositrio para substituir pelo que est no PA
      try {
        AlgorithmVersionId versionId = paVersionInfo.getId();
        AlgorithmVersionInfo repVersionInfo = repAlgoInfo.getVersionInfo(
          versionId);
        if (repVersionInfo == null) {
          //A verso no existe no algoritmo do repositrio, ento  s cri-la
          createVersion(token, repAlgo, paVersionInfo);
          System.out.println(String.format("Criando a verso %s do algoritmo.",
            versionId));
        }
        else {
          if (toReplace) {
            System.out.println(String.format(
              "Substituindo a verso %s do algoritmo.", versionId));
            boolean toNotify = false;
            removeVersion(paAlgoId, versionId, toNotify);
            createVersion(token, repAlgo, paVersionInfo);
          }
          else {
            System.out.println(String.format(
              "Mantendo a verso %s do algoritmo.", versionId));
          }
        }
      }
      catch (Exception e) {
        String clientMsg = getMessageFormatted(
          "AlgoService.error.algo.create_failed", paAlgoInfo.getName());
        Server.logSevereMessage(clientMsg, e);
        throw new ServiceFailureException(clientMsg);
      }
    }
    return true;
  }

  /**
   * Adiciona os algoritmos de um PA no repositrio de algoritmos, com base na
   * operao de substituio ("replace") ou de conservao ("keep") dos
   * algoritmos.
   *
   * @param importAlgoPackDataInfo informaes dos dados de algoritmos do PA.
   * @param toReplace indica se os algoritmos que j existem no repositrio
   *        devem ou no ser substitudos pelos algoritmos com mesmos ids
   *        oriundos do PA.
   * @return retorna true se todos os algoritmos foram importados do PA com
   *         sucesso, ou retorna false se o PA for invlido
   * @throws RemoteException falha na criao do algoritmo no repositrio, a
   *         partir dos dados oriundos do PA
   */
  private boolean addAlgorithmsFromAlgoPackInRepository(
    ImportAlgorithmsPackDataInfo importAlgoPackDataInfo, boolean toReplace)
    throws RemoteException {
    String token = importAlgoPackDataInfo.getToken().toString();
    if (!importAlgoPackDataInfo.isDescompressed()) {
      if (!isValidAlgorithmsPack(token)) {
        return false;
      }
    }

    System.out.println(String.format(
      "\nEfetuando a importao dos algoritmos do Pacote de Algoritmos."));
    if (toReplace) {
      System.out.println(String.format(
        "A operao solicitada requer a substituio dos algoritmos j "
          + "existentes no repositrio.\n"));
    }
    else {
      System.out.println(String.format(
        "A operao solicitada requer a conservao dos algoritmos j "
          + "existentes no repositrio.\n"));
    }

    //Le o algoritmo do PA e busca o mesmo no repositrio de algoritmos
    AlgorithmsPack algoPackage = importAlgoPackDataInfo.getAlgoPackage();
    List<AlgorithmInfo> paAlgorithms = algoPackage.getAlgorithms();
    for (AlgorithmInfo paAlgoInfo : paAlgorithms) {
      String paAlgoId = paAlgoInfo.getId();
      //Busca o algoritmo no repositrio
      Algorithm repAlgo = registeredAlgorithms.get(paAlgoId);
      if (repAlgo != null) {
        //Remover o algoritmo do repositrio para substituir pelo que est no PA
        try {
          if (toReplace) {
            System.out.println(String.format(
              "Substituindo o algoritmo %s no repositrio.", paAlgoId));
            removeAlgorithm(paAlgoId);
            createAlgorithm(token, paAlgoInfo);
          }
          else {
            System.out.println(String.format(
              "Mantendo o algoritmo %s j existente no repositrio.",
              paAlgoId));
          }
        }
        catch (Exception e) {
          String clientMsg = getMessageFormatted(
            "AlgoService.error.algo.create_failed", paAlgoInfo.getName());
          Server.logSevereMessage(clientMsg, e);
          throw new ServiceFailureException(clientMsg);
        }
      }
      else {
        //Cria um novo algoritmo no repositrio a partir do PA
        try {
          System.out.println(String.format(
            "Criando o novo algoritmo %s no repositrio.", paAlgoId));
          createAlgorithm(token, paAlgoInfo);
        }
        catch (Exception e) {
          String clientMsg = getMessageFormatted(
            "AlgoService.error.algo.create_failed", paAlgoInfo.getName());
          Server.logSevereMessage(clientMsg, e);
          throw new ServiceFailureException(clientMsg);
        }
      }
    }
    return true;
  }

  /**
   * Cria o novo algoritmo lido do PA no repositrio de algoritmos.
   *
   * @param token identificador nico do dado (PA)
   * @param paAlgoInfo informaes do algoritmo do PA
   * @throws ServerException
   * @throws IOException
   */
  private void createAlgorithm(String token, AlgorithmInfo paAlgoInfo)
    throws ServerException, IOException {
    String paAlgoId = paAlgoInfo.getId();
    String paAlgoName = paAlgoInfo.getName();

    //Propriedades do algoritmo
    Hashtable<String, String> paAlgoPropertyValues = paAlgoInfo
      .getPropertyValues();

    Algorithm newAlgorithm = null;
    newAlgorithm = Algorithm.createAlgorithm(paAlgoId, paAlgoName,
      paAlgoPropertyValues);
    createVersions(token, newAlgorithm, paAlgoInfo.getVersions());
    AlgorithmInfo newAlgoInfo = newAlgorithm.getInfo();
    registeredAlgorithms.put(newAlgoInfo.getId(), newAlgorithm);

    createAndBindCategoriesToAlgorithmFromPack(paAlgoInfo);

    Server.logInfoMessage("Algoritmo " + paAlgoName + " criado em "
      + newAlgoInfo.getDirectory());
    String[] path = new String[] { getSymbolicRootName() };
    String description = getMessageFormatted(HISTORY_CREATE_ALGORITHM,
      paAlgoName);
    appendHistory(path, description);
  }

  /**
   * Criao das verses do novo algoritmo do Pacote de Algoritmos criado.
   *
   * @param token identificador nico do dado do PA sendo importado
   * @param newAlgorithm novo algoritmo correspondente ao do PA
   * @param versions verses do algoritmo do PA
   * @throws IOException
   */
  private void createVersions(String token, Algorithm newAlgorithm,
    Vector<AlgorithmVersionInfo> versions) throws IOException {

    for (AlgorithmVersionInfo algoVersionInfo : versions) {
      createVersion(token, newAlgorithm, algoVersionInfo);
    }

  }

  /**
   * Criao de uma nova verso para o algoritmo do Pacote de Algoritmos que
   * est sendo importado.
   *
   * @param token identificador nico do dado do PA sendo importado
   * @param repAlgorithm algoritmo do repositrio, que corresponde ao do PA
   * @param versionInfo verso do algoritmo do PA que vai ser criada no
   *        algoritmo do repositrio
   * @throws IOException
   */
  private void createVersion(String token, Algorithm repAlgorithm,
    AlgorithmVersionInfo versionInfo) throws IOException {
    Map<String, String> versionProperties = versionInfo.getPropertyValues();

    Server.logInfoMessage(
      "Tentando criar uma verso para o algoritmo criado a partir do PA: "
        + repAlgorithm.getInfo().getId());

    AlgorithmVersionId versionId = versionInfo.getId();
    try {
      repAlgorithm.createVersionStructure(versionId, versionProperties);
      copyVersionDocumentationFile(token, repAlgorithm, versionInfo);
      copyVersionConfigurationFile(token, repAlgorithm, versionInfo);
      copyVersionExecutablesFile(token, repAlgorithm, versionInfo);
    }
    catch (ServerException se) {
      String versionStr = String.format("%d.%d.%d", versionId.getMajor(),
        versionId.getMinor(), versionId.getPatch());
      Server.logSevereMessage(getMessageFormatted(
        "AlgoService.error.algo.version.create_failed", versionStr, repAlgorithm
          .getInfo().getName()), se);
      throw new ServiceFailureException(versionStr);
    }
    String[] path = new String[] { getSymbolicRootName(), repAlgorithm.getInfo()
      .getName() };
    String description = getMessageFormatted(HISTORY_CREATE_VERSION, versionId);
    appendHistory(path, description);
    Server.logInfoMessage("Verso " + versionId + " criada para algoritmo "
      + repAlgorithm.getInfo().getName());
  }

  /**
   * Copia os arquivos de documentao do algoritmo do PA para o novo algoritmo
   * no repositrio.
   *
   * @param token identificador nico do dado (PA)
   * @param newAlgorithm novo algoritmo criado a partir do algoritmo do PA
   * @param paVersionInfo
   * @throws IOException
   */
  private void copyVersionDocumentationFile(String token,
    Algorithm newAlgorithm, AlgorithmVersionInfo paVersionInfo)
    throws IOException {
    //Dados do novo algoritmo no repositrio
    AlgorithmInfo repAlgoInfo = newAlgorithm.getInfo();
    final String repAlgoId = repAlgoInfo.getId();

    //Dados da verso do novo algoritmo
    final AlgorithmVersionId versionId = paVersionInfo.getId();
    final AlgorithmVersionInfo repVersionInfo = repAlgoInfo.getVersionInfo(
      versionId);

    //Diretrio da verso e da documentao no algoritmo do PA
    File paVersionDir = getAlgoPackVersionDir(token, repAlgoId, versionId);
    File paDocDir = getAlgoPackDocumentationDir(paVersionDir);

    //Diretrio de documentao da verso no repositrio de algoritmos
    File repDocDir = getDocumentationDir(repVersionInfo);

    String[] docHistoryPath = new String[] { getSymbolicRootName(), repAlgoInfo
      .getName(), versionId.toString(),
        AlgorithmVersionInfo.DOCUMENTATION_DIR };
    String docHistoryDescPattern = getString(HISTORY_UPDATE_DOC);

    copyFiles(repVersionInfo, paDocDir, repDocDir, docHistoryPath,
      docHistoryDescPattern);

    //Atualiza as informaes da verso com os arquivos de documentao copiados
    final List<FileInfo> documentation = FileInfo.createFilesInfo(repDocDir);
    repVersionInfo.setDocumentation(documentation);
  }

  /**
   * Copia os arquivos de configurao do algoritmo do PA para o novo algoritmo
   * no repositrio.
   *
   * @param token identificador nico do dado (PA)
   * @param newAlgorithm novo algoritmo criado a partir do algoritmo do PA
   * @param paVersionInfo
   * @throws IOException
   */
  private void copyVersionConfigurationFile(String token,
    Algorithm newAlgorithm, AlgorithmVersionInfo paVersionInfo)
    throws IOException {
    //Dados do novo algoritmo no repositrio
    AlgorithmInfo repAlgoInfo = newAlgorithm.getInfo();
    final String repAlgoId = repAlgoInfo.getId();

    //Dados da verso do novo algoritmo
    final AlgorithmVersionId versionId = paVersionInfo.getId();
    final AlgorithmVersionInfo repVersionInfo = repAlgoInfo.getVersionInfo(
      versionId);

    //Diretrio da verso e da configurao no algoritmo do PA
    File paVersionDir = getAlgoPackVersionDir(token, repAlgoId, versionId);
    File paConfigDir = getAlgoPackConfigurationDir(paVersionDir);

    //Diretrio de configurao da verso no repositrio de algoritmos
    File repConfigDir = getConfigurationDir(repVersionInfo);

    String[] configHistoryPath = new String[] { getSymbolicRootName(),
        repAlgoInfo.getName(), versionId.toString(),
        AlgorithmVersionInfo.CONFIGURATOR_DIR };
    String configHistoryDescPattern = getString(HISTORY_UPDATE_CONFIGURATOR);

    copyFiles(repVersionInfo, paConfigDir, repConfigDir, configHistoryPath,
      configHistoryDescPattern);

    //Atualiza as informaes da verso com os arquivos de configurao copiados
    final List<FileInfo> configurators = FileInfo.createFilesInfo(repConfigDir);
    repVersionInfo.setConfigurators(configurators);
  }

  /**
   * Copia os arquivos de executveis do algoritmo do PA para o novo algoritmo
   * no repositrio.
   *
   * @param token identificador nico do dado (PA)
   * @param newAlgorithm novo algoritmo criado a partir do algoritmo do PA
   * @param paVersionInfo
   * @throws IOException
   */
  private void copyVersionExecutablesFile(String token, Algorithm newAlgorithm,
    AlgorithmVersionInfo paVersionInfo) throws IOException {
    //Dados do novo algoritmo no repositrio
    AlgorithmInfo repAlgoInfo = newAlgorithm.getInfo();
    final String repAlgoId = repAlgoInfo.getId();

    //Dados da verso do novo algoritmo
    final AlgorithmVersionId versionId = paVersionInfo.getId();
    final AlgorithmVersionInfo repVersionInfo = repAlgoInfo.getVersionInfo(
      versionId);

    //Diretrio da verso e de executveis no algoritmo do PA
    File paVersionDir = getAlgoPackVersionDir(token, repAlgoId, versionId);
    File paBinDir = getAlgoPackExecutableDir(paVersionDir);

    //Diretrio de configurao da verso no repositrio de algoritmos
    File repBinDir = getExecutableDir(repVersionInfo);

    copyPlatformsDirectories(repVersionInfo, paBinDir, repBinDir);
  }

  /**
   * Copia arquivos entre um diretrio origem no Pacote de algoritmos e um de
   * destino no repositrio de algoritmos.
   *
   * @param versionInfo
   * @param paSourceDir
   * @param repTargetDir
   * @param historyPath
   * @param historyDescPattern
   * @throws IOException
   */
  private void copyFiles(final AlgorithmVersionInfo versionInfo,
    final File paSourceDir, final File repTargetDir, final String[] historyPath,
    final String historyDescPattern) throws IOException {
    //Obtm os arquivos de documentao no diretrio da verso no PA
    File[] sourceFiles = paSourceDir.listFiles();
    User user = Service.getUser();

    if (sourceFiles == null) {
      return;
    }
    for (File sourceFile : sourceFiles) {
      if (!sourceFile.isDirectory()) {
        // Ser sempre um arquivo. Diretrios so encapsulados em arquivos zip.
        FileInfo file = new FileInfo(sourceFile.getName(), false);
        if (isZip(file.getName())) {
          IPathFactory pathFactory = new IPathFactory() {
            @Override
            public String getPath(FileInfo fileInfo) {
              try {
                return paSourceDir.getCanonicalPath() + File.separator
                  + fileInfo.getPath();
              }
              catch (IOException e) {
                throw new ServiceFailureException(String.format(
                  "Falha para identificar o path do diretrio %s.",
                  paSourceDir), e);
              }
            }
          };
          if (!extractZipFile(pathFactory, file, user.getId())) {
            throw new ServiceFailureException(String.format(
              "Falha ao extrair o contedo do arquivo compactado %s, "
                + "localizado em <%s>.", file, pathFactory));
          }
        }
        else {
          File targetFile = new File(repTargetDir, sourceFile.getName());
          if (!targetFile.exists()) {
            targetFile.createNewFile();
            copyFile(new FileInputStream(sourceFile), new FileOutputStream(
              targetFile));
          }
          String historyDesc = MessageFormat.format(historyDescPattern,
            sourceFile.getName());

          appendHistory(user.getId(), historyPath, historyDesc);
        }
      }
    }
  }

  /**
   * Copia os diretrios das plataformas entre um diretrio origem no Pacote de
   * algoritmos e um de destino no repositrio de algoritmos.
   *
   * @param versionInfo
   * @param paExecutableDir
   * @param repExecutableDir
   * @throws IOException
   */
  private void copyPlatformsDirectories(final AlgorithmVersionInfo versionInfo,
    final File paExecutableDir, final File repExecutableDir)
    throws IOException {
    //Obtm os diretrios com as plataformas dos binrios no diretrio da
    // verso no PA
    File[] platformDirs = paExecutableDir.listFiles();
    AlgorithmInfo algoInfo = versionInfo.getInfo();
    AlgorithmVersionId versionId = versionInfo.getId();

    if (platformDirs == null) {
      return;
    }
    for (File platformDir : platformDirs) {
      if (!platformDir.isDirectory()) {
        throw new ServiceFailureException(String.format(
          "A plataforma %s no  um diretrio na verso %s do algoritmo %s do "
            + "" + "PA.", platformDir, versionId, algoInfo.getId()));
      }
      else {
        String[] binHistoryPath = new String[] { getSymbolicRootName(), algoInfo
          .getName(), versionId.toString(), AlgorithmVersionInfo.BIN_DIR };
        String binHistoryDescPattern = getString(HISTORY_UPDATE_EXECUTABLE);

        File targetDir = new File(repExecutableDir, platformDir.getName());
        if (!targetDir.exists()) {
          targetDir.mkdirs();
          copyFiles(versionInfo, platformDir, targetDir, binHistoryPath,
            binHistoryDescPattern);
        }

        //Atualiza as informaes da verso com os arquivos de plataformas
        // copiados
        final List<FileInfo> executables = FileInfo.createFilesInfo(
          platformDir);
        for (FileInfo file : executables) {
          if (!file.isDirectory()) {
            String path = null;
            try {
              path = new File(platformDir, file.getPath()).getAbsolutePath();
              FileSystem.enableExecutionPermission(path);
            }
            catch (final ServerException e) {
              throw new ServiceFailureException(String.format(
                "Falha ao habilitar permisso de execuo sobre o arquivo %s.",
                path), e);
            }
          }
        }
        versionInfo.setPlatformExecutables(platformDir.getName(), executables);

      }
    }
  }

  /**
   * Obtm o diretrio de documentao de uma verso de algoritmo do
   * repositrio.
   *
   * @param versionInfo informaes da verso do algoritmo
   * @return o diretrio de documentao de uma verso de algoritmo
   */
  private File getDocumentationDir(AlgorithmVersionInfo versionInfo) {
    File repDocDir = new File(versionInfo.getDocDirPath());
    if (!repDocDir.exists() || !repDocDir.isDirectory()) {
      String clientMessage = String.format(
        "O diretrio de documentao no existe para a verso % do algoritmo "
          + "%s no repositrio.", versionInfo.getId(), versionInfo.getInfo()
            .getId());
      throw new ServiceFailureException(clientMessage);
    }
    return repDocDir;
  }

  /**
   * Obtm o diretrio de configurao de uma verso de algoritmo do
   * repositrio.
   *
   * @param versionInfo informaes da verso do algoritmo
   * @return o diretrio de configurao de uma verso de algoritmo
   */
  private File getConfigurationDir(AlgorithmVersionInfo versionInfo) {
    File repConfigDir = new File(versionInfo.getConfiguratorDirPath());
    if (!repConfigDir.exists() || !repConfigDir.isDirectory()) {
      String clientMessage = String.format(
        "O diretrio de configurao no existe para a verso % do algoritmo "
          + "%s no repositrio.", versionInfo.getId(), versionInfo.getInfo()
            .getId());
      throw new ServiceFailureException(clientMessage);
    }
    return repConfigDir;
  }

  /**
   * Obtm o diretrio executvel de uma verso de algoritmo do repositrio.
   *
   * @param versionInfo informaes da verso do algoritmo
   * @return o diretrio executvel de uma verso de algoritmo
   */
  private File getExecutableDir(AlgorithmVersionInfo versionInfo) {
    File executableDir = new File(versionInfo.getExecutableDirPath());
    if (!executableDir.exists() || !executableDir.isDirectory()) {
      String clientMessage = String.format(
        "O diretrio de executveis no existe para a verso % do algoritmo "
          + "%s" + " no repositrio.", versionInfo.getId(), versionInfo
            .getInfo().getId());
      throw new ServiceFailureException(clientMessage);
    }
    return executableDir;
  }

  /**
   * Obtm o diretrio de documentao da verso do algoritmo do Pacote de
   * Algoritmos.
   *
   * @param paVersionDir diretrio de verso do algoritmo no PA
   * @return o diretrio de documentao da verso do algoritmo do Pacote de
   *         Algoritmos
   */
  private File getAlgoPackDocumentationDir(File paVersionDir) {
    File paDocDirectory = new File(paVersionDir,
      AlgorithmVersionInfo.DOCUMENTATION_DIR + "/");
    if (!paDocDirectory.exists() || !paDocDirectory.isDirectory()) {
      String clientMessage = String.format(
        "O caminho <%s> para o diretrio de documentao no existe no PA.",
        paDocDirectory);
      throw new ServiceFailureException(clientMessage);
    }
    return paDocDirectory;
  }

  /**
   * Obtm o diretrio de configurao da verso do algoritmo do Pacote de
   * Algoritmos.
   *
   * @param paVersionDir diretrio de verso do algoritmo no PA
   * @return o diretrio de configurao da verso do algoritmo do Pacote de
   *         Algoritmos
   */
  private File getAlgoPackConfigurationDir(File paVersionDir) {
    File paConfigDir = new File(paVersionDir,
      AlgorithmVersionInfo.CONFIGURATOR_DIR + "/");
    if (!paConfigDir.exists() || !paConfigDir.isDirectory()) {
      String clientMessage = String.format(
        "O caminho <%s> para o diretrio de configurao no existe no PA.",
        paConfigDir);
      throw new ServiceFailureException(clientMessage);
    }
    return paConfigDir;
  }

  /**
   * Obtm o diretrio de configurao da verso do algoritmo do Pacote de
   * Algoritmos.
   *
   * @param paVersionDir diretrio de verso do algoritmo no PA
   * @return o diretrio de configurao da verso do algoritmo do Pacote de
   *         Algoritmos
   */
  private File getAlgoPackExecutableDir(File paVersionDir) {
    File paBinDir = new File(paVersionDir, AlgorithmVersionInfo.BIN_DIR + "/");
    if (!paBinDir.exists() || !paBinDir.isDirectory()) {
      String clientMessage = String.format(
        "O caminho <%s> para o diretrio de executveis no existe no PA.",
        paBinDir);
      throw new ServiceFailureException(clientMessage);
    }
    return paBinDir;
  }

  /**
   * Obtm o diretrio de uma verso de um algoritmo, no Pacote de Algoritmos.
   *
   * @param token identificador nico do dado (PA)
   * @param algoId identificador do algoritmo
   * @param versionId identificador da verso
   * @return retorna o diretrio de uma verso de um algoritmo, no Pacote de
   *         Algoritmos
   * @throws IOException
   */
  private File getAlgoPackVersionDir(String token, String algoId,
    AlgorithmVersionId versionId) throws IOException {
    String versionDir = AlgorithmVersionInfo.getDirectoryFor(versionId
      .getMajor(), versionId.getMinor(), versionId.getPatch());

    String paTokenDirPath = getAndCheckTempPackRootDir(token)
      .getCanonicalPath();
    File paTokenDirectory = new File(paTokenDirPath);
    String paAlgoDirPath = algoId + "/";
    String paVersionPath = paAlgoDirPath + versionDir + "/";
    File paVersionDir = new File(paTokenDirectory, paVersionPath);
    return paVersionDir;
  }

  /**
   * Verifica se o pacote de algoritmos  vlido.
   *
   * @param importDataToken identificador nico do dado (PA)
   * @return retorna true se o PA for vlido, caso contrrio, retorna false
   * @throws RemoteException
   */
  private boolean isValidAlgorithmsPack(String importDataToken)
    throws RemoteException {
    if (validateAlgorithmsPack(importDataToken).equals(
      VALID_ALGORITHMS_PACK_MSG)) {
      return true;
    }
    return false;
  }

  /**
   * Verifica se o usurio tem permisso para importar o dado PA especificado
   * pelas suas informaes de importao.
   *
   * @param importAlgoPackDataInfo informaes do dado PA a ser importado
   * @throws PermissionException usurio sem permisso para administrar o
   *         algoritmo
   */
  private void checkUserImportAlgorithmsPackPermission(
    ImportAlgorithmsPackDataInfo importAlgoPackDataInfo) {
    if (importAlgoPackDataInfo == null) {
      throw new PermissionException(getUserImportAlgoPackPermissionErrorMessage(
        importAlgoPackDataInfo));
    }
    User user = Service.getUser();
    if (!user.equals(importAlgoPackDataInfo.getUser())) {
      throw new PermissionException(getUserImportAlgoPackPermissionErrorMessage(
        importAlgoPackDataInfo));
    }
  }

  /**
   * Obtm mensagem de erro para o caso do usurio no ter permisso para
   * importar o dado PA especificado.
   *
   * @param importAlgoPackDataInfo informaes sobre o dado PA a ser importado
   * @return a mensagem de erro correspondente
   */
  private String getUserImportAlgoPackPermissionErrorMessage(
    ImportAlgorithmsPackDataInfo importAlgoPackDataInfo) {
    String msg = getString(ERROR_IMPORT_PA_USER_PERMISSION);
    Object[] args = new Object[] { importAlgoPackDataInfo.getUser().getLogin(),
        importAlgoPackDataInfo.getToken() };
    return MessageFormat.format(msg, args);
  }

  /**
   * {@inheritDoc}
   *
   * @throws RemoteException em caso de erro.
   */
  @Override
  public synchronized void finishImportAlgorithmsPack(String importDataToken)
    throws RemoteException {
    if (importDataToken == null) {
      throw new IllegalArgumentException(getParameterNullMessage(
        importDataToken));
    }
    ImportAlgorithmsPackDataInfo importAlgoPackDataInfo = algoPackTokenMap.get(
      importDataToken);
    if (importAlgoPackDataInfo == null) {
      throw new ServiceFailureException(getMessageFormatted(
        "AlgoService.error.algorithm_pack.invalid_token", importDataToken));
    }
    checkUserImportAlgorithmsPackPermission(importAlgoPackDataInfo);
    removeAlgoPackTokensDirs(importDataToken);
  }

  /**
   * Remove os diretrios correspondentes aos tokens de dados (PAs) que no so
   * mais vlidos.
   *
   * @param importDataToken identificador nico do dado (PA) a ser importado
   */
  private void removeAlgoPackTokensDirs(String importDataToken) {
    String paDirPath;
    try {
      paDirPath = getAndCheckTempPackRootDir(importDataToken)
        .getCanonicalPath();
    }
    catch (IOException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
    File tempDirFile = new File(paDirPath);
    if (!FileUtils.delete(tempDirFile)) {
      String detailedMessage = String.format(
        "Erro ao tentar excluir o diretrio %s.\n" + "Por favor, tente "
          + "exclu-lo manualmente.\n", tempDirFile.getAbsolutePath());
      Server.logWarningMessage(detailedMessage);
    }
    algoPackTokenMap.remove(importDataToken);
  }

  /**
   * Realiza a leitura do arquivo xml de metadados do pacote de algoritmos.
   *
   * @param importAlgoPackDataInfo informaes do dado (PA) que est sendo
   *        importado
   * @return a estrutura correspondente ao metadados do PA
   * @throws IOException erro de IO
   */
  private AlgorithmsPack readAlgorithmsPackMetadata(
    ImportAlgorithmsPackDataInfo importAlgoPackDataInfo) throws IOException {
    String importDataToken = importAlgoPackDataInfo.getToken().toString();
    String paFileUserLogin = importAlgoPackDataInfo.getUser().getLogin();

    String filePath = getAndCheckTempPackRootDir(importDataToken)
      .getCanonicalPath() + File.separator + ALGO_PACK_FILE_NAME;

    File paFile = new File(filePath);
    ZipFile zipFile = new ZipFile(paFile);

    Enumeration<? extends ZipEntry> e = zipFile.entries();
    if (!e.hasMoreElements()) {
      String userMessage =
        "Formato invlido do Pacote de Algoritmos: arquivo est vazio.";
      String detailedMessage = String.format(
        "%sDetalhes: o arquivo zip est vazio.\n" + "Arquivo utilizado: %s.\n"
          + "Identificador do usurio: %s.\n", userMessage, filePath,
        paFileUserLogin);
      Server.logSevereMessage(detailedMessage);
      return null;
    }

    ZipEntry xmlEntry = zipFile.getEntry(ALGO_PACK_METADATA_FILE_NAME);
    if (xmlEntry == null) {
      String clientMessage = String.format(
        "Arquivo de metadados %s no encontrado no Pacote de Algoritmos.",
        ALGO_PACK_METADATA_FILE_NAME);
      Server.logSevereMessage(clientMessage);
      throw new ServiceFailureException(clientMessage);
    }

    System.out.println("Lendo o arquivo xml de metadados do PA (zip): "
      + xmlEntry.getName());

    File file = new File(getAndCheckTempPackRootDir(importDataToken)
      .getCanonicalPath(), xmlEntry.getName());
    AlgorithmsPack algorithmsPack = null;

    if (!file.exists()) {
      file.createNewFile();
      copyFile(zipFile.getInputStream(xmlEntry), new FileOutputStream(file));
      algorithmsPack = readXMLAlgoPackageFile(file);
      importAlgoPackDataInfo.setAlgoPackage(algorithmsPack);
    }
    //Se no fechar o zip, o arquivo no  removido ao final da importao
    zipFile.close();
    return algorithmsPack;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String validateAlgorithmsPack(String importDataToken)
    throws RemoteException {
    if (importDataToken == null) {
      throw new IllegalArgumentException(getParameterNullMessage(
        importDataToken));
    }
    String errorMessage = INVALID_ALGORITHMS_PACK_MSG;

    ImportAlgorithmsPackDataInfo importAlgoPackDataInfo = algoPackTokenMap.get(
      importDataToken);

    if (importAlgoPackDataInfo == null) {
      throw new ServiceFailureException(getMessageFormatted(
        "AlgoService.error.algorithm_pack.invalid_token", importDataToken));
    }
    checkUserImportAlgorithmsPackPermission(importAlgoPackDataInfo);

    AlgorithmsPack algorithmsPack = importAlgoPackDataInfo.getAlgoPackage();
    try {
      if (algorithmsPack == null) {
        algorithmsPack = readAlgorithmsPackMetadata(importAlgoPackDataInfo);
      }
    }
    catch (IOException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }

    //Identifica os diretrios de algoritmos inexistentes no PA
    List<AlgorithmInfo> paAlgorithms = algorithmsPack.getAlgorithms();
    List<AlgorithmInfo> validsAlgorithms = new Vector<AlgorithmInfo>(
      paAlgorithms);
    Iterator<AlgorithmInfo> validsAlgorithmsIterator = validsAlgorithms
      .iterator();
    try {
      //Valida os diretrios dos algoritmos
      errorMessage += validateAlgoPackAlgorithmsDirs(validsAlgorithmsIterator,
        importDataToken);

      //Valida os diretrios das verses dos algoritmos com diretrios vlidos
      validsAlgorithmsIterator = validsAlgorithms.iterator();
      errorMessage += validateAlgoPackVersionsDirs(validsAlgorithmsIterator,
        importDataToken);

      //Valida a estrutura de diretrios de cada verso com diretrio vlido
      validsAlgorithmsIterator = validsAlgorithms.iterator();
      try {
        errorMessage += validateAlgoPackVersionsStructure(
          validsAlgorithmsIterator, importDataToken);
      }
      catch (ServerException e) {
        throw new ServiceFailureException(e.getMessage(), e);
      }

    }
    catch (IOException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }

    //Verifica se algum algoritmo do PA no est vlido
    boolean isValid = paAlgorithms.size() == validsAlgorithms.size();
    String resultMessage = (isValid) ? VALID_ALGORITHMS_PACK_MSG : errorMessage;
    Server.logSevereMessage(resultMessage);
    return resultMessage;
  }

  /**
   * Valida se a estrutura de verses de todos os algoritmos existentes no PA
   * est de acordo com o padro definido para o PA. Atualmente esse padro
   * define que deve haver um diretrio com binrios, um de documentao e um de
   * configurador.
   *
   * @param validsAlgorithmsIterator iterador correspondente aos algoritmos do
   *        pacote de algoritmos. Esse iterador vai removendo os elementos que
   *        no esto vlidos.
   * @param importDataToken identificador nico do dado (PA)
   * @return retorna uma mensagem correspondente a validao. Se a estrutura de
   *         verses de todos os algoritmos forem vlidos retorna uma string
   *         vazia, caso contrrio, retorna uma string contendo os problemas de
   *         validao encontrados.
   * @throws IOException
   * @throws RemoteException
   * @throws ServerException
   */
  private String validateAlgoPackVersionsStructure(
    Iterator<AlgorithmInfo> validsAlgorithmsIterator, String importDataToken)
    throws RemoteException, IOException, ServerException {
    String algoMessage = "";

    String paTokenDirPath = getAndCheckTempPackRootDir(importDataToken)
      .getCanonicalPath();
    String paFilePath = paTokenDirPath + File.separator + ALGO_PACK_FILE_NAME;

    File paFile = new File(paFilePath);
    Unzip unzipFile = new Unzip(paFile);

    while (validsAlgorithmsIterator.hasNext()) {
      AlgorithmInfo algoInfo = validsAlgorithmsIterator.next();
      String algoId = algoInfo.getId();
      Object[] versionIds = algoInfo.getVersionIds();

      String message = "";
      for (Object object : versionIds) {
        AlgorithmVersionId versionId = (AlgorithmVersionId) object;
        int major = versionId.getMajor();
        int minor = versionId.getMinor();
        int patch = versionId.getPatch();
        String versionDir = AlgorithmVersionInfo.getDirectoryFor(major, minor,
          patch);
        String algoDirPath = algoId + "/";
        String paVersionPath = algoDirPath + versionDir + "/";

        //Diretrio do PA onde todo o arquivo ser descompactado
        File paTokenDirectory = new File(paTokenDirPath);
        ImportAlgorithmsPackDataInfo importAlgoPackDataInfo = algoPackTokenMap
          .get(importDataToken);

        if (!importAlgoPackDataInfo.isDescompressed()) {
          unzipFile.decompress(paTokenDirectory);
          importAlgoPackDataInfo.setDescompressed(true);
        }

        //Diretrio da verso, onde o PA foi descompactado
        File versionDirectory = new File(paTokenDirectory, paVersionPath);

        message += validateAlgoPackDocumentationDirectory(versionDirectory,
          versionId);
        message += validateAlgoPackConfigurationDirectory(versionDirectory,
          versionId);
        message += validateAlgoPackBinaryDirectory(versionDirectory, versionId);

      }
      //Finalizou o algoritmo
      if (!message.isEmpty()) {
        algoMessage += (message.isEmpty()) ? ""
          : String.format("\n>> Algoritmo com id <%s>:\n", algoId) + message;
        validsAlgorithmsIterator.remove();
      }
    }
    String resultMessage = (algoMessage.isEmpty()) ? ""
      : "\n*** Verses com estrutura (binrios, docs e "
        + "configurador) invlida no PA:\n" + algoMessage;
    return resultMessage;
  }

  /**
   * Valida o diretrio de documentao existente no diretrio de verso
   * especificado.
   *
   * @param versionDirectory diretrio da verso do algoritmo
   * @param versionId identificador da verso do algoritmo
   * @return retorna uma mensagem correspondente  validao. Se estiver vazia,
   *         o diretrio de documentao est ok, caso contrrio, uma mesnagem
   *         ser retornada informando o problema a ser tratado.
   */
  private String validateAlgoPackDocumentationDirectory(File versionDirectory,
    AlgorithmVersionId versionId) {
    String message = "";
    String identation = "   ";

    File docDirectory = new File(versionDirectory,
      AlgorithmVersionInfo.DOCUMENTATION_DIR + "/");
    if (!docDirectory.exists() || !docDirectory.isDirectory()) {
      message += String.format(
        "%sVerso %s: Diretrio de documentao <%s> ausente.\n", identation,
        versionId, AlgorithmVersionInfo.DOCUMENTATION_DIR);
    }
    return message;
  }

  /**
   * Valida o diretrio de configurao existente no diretrio de verso
   * especificado.
   *
   * @param versionDirectory diretrio da verso do algoritmo
   * @param versionId identificador da verso do algoritmo
   * @return retorna uma mensagem correspondente  validao. Se estiver vazia,
   *         o diretrio de configurao est ok, caso contrrio, uma mesnagem
   *         ser retornada informando o problema a ser tratado.
   */
  private String validateAlgoPackConfigurationDirectory(File versionDirectory,
    AlgorithmVersionId versionId) {
    String message = "";
    String identation = "   ";

    File configDirectory = new File(versionDirectory,
      AlgorithmVersionInfo.CONFIGURATOR_DIR + "/");
    if (!configDirectory.exists() || !configDirectory.isDirectory()) {
      message += String.format(
        "%sVerso %s: Diretrio de configurao <%s> ausente.\n", identation,
        versionId, AlgorithmVersionInfo.CONFIGURATOR_DIR);
    }
    return message;
  }

  /**
   * Valida o diretrio de binrios existente no diretrio de verso
   * especificado.
   *
   * @param versionDirectory diretrio da verso do algoritmo
   * @param versionId identificador da verso do algoritmo
   * @return retorna uma mensagem correspondente  validao. Se estiver vazia,
   *         o diretrio de binrios est ok, caso contrrio, uma mesnagem ser
   *         retornada informando o problema a ser tratado.
   */
  private String validateAlgoPackBinaryDirectory(File versionDirectory,
    AlgorithmVersionId versionId) {
    String message = "";
    String identation = "   ";

    File binaryDirectory = new File(versionDirectory,
      AlgorithmVersionInfo.BIN_DIR + "/");
    if (!binaryDirectory.exists() || !binaryDirectory.isDirectory()) {
      message += String.format(
        "%sVerso %s: Diretrio de binrios <%s> ausente.\n", identation,
        versionId, AlgorithmVersionInfo.BIN_DIR);
    }
    File[] platformDirectories = binaryDirectory.listFiles();
    if (platformDirectories != null) {
      for (File platformDirectory : platformDirectories) {
        if (!platformDirectory.isDirectory()) {
          message += String.format(
            "%sVerso %s: A plataforma <%s> deveria ser um diretrio.\n",
            identation, versionId, platformDirectory.getName());
        }
      }
    }
    return message;
  }

  /**
   * Valida se existe no PA os diretrios correspondentes aos identificadores de
   * algoritmos lidos do arquivo xml de metadados do PA.
   *
   * @param validsAlgorithmsIterator iterador correspondente aos algoritmos do
   *        pacote de algoritmos. Esse iterador vai removendo os elementos que
   *        no esto vlidos.
   * @param importDataToken identificador nico do dado (PA)
   * @return retorna uma mensagem correspondente a validao. Se todos os
   *         algoritmos forem vlidos retorna uma string vazia, caso contrrio,
   *         retorna uma string contendo os problemas de validao encontrados.
   * @throws IOException
   * @throws RemoteException
   */
  private String validateAlgoPackAlgorithmsDirs(
    Iterator<AlgorithmInfo> validsAlgorithmsIterator, String importDataToken)
    throws RemoteException, IOException {
    String message = "";
    while (validsAlgorithmsIterator.hasNext()) {
      AlgorithmInfo algoInfo = validsAlgorithmsIterator.next();
      String algoId = algoInfo.getId();
      String algoDirPath = algoId + "/";
      if (!validateAlgoPackDir(algoDirPath, importDataToken)) {
        message += String.format(">> Algoritmo com id: %s\n", algoId);
        validsAlgorithmsIterator.remove();
      }
    }
    String resultMessage = (message.isEmpty()) ? ""
      : "\n*** No h diretrios correspondentes aos "
        + "identificadores dos algoritmos especificados no metadados do PA:\n"
        + message;
    return resultMessage;
  }

  /**
   * Valida se os diretrios das verses dos algoritmos existentes no PA
   * correspondem aos identificadores de verses lidos do arquivo xml de
   * metadados do PA.
   *
   * @param validsAlgorithmsIterator iterador correspondente aos algoritmos do
   *        pacote de algoritmos. Esse iterador vai removendo os elementos que
   *        no esto vlidos.
   * @param importDataToken identificador nico do dado (PA)
   * @return retorna uma mensagem correspondente a validao. Se todos os
   *         algoritmos forem vlidos retorna uma string vazia, caso contrrio,
   *         retorna uma string contendo os problemas de validao encontrados.
   * @throws IOException
   * @throws RemoteException
   */
  private String validateAlgoPackVersionsDirs(
    Iterator<AlgorithmInfo> validsAlgorithmsIterator, String importDataToken)
    throws RemoteException, IOException {
    String algoMessage = "";
    String identation = "   ";

    while (validsAlgorithmsIterator.hasNext()) {
      AlgorithmInfo algoInfo = validsAlgorithmsIterator.next();
      String algoId = algoInfo.getId();
      Object[] versionIds = algoInfo.getVersionIds();

      boolean validsVersionsDirs = true;
      String message = "";
      for (Object object : versionIds) {
        AlgorithmVersionId versionId = (AlgorithmVersionId) object;
        int major = versionId.getMajor();
        int minor = versionId.getMinor();
        int patch = versionId.getPatch();
        String versionDir = AlgorithmVersionInfo.getDirectoryFor(major, minor,
          patch);
        String algoDirPath = algoId + "/";
        String paVersionPath = algoDirPath + versionDir + "/";
        if (!validateAlgoPackDir(paVersionPath, importDataToken)) {
          validsVersionsDirs = false;
          message += String.format("%sDiretrio de verso no encontrado: %s\n",
            identation, versionDir);
        }
      }
      if (!validsVersionsDirs) {
        algoMessage += (message.isEmpty()) ? ""
          : String.format("\n>> Algoritmo com id <%s>:\n", algoId) + message;
        validsAlgorithmsIterator.remove();
      }
    }
    String resultMessage = (algoMessage.isEmpty()) ? ""
      : "\n*** No h diretrios correspondentes aos ids das "
        + "verses dos algoritmos especificados no metadados do PA:\n"
        + algoMessage;
    return resultMessage;
  }

  /**
   * Valida se existe um diretrio no PA que corresponde ao caminho
   * especificado.
   *
   * @param dirPath caminho do diretrio relativo a raiz do PA
   * @param importDataToken identificador nico do dado (PA)
   * @return retorna true se o diretrio existe no PA, caso contrrio, retorna
   *         false
   * @throws RemoteException
   * @throws IOException
   */
  private boolean validateAlgoPackDir(String dirPath, String importDataToken)
    throws RemoteException, IOException {
    ZipEntry algoEntry = getAlgoPathEntry(dirPath, importDataToken);
    if (algoEntry == null) {
      return false;
    }
    if (!algoEntry.isDirectory()) {
      return false;
    }
    return true;
  }

  /**
   * Obtm a entrada para o arquivo compactado que representa o PA.
   *
   * @param dirPath caminho do diretrio relativo a raiz do PA
   * @param importDataToken identificador nico do dado (PA)
   * @return retorna um objeto para o arquivo compactado que representa o PA
   * @throws RemoteException erro de rmi
   * @throws IOException erro de IO
   */
  private ZipEntry getAlgoPathEntry(String dirPath, String importDataToken)
    throws RemoteException, IOException {
    String filePath = getAndCheckTempPackRootDir(importDataToken)
      .getCanonicalPath() + File.separator + ALGO_PACK_FILE_NAME;

    File paFile = new File(filePath);
    ZipFile zipFile = new ZipFile(paFile);

    Enumeration<? extends ZipEntry> e = zipFile.entries();
    if (!e.hasMoreElements()) {
      return null;
    }

    ZipEntry algoEntry = zipFile.getEntry(dirPath);
    if (algoEntry == null) {
      return null;
    }
    return algoEntry;
  }

  /**
   * Envia um evento para todos os clientes ativos.
   *
   * @param event
   */
  private void sendEvent(AlgoEvent event) {
    MessageService.getInstance().sendToAll(new Message(event));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public RemoteFileChannelInfo openExecFileChannel(Object algoId,
    Object versionId, String platformName, String fileName, boolean readOnly)
    throws RemoteException {
    if (!readOnly) {
      this.checkAlgorithmAdminPermission(algoId);
    }
    if (!execFileExists(algoId, versionId, platformName, fileName)) {
      Algorithm algo = registeredAlgorithms.get(algoId);
      String userMessage = getMessageFormatted(
        "AlgoService.error.algo.exec_file_not_exists", fileName, algo.getInfo()
          .getId(), versionId, platformName);
      Server.logSevereMessage(userMessage);
      throw new ServiceFailureException(userMessage);
    }
    String platformPath = getPlatformPath(algoId, versionId, platformName);
    String targetPath = platformPath + File.separator + fileName;
    Map<String, Object> notifyArgs = new HashMap<String, Object>();
    notifyArgs.put(ALGO_FILE_TYPE_KEY, AlgoFileType.EXECUTABLE);
    notifyArgs.put(USER_ID_KEY, Service.getUser().getId());
    notifyArgs.put(ALGO_ID_KEY, algoId);
    notifyArgs.put(VERSION_ID_KEY, versionId);
    notifyArgs.put(PLATFORM_NAME_KEY, platformName);
    notifyArgs.put(FILE_NAME_KEY, fileName);
    return createChannel(targetPath, notifyArgs, readOnly ? FileChannelOp.READ
      : FileChannelOp.WRITE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public RemoteFileChannelInfo openDocFileChannel(Object algoId,
    Object versionId, String fileName, boolean readOnly)
    throws RemoteException {
    if (!readOnly) {
      this.checkAlgorithmAdminPermission(algoId);
    }
    if (!docFileExists(algoId, versionId, fileName)) {
      Algorithm algo = registeredAlgorithms.get(algoId);
      String userMessage = getMessageFormatted(
        "AlgoService.error.algo.doc_file_not_exists", fileName, algo.getInfo()
          .getId(), versionId);
      Server.logSevereMessage(userMessage);
      throw new ServiceFailureException(userMessage);
    }
    String docDirPath = getDocumentationDirPath(algoId, versionId);
    String sourcePath = docDirPath + File.separator + fileName;
    return createChannel(sourcePath, null, readOnly ? FileChannelOp.READ
      : FileChannelOp.WRITE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public RemoteFileChannelInfo openConfFileChannel(Object algoId,
    Object versionId, String fileName, boolean readOnly)
    throws RemoteException {
    if (!readOnly) {
      this.checkAlgorithmAdminPermission(algoId);
    }
    if (!configFileExists(algoId, versionId, fileName)) {
      Algorithm algo = registeredAlgorithms.get(algoId);
      String userMessage = getMessageFormatted(
        "AlgoService.error.algo.config_file_not_exists", fileName, algo
          .getInfo().getId(), versionId);
      Server.logSevereMessage(userMessage);
      throw new ServiceFailureException(userMessage);
    }
    String configDirPath = getConfiguratorDirPath(algoId, versionId);
    String sourcePath = configDirPath + File.separator + fileName;
    return createChannel(sourcePath, null, readOnly ? FileChannelOp.READ
      : FileChannelOp.WRITE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public RemoteFileChannelInfo openReleaseNotesFileChannel(Object algoId,
    Object versionId, String fileName, boolean readOnly)
    throws RemoteException {
    if (!readOnly) {
      this.checkAlgorithmAdminPermission(algoId);
    }
    if (!releaseNotesFileExists(algoId, versionId, fileName)) {
      Algorithm algo = registeredAlgorithms.get(algoId);
      String userMessage = getMessageFormatted(
        "AlgoService.error.algo.release_notes_file_not_exists", fileName, algo
          .getInfo().getId(), versionId);
      Server.logSevereMessage(userMessage);
      throw new ServiceFailureException(userMessage);
    }
    String configDirPath = getReleaseNotesDirPath(algoId, versionId);
    String sourcePath = configDirPath + File.separator + fileName;
    return createChannel(sourcePath, null, readOnly ? FileChannelOp.READ
      : FileChannelOp.WRITE);
  }

}
