package csbase.server.services.restservice.websocket.notificationcenter;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import org.glassfish.grizzly.http.HttpRequestPacket;
import org.glassfish.grizzly.websockets.DataFrame;
import org.glassfish.grizzly.websockets.ProtocolHandler;
import org.glassfish.grizzly.websockets.WebSocket;
import org.glassfish.grizzly.websockets.WebSocketListener;
import org.json.JSONArray;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.EvictingQueue;

import csbase.logic.CommandInfo;
import csbase.logic.CommandNotification;
import csbase.logic.ProjectEvent;
import csbase.server.Server;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.restservice.websocket.CSBaseWebSocket;
import csbase.server.services.restservice.websocket.CSBaseWebSocketApplication;
import csbase.server.services.restservice.websocket.Message;
import csbase.server.services.restservice.websocket.Project;
import csbase.server.services.restservice.websocket.User;
import csbase.server.services.restservice.websocket.utils.PersistentMap;
import csbase.server.services.restservice.websocket.utils.WebSocketUtils;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.MessageBroker;
import csbase.util.messages.filters.BodyTypeFilter;

import ibase.common.ServiceUtil;

/**
 * WebSocket de notificaes para o cliente web.
 * 
 * Se encarrega de enviar as notificaes de final de comando e de atualizaes da rvore de projeto. 
 *
 * @author Tecgraf/PUC-Rio
 */
public class CSBaseNotificationCenter extends CSBaseWebSocketApplication {

	/** Notification persistence file name */
	private static final String NOTIFICATIONS_FILE = "wsnotifications.dat";

	/** Notification history max size */
	private static final int NOTIFICATION_HISTORY_MAX_SIZE = 20;

	/** Notifications persistence */
	private PersistentMap<String, EvictingQueue<String>> notifications = new PersistentMap<>(
			WebSocketUtils.generatePath(NOTIFICATIONS_FILE));

	/** Controle dos consumerIds para cada conexo */ 
  protected ConcurrentHashMap<String, Serializable> consumerIds = new ConcurrentHashMap<>();
	
  /**
   * Construtor
   *
   */
	public CSBaseNotificationCenter() {
		super();

		// CommandNotification listener
		setOnJobTerminateListener();		
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onConnect(WebSocket socket) {
		super.onConnect(socket);

		if (socket instanceof CSBaseWebSocket) {
			CSBaseWebSocket ws = ((CSBaseWebSocket) socket);

			// Initialize notifications history
			notifications.putIfAbsent(ws.getUser().getLogin(), EvictingQueue.create(NOTIFICATION_HISTORY_MAX_SIZE));

			// Upon connection, send the notification history to the current
			// user
			ws.send(createNotificationHistoryMessage(ws.getUser().getLogin()).toString());
		}
	}

	/**
	 * Configura o listener de mensagens de trmino de comando para serem repassadas aos usurios interessados.
	 */
	private void setOnJobTerminateListener() {

		// Initialize Jackson ObjectMapper (POJO to JSON conversion)
		ObjectMapper mapper = new ObjectMapper();

		// POJO to JSON with non null values only
		mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);

		// Set CommandNotification listener
		MessageService.getInstance().setServerMessageListener((IMessageListener) messages -> {
			for (csbase.util.messages.Message message : messages) {
				try {
					Server.logInfoMessage("CommandNotification received in CSBaseNotificationCenter\n\t"
							+ message.getBody().getClass().getSimpleName() + ": " + message.getBody());
					if (message.getBody() instanceof CommandNotification) {

						// Cast body to CommandNotification
						CommandNotification commandNotification = (CommandNotification) message.getBody();

						// POJO to JSON conversion
						JSONObject content = new JSONObject(mapper.writeValueAsString(commandNotification));

						// Hack to get project name of incoming notification
						ProjectService ps = ProjectService.getInstance();
						String project = ps.getProjectName(commandNotification.getProjectId());
						content.put("project", project);

						// Get CommandInfo object
						CommandInfo commandInfo = CommandPersistenceService.getInstance().getCommandInfo(
								commandNotification.getProjectId(), commandNotification.getCommandId().toString());
						String user = commandInfo.getUserId().toString();
						content.put("user", user);

						// Encapsulate content into a Message object
						Message commandTerminate = createCommandTerminateMessage(content);

						// Send to user that "owns" the notification
						this.connections.forEach((targetUser, targetUserWebSockets) -> {
							if (targetUser.equals(user)) {
								notifications.compute(targetUser, (savedUser, savedUserNotifications) -> {
									savedUserNotifications.add(content.toString());
									return savedUserNotifications;
								});
								Server.logInfoMessage("Sending notification for user " + user + " "
										+ targetUserWebSockets.size() + " connections");
								broadcaster.broadcast(targetUserWebSockets, commandTerminate.toString());
							}
						});

					}
				} catch (Exception e) {
					e.printStackTrace();
					Server.logSevereMessage(
							"Error processing Message " + message + "\n\tMessage body: " + message.getBody(), e);
				}
			}
		} , new BodyTypeFilter(CommandNotification.class));
	}

	/**
	 * Cria a mensagem de trmino de comando a ser enviada ao usurio.
	 * 
	 * @param content o contedo da mensagem no formato JSON.
	 * 
	 * @return a mensagem.
	 * 
	 * @throws JsonProcessingException erro ao processar o JSON.
	 */
	private static Message createCommandTerminateMessage(JSONObject content) throws JsonProcessingException {
		Message commandTerminate = new Message();
		commandTerminate.setType(Message.TYPE_COMMAND_TERMINATE);
		commandTerminate.setContent(content);
		return commandTerminate;
	}

	/**
	 * Cria a mensagem de histrico de notificaes para ser enviada ao usurio.
	 *  
	 * @param user o usurio destino.
	 * 
	 * @return a mensagem.
	 */
	private Message createNotificationHistoryMessage(String user) {
		JSONArray jsonArray = new JSONArray();
		notifications.get(user).forEach(jsonArray::put);
		Message history = new Message();
		history.setType(Message.TYPE_NOTIFICATION_HISTORY);
		history.setContent(jsonArray);
		return history;
	}
	
	/**
	 * Processa a mensagem vinda do cliente web. Indica abertura ou fechamento do projeto.
	 */
	@Override
	public void onMessage(WebSocket websocket, String json) {

		try {
			// Cast to CSBaseWebSocket
			CSBaseWebSocket ws = (CSBaseWebSocket) websocket;
			
			// Parse message JSON
			Message incomingMessage = new Message(json);

			incomingMessage.setUser(ws.getUser());
			
			// Project change message
			if (incomingMessage.isProjectChange().booleanValue()) {

				// Instantiate project if message content is not null
				Object content = incomingMessage.getContent();
				Project project = content != null ? new Project((JSONObject) content) : null;

				// Update connection
				ws.setProject(project);
				connections.compute(ws.getUser().getLogin(), (user, userWebSockets) -> {
					userWebSockets.removeIf(userWebSocket -> userWebSocket.getId().equals(ws.getId()));
					userWebSockets.add(ws);
					return userWebSockets;
				});
			} 
		} catch (Exception e) {
			e.printStackTrace();
			Server.logSevereMessage(
					"Error processing message " + json + " sent by " + ((CSBaseWebSocket) websocket).getUser(), e);
		} 
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * Sobrecarga necessria para criar os listeners de projeto na criao do socket.
	 */
	@Override
	public WebSocket createSocket(ProtocolHandler handler, HttpRequestPacket requestPacket,
	  WebSocketListener... listeners) {
	  CSBaseWebSocket ws = (CSBaseWebSocket) super.createSocket(handler, requestPacket, listeners);
	  
	  if(consumerIds.get(ws.getUser().getLogin()) == null) {
	    MessageBroker broker = Server.getInstance().getMessageBroker();
	    Serializable consumerId = broker.createConsumerId();
	    broker.setMessageListener(ws.getUser().getLogin(), consumerId, messages -> {

	      User user = ws.getUser();

	      // Initialize Jackson ObjectMapper (POJO to JSON conversion)
	      ObjectMapper mapper = new ObjectMapper();

	      // POJO to JSON with non null values only
	      mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
	      
	      for (csbase.util.messages.Message message : messages) {
	        try {
	          Server.logInfoMessage("ProjectEvent received in CSBaseNotificationCenter\n\t"
	              + message.getBody().getClass().getSimpleName() + ": " + message.getBody());
	          if (message.getBody() instanceof ProjectEvent) {

	            // Cast body to CommandNotification
	            final ProjectEvent projectEvent = (ProjectEvent) message.getBody();
	            
	            switch(projectEvent.event) {
	              /** O evento de fechamento de um projeto */
	              case ProjectEvent.PROJECT_CLOSED:
	              /** O evento de modificar informaes do projeto. */
	              case ProjectEvent.INFO_MODIFIED:
	              /** O evento de remoo do projeto. */
	              case ProjectEvent.PROJECT_DELETED:
	                /** O evento de alterao do estado de um arquivo. */
	              case ProjectEvent.FILE_STATE_CHANGED:
	                continue;
	              /** O evento de incluir um arquivo na rvore do projeto. */
	              case ProjectEvent.NEW_FILE:
	              /** O evento de remover um arquivo da rvore do projeto. */
	              case ProjectEvent.FILE_DELETED:
	              /** O evento de renomear um arquivo. */
	              case ProjectEvent.NEW_FILE_NAME:
	              /**
	               * O evento de mover um arquivo para um outro diretrio da rvore do projeto.
	               */
	              case ProjectEvent.FILE_MOVED:
	              /** O evento de atualizao da rvore do projeto. */
	              case ProjectEvent.TREE_CHANGED:
	              /** O evento de remover vrios arquivos da rvore do projeto. */
	              case ProjectEvent.FILES_DELETED:
	              /**
	               * O evento de incluir vrios arquivos simultaneamente na rvore do projeto.
	               */
	              case ProjectEvent.NEW_FILES:
	              /** O evento de atualizao de um diretrio do projeto. */
	              case ProjectEvent.DIR_REFRESHED:
	            }

	            
	            List<CSBaseWebSocket> sendList = new ArrayList<>();
	            
	            connections.get(user.getLogin()).forEach((CSBaseWebSocket socket) -> {
	              if(socket.getProject() != null && projectEvent.projectId.equals(ServiceUtil.decodeFromBase64(socket.getProject().getId()))) {
	                sendList.add(socket);
	              }
	            });
	            
	            if(sendList.size() > 0) {
	              Message projectChange = createProjectChangeMessage(user, sendList.get(0).getProject());
	              broadcaster.broadcast(sendList, projectChange.toString());
	            }
	          }
	        } catch (Exception e) {
	          e.printStackTrace();
	          Server.logSevereMessage(
	              "Error processing Message " + message + "\n\tMessage body: " + message.getBody(), e);
	        }
	      }  
	    }, new BodyTypeFilter(ProjectEvent.class));
	    consumerIds.put(ws.getUser().getLogin(), consumerId);
	  }
	  
    return ws;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onClose(WebSocket socket, DataFrame frame) {
	  super.onClose(socket, frame);
	  
	  if(socket instanceof CSBaseWebSocket) {
	    CSBaseWebSocket ws = (CSBaseWebSocket) socket;
	    List<CSBaseWebSocket> conn = connections.get(ws.getUser().getLogin());
	    if((conn == null || conn.isEmpty()) && consumerIds.get(ws.getUser().getLogin()) != null) {
	      Server.getInstance().getMessageBroker().removeMessageListener(ws.getUser().getLogin(), consumerIds.get(ws.getUser().getLogin()));
	      consumerIds.remove(ws.getUser().getLogin());
	    }
	  }
	  
	}

	/**
	 * Cria uma mensagem de alterao de projeto para ser enviada ao cliente.
	 * 
	 * @param user o usurio destino desta mensagem.
	 * @param project o projeto que foi modificado.
	 * 
	 * @return a mensagem criada.
	 */
	private static Message createProjectChangeMessage(User user, Project project) {
		Message projectChange = new Message();
		projectChange.setType(Message.TYPE_PROJECT_CHANGE);
		projectChange.setUser(user);
		projectChange.setContent(project);
		return projectChange;
	}
}
