/*
 * $Id$
 */

package csbase.server.services.algorithmservice;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import csbase.exception.InfoException;
import csbase.exception.OperationFailureException;
import csbase.exception.ServiceFailureException;
import csbase.logic.FileInfo;
import csbase.logic.IPathFactory;
import csbase.logic.Utilities;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmOutline;
import csbase.logic.algorithms.AlgorithmProperty;
import csbase.logic.algorithms.AlgorithmProperty.PropertyType;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.server.FileSystem;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.util.FileSystemUtils;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Classe que representa um algoritmo no servidor.
 *
 * @author Tecgraf/PUC-Rio
 */
class Algorithm {
  /**
   * Nome do arquivo que contm o identificador do algoritmo.
   */
  private static final String NAME_FILE_NAME = "name.txt";

  /**
   * Nome da propriedade que determina a ordem das propriedades dos algoritmos e
   * verses.
   */
  private static final String SORT_ORDER_PROPERTY = "ordem";

  /**
   * Informaes do algoritmo que podem ser serializadas para o cliente.
   */
  private final ModifiableAlgorithmInfo info;

  /**
   * Lista de propriedades dos algoritmos. A lista mantm a ordem de
   * apresentao de tais propriedades. A propriedades mapeiam a chave no label
   * (String de apresentao)
   */
  private static List<AlgorithmProperty> algorithmProperties;

  /**
   * Lista de propriedades das verses. A lista mantm a ordem de apresentao
   * de tais propriedades. A propriedades mapeiam a chave no label (String de
   * apresentao)
   */
  private static List<AlgorithmProperty> versionProperties;

  /**
   * Obtm as informaes de um algoritmo.
   *
   * @return um objeto do tipo <code>AlgorithmInfo</code> (serializvel e
   *         imutvel) com as informaes do algoritmo.
   */
  public AlgorithmInfo getInfo() {
    return info.toAlgorithmInfo();
  }

  /**
   * Obtm o resumo de um algoritmo.
   *
   * @return um objeto do tipo <code>AlgorithmOutline</code> com o resumo
   */
  public AlgorithmOutline getOutline() {
    return info.getOutline();
  }

  /**
   * Obtm as informaes de um algoritmo j presente (completo) no repositrio
   * e constri sua representao no servidor.
   *
   * @param dir diretrio do algoritmo no repositrio
   * @return a representao do algoritmo
   * @throws ServerException se no for possvel ler as informaes sobre o
   *         algoritmo.
   */
  public static Algorithm includeAlgorithm(final String dir)
    throws ServerException {
    /* Obtem o nome e outros dados do algoritmo */
    final String algoDirPath = getAlgorithmRepositoryPath() + File.separator
      + dir;
    final String name = readName(algoDirPath);
    final Hashtable<String, String> propertyValues =
      loadAlgorithmPropertyValues(algoDirPath);

    /* Cria a representao do algoritmo */
    final Algorithm algo = new Algorithm(name, dir, propertyValues);

    /* Obtm as verses do algoritmo */
    algo.readVersions();
    return algo;
  }

  /**
   * Obtm o nome de um algoritmo ou verso a partir do arquivo correspondente.
   *
   * @param dirPath path para o diretrio do algoritmo ou verso no repositrio
   * @return o nome do algoritmo ou verso.
   * @throws ServerException em caso de erro.
   */
  private static String readName(final String dirPath) throws ServerException {
    BufferedReader br = null;
    try {
      br = new BufferedReader(new FileReader((dirPath + File.separator
        + NAME_FILE_NAME)));
      String name = br.readLine();
      if (name == null || name.trim().isEmpty()) {
        throw new ServerException("Falha na obteno de nome de "
          + "algoritmo/verso no arquivo " + (dirPath + File.separator
            + NAME_FILE_NAME) + ": Nome vazio.");
      }
      return name;
    }
    catch (final IOException e) {
      throw new ServerException("Falha na obteno de nome de "
        + "algoritmo/verso no arquivo " + (dirPath + File.separator
          + NAME_FILE_NAME), e);
    }
    finally {
      if (br != null) {
        try {
          br.close();
        }
        catch (final IOException ieo) {
          ieo.printStackTrace();
        }
      }
    }
  }

  /**
   * Atribui um novo nome ao algoritmo.
   *
   * @param newName o novo nome do algoritmo
   * @throws ServerException em caso de erro.
   */
  protected void rename(final String newName) throws ServerException {
    final String algoDirPath = getAlgorithmRepositoryPath() + File.separator
      + getInfo().getDirectory();
    final String aNameFile = algoDirPath + File.separator + NAME_FILE_NAME;
    writeSingleLineFile(aNameFile, newName);
    info.rename(newName);
  }

  /**
   * Altera o arquivo com o nome de um algoritmo.
   *
   * @param dirPath path para o diretrio do algoritmo no repositrio
   * @param name o nome do algoritmo
   * @throws ServerException em caso de erro.
   */
  private static void writeName(final String dirPath, final String name)
    throws ServerException {
    final String aNameFile = dirPath + File.separator + NAME_FILE_NAME;
    writeSingleLineFile(aNameFile, name);
  }

  /**
   * Obtm as informaes das verses do algoritmo disponveis no repositrio.
   *
   * @throws ServerException em caso de erro.
   */
  private void readVersions() throws ServerException {
    final String versionsDirectoryPath = getAlgorithmRepositoryPath()
      + File.separator + this.info.getDirectory() + File.separator
      + AlgorithmInfo.VERSIONS_DIR;
    final File versionsDirectoryFile = new File(versionsDirectoryPath);
    if (!versionsDirectoryFile.exists()) {
      final String message = MessageFormat.format("O diretrio {0} no existe.",
        new Object[] { versionsDirectoryFile.getAbsolutePath() });
      throw new ServerException(message);
    }
    if (!versionsDirectoryFile.isDirectory()) {
      final String message = MessageFormat.format(
        "O arquivo {0} deveria ser um diretrio.", new Object[] {
            versionsDirectoryFile.getAbsolutePath() });
      throw new ServerException(message);
    }
    final File[] versionDirs = versionsDirectoryFile.listFiles();
    if (versionDirs.length <= 0) {
      final String message = MessageFormat.format(
        "O algoritmo {0} no possui verses.\nDiretrio de verses: {1}.",
        new Object[] { this.info.getName(), versionsDirectoryFile
          .getAbsoluteFile() });
      Server.logWarningMessage(message);
      return;
    }
    for (final File versionDir : versionDirs) {
      try {
        readVersion(versionsDirectoryPath, versionDir);
      }
      catch (final ServerException exception) {
        final String message = MessageFormat.format(
          "Erro ao tentar ler informaes sobre o algoritmo {0} diretrio de "
            + "verso {1}.", new Object[] { this.info, versionDir
              .getAbsolutePath() });
        Server.logSevereMessage(message, exception);
      }
    }
  }

  /**
   * L dados de verso.
   *
   * @param versionsDirectoryPath path de verso.
   * @param versionDirectoryFile arquivo de verso.
   * @throws ServerException em caso de erro.
   */
  void readVersion(final String versionsDirectoryPath,
    final File versionDirectoryFile) throws ServerException {
    if (!versionDirectoryFile.isDirectory()) {
      final String message = MessageFormat.format(
        "O arquivo {0} deveria ser um diretrio.", new Object[] {
            versionDirectoryFile.getAbsolutePath() });
      throw new ServerException(message);
    }
    final String versionDirectoryName = versionDirectoryFile.getName().trim();
    if (!versionDirectoryName.equalsIgnoreCase(
      FileSystemUtils.VERSION_CONTROL_DIR)) {
      final String versionDirectoryPath = versionsDirectoryPath + File.separator
        + versionDirectoryName;
      List<FileInfo> configurators = getConfigurators(versionDirectoryPath);
      ConfiguratorType type = ConfiguratorType.SIMPLE;
      for (FileInfo file : configurators) {
        if (file.getName().equals(AlgorithmVersionInfo.FLOW_CONFIG_FILE)) {
          type = ConfiguratorType.FLOW;
          break;
        }
      }
      List<FileInfo> documentation = getDocumentation(versionDirectoryPath);
      FileInfo releaseNotes = getReleaseNotes(versionDirectoryPath);
      final Hashtable<String, String> versionPropertyValues =
        loadVersionPropertyValues(versionDirectoryPath);
      final Map<String, List<FileInfo>> supportedPlatforms;
      /*
       * Se for fluxo instalado no faz sentido verificar as plataformas
       * suportadas.
       */
      if (type == ConfiguratorType.FLOW) {
        supportedPlatforms = Collections.emptyMap();
      }
      else {
        supportedPlatforms = getSupportedPlatforms(versionDirectoryPath);
      }
      final AlgorithmVersionInfo versionInfo = new AlgorithmVersionInfo(
        getInfo(), versionDirectoryName, supportedPlatforms,
        versionPropertyValues, type);
      versionInfo.setDocumentation(documentation);
      versionInfo.setConfigurators(configurators);
      versionInfo.setReleaseNotes(releaseNotes);
      this.info.includeVersion(versionInfo.getId(), versionInfo);
    }
  }

  /**
   * Consulta a lista de configuradores.
   *
   * @param versionDirectoryPath caminho para o diretrio de verses do
   *        algoritmo.
   * @return a lista
   * @throws ServerException em caso de erro
   */
  private List<FileInfo> getConfigurators(final String versionDirectoryPath)
    throws ServerException {
    final File directory = new File(versionDirectoryPath + File.separator
      + AlgorithmVersionInfo.CONFIGURATOR_DIR);
    if (!directory.exists()) {
      final String message = MessageFormat.format("O diretrio {0} no existe.",
        new Object[] { directory.getAbsolutePath() });
      throw new ServerException(message);
    }
    if (!directory.isDirectory()) {
      final String message = MessageFormat.format(
        "O arquivo {0} deveria ser um diretrio.", new Object[] { directory
          .getAbsolutePath() });
      throw new ServerException(message);
    }

    return FileInfo.createFilesInfo(directory);
  }

  /**
   * Consulta a documentao.
   *
   * @param versionDirectoryPath path
   * @return lista de informaes
   * @throws ServerException em caso de erro
   */
  private List<FileInfo> getDocumentation(final String versionDirectoryPath)
    throws ServerException {
    final File directory = new File(versionDirectoryPath + File.separator
      + AlgorithmVersionInfo.DOCUMENTATION_DIR);
    if (!directory.exists()) {
      final String message = MessageFormat.format("O diretrio {0} no existe.",
        new Object[] { directory.getAbsolutePath() });
      throw new ServerException(message);
    }
    if (!directory.isDirectory()) {
      final String message = MessageFormat.format(
        "O arquivo {0} deveria ser um diretrio.", new Object[] { directory
          .getAbsolutePath() });
      throw new ServerException(message);
    }

    return FileInfo.createFilesInfo(directory);
  }

  /**
   * Obtm o release notes.
   *
   * @param versionDirectoryPath caminho para a verso.
   * @return arquivo de release notes.
   * @throws ServerException em caso de erro
   */
  private FileInfo getReleaseNotes(final String versionDirectoryPath)
    throws ServerException {
    final File directory = new File(versionDirectoryPath + File.separator
      + AlgorithmVersionInfo.RELEASE_NOTES_DIR);
    if (!directory.exists()) {
      return null;
    }
    if (!directory.isDirectory()) {
      final String message = MessageFormat.format(
        "O arquivo {0} deveria ser um diretrio.", new Object[] { directory
          .getAbsolutePath() });
      throw new ServerException(message);
    }

    return getReleaseNotes(FileInfo.createFilesInfo(directory));
  }

  /**
   * Descobre quais as plataformas suportadas por uma verso de algoritmo. Os
   * nomes dos subdiretrios de "BIN_DIR" so os identificadores das plataformas
   * suportadas.
   *
   * @param vDirPath path para o diretrio da verso do algoritmo
   * @return uma Hashtable contendo as identificaes (Strings) das plataformas
   *         suportadas (chave) e os executaveis associados as plataformas
   *         (valor da hash).
   * @throws ServerException Em caso de erro.
   */
  private Map<String, List<FileInfo>> getSupportedPlatforms(
    final String vDirPath) throws ServerException {

    final Map<String, List<FileInfo>> supportedPlatforms =
      new HashMap<String, List<FileInfo>>();
    final String path = vDirPath + File.separator
      + AlgorithmVersionInfo.BIN_DIR;
    final File binDir = new File(path);
    if (!binDir.exists()) {
      final String message = MessageFormat.format(
        "O diretrio - {0} - que armazena as plataformas no existe.",
        new Object[] { binDir.getAbsolutePath() });

      Server.logWarningMessage(message);
      return supportedPlatforms;
    }

    if (!binDir.isDirectory()) {
      throw new ServerException(MessageFormat.format(
        "O arquivo {0} deveria ser um diretrio.", new Object[] { binDir
          .getAbsolutePath() }));
    }
    final File[] platformsDir = binDir.listFiles();
    if (platformsDir.length == 0) {
      final String message = MessageFormat.format(
        "O diretrio - {0} - que armazena as plataformas est vazio.",
        new Object[] { binDir.getAbsolutePath() });
      Server.logWarningMessage(message);
      return supportedPlatforms;
    }

    for (final File platform : platformsDir) {
      if (platform.isDirectory() && FileInfo.isNameValid(platform.getName())) {

        final List<FileInfo> executablesInfo = FileInfo.createFilesInfo(
          platform);
        supportedPlatforms.put(platform.getName(), executablesInfo);
      }
    }

    return supportedPlatforms;
  }

  /**
   * Cria no repositrio a estrutura para um novo algoritmo.
   *
   * @param name nome do novo algoritmo
   * @param propertyValues valores.
   * @return representao do algoritmo
   * @throws ServerException se a estrutura no for criada com sucesso.
   */
  public static Algorithm createAlgorithm(final String name,
    final Hashtable<String, String> propertyValues) throws ServerException {
    return createAlgorithm(null, name, propertyValues);
  }

  /**
   * Cria no repositrio a estrutura para um novo algoritmo.
   *
   * @param algoId identificador do algoritmo
   * @param name nome do novo algoritmo
   * @param propertyValues valores.
   * @return representao do algoritmo
   * @throws ServerException se a estrutura no for criada com sucesso.
   */
  public static Algorithm createAlgorithm(final String algoId,
    final String name, final Hashtable<String, String> propertyValues)
    throws ServerException {
    return createAlgorithmStructure(algoId, name, propertyValues);
  }

  /**
   * Cria no repositrio a estrutura para um novo algoritmo.
   *
   * @param algoId identificador do algoritmo
   * @param name nome do novo algoritmo
   * @param propertyValues valores.
   * @return representao do algoritmo
   * @throws ServerException se a estrutura no for criada com sucesso.
   */
  private static Algorithm createAlgorithmStructure(final String algoId,
    final String name, final Hashtable<String, String> propertyValues)
    throws ServerException {
    String algoDirName = defineAlgoDirName(algoId, name);
    final String algoDirPath = getAlgorithmRepositoryPath() + File.separator
      + algoDirName;
    try {
      createAlgoRootDir(algoDirPath);

      /* cria a estrutura de subdiretrios do algoritmo */
      createAlgoSubDirectories(algoDirPath);

      /* cria os arquivos com o nome do algoritmo e data atualizao */
      writeName(algoDirPath, name);
      final String pathToPropFile = algoDirPath + File.separator
        + AlgorithmInfo.PROPERTY_VALUES_FILE;

      saveOwnerPropertyValue(Service.getUser().getLogin(), propertyValues);
      savePropertyValues(pathToPropFile, propertyValues);
      return new Algorithm(name, algoDirName, propertyValues);
    }
    catch (final ServerException se) {
      try {
        FileSystem.removeFile(new File(algoDirPath), true);
      }
      catch (final Exception e) {
        Server.logSevereMessage("Falha na remoo de arquivo/diretrio \n"
          + "\tExceo: " + e);
      }
      throw se;
    }
  }

  /**
   * Define o nome do diretrio "raiz" de um algoritmo, utilizando como "base" o
   * nome do algoritmo. O nome do diretrio no dever conter caracteres
   * especiais e deve ser nico no repositrio.
   *
   * @param id identificador do algoritmo
   * @param name nome do algoritmo
   * @return uma String com o nome do diretrio do algoritmo
   */
  private static String defineAlgoDirName(String id, String name) {
    String baseName = id;
    if (baseName == null || baseName.isEmpty()) {
      baseName = name;
    }
    /* Transforma o nome do algoritmo num nome de diret<F3>rio v<E1>lido */
    String algoDirName = FileUtils.fixDirectoryName(baseName);

    /* Garante a unicidade do nome no repositrio */
    while (getService().algorithmIdRegistered(algoDirName)) {
      algoDirName = algoDirName + "_";
    }
    return algoDirName;
  }

  /**
   * Cria o diretrio "raiz" de um algoritmo.
   *
   * @param algoDirPath path para o diretrio "raiz" do algoritmo
   * @throws ServerException em caso de erro.
   */
  private static void createAlgoRootDir(final String algoDirPath)
    throws ServerException {
    final File algoDir = new File(algoDirPath);
    if (algoDir.exists()) {
      throw new ServerException("Diretrio " + algoDirPath + " j existe!");
    }
    if (!algoDir.mkdir()) {
      throw new ServerException("Falha na criao do diretrio " + algoDirPath);
    }
  }

  /**
   * Cria a estrutura de subdiretrios para um algoritmo.
   *
   * @param algoDirPath path para o diretrio "raiz" do algoritmo
   * @throws ServerException se no for possvel criar os subdiretrios.
   */
  private static void createAlgoSubDirectories(final String algoDirPath)
    throws ServerException {
    final String verDirPath = algoDirPath + File.separator
      + AlgorithmInfo.VERSIONS_DIR;
    final File verDir = new File(verDirPath);
    if (!verDir.mkdir()) {
      throw new ServerException("Falha na criao do diretrio de verses "
        + verDirPath);
    }
  }

  /**
   * Remove o algoritmo do repositrio.
   *
   * @throws ServerException em caso de erro.
   */
  public void remove() throws ServerException {
    /* remover os diretorios */
    final String algoDirPath = getAlgorithmRepositoryPath() + File.separator
      + info.getDirectory();
    final File algoFile = new File(algoDirPath);
    try {
      FileSystem.removeFile(algoFile, true);
    }
    catch (final Exception e) {
      throw new ServerException(e.getMessage(), e);
    }
  }

  /**
   * Obtm o caminho para o repositrio de algoritmos.
   *
   * @return o caminho para o repositrio de algoritmos.
   */
  private static String getAlgorithmRepositoryPath() {
    return getService().getAlgorithmRepositoryPath();
  }

  /**
   * Recarrega as informaes de um algoritmo.
   *
   * @throws ServerException em caso de erro na leitura das informaes do
   *         algoritmo.
   */
  public void reload() throws ServerException {
    /* Reobtem as verses do algoritmo */
    readVersions();
  }

  /**
   * Remove uma verso de algoritmo do repositrio.
   *
   * @param versionId verso a remover
   * @throws ServerException em caso de erro.
   */
  public void removeVersion(final Object versionId) throws ServerException {
    final AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);

    /* remover os diretorios */
    final String versionDirPath = versionInfo.getDirPath();
    final File versionFile = new File(versionDirPath);
    try {
      FileSystem.removeFile(versionFile, true);
    }
    catch (final Exception e) {
      throw new ServerException(e.getMessage(), e);
    }
    info.deleteVersion(versionId);
  }

  /**
   * Cria no repositrio a estrutura para uma nova verso de algoritmo.
   *
   * @param major numero da verso
   * @param minor numero da reviso
   * @param patch numero da correo
   * @param propertyValues Mapa com as propriedades dos algoritmos.
   * @return identificador da verso criada
   * @throws ServerException se no for possvel criar os diretrios para
   *         armazenar os arquivos da nova verso.
   */
  public Object createVersion(final int major, final int minor, final int patch,
    final Map<String, String> propertyValues) throws ServerException {
    final String algoDir = info.getDirectory();
    final String algoDirPath = getAlgorithmRepositoryPath() + File.separator
      + algoDir;
    final String versionDir = AlgorithmVersionInfo.getDirectoryFor(major, minor,
      patch);
    final String versionDirPath = algoDirPath + File.separator
      + AlgorithmInfo.VERSIONS_DIR + File.separator + versionDir;
    createVersionRootDir(versionDirPath);
    try {
      /* cria a estrutura de subdiretrios do algoritmo */
      createVersionSubDirectories(versionDirPath);
    }
    catch (final ServerException se) {
      try {
        FileSystem.removeFile(new File(versionDirPath), true);
      }
      catch (final Exception e) {
        Server.logSevereMessage("Falha na remoo de arquivo/diretrio.", e);
      }
      throw se;
    }

    /* cria as informaes da verso */
    final Map<String, List<FileInfo>> supportedPlatforms =
      new Hashtable<String, List<FileInfo>>();
    final String pathToPropFile = versionDirPath + File.separator
      + AlgorithmVersionInfo.PROPERTY_VALUES_FILE;
    saveOwnerPropertyValue(getOwner(), propertyValues);
    savePropertyValues(pathToPropFile, propertyValues);
    final AlgorithmVersionInfo versionInfo = new AlgorithmVersionInfo(getInfo(),
      new AlgorithmVersionId(major, minor, patch), supportedPlatforms,
      propertyValues, null);
    final AlgorithmVersionId versionId = versionInfo.getId();
    info.includeVersion(versionId, versionInfo);
    return versionId;
  }

  /**
   * Cria a estrutura de uma verso a partir do seu identificador e suas
   * propriedades.
   *
   * @param versionId identificador da verso
   * @param propertyValues valores das propriedades da verso
   * @throws ServerException se no for possvel criar os diretrios para
   *         armazenar a verso
   */
  public void createVersionStructure(AlgorithmVersionId versionId,
    Map<String, String> propertyValues) throws ServerException {
    int major = versionId.getMajor();
    int minor = versionId.getMinor();
    int patch = versionId.getPatch();

    final String algoDir = info.getDirectory();
    final String algoDirPath = getAlgorithmRepositoryPath() + File.separator
      + algoDir;
    final String versionDir = AlgorithmVersionInfo.getDirectoryFor(major, minor,
      patch);
    final String versionDirPath = algoDirPath + File.separator
      + AlgorithmInfo.VERSIONS_DIR + File.separator + versionDir;
    createVersionRootDir(versionDirPath);
    try {
      /* cria a estrutura de subdiretrios do algoritmo */
      createVersionSubDirectories(versionDirPath);
    }
    catch (final ServerException se) {
      try {
        FileSystem.removeFile(new File(versionDirPath), true);
      }
      catch (final Exception e) {
        Server.logSevereMessage("Falha na remoo de arquivo/diretrio.", e);
      }
      throw se;
    }

    /* cria as informaes da verso */
    final Map<String, List<FileInfo>> supportedPlatforms =
      new Hashtable<String, List<FileInfo>>();
    final String pathToPropFile = versionDirPath + File.separator
      + AlgorithmVersionInfo.PROPERTY_VALUES_FILE;
    saveOwnerPropertyValue(getOwner(), propertyValues);
    savePropertyValues(pathToPropFile, propertyValues);
    final AlgorithmVersionInfo versionInfo = new AlgorithmVersionInfo(getInfo(),
      versionId, supportedPlatforms, propertyValues, null);
    info.includeVersion(versionId, versionInfo);
  }

  /**
   * Cria o diretrio "raiz" de uma verso.
   *
   * @param versionDirPath path para o diretrio "raiz" da verso
   * @throws ServerException se o diretrio da verso no puder ser criado.
   */
  private void createVersionRootDir(final String versionDirPath)
    throws ServerException {
    final File versionDir = new File(versionDirPath);
    if (versionDir.exists()) {
      throw new ServerException("Diretrio " + versionDirPath + " j existe!");
    }
    if (!versionDir.mkdir()) {
      throw new ServerException("Falha na criao do diretrio "
        + versionDirPath);
    }
  }

  /**
   * Cria a estrutura de subdiretrios para uma verso de algoritmo
   *
   * @param versionDirPath path para o diretrio "raiz" da verso
   * @throws ServerException Se no for possvel criar todos os subdiretrios.
   */
  private void createVersionSubDirectories(final String versionDirPath)
    throws ServerException {
    final String confDirPath = versionDirPath + File.separator
      + AlgorithmVersionInfo.CONFIGURATOR_DIR;
    final File confDir = new File(confDirPath);
    if (!confDir.mkdir()) {
      throw new ServerException("Falha na criao do diretrio do "
        + "configurador " + confDirPath);
    }
    final String binDirPath = versionDirPath + File.separator
      + AlgorithmVersionInfo.BIN_DIR;
    final File binDir = new File(binDirPath);
    if (!binDir.mkdir()) {
      throw new ServerException("Falha na criao do diretorio de "
        + "executveis: " + binDirPath);
    }
    final String docDirPath = versionDirPath + File.separator
      + AlgorithmVersionInfo.DOCUMENTATION_DIR;
    final File docDir = new File(docDirPath);
    if (!docDir.mkdir()) {
      throw new ServerException(
        "Falha na criao do diretorio de arquivos de documentao: "
          + docDirPath);
    }
    final String notesDirPath = versionDirPath + File.separator
      + AlgorithmVersionInfo.RELEASE_NOTES_DIR;
    final File notesDir = new File(notesDirPath);
    if (!notesDir.mkdir()) {
      throw new ServerException(
        "Falha na criao do diretorio de arquivos de release notes: "
          + notesDirPath);
    }
  }

  /**
   * Atualiza os arquivos de configurao de uma verso de um algoritmo.
   *
   * @param versionId identificador da verso
   * @throws OperationFailureException Se no for possvel encontrar os
   *         executveis da plataforma determinada.
   */
  protected void updateConfigurations(final Object versionId)
    throws OperationFailureException {

    final AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);
    final File configDir = new File(versionInfo.getConfiguratorDirPath());
    if (!configDir.exists()) {
      String msg = AlgorithmService.getInstance().getString(
        "server.algoservice.error.config_version_not_found");
      final Object[] args = new Object[] { info.getId(), versionId };
      msg = MessageFormat.format(msg, args);
      throw new OperationFailureException(msg);
    }

    final String versionsDirectoryPath = getAlgorithmRepositoryPath()
      + File.separator + this.info.getDirectory() + File.separator
      + AlgorithmInfo.VERSIONS_DIR;
    String algoVersionPath = versionInfo.getDirPath();
    File versionDirectory = new File(algoVersionPath);
    try {
      // Rel a verso inteira, pois o configurador pode alterar o tipo da
      // verso (se  algoritmo simples ou fluxo).
      readVersion(versionsDirectoryPath, versionDirectory);
    }
    catch (final ServerException exception) {
      final String message = MessageFormat.format(
        "Erro ao tentar ler informaes sobre o algoritmo {0} diretrio de "
          + "verso {1}.", new Object[] { this.info, versionDirectory
            .getAbsolutePath() });
      Server.logSevereMessage(message, exception);
    }
  }

  /**
   * Atualiza os arquivos de documentao de uma verso de um algoritmo.
   *
   * @param versionId identificador da verso
   * @throws OperationFailureException Se no for possvel encontrar os
   *         executveis da plataforma determinada.
   */
  protected void updateDocumentations(final Object versionId)
    throws OperationFailureException {

    final AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);
    final File docDir = new File(versionInfo.getDocDirPath());
    if (!docDir.exists()) {
      String msg = AlgorithmService.getInstance().getString(
        "server.algoservice.error.doc_version_not_found");
      final Object[] args = new Object[] { info.getId(), versionId };
      msg = MessageFormat.format(msg, args);
      throw new OperationFailureException(msg);
    }

    final List<FileInfo> documentation = FileInfo.createFilesInfo(docDir);
    versionInfo.setDocumentation(documentation);
  }

  /**
   * Atualiza o arquivo de release notes de uma verso de um algoritmo.
   *
   * @param versionId identificador da verso.
   * @param fileName nome do arquivo.
   * @throws OperationFailureException Se no for possvel encontrar os
   *         executveis da plataforma determinada.
   */
  protected void updateReleaseNotes(final Object versionId,
    final String fileName) throws OperationFailureException {

    final AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);
    final File textDir = new File(versionInfo.getReleaseNotesDirPath());
    if (!textDir.exists()) {
      String msg = AlgorithmService.getInstance().getString(
        "server.algoservice.error.release_notes_version_not_found");
      final Object[] args = new Object[] { info.getId(), versionId };
      msg = MessageFormat.format(msg, args);
      throw new OperationFailureException(msg);
    }

    final List<FileInfo> files = FileInfo.createFilesInfo(textDir);
    versionInfo.setReleaseNotes(getReleaseNotes(files));
  }

  /**
   * Obtm o arquivo de release notes entre uma lista de possveis arquivos que
   * possam existir no diretrio.
   * 
   * @param files arquivos no diretrio
   * @return arquivo de release notes.
   */
  private FileInfo getReleaseNotes(List<FileInfo> files) {
    for (FileInfo file : files) {
      if (file.getPath().equals("releasenotes.txt")) {
        return file;
      }
    }
    return null;
  }

  /**
   * Atualiza os executveis de uma plataforma de uma verso de algoritmo.
   *
   * @param versionId identificador da verso
   * @param platform plataforma cujos executveis deseja-se atualizar.
   * @throws OperationFailureException Se no for possvel encontrar os
   *         executveis da plataforma determinada ou se no for possvel dar
   *         permisso de execuo para os arquivos.
   */
  protected void updateExecutables(final Object versionId,
    final String platform) throws OperationFailureException {
    final AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);
    final String platDirPath = versionInfo.getPlatformPath(platform);
    final File platDir = new File(platDirPath);
    if (!platDir.exists()) {
      String msg = AlgorithmService.getInstance().getString(
        "server.algoservice.error.plat_version_not_found");
      final Object[] args = new Object[] { info.getId(), versionId, platform };
      msg = MessageFormat.format(msg, args);
      throw new OperationFailureException(msg);
    }

    // configuro os arquivos como executveis
    FileVisitor<Path> fileVisitor = new SimpleFileVisitor<Path>() {
      @Override
      public FileVisitResult visitFile(Path path, BasicFileAttributes attrs) throws IOException {
        try {
          FileSystem.enableExecutionPermission(path.toString());
        }
        catch (ServerException e) {
          throw new IOException(e.getMessage(), e);
        }
        return super.visitFile(path, attrs);
      }
    };
    try {
      // garante que arquivos em subdiretrios tambm sero definidos como executveis.
      Files.walkFileTree(Paths.get(platDir.getAbsolutePath()), fileVisitor);
    }
    catch (IOException e) {
      throw new OperationFailureException(e);
    }
    versionInfo.setPlatformExecutables(platform, FileInfo.createFilesInfo(platDir));
  }

  /**
   * Apaga arquivo.
   *
   * @param file arquivo
   * @return indicativo
   */
  private boolean deleteFile(final File file) {
    if (file.isDirectory()) {
      for (final File child : file.listFiles()) {
        if (!deleteFile(child)) {
          return false;
        }
      }
    }
    return file.delete();
  }

  /**
   * Remove arquivos executveis de uma verso de algoritmo.
   *
   * @param version Verso que detm os arquivos.
   * @param platform Plataforma que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * @throws Exception caso no tenha sido possvel remover o arquivo.
   */
  public void removeExecutableFiles(final AlgorithmVersionInfo version,
    final String platform, final FileInfo[] files) throws Exception {
    for (final FileInfo file : files) {
      final String path = version.getExecFilePath(platform, file.getPath());
      final File serverFile = new File(path);
      try {
        // Remove o arquivo do servidor
        if (!deleteFile(serverFile)) {
          final String msg = AlgorithmService.getInstance().getString(
            "server.algoservice.error.copy_server");
          throw new Exception(msg);
        }
      }
      catch (final Exception e) {
        String msg = AlgorithmService.getInstance().getString(
          "server.algoservice.error.remove_exe");
        final Object[] args = new Object[] { serverFile.getName(), info.getId(),
            version.getId(), platform };
        msg = MessageFormat.format(msg, args);
        throw new Exception(msg + e.getMessage(), e);
      }
    }

    // Recarrega o algoritmo
    reload();
  }

  /**
   * Remove arquivos de configurao de uma verso de algoritmo.
   *
   * @param version Verso que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * @throws Exception caso no tenha sido possvel remover o arquivo.
   */
  public void removeConfigurationFiles(final AlgorithmVersionInfo version,
    final FileInfo[] files) throws Exception {
    for (final FileInfo file : files) {
      final String path = version.getConfiguratorDirPath() + File.separator
        + file.getPath();
      final File serverFile = new File(path);
      try {
        // Remove o arquivo do servidor
        if (!deleteFile(serverFile)) {
          final String msg = AlgorithmService.getInstance().getString(
            "server.algoservice.error.copy_server");
          throw new Exception(msg);
        }
      }
      catch (final Exception e) {
        String msg = AlgorithmService.getInstance().getString(
          "server.algoservice.error.remove_config");
        final Object[] args = new Object[] { info.getId(), version.getId() };
        msg = MessageFormat.format(msg, args);
        throw new Exception(msg + e.getMessage(), e);
      }
    }

    // Recarrega o algoritmo
    reload();
  }

  /**
   * Remove arquivos de documentao de uma verso de algoritmo.
   *
   * @param version Verso que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * @throws Exception caso no tenha sido possvel remover o arquivo.
   */
  public void removeDocumentationFiles(final AlgorithmVersionInfo version,
    final FileInfo[] files) throws Exception {
    for (final FileInfo file : files) {
      final String path = version.getDocDirPath() + File.separator + file
        .getPath();
      final File serverFile = new File(path);
      try {
        // Remove o arquivo do servidor
        if (!deleteFile(serverFile)) {
          final String msg = AlgorithmService.getInstance().getString(
            "server.algoservice.error.copy_server");
          throw new Exception(msg);
        }
      }
      catch (final Exception e) {
        String msg = AlgorithmService.getInstance().getString(
          "server.algoservice.error.remove_doc");
        final Object[] args = new Object[] { serverFile.getName(), info.getId(),
            version.getId() };
        msg = MessageFormat.format(msg, args);
        throw new Exception(msg + e.getMessage(), e);
      }
    }

    // Recarrega o algoritmo
    reload();
  }

  /**
   * Remove arquivos de release notes de uma verso de algoritmo.
   *
   * @param version Verso que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * @throws Exception caso no tenha sido possvel remover o arquivo.
   */
  public void removeReleaseNotesFiles(final AlgorithmVersionInfo version,
    final FileInfo[] files) throws Exception {
    for (final FileInfo file : files) {
      final String path = version.getReleaseNotesDirPath() + File.separator
        + file.getPath();
      final File serverFile = new File(path);
      try {
        // Remove o arquivo do servidor
        if (!deleteFile(serverFile)) {
          final String msg = AlgorithmService.getInstance().getString(
            "server.algoservice.error.copy_server");
          throw new Exception(msg);
        }
      }
      catch (final Exception e) {
        String msg = AlgorithmService.getInstance().getString(
          "server.algoservice.error.remove_notes");
        final Object[] args = new Object[] { serverFile.getName(), info.getId(),
            version.getId() };
        msg = MessageFormat.format(msg, args);
        throw new Exception(msg + e.getMessage(), e);
      }
    }

    // Recarrega o algoritmo
    reload();
  }

  /**
   * <p>
   * Faz a cpia do arquivo dado para o caminho especificado pelo toPath. Em
   * seguida, faz a cpia do mesmo arquivo para o diretrio do cliente. Se
   * {@code setExecutable} for verdadeiro a cpia criada/sobrescrita ter
   * permisso de execuo.
   * </p>
   * <p>
   * Ao final da execuo os algoritmos sero recarregados.
   * </p>
   *
   * @param fromPath caminho para o arquivo a ser copiado.
   * @param toPath caminho de destino.
   * @param setExecutable indica se deve habilitar a permisso de execuo.
   * @return true se for criado um arquivo novo para a cpia, false se j
   *         existisse um arquivo com o mesmo nome no diretrio (este ento 
   *         sobrescrito).
   * @throws Exception Se no for possvel copiar o arquivo ou recarregar as
   *         informaes do algoritmo.
   */
  boolean copyFile(final String fromPath, final String toPath,
    final boolean setExecutable) throws Exception {
    try {
      return copyFileWithoutReloading(fromPath, toPath, setExecutable);
    }
    finally {
      reload();
    }
  }

  /**
   * Copia arquivos de uma pasta para a outra com a possibilidade de dar
   * permisso de execuo s cpias. Para definir o caminho do arquivo de
   * destino, ser construido um FileInfo usando o targetDir como path se este
   * for diferente de <tt>null</tt> e mantendo o nome do arquivo original. Este
   * novo FileInfo ser passado como parmetro para o
   * {@link IPathFactory#getPath(FileInfo) targetPathFactory.getPath(FileInfo)}
   * que retornar o caminho fsico do novo arquivo. Ao final da execuo os
   * algoritmos sero recarregados.
   *
   * @param files Arquivos a serem transferidos.
   * @param sourcePathFactory Fbrica para criar o caminho dos arquivos a serem
   *        transferidos.
   * @param targetDir diretrio de destino (pode ser <tt>null</tt>).
   * @param targetPathFactory Fbrica para criar o caminho dos arquivos de
   *        destino. Este caminho  relativo ao valor do parmetro targetDir.
   * @param setExecutables <tt>true</tt> se as cpias devero ter permisso de
   *        execuo ou no.
   * @param records Uma lista para gravar as informaoes dos arquivos copiados
   *        (pode ser <tt>null</tt>).
   * @throws InfoException Se no for possvel copiar os arquivos.
   * @throws Exception Se no for possvel copiar os arquivos ou recarregar as
   *         informaes do algoritmo.
   */
  public void copyFiles(final List<FileInfo> files,
    final IPathFactory sourcePathFactory, final FileInfo targetDir,
    final IPathFactory targetPathFactory, final boolean setExecutables,
    final List<ICopyRecord> records) throws InfoException, Exception {

    try {
      copyFilesWithoutReloading(files, sourcePathFactory, targetDir,
        targetPathFactory, setExecutables, records);
    }
    finally {
      reload();
    }
  }

  /**
   * Copia arquivos de uma pasta para a outra com a possibilidade de dar
   * permisso de execuo s cpias. Para definir o caminho do arquivo de
   * destino, ser construido um FileInfo usando o targetDir como path se este
   * for diferente de <tt>null</tt> e mantendo o nome do arquivo original. Este
   * novo FileInfo ser passado como parmetro para o
   * {@link IPathFactory#getPath(FileInfo) targetPathFactory.getPath(FileInfo)}
   * que retornar o caminho fsico do novo arquivo. Ao final da execuo os
   * algoritmos <b>NO</b> sero recarregados.
   *
   * @param files Arquivos a serem transferidos.
   * @param sourcePathFactory Fbrica para criar o caminho dos arquivos a serem
   *        transferidos.
   * @param targetDir diretrio de destino.
   * @param targetPathFactory Fbrica para criar o caminho dos arquivos de
   *        destino. Este caminho  relativo ao valor do parmetro targetDir.
   * @param setExecutables <tt>true</tt> se as cpias devero ter permisso de
   *        execuo ou no.
   * @param records Uma lista com informaoes dos arquivos copiados.
   * @throws InfoException Se no for possvel copiar os arquivos.
   * @throws Exception Se no for possvel copiar os arquivos.
   */
  void copyFilesWithoutReloading(final List<FileInfo> files,
    final IPathFactory sourcePathFactory, final FileInfo targetDir,
    final IPathFactory targetPathFactory, final boolean setExecutables,
    final List<ICopyRecord> records) throws InfoException, Exception {

    for (final FileInfo sourceFileInfo : files) {
      FileInfo targetFileInfo;
      if (null != targetDir) {
        final String[] newDirPath = Utilities.splitProjectPath(targetDir
          .getPath());
        targetFileInfo = new FileInfo(newDirPath, sourceFileInfo.getName(),
          sourceFileInfo.isDirectory());

        // Verifica se o diretrio origem  ancestral do diretrio destino
        boolean isSourceAncestor = targetDir.getPath().contains(sourceFileInfo
          .getPath());

        if (sourceFileInfo.isDirectory() && isSourceAncestor) {
          String msg = AlgorithmService.getInstance().getString(
            "server.algoservice.error.copy_file_into_descendant");
          final Object[] args = new Object[] { sourceFileInfo.getName(),
              targetDir.getName() };
          String infoMsg = MessageFormat.format(msg, args);
          throw new InfoException(infoMsg);
        }
      }
      else {
        targetFileInfo = new FileInfo(sourceFileInfo.getName(), sourceFileInfo
          .isDirectory());
      }
      final String testTargetPath = targetPathFactory.getPath(targetFileInfo);
      File testFile = new File(testTargetPath);

      // Se ja existe um arquivo presente no diretrio alvo com o mesmo nome
      if (testFile.exists()) {
        // e se o arquivo no  um diretrio, devemos duplicar o arquivo alvo
        if (!testFile.isDirectory()) {
          testFile = new File(getCopyName(testTargetPath));
          if (null != targetDir) {
            final String[] newTargetDirPath = Utilities.splitProjectPath(
              targetDir.getPath());
            targetFileInfo = new FileInfo(newTargetDirPath, testFile.getName(),
              testFile.isDirectory());
          }
          else {
            targetFileInfo = new FileInfo(testFile.getName(), testFile
              .isDirectory());
          }
        }
        // e se o arquivo  um diretrio, devemos avisar que no podem 
        // existir diretrios duplicados
        else {
          String msg = AlgorithmService.getInstance().getString(
            "server.algoservice.error.copy_file_directory_exists");
          final Object[] args = new Object[] { sourceFileInfo.getName() };
          String infoMsg = MessageFormat.format(msg, args);
          throw new InfoException(infoMsg);
        }
      }
      final String targetPath = targetPathFactory.getPath(targetFileInfo);
      if (sourceFileInfo.isDirectory()) {
        final File targetFile = new File(targetPath);
        if (!targetFile.exists()) {
          targetFile.mkdir();
          if (null != records) {
            records.add(new CopyDirectoryRecord(targetFileInfo));
          }
        }
        copyFilesWithoutReloading(sourceFileInfo.getChildren(),
          sourcePathFactory, targetFileInfo, targetPathFactory, setExecutables,
          records);
      }
      else {
        final String sourcePath = sourcePathFactory.getPath(sourceFileInfo);
        final boolean created = copyFileWithoutReloading(sourcePath, targetPath,
          setExecutables);
        if (null != records) {
          records.add(new CopyFileRecord(sourceFileInfo, targetFileInfo,
            created));
        }
      }
    }
  }

  /**
   * Faz a cpia do arquivo dado para o caminho especificado pelo toPath. Em
   * seguida, faz a cpia do mesmo arquivo para o diretrio do cliente. Se
   * {@code setExecutable} for verdadeiro a cpia criada/sobrescrita ter
   * permisso de execuo. Ao final da execuo os algoritmos <b>NO</b> sero
   * recarregados.
   *
   * @param fromPath caminho para o arquivo a ser copiado.
   * @param toPath caminho de destino.
   * @param setExecutable indica se deve habilitar a permisso de execuo.
   * @return true se for criado um arquivo novo para a cpia, false se j
   *         existisse um arquivo com o mesmo nome no diretrio (este ento 
   *         sobrescrito).
   * @throws Exception Se no for possvel copiar o arquivo.
   */
  boolean copyFileWithoutReloading(final String fromPath, final String toPath,
    final boolean setExecutable) throws Exception {
    // O padro  rescrever os arquivos.
    boolean copyCreated = false;
    final File from = new File(fromPath);
    File serverTo = new File(toPath);
    serverTo.createNewFile(); // gera erro quando for um diretrio.
    copyCreated = true;
    if (FileSystem.copyFile(from, serverTo)) {
      if (setExecutable) {
        FileSystem.enableExecutionPermission(toPath);
      }
    }
    else {
      final String msg = AlgorithmService.getInstance().getString(
        "server.algoservice.error.copy_server");
      throw new Exception(msg);
    }
    return copyCreated;
  }

  /**
   * Obtm o nome da cpia. O nome da cpia  formado por
   * [prefixo][sequencial].[extenso]. O prefixo  o nome do arquivo original
   * sem extenso. O sequencial  formado por um nmero entre parnteses. Esse
   * nmero  o nmero seguinte da ltima cpia existente.
   *
   * @param path o caminho do arquivo base para obter o nome da cpia.
   * @return O nome da cpia.
   */
  private String getCopyName(String path) {
    int maxCopyNumber = 0;
    File file = new File(path);
    File father = file.getParentFile();
    File[] brothers = father.listFiles();
    String[] parts = splitFileName(file.getName());
    String preffix = parts[0];
    String extension = parts[2];

    // Obtendo o maior nmero de cpia.
    for (int i = 0; i < brothers.length; i++) {
      File brother = brothers[i];
      String[] brotherParts = splitFileName(brother.getName());
      if (brotherParts[0].equals(preffix) && brotherParts[2].equals(extension)
        && (brotherParts[1].length() != 0)) {
        int copyNumber;
        try {
          copyNumber = Integer.parseInt(brotherParts[1]);
          if (copyNumber > maxCopyNumber) {
            maxCopyNumber = copyNumber;
          }
        }
        catch (NumberFormatException ex) {
          throw new ServiceFailureException(
            "Erro ao construir o nome do arquivo.", ex);
        }
      }
    }
    int copyNumber = maxCopyNumber + 1;
    String copyName = preffix + "_" + copyNumber;
    if (extension.length() != 0) {
      copyName += ("." + extension);
    }
    return copyName;
  }

  /**
   * Obtm os nomes e ndices que delimitam o prefixo, o sequencial e a extenso
   * deste arquivo.
   * 
   * @param name nome do arquivo a ser dividido.
   *
   * @return O array dos textos.
   *         <ul>
   *         <li>ndice 0: Ser utilizado para o prefixo;
   *         <li>ndice 1: Ser utilizado para o sequencial (sem os "()");
   *         <li>ndice 2: Ser utilizado para a extenso.
   *         </ul>
   *         Se no houver uma parte, a posio dela no array ser uma string
   *         vazia.
   */
  private String[] splitFileName(String name) {
    int ixLastUnderscore = name.lastIndexOf("_");
    int ixDot = name.lastIndexOf(".");
    int ixEndPreffix = name.length();
    String preffix = "";
    String sequencial = "";
    String extension = "";
    if (ixDot != -1) {
      extension = name.substring(ixDot + 1);
      // removendo a extenso do arquivo.
      name = name.substring(0, ixDot);
      ixEndPreffix = ixDot;
    }
    ixLastUnderscore = name.lastIndexOf("_");
    if (ixLastUnderscore != -1) {
      String sequencialCandidate;
      sequencialCandidate = name.substring(ixLastUnderscore + 1);
      if (sequencialCandidate.matches("\\d+")) {
        sequencial = sequencialCandidate;
        ixEndPreffix = ixLastUnderscore;
      }
    }
    preffix = name.substring(0, ixEndPreffix);
    return new String[] { preffix, sequencial, extension };
  }

  /**
   * Inclui uma plataforma para uma verso de algoritmo.
   *
   * @param versionId identificador da verso
   * @param platform plataforma a incluir
   * @throws ServerException Se no for possvel criar os diretrios para
   *         armazenar os arquivos referentes  plataforma.
   */
  public void includePlatform(final Object versionId, final String platform)
    throws ServerException {
    final AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);
    final String platDirPath = versionInfo.getPlatformPath(platform);
    final File platDir = new File(platDirPath);
    if (!platDir.mkdirs()) {
      throw new ServerException("Falha na criao do diretorio de plataforma: "
        + platDirPath);
    }
    versionInfo.setPlatform(platform);
  }

  /**
   * Remove uma plataforma de uma verso de algoritmo.
   *
   * @param versionId identificador da verso
   * @param platform plataforma a remover
   * @throws ServerException se no for possvel remover os arquivos referentes
   *          plataforma.
   */
  public void removePlatform(final Object versionId, final String platform)
    throws ServerException {
    final AlgorithmVersionInfo versionInfo = info.getVersionInfo(versionId);
    final String platDirPath = versionInfo.getPlatformPath(platform);
    final File platDir = new File(platDirPath);
    try {
      FileSystem.removeFile(platDir, true);
      final Map<String, List<FileInfo>> supportedPlatforms = versionInfo
        .getPlatforms();
      supportedPlatforms.remove(platform);
    }
    catch (final Exception e) {
      throw new ServerException(e.getMessage(), e);
    }
  }

  /**
   * Escreve o contedo de um arquivo texto.
   *
   * @param fileName path do arquivo
   * @param content contedo a ser escrito
   * @throws ServerException se no for possvel escrever no arquivo.
   */
  private static void writeSingleLineFile(final String fileName,
    final String content) throws ServerException {
    try {
      final FileWriter fw = new FileWriter(fileName);
      fw.write(content);
      fw.close();
    }
    catch (final IOException e) {
      throw new ServerException("Falha na escrita do arquivo " + fileName, e);
    }
  }

  /**
   * Obtm referncia para o servio de gerncia de algoritmos.
   *
   * @return referncia para o servio de gerncia de algoritmos.
   */
  private static AlgorithmService getService() {
    return AlgorithmService.getInstance();
  }

  /**
   * Constri a representao de um algoritmo no SSI. Deve ser utilizado apenas
   * para <b>operacoes de inclusao do servico</b>!
   *
   * @param name nome do algoritmo
   * @param dir diretrio do algoritmo no repositrio
   * @param attributes Atributos do algoritmo lidos do arquivo de propriedades
   */
  private Algorithm(final String name, final String dir,
    final Hashtable<String, String> attributes) {
    /* O identificador do algoritmo  o nome de seu diretrio */
    final String id = dir;
    this.info = new ModifiableAlgorithmInfo(id, name, dir,
      getAlgorithmRepositoryPath(), attributes);
  }

  /**
   * Retorna a lista de <code>Property</code> dos algoritmos. Cada
   * <code>Property</code> mapeia a chave da propriedade ao Label (String de
   * apresentao).
   *
   * @return Lista com as propriedades dos algoritmos.
   */
  public static List<AlgorithmProperty> getAlgorithmProperties() {
    if (algorithmProperties == null) {
      final AlgorithmService algoService = AlgorithmService.getInstance();
      final String algPropertiesPath = algoService
        .getAlgorithmPropertyNamesPath();
      algorithmProperties = getProperties(algPropertiesPath);
      validateAlgorithmProperties();
    }
    return algorithmProperties;
  }

  /**
   * Valida as propriedades de algoritmos lidas, para ver se no existe alguma
   * que seja uma propriedade reservada de algoritmos no sistema. Se for, uma
   * exceo  lanada e o servidor  interrompido.
   */
  private static void validateAlgorithmProperties() {
    for (AlgorithmProperty algoProperty : algorithmProperties) {
      if (algoProperty.getKey().equals(
        AlgorithmInfo.OWNER_ALGORITHM_PROPERTY)) {
        final String fmt =
          "Uma propriedade de algoritmo reservada foi detectada: %s";
        final String msg = String.format(fmt, algoProperty);
        throw new ServiceFailureException(msg);
      }
    }
  }

  /**
   * Retorna a lista de <code>Property</code> das verses dos algoritmos. Cada
   * <code>Property</code> mapeia a chave da propriedade ao Label (String de
   * apresentao).
   *
   * @return Lista com as propriedades das verses dos algoritmos.
   */
  public static List<AlgorithmProperty> getVersionProperties() {
    if (versionProperties == null) {
      final AlgorithmService algoService = AlgorithmService.getInstance();
      final String versionPropertiesPath = algoService
        .getVersionPropertyNamesPath();
      versionProperties = getProperties(versionPropertiesPath);
    }
    return versionProperties;
  }

  /**
   * Mtodo para leitura dos nomes das propriedades dos algoritmos e das
   * verses. Tais arquivos de propriedades possuem uma propriedade que
   * determina a ordem de leitura das chaves de propridades (ordem =
   * propriedade1, propriedade2, propriedadeX) As demais propriedades neste
   * arquivo mapeiam a chave no label de exibio. propriedade1 = Meu ttulo 
   * propriedade2 = Tenho o nome propriedadeX = Eu sou ... A propriedade ordem 
   * importante para que as propriedades sejam exibidas sempre na mesma ordem.
   * Caso ela no exista, no garantimos a ordem de exibico das propriedades.
   *
   * @param propertyPath Nome do arquivo de propriedades
   * @return Um lista de <code>Property<code> que possui as chaves das
   *         propriedades ordenadas
   */
  private static List<AlgorithmProperty> getProperties(
    final String propertyPath) {
    final Properties properties = readPropertiesFile(propertyPath);
    final List<String> keys = new ArrayList<String>();
    final String sortedProperties = properties.getProperty(SORT_ORDER_PROPERTY);
    final List<AlgorithmProperty> propertyLabels =
      new ArrayList<AlgorithmProperty>();
    // Existe a propriedade ordem
    if (sortedProperties != null) {
      final String[] sortedPropertiesArray = sortedProperties.split(",");
      for (final String element : sortedPropertiesArray) {
        keys.add(element.trim());
      }
      String propertyKey = null;
      String propertyLabel = null;
      PropertyType propertyType;
      for (int i = 0; i < keys.size(); i++) {
        propertyKey = keys.get(i);
        propertyLabel = properties.getProperty(propertyKey);
        final String type = properties.getProperty(propertyKey
          + AlgorithmProperty.TYPE_SUFFIX);

        /* Trata da possibilidade da ordem conter chaves inexistentes */
        if (propertyLabel != null) {
          if (type == null) {
            propertyLabels.add(new AlgorithmProperty(propertyKey,
              propertyLabel));
          }
          else {
            propertyType = PropertyType.valueOf(type.toUpperCase());
            propertyLabels.add(new AlgorithmProperty(propertyKey, propertyLabel,
              propertyType));
          }
        }
      }
    }
    // A propriedade ordem no existe, consideramos ordem de leitura
    else {
      for (final Enumeration<?> e = properties.propertyNames(); e
        .hasMoreElements();) {
        final String propertyKey = (String) e.nextElement();
        if (propertyKey.endsWith(AlgorithmProperty.TYPE_SUFFIX)) {
          continue;
        }
        final String propertyLabel = properties.getProperty(propertyKey);
        final String type = properties.getProperty(propertyKey
          + AlgorithmProperty.TYPE_SUFFIX);
        /* Trata da possibilidade da ordem conter chaves inexistentes */
        if (propertyLabel != null) {
          if (type == null) {
            propertyLabels.add(new AlgorithmProperty(propertyKey,
              propertyLabel));
          }
          else {
            final PropertyType propertyType = PropertyType.valueOf(type
              .toUpperCase());
            propertyLabels.add(new AlgorithmProperty(propertyKey, propertyLabel,
              propertyType));
          }
        }
      }
    }
    return propertyLabels;
  }

  /**
   * Recupera os atributos estendidos de um algoritmo do local onde eles foram
   * persistidos.
   *
   * @param algoDirPath caminho para o diretorio do algoritmo
   * @return chaves e valores dos atributos estendidos.
   */
  private static Hashtable<String, String> loadAlgorithmPropertyValues(
    final String algoDirPath) {
    final String pathToPropFile = algoDirPath + File.separator
      + AlgorithmInfo.PROPERTY_VALUES_FILE;
    final List<AlgorithmProperty> algProperties = getAlgorithmProperties();
    return loadPropertiesValues(pathToPropFile, algProperties);
  }

  /**
   * Obtm os valores das propriedades no caminho especificado. As chaves das
   * propriedades lidas esto no atributo <code>properties</code>.
   *
   * @param propPath Caminho do arquivo de propriedades.
   * @param propertiesList Lista com as propriedades que devem ser lidas do
   *        arquivo dado
   * @return uma hashtable com os valores das propriedades associados s chaves
   */
  private static Hashtable<String, String> loadPropertiesValues(
    final String propPath, final List<AlgorithmProperty> propertiesList) {
    final Hashtable<String, String> result = new Hashtable<String, String>();
    final Properties properties = readPropertiesFile(propPath);
    for (final AlgorithmProperty property : propertiesList) {
      final String propKey = property.getKey();
      final PropertyType type = property.getType();
      if (propKey != null) {
        final String propValue = properties.getProperty(propKey);
        if (propValue != null && propValue.length() > 0) {
          if (type == PropertyType.DOUBLE) {
            // Tenta converter. Se falhar ignora propriedade e loga.
            try {
              new Double(propValue);
            }
            catch (final NumberFormatException e) {
              Server.logSevereMessage("Falha na leitura do valor " + propValue
                + " da varivel " + propKey + " em: " + propPath);
              continue;
            }
          }
          result.put(propKey, propValue);
        }
      }
    }
    readHideAlgorithmProperty(properties, result);
    readOwnerAlgorithmProperty(properties, result);
    return result;
  }

  /**
   * Procura pela propriedade ocultar, usada pelo ExecutorFrame, para que as
   * aplicaes X no apaream no menu de algoritmos.
   *
   * @param properties propriedades lidas de um arquivo de propriedades do
   *        algoritmo
   * @param algoPropertiesTable hashing com os valores das propriedades
   *        associados s chaves
   */
  private static void readHideAlgorithmProperty(final Properties properties,
    final Hashtable<String, String> algoPropertiesTable) {
    final String propValue = properties.getProperty(
      AlgorithmInfo.HIDE_ALGORITHM_PROPERTY);
    if (propValue != null && propValue.trim().equals(
      AlgorithmInfo.HIDE_ALGORITHM_VALUE)) {
      algoPropertiesTable.put(AlgorithmInfo.HIDE_ALGORITHM_PROPERTY,
        AlgorithmInfo.HIDE_ALGORITHM_VALUE);
    }
  }

  /**
   * Procura pela propriedade que indica o usurio (login) que criou esse
   * algoritmo.
   *
   * @param properties propriedades lidas de um arquivo de propriedades do
   *        algoritmo
   * @param algoPropertiesTable hashing com os valores das propriedades
   *        associados s chaves
   */
  private static void readOwnerAlgorithmProperty(final Properties properties,
    final Hashtable<String, String> algoPropertiesTable) {
    final String propValue = properties.getProperty(
      AlgorithmInfo.OWNER_ALGORITHM_PROPERTY);
    if (propValue != null) {
      algoPropertiesTable.put(AlgorithmInfo.OWNER_ALGORITHM_PROPERTY,
        propValue);
    }
  }

  /**
   * Recupera as propriedades da versao do algoritmo do local onde elas foram
   * persistidas
   *
   * @param versionDirectoryPath caminho para o diretorio da versao
   * @return tabela com os nomes e valores dos atributos estendidos
   */
  private Hashtable<String, String> loadVersionPropertyValues(
    final String versionDirectoryPath) {
    final String pathToPropFile = versionDirectoryPath + File.separator
      + AlgorithmVersionInfo.PROPERTY_VALUES_FILE;
    final List<AlgorithmProperty> vp = getVersionProperties();
    return loadPropertiesValues(pathToPropFile, vp);
  }

  /**
   * L as propriedades de um arquivo e carrega em um objeto-coleo de
   * propriedades. Nao deve ser utilizado diretamente. As propriedaes do
   * algoritmo e da versao podem ser obtidos atraves dos metodos
   * loadAlgorithmPropertyValues e loadVersionPropertyValues
   *
   * @param path caminho para o arquivo que contm as propriedades.
   * @return o objeto properties.
   */
  private static Properties readPropertiesFile(final String path) {
    InputStream inputStream = null;
    final Properties properties = new Properties();
    try {
      inputStream = new FileInputStream(path);
      properties.load(inputStream);
    }
    catch (final FileNotFoundException e) {
      Server.logFineMessage("Arquivo de propriedades " + path
        + " no foi encontrado.");
    }
    catch (final IOException e) {
      Server.logSevereMessage("Erro ao ler arquivo de propriedades " + path, e);
    }
    finally {
      if (inputStream != null) {
        try {
          inputStream.close();
        }
        catch (final IOException ioe) {
          Server.logSevereMessage("Erro ao fechar arquivo de propriedades "
            + path, ioe);
        }
      }
    }
    return properties;
  }

  /**
   * Define novos atributos estendidos para o algoritmo e persiste a mudanca
   *
   * @param newValues Novos atributos.
   */
  public void setAlgorithmFieldValuesList(
    final Hashtable<String, String> newValues) {
    final String algoDirPath = getAlgorithmRepositoryPath() + File.separator
      + info.getDirectory();

    /* Periste Mudancas */
    final String pathToPropFile = algoDirPath + File.separator
      + AlgorithmInfo.PROPERTY_VALUES_FILE;
    saveOwnerPropertyValue(getOwner(), newValues);
    savePropertyValues(pathToPropFile, newValues);

    /* Altera dados em memoria */
    this.info.setPropertyValues(newValues);
  }

  /**
   * Define novos atributos estendidos para o algoritmo e persiste a mudanca
   *
   * @param id Identificador da verso do algoritmo.
   * @param newValues Novos atributos.
   */
  public void setVersionFieldValuesList(final AlgorithmVersionId id,
    final Hashtable<String, String> newValues) {
    final AlgorithmVersionInfo version = this.getInfo().getVersionInfo(id);
    final String versionDirPath = version.getDirPath();

    /* Persiste mudancas */
    final String pathToPropFile = versionDirPath + File.separator
      + AlgorithmVersionInfo.PROPERTY_VALUES_FILE;
    saveOwnerPropertyValue(getOwner(), newValues);
    savePropertyValues(pathToPropFile, newValues);

    /* Altera dados em memoria */
    version.setPropertyValues(newValues);
  }

  /**
   * Persiste os atributos estendidos do algoritmo
   *
   * @param pathToPropFile caminho para o arquivo de propriedades
   * @param propertyValues valores dos atributos estendidos
   */
  private static void savePropertyValues(final String pathToPropFile,
    final Map<String, String> propertyValues) {
    final Properties properties = new Properties();
    for (final String attributeKey : propertyValues.keySet()) {
      final String attributeValue = propertyValues.get(attributeKey);
      properties.put(attributeKey, attributeValue);
    }
    try {
      writePropertiesFile(pathToPropFile, properties);
    }
    catch (final ServerException e) {
      Server.logInfoMessage(e.getMessage());
    }
  }

  /**
   * Salva o valor da propriedade do algoritmo que indica quem o criou. Quando o
   * algoritmo  criado, o valor deve ser o login do usurio que solicitou o
   * servio. Nas alteraes do algoritmo, essa propriedade deve manter o valor
   * original de quem o criou e no de quem est alterando.
   *
   * @param owner usurio que criou esse algoritmo
   * @param propertyValues valores das propriedades do algoritmo
   */
  private static void saveOwnerPropertyValue(String owner,
    Map<String, String> propertyValues) {
    if (owner != null) {
      propertyValues.put(AlgorithmInfo.OWNER_ALGORITHM_PROPERTY, owner);
    }
  }

  /**
   * Obtm o valor da propriedade que indica o usurio (login) que criou esse
   * algoritmo.
   *
   * @return retorna o login do usurio que criou esse algoritmo
   */
  private String getOwner() {
    return getInfo().getOwner();
  }

  /**
   * Armazena um arquivo de propriedades num dado local. Nao deve ser utilizado
   * diretamente! As propriedades estendidas do algoritmo podem ser salvas
   * utilizando os metodos saveAlgorithmPropertyValues e
   * saveVersionPropertyValues
   *
   * @param pathToPropFile nome e caminho completo do arquivo de propriedades
   * @param properties propriedades a serem salvas
   * @throws ServerException em caso de erro.
   */
  private static void writePropertiesFile(final String pathToPropFile,
    final Properties properties) throws ServerException {
    FileOutputStream out = null;
    try {
      out = new FileOutputStream(pathToPropFile);
    }
    catch (final FileNotFoundException e) {
      final String message = "Erro ao escrever arquivo de propriedades "
        + pathToPropFile;
      throw new ServerException(message, e);
    }
    finally {
      try {
        properties.store(out, "Arquivo de propriedades do algoritmo");
        if (out != null) {
          out.close();
        }
      }
      catch (final IOException e) {
        final String message = "Erro ao escrever arquivo de propriedades "
          + pathToPropFile;
        throw new ServerException(message, e);
      }
    }
  }

  /**
   * Interface para a funcionalidade de copiar um arquivo/diretrio.
   */
  public static interface ICopyRecord {
    /**
     * Verifica se  ou no um diretrio.
     *
     * @return retorna true se for um diretrio, caso contrrio, retorna false
     */
    boolean isDirectory();

    /**
     * Verifica se o arquivo/diretrio foi criado.
     *
     * @return retorna true o arquivo/diretrio foi criado, caso contrrio,
     *         retorna false
     */
    boolean wasCreated();
  }

  /**
   * Mantm o diretrio a ser copiado.
   */
  public static class CopyDirectoryRecord implements ICopyRecord {

    /**
     * O path.
     */
    private final FileInfo path;

    /**
     * Construtor
     *
     * @param path o path
     */
    private CopyDirectoryRecord(final FileInfo path) {
      this.path = path;
    }

    /**
     * Obtm o diretrio.
     *
     * @return o caminho do diretrio
     */
    public FileInfo getDirectory() {
      return path;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isDirectory() {
      return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean wasCreated() {
      return true;
    }
  }

  /**
   * Encapsula o resultado da cpia de um arquivo.
   */
  public static class CopyFileRecord implements ICopyRecord {
    /**
     * Representao do arquivo fonte.
     */
    private final FileInfo sourceFile;
    /**
     * Representao do arquivo de destino.
     */
    private final FileInfo targetFile;
    /**
     * Indica se o arquivo foi criado ou sobrescrito.
     */
    private final boolean created;

    /**
     * Construtor
     *
     * @param sourceFile arquivo origem
     * @param targetFile arquivo destino
     * @param created indicativo se arquivo foi criado ou sobrescrito.
     */
    private CopyFileRecord(final FileInfo sourceFile, final FileInfo targetFile,
      final boolean created) {
      this.sourceFile = sourceFile;
      this.targetFile = targetFile;
      this.created = created;
    }

    /**
     * Obtm uma representao do arquivo fonte.
     *
     * @return uma representao do arquivo fonte.
     */
    public FileInfo getSourceFile() {
      return sourceFile;
    }

    /**
     * Obtm uma representao do arquivo de destino.
     *
     * @return uma representao do arquivo de destino.
     */
    public FileInfo getTargetFile() {
      return targetFile;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isDirectory() {
      return false;
    }

    /**
     * Indica se o arquivo foi criado ou sobrescrito.
     *
     * @return <tt>true</tt> se o arquivo foi criado ou <tt>false</tt> caso
     *         tenha sido sobrescrito.
     */
    @Override
    public boolean wasCreated() {
      return created;
    }
  }

}
