package csbase.server.services.repositoryservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import csbase.exception.ServiceFailureException;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Repositrio localizado fisicamente na mquina do servidor.
 *
 * NOTA: Essa classe  de visibilidade restrita ao pacote, pois s o
 * {@link RepositoryService} pode manipular suas instncias.
 *
 * @see LocalFile
 *
 * @author Tecgraf/PUC-Rio
 */
class LocalRepository extends AbstractRepository {

  /** Conjunto de arquivos bloqueados. */
  private Set<String> locks;

  /**
   * Construtor.
   *
   * @param uri identificador nico do repositrio.
   */
  LocalRepository(String uri) {
    super(handleURI(uri));

    locks = new HashSet<String>();
  }

  /** {@inheritDoc} */
  @Override
  public IRepositoryFile getFile(String... path) {
    // Evita que seja possvel obter um IRepositoryFile que represente todo
    // o repositrio.
    if (!isValidPath(path)) {
      return null;
    }
    String filePath = FileUtils.joinPath(getFileSeparator(), path);
    String fullPath = FileUtils.joinPath(getFileSeparator(), uri, filePath);
    File file = new File(fullPath);
    if (file.exists()) {
      filePath = clearPath(filePath);
      return new LocalFile(file, this, filePath);
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public char getFileSeparator() {
    return File.separatorChar;
  }

  /**
   * Manipula uma URI passada.
   *
   * @param uri referncia para o repositrio.
   * @return o caminho do repositrio.
   */
  private static String handleURI(String uri) {
    File file = new File(uri);
    return file.getPath();
  }

  /** {@inheritDoc} */
  @Override
  public IRepositoryFile getDir(String... path) {
    IRepositoryFile file = getFile(path);
    return file != null && file.isDirectory() ? file : null;
  }

  /** {@inheritDoc} */
  @Override
  public List<IRepositoryFile> listFiles() {
    File file = new File(uri);
    if (!file.isDirectory()) {
      return null;
    }

    List<IRepositoryFile> files = new LinkedList<IRepositoryFile>();
    for (File child : file.listFiles()) {
      files.add(new LocalFile(child, this, child.getName()));
    }
    return files;
  }

  /** {@inheritDoc} */
  @Override
  public synchronized IRepositoryFile createFile(String[] path, String name)
    throws RepositoryFileException {
    String parentPath;
    if (path == null || path.length == 0) {
      parentPath = ".";
    }
    else {
      parentPath = FileUtils.joinPath(getFileSeparator(), path);
    }
    String fullParrentPath = FileUtils.joinPath(getFileSeparator(), uri,
      parentPath);
    File parent = new File(fullParrentPath);
    File newFile = new File(parent, name);

    if (newFile.exists()) {
      String infoMsg = String.format(
        "J existe um arquivo com o nome %s em %s.", name, parent.getName());
      throw new RepositoryFileException(infoMsg);
    }
    try {
      newFile.createNewFile();

      String relativeFilePath = FileUtils.joinPath(getFileSeparator(),
        parentPath, name);
      relativeFilePath = clearPath(relativeFilePath);
      return new LocalFile(newFile, this, relativeFilePath);
    }
    catch (IOException e) {
      String errMsg = String.format("LocalRepository: erro ao criar %s", newFile
        .getAbsolutePath());
      throw new ServiceFailureException(errMsg, e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public IRepositoryFile createDir(String... path) throws IOException {
    String filePath = FileUtils.joinPath(getFileSeparator(), path);
    String fullPath = FileUtils.joinPath(getFileSeparator(), uri, filePath);

    File file = new File(fullPath);
    if (!file.mkdir()) {
      return null;
    }
    filePath = clearPath(filePath);
    return new LocalFile(file, this, filePath);
  }

  /** {@inheritDoc} */
  @Override
  public boolean renameFile(String srcFilePath, String newFileName)
    throws IOException {
    if (!isValidPath(srcFilePath)) {
      return false;
    }
    String fullPath = FileUtils.joinPath(getFileSeparator(), uri, srcFilePath);
    String targetFullPath = FileUtils.joinPath(getFileSeparator(), uri,
      FileUtils.getFilePath(srcFilePath), newFileName);

    File srcFile = new File(fullPath);
    File destFile = new File(targetFullPath);
    if (!(srcFile.exists())) {
      String msg = String.format(
        "No foi possvel renomear %s: arquivo no existe.", srcFile.getName());
      throw new IOException(msg);
    }
    if (destFile.exists()) {
      if (!(destFile.delete())) {
        String msg = String.format("%s no foi renomeado para %s", srcFile
          .getName(), newFileName);
        throw new IOException(msg);
      }
    }
    if (srcFile.renameTo(destFile)) {
      srcFile.delete();
      return true;
    }
    return false;
  }

  /** {@inheritDoc} */
  @Override
  public boolean removeFile(String... path) throws IOException {
    String filePath = FileUtils.joinPath(getFileSeparator(), path);
    String fullPath = FileUtils.joinPath(getFileSeparator(), uri, filePath);
    File file = new File(fullPath);
    if (!isValidPath(path)) {
      return false;
    }
    IRepositoryFile localFile = new LocalFile(file, this, filePath);
    if (isLocked(localFile)) {
      localFile.close();
      return false;
    }
    return delete(file);
  }

  /**
   * Verifica se o parmetro path passado  vlido. Um path  considerado vlido
   * se no for a raiz do repositrio, nem terminar com ".", "./", ".." ou
   * "../".
   *
   * @param path caminho a ser analizado.
   * @return true se o caminho for a raiz do repositrio.
   */
  private boolean isValidPath(String... path) {
    String filePath = FileUtils.joinPath(getFileSeparator(), path);
    String fullPath = FileUtils.joinPath(getFileSeparator(), uri, filePath);
    File file = new File(fullPath);
    File uriFile = new File(uri);
    if (uriFile.getAbsolutePath().equals(file.getAbsolutePath())) {
      return false;
    }
    String[] patterns = new String[] { ".", "./", "..", "../" };
    for (String suffix : patterns) {
      if (fullPath.endsWith(suffix)) {
        // TODO: Verificar se os testes de JUnit passam para LocalRepository.
        //      if (fullPath.endsWith(suffix) || filePath.startsWith(suffix)) {
        //        System.out.println("[DEBUG] No consegui apagar: " + filePath);
        return false;
      }
    }
    return true;
  }

  /** {@inheritDoc} */
  @Override
  public boolean fileExists(String... path) {
    String filePath = FileUtils.joinPath(getFileSeparator(), path);
    String fullPath = FileUtils.joinPath(getFileSeparator(), uri, filePath);
    File file = new File(fullPath);
    return file.exists();
  }

  /** {@inheritDoc} */
  @Override
  public boolean lock(IRepositoryFile file) {
    synchronized (locks) {
      return locks.add(file.getPath());
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean release(IRepositoryFile file) {
    synchronized (locks) {
      return locks.remove(file.getPath());
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean isLocked(IRepositoryFile file) {
    synchronized (locks) {
      return locks.contains(file.getPath());
    }
  }

  /**
   * Remove um diretrio (ou arquivo), removendo recursivamente o seu contedo
   * (no caso de diretrios).
   *
   * @param file arquivo/diretrio a remover.
   *
   * @return <code>true</code> se o arquivo/diretrio existir e for removido com
   *         sucesso; <code>false</code> caso o arquivo no exista.
   */
  private boolean delete(File file) {
    if (file == null) {
      throw new IllegalArgumentException("file == null");
    }
    if (!file.exists()) {
      return false;
    }
    if (file.isDirectory()) {
      File[] children = file.listFiles();
      if (!(children == null) && (children.length > 0)) {
        for (int i = 0; i < children.length; i++) {
          delete(children[i]);
        }
      }
    }
    return file.delete();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IRepositoryFile copyFile(String srcPath, String[] targetPath)
    throws IOException {
    return copyFile(srcPath, targetPath, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IRepositoryFile copyFile(String srcPath, String[] targetPath, String fileName)
    throws IOException {
    IRepositoryFile origin = getFile(srcPath);
    if (origin == null) {
      throw new IOException("Arquivo de origem no pode ser null.");
    }
    // Verifica se queremos copiar para a raiz ou nao
    String stringTargetPath;
    if (!isRoot(targetPath)) {
      IRepositoryFile target = getDir(targetPath);
      if (target == null) {
        throw new IOException("No foi possvel obter o caminho de destino.");
      }
      if (target.getPath().startsWith(origin.getPath())) {
        throw new IOException(
          "Um diretrio no pode ser copiado para dentro de si mesmo.");
      }
      stringTargetPath = target.getPath();
    }
    else {
      stringTargetPath = FileUtils.joinPath(getFileSeparator(), targetPath);
    }
    IRepositoryFile newFile = null;
    if (origin.isDirectory()) {
      newFile = createDir(stringTargetPath, origin.getName()); 
      if (newFile == null) {
        throw new IOException("No foi possvel criar o novo arquivo.");
      }
      List<IRepositoryFile> children = origin.getChildren();
      for (IRepositoryFile child : children) {
        copyFile(child.getPath(), new String[] { newFile.getPath() }); 
      }
    }
    else {
      String filePath;
      if(fileName != null) {
        filePath = FileUtils.joinPath(getFileSeparator(), stringTargetPath, fileName);
      }
      else {
        filePath = FileUtils.joinPath(getFileSeparator(), stringTargetPath, origin.getName());
      }
      String fullPath = FileUtils.joinPath(getFileSeparator(), uri, filePath);
      File targetFile = new File(fullPath);
      if (!targetFile.createNewFile()) {
        throw new IOException("No foi possvel criar o novo arquivo: " + fullPath);
      }
      filePath = clearPath(filePath);
      newFile = new LocalFile(targetFile, this, filePath);
      ((LocalFile) origin).copyRegularFile(targetFile);
      targetFile.setLastModified(origin.getModificationDate());
    }
    origin.close();
    return newFile;
  }

  /**
   * Limpa os caracteres '.' e '/' iniciais do path relativo.
   *
   * @param filePath o path.
   * @return o path limpo.
   */
  private String clearPath(String filePath) {
    String newFilePath = filePath;
    while (newFilePath.startsWith(String.valueOf(getFileSeparator()))
      || newFilePath.startsWith(".")) {
      newFilePath = newFilePath.substring(1);
    }
    return newFilePath;
  }

  /**
   * Verifica se um caminho passado  referente  raiz do repositrio.
   *
   * @param path caminho a ser verificado.
   * @return true se path for a raiz. false caso contrrio.
   */
  private boolean isRoot(String... path) {
    String stringPath = FileUtils.joinPath(getFileSeparator(), path);
    return String.valueOf(getFileSeparator()).equals(stringPath);
  }

  /**
   * Copia o contedo de um arquivo.
   *
   * @param targetFileStream o fileStrem de destino.
   * @param sourceFileStream o fileStream de origem.
   *
   * @throws IOException Exceo lanada.
   * 
   * TODO: Mtodo para quando for feita a transferncia inter-repsitorios.
   */
  private void copyContent(OutputStream targetFileStream,
    InputStream sourceFileStream) throws IOException {
    // Bloco de tamanho normal 256KB
    final int bufferSize = 256 * 1024;
    BufferedOutputStream out = null;
    BufferedInputStream in = null;
    try {
      out = new BufferedOutputStream(targetFileStream);
      in = new BufferedInputStream(sourceFileStream);
      byte[] buffer = new byte[bufferSize];
      int len = 0;
      while ((len = in.read(buffer)) != -1) {
        out.write(buffer, 0, len);
      }
    }
    finally {
      if (out != null) {
        out.flush();
        out.close();
      }
      if (in != null) {
        in.close();
      }
    }
  }
}
