/*
 * $Id: XMLConversionService.java 174162 2016-06-06 17:18:22Z ururahy $
 */
package csbase.server.services.xmlconversionservice;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.lang.reflect.Constructor;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.Calendar;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.xml.conversion.XMLConverter;
import tecgraf.javautils.xml.conversion.XMLConverter.XMLConversionStatus;
import tecgraf.javautils.xml.conversion.exception.XMLConversionException;
import tecgraf.javautils.xml.conversion.exception.XMLConversionException.XMLConversionExceptionType;
import csbase.exception.BugException;
import csbase.exception.ServiceFailureException;
import csbase.logic.Utilities;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.XMLConversionServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.projectservice.ProjectService;

/**
 * A classe <code>XMLConversionService</code> implementa o servio de converso
 * de arquivos XML. Este servio  usado para converter arquivos XML para
 * torn-los compatveis com a ltima verso do DTD. Isto  feito usando-se
 * {@link XMLConverter conversores XML}
 * <p>
 * <b>IMPORTANTE</b>: atualmente somos forados a sobrescrever o arquivo
 * original ( possvel antes fazer um backup) pois o CSBase no permite
 * transmitirmos dados diretamente por channel do servidor para o cliente sem
 * que este contedo esteja gravado em um <code>ServerProjectFile</code>.
 * <p>
 * Idealmente o resultado da converso seria transferido diretamente do
 * documento DOM em memria para o cliente via rede, sem a necessidade de
 * grav-lo previamente como um arquivo XML. Com isto o cliente poderia ler o
 * arquivo diretamente da rede, marc-lo como "sujo" e deixar a gravao por
 * conta do usurio, no cliente.
 */
public class XMLConversionService extends Service implements
  XMLConversionServiceInterface {

  /**
   * Cria uma instncia do servio.
   * 
   * @throws ServerException se houver algum erro na criao do servio
   */
  public XMLConversionService() throws ServerException {
    super(SERVICE_NAME);
    ClientRemoteLocator.xmlConversionService = this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() throws ServerException {
    // vazio
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void shutdownService() throws ServerException {
    // vazio
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public XMLConversionStatus convertXMLToFile(Object projectId,
    String[] clientPath, boolean validate, String dtdPrefix,
    Class<? extends XMLConverter> clazz) throws RemoteException {
    ProjectService projectService = ProjectService.getInstance();
    /*
     * cada converso instancia um novo conversor; desta forma no temos
     * condies de corrida (a no ser, claro, que dois requests simultneos
     * atuem sobre o mesmo arquivo)
     */
    XMLConversionStatus status = null;
    String xmlData = null;
    OutputStream outputStream = null;
    try {
      XMLConverter converter = createConverter(dtdPrefix, clazz);
      xmlData = readXMLAsString(projectId, clientPath);
      preConvert(projectId, clientPath, dtdPrefix, xmlData);
      status = converter.convert(xmlData, validate).getConversionStatus();
      if (status == XMLConversionStatus.NOT_NEEDED) {
        /*
         * Chama o postConvert de qualquer jeito para ele limpar algo que se
         * faa necessrio
         */
        postConvert(projectId, clientPath, dtdPrefix, outputStream, xmlData,
          status);
        return status;
      }
      outputStream = projectService.getOutputStream(projectId, clientPath);
      converter.saveTo(outputStream);
    }
    catch (IOException e) {
      throw new ServiceFailureException(
        getString("XMLConversionService.error.io"), e);
    }
    catch (Exception e) {
      throw new ServiceFailureException(e.toString(), e);
    }
    finally {
      // Deve chamar o postconvert, mesmo com exceo
      postConvert(projectId, clientPath, dtdPrefix, outputStream, xmlData,
        status);
      if (outputStream != null) {
        FileUtils.close(outputStream);
      }
    }

    /*
     * acrescentamos  descrio a informao de que o arquivo foi convertido
     * com sucesso
     */
    long date = Calendar.getInstance().getTimeInMillis();
    String text =
      MessageFormat.format(getString("XMLConversionService.success.msg"),
        new Object[] { Utilities.getFormattedDate(date),
            Service.getUser().getId() });
    projectService.appendFileDescription(projectId, clientPath, text);
    return status;
  }

  /**
   * Cria o conversor a ser utilizado.
   *
   * @param clazz a classe do conversor a ser utilizada
   * @param dtdPrefix prefixo para o DTD
   * @return o conversor a ser utilizado
   * @throws Exception se houver algum problema na criao do conversor
   */
  private <T extends XMLConverter> T createConverter(String dtdPrefix,
    Class<T> clazz) throws Exception {
    T converter = null;
    try {
      Constructor<T> constructor = clazz.getConstructor(String.class);
      converter = constructor.newInstance(dtdPrefix);
    }
    catch (NoSuchMethodException e) {
      String text =
        MessageFormat.format(
          getString("XMLConversionService.error.impl.no_constructor_found"),
          new Object[] { clazz.getName() });
      throw new BugException(text, e);
    }

    return converter;
  }

  /**
   * Mtodo com as aes de ps converso.
   * 
   * @param projectId - identificador do projeto
   * @param clientPath - path relativo para o arquivo no projeto
   * @param dtdPrefix - string que ser prefixada na declarao do DTD. Pode ser
   *        <code>null</code> caso no se deseje um prefixo especfico
   * @param outputStream - stream do arquivo convertido
   * @param xmlData - contedo do arquivo xml
   * @param status - o status da converso
   */
  protected void postConvert(Object projectId, String[] clientPath,
    String dtdPrefix, OutputStream outputStream, String xmlData,
    XMLConversionStatus status) {
    // implementao default vazia
  }

  /**
   * Executa as aes antes da converso
   * 
   * @param projectId - identificador do projeto
   * @param clientPath - path relativo para o arquivo no projeto
   * @param dtdPrefix - string que ser prefixada na declarao do DTD. Pode ser
   *        <code>null</code> caso no se deseje um prefixo especfico
   * @param xmlData - contedo do arquivo xml
   */
  protected void preConvert(Object projectId, String[] clientPath,
    String dtdPrefix, String xmlData) {
    // implementao default vazia
  }

  /**
   * L os dados de um arquivo XML como uma string.
   * 
   * @param projectId id do projeto
   * @param clientPath path para o arquivo
   * @return string contendo os dados xml do arquivo
   * @throws IOException
   */
  private String readXMLAsString(Object projectId, String[] clientPath)
    throws IOException {
    String xmlData;
    InputStream inputStream =
      ProjectService.getInstance().getInputStream(projectId, clientPath);
    xmlData = ServiceUtils.readXMLAsString(inputStream, getXMLEndMarker());
    return xmlData;
  }

  /**
   * Retorna o marcador da tag final do arquivo XML. Se no for null  porque o
   * arquivo xml possui, aps o fechamento da tag principal, outros contedo de
   * interesse para apliaes especficas. Se for null, ler todo o inputStream,
   * que dever ser apenas XML.
   * 
   * @return marcador da tag final do arquivo XML
   */
  protected String getXMLEndMarker() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String convertXMLToString(Object projectId, String[] clientPath,
    boolean validate, String dtdPrefix, Class<? extends XMLConverter> clazz)
      throws RemoteException {
    /*
     * cada converso instancia um novo conversor; desta forma no temos
     * condies de corrida (a no ser, claro, que dois requests simultneos
     * atuem sobre o mesmo arquivo)
     */
    try {
      XMLConverter converter = createConverter(dtdPrefix, clazz);
      String xmlData = readXMLAsString(projectId, clientPath);
      converter.convert(xmlData, validate);
      return converter.saveToString();
    }
    catch (IOException e) {
      throw new ServiceFailureException(
        getString("XMLConversionService.error.io"), e);
    }
    catch (Exception e) {
      throw new ServiceFailureException(e.toString(), e);
    }
  }

  /**
   * Grava o contedo de um arquivo em um stream de sada.
   *
   * @param src origem dos dados
   * @param outputStream stream de sada
   * 
   * @throws XMLConversionException
   */
  protected void append(File src, OutputStream outputStream)
    throws XMLConversionException {
    BufferedWriter writer = null;
    BufferedReader reader = null;
    try {
      writer = new BufferedWriter(new OutputStreamWriter(outputStream));
      reader = new BufferedReader(new FileReader(src));
      String line = reader.readLine();
      while (line != null) {
        writer.write(line);
        // FIXME estamos fixando o terminador de linha para o UNIX
        writer.write('\n');
        line = reader.readLine();
      }
    }
    catch (Exception e) {
      String msg =
        "Erro: src: " + src.getAbsolutePath() + " - outputStream: "
          + outputStream.toString();
      Server.logSevereMessage(msg, e);
      throw new XMLConversionException(XMLConversionExceptionType.WRITE);
    }
    finally {
      FileUtils.close(reader);
      FileUtils.close(writer);
    }
  }

  /**
   * Retorna a instncia do servio.
   * 
   * @return instncia do servio
   */
  public static XMLConversionService getInstance() {
    return (XMLConversionService) getInstance(SERVICE_NAME);
  }

  /**
   * Cria a instncia do servio.
   * 
   * @throws ServerException
   */
  public static void createService() throws ServerException {
    new XMLConversionService();
  }

}
