package csbase.server.services.ftcservice;

import java.nio.channels.FileChannel;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import tecgraf.ftc.common.exception.FailureException;
import tecgraf.ftc.common.exception.PermissionException;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.ftc.server.FileChannelAccessInfo;
import tecgraf.ftc.server.FileServer;
import tecgraf.ftc.server.FileServerConfig;
import tecgraf.ftc.server.FileServerOwner;
import csbase.server.Server;
import csbase.server.services.repositoryservice.IRepositoryFile;

/**
 * Controlador do servidor FTC do CSBase.
 *
 * @author Tecgraf
 */
class FTCController implements FileServerOwner {
  /**
   * Referncia direta para o servio que criou esse controlador.
   */
  private FTCService service = null;
  /**
   * Configurao do servidor FTC.
   */
  private FileServerConfig config = null;
  /**
   * Servidor FTC.
   */
  private FileServer server = null;
  /**
   * Thread que executa o servidor FTC.
   */
  private Thread serverThread = null;
  /**
   * Thread de limpeza das requisies.
   */
  private Thread cleanThread = null;
  /**
   * Indica se o controlador deve ser interrompido.
   */
  private volatile boolean stopped = false;
  /**
   * Mapa de indentificadores em requisies de acesso a arquivos.
   */
  private Map<byte[], RequestData> requestMap = null;
  /**
   * Tempo mximo que uma requisio ser guardada at ser atendida. Passado
   * esse tempo, caso a requisio seja feita, ela falhar.
   */
  private long remoteFileChannelInfoTimeout = 600000;
  /**
   * Gerador de nmeros aleatrios usado para gerar os identificadores.
   */
  private SecureRandom secureRandom = null;
  /**
   * Nome do algoritmo de gerao de nmeros pseudo-aleatrios.
   */
  private static final String PRNG_ALGORITHM = "SHA1PRNG";
  /**
   * O tamanho a ser usado para os identificadores.
   */
  private static final int ID_SIZE = 127;

  /**
   * Modela a requisio de acesso a um arquivo.
   *
   * @author Tecgraf
   */
  private class RequestData {
    /**
     * O arquivo a ser acessado.
     */
    IRepositoryFile file;
    /**
     * Indicativo de acesso somente leitura ou leitura/escrita.
     */
    boolean readOnly;
    /**
     * Registra o momento de construo desta requisio.
     */
    long t0;
    /**
     * Indica se essa requisio j est sendo usada.
     */
    boolean channelOpened;

    /**
     * Constri a representao de uma requisio de acesso a um arquivo.
     *
     * @param file O arquivo a ser acessado.
     * @param readOnly Indicativo de acesso somente leitura ou leitura/escrita.
     */
    RequestData(IRepositoryFile file, boolean readOnly) {
      this.file = file;
      this.readOnly = readOnly;
      this.t0 = System.currentTimeMillis();
      this.channelOpened = false;
    }

    /**
     * Informa se a requisio j expirou o tempo mximo de espera pela
     * requisio remota, sem que esta tenha sido feita.
     *
     * @return True se expirou o tempo mximo de espera, sem que uma conexo
     *         remota tenha sido estabelecida, ou false caso contrrio.
     */
    boolean expired() {
      if (channelOpened) {
        return false;
      }
      long now = System.currentTimeMillis();
      return (now - t0) > remoteFileChannelInfoTimeout;
    }
  }

  /**
   * Cria um controlador de um servidor FTC.
   *
   * @param service A instncia de FTCService que criou o controlador.
   */
  FTCController(FTCService service) {
    this.service = service;
    config = new FTCConfig();
    requestMap = new HashMap<byte[], RequestData>();
    try {
      secureRandom = SecureRandom.getInstance(PRNG_ALGORITHM);
    }
    catch (NoSuchAlgorithmException e) {
      secureRandom = new SecureRandom();
    }
  }

  /**
   * Dispara a execuo do servidor FTC e da limpeza das requisies, ambos em
   * threads prprias. Essa mtodo no bloqueia.
   *
   * @throws Exception Caso ocorra alguma falha.
   */
  void start() throws Exception {
    stopped = false;
    server = new FileServer(this);
    serverThread = new Thread(service.getName() + " - FTC server") {
      @Override
      public void run() {
        server.dispatch();
      }
    };
    serverThread.start();
    cleanThread = new Thread(service.getName() + " - FTC cleaner") {
      @Override
      public void run() {
        while (!stopped) {
          try {
            Thread.sleep(remoteFileChannelInfoTimeout);
          }
          catch (Throwable t) {
            // do nothing
          }
          clean();
        }
      }
    };
    cleanThread.start();
    Server.logInfoMessage("FTCController iniciado.");
  }

  /**
   * Percorre a estrutura de requisies e remove as que no foram solicitadas
   * dentro do tempo mximo de espera configurado.
   */
  private void clean() {
    synchronized (requestMap) {
      if (requestMap.size() == 0) {
        return;
      }
      Iterator<byte[]> it = requestMap.keySet().iterator();
      while (it.hasNext()) {
        byte[] id = it.next();
        RequestData data = requestMap.get(id);
        if (data.expired()) {
          it.remove();
        }
      }
    }
  }

  /**
   * Solicita a interrupo do servidor FTC. Esse mtodo espera a efetiva
   * interrupo do servidor durante, no mximo, 5s.
   */
  void stop() {
    stopped = true;
    cleanThread.interrupt();
    server.stop();
    synchronized (requestMap) {
      requestMap.clear();
    }
    try {
      serverThread.join(5000);
    }
    catch (InterruptedException e) {
      // do nothing
    }
    Server.logInfoMessage("FTCController finalizado.");
  }

  /**
   * Retorna o objeto que modela a requisio para um dado identificador.
   *
   * @param fileId O identificador da requisio.
   * @return O objeto que representa a requisio.
   */
  private RequestData getRequestData(byte[] fileId) {
    synchronized (requestMap) {
      return requestMap.get(fileId);
    }
  }

  /**
   * Retorna o arquivo de uma requisio de um dado identificador.
   *
   * @param fileId O identificador da requisio.
   * @param remove Indica se a requisio correspondente deve ser removida.
   * @return O arquivo a ser acessado na requisio ou <code>null</code> se o
   *         arquivo no for encontrado
   */
  private IRepositoryFile getFile(byte[] fileId, boolean remove) {
    RequestData data = getRequestData(fileId);
    if (remove) {
      synchronized (requestMap) {
        requestMap.remove(fileId);
      }
    }
    return (data == null ? null : data.file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileChannel createFileChannel(Object requester, byte[] fileId,
    boolean readOnly) throws PermissionException, FailureException {
    FileChannel channel = null;
    try {
      RequestData data = getRequestData(fileId);
      if (data == null) {
        throw new Exception("Identificador no reconhecido.");
      }
      if (data.readOnly && !readOnly) {
        throw new Exception("Tentativa de escrita em arquivo de leitura: "
          + data.file.getPath());
      }

      channel =
        ((FTCRequester) requester).createFileChannel(getFile(fileId, false),
          readOnly);
      data.channelOpened = true;
    }
    catch (Exception e) {
      Server.logSevereMessage(e.getMessage(), e);
    }
    return channel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isLocked(Object requester, byte[] fileId) {
    try {
      return ((FTCRequester) requester).isLocked(getFile(fileId, false));
    }
    catch (Exception e) {
      Server.logSevereMessage(e.getMessage(), e);
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void fileChannelClosed(Object requester, byte[] fileId) {
    try {
      ((FTCRequester) requester).fileChannelClosed(getFile(fileId, true));
    }
    catch (Exception e) {
      Server.logSevereMessage(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileServerConfig getConfig() {
    return config;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setConfig(FileServerConfig config) {
    this.config = config;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void exceptionRaised(Exception e, byte[] fileId) {
    IRepositoryFile file = getFile(fileId, false);
    String msg;
    if (file == null) {
      msg =
        "Identificador de arquivo no encontrado: " + Arrays.toString(fileId);
    }
    else {
      msg = "Falha no acesso ao arquivo " + file.getPath();
    }
    Server.logSevereMessage(msg, e);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void exceptionRaised(Exception e) {
    Server.logSevereMessage(e.getMessage(), e);
  }

  /**
   * Cria um novo identificador para uma requisio de acesso a um arquivo. Os
   * identificadores so aleatrios.
   *
   * @return O novo identificador
   */
  private byte[] nextId() {
    byte[] id = new byte[ID_SIZE];
    secureRandom.nextBytes(id);
    return id;
  }

  /**
   * Cria a informao de canal para acesso a um determinado arquivo.
   *
   * @param requester Representao do fornecedor do arquivo, que  quem pode
   *        solicitar o acesso a ele.
   * @param file Referncia ao arquivo a ser acessado.
   * @param readOnly Indicativo de acesso somente leitura ou leitura/escrita.
   * @return O objeto contendo toda a informao necessria para que um canal
   *         remoto de acesso seja criado para acessar o arquivo em questo.
   * @throws Exception Caso ocorra algum erro na operao.
   */
  RemoteFileChannelInfo createFileChannelInfo(FTCRequester requester,
    IRepositoryFile file, boolean readOnly) throws Exception {
    byte[] fileId = nextId();
    FileChannelAccessInfo srvInfo =
      server.createFileChannelInfo(requester, fileId);
    RequestData data = new RequestData(file, readOnly);
    synchronized (requestMap) {
      requestMap.put(fileId, data);
    }
    String host =
      service.isPropertyNull("hostName") ? Server.getInstance().getHostName()
        : service.getStringProperty("hostName");
      RemoteFileChannelInfo info =
        new RemoteFileChannelInfo(srvInfo.getFileIdentifier(), !readOnly, host,
          srvInfo.getPort(), srvInfo.getAccessKey());
      return info;
  }

  /**
   * Ajusta o tempo mximo que uma requisio ser guardada at ser atendida.
   *
   * @param remoteFileChannelInfoTimeout tempo em milisegundos.
   */
  void setRemoteFileChannelInfoTimeout(long remoteFileChannelInfoTimeout) {
    this.remoteFileChannelInfoTimeout = remoteFileChannelInfoTimeout;
  }
}
