/**
 * $Id: DiskMonitor.java 175310 2016-08-03 14:38:03Z fpina $
 */
package csbase.server.services.diagnosticservice.monitors.disk;

import java.text.NumberFormat;
import java.util.Locale;

import csbase.logic.diagnosticservice.DiskStatus;
import csbase.logic.diagnosticservice.ResourceException;
import csbase.logic.diagnosticservice.Status;
import csbase.logic.diskusageservice.DiskOccupation;
import csbase.remote.DiagnosticServiceInterface;
import csbase.remote.DiskUsageServiceInterface;
import csbase.server.services.diagnosticservice.MessageTranslator;
import csbase.server.services.diagnosticservice.monitors.AbstractMonitor;
import csbase.server.services.diskusageservice.DiskUsageService;

/**
 * Classe abstrata para criar monitores de disco.
 *
 * @author Tecgraf
 */
public class DiskMonitor extends AbstractMonitor {

  /**
   * Construtor.
   *
   * @throws ResourceException caso algum parmetro esteja incorreto
   */
  public DiskMonitor() throws ResourceException {
    super(DiagnosticServiceInterface.SERVER_DISK_RESOURCE);
    this.addValidation(new DiskOcupationValidation());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized Status checkResource(Locale locale)
    throws ResourceException {
    DiskOccupation serverDiskOccupation = DiskUsageService.getInstance()
      .getAreaOccupation(DiskUsageServiceInterface.DIR_SERVER_ID);
    addPropertiesForServerDisk(locale, serverDiskOccupation);
    Status status = super.checkResource(locale);
    return new DiskStatus(serverDiskOccupation, status.getResourceName(), status
      .getCode(), status.getValidationStatusList(), status.getProperties());
  }

  /**
   * Adiciona as propriedades referentes a rea de disco do servidor.
   *
   * @param locale o locale para traduo dos textos
   * @param serverDiskOccupation a ocupao do disco
   */
  private void addPropertiesForServerDisk(Locale locale,
    DiskOccupation serverDiskOccupation) {
    this.cleanProperties();
    double usedSpacePerc = serverDiskOccupation.getUsedSpacePerc() / 100;
    double freeSpacePerc = serverDiskOccupation.getFreeSpacePerc() / 100;
    long usedSpaceGb = (long) serverDiskOccupation.getUsedSpaceMb() / 1024;
    long freeSpaceGb = (long) serverDiskOccupation.getFreeSpaceMb() / 1024;
    long alertLimit = serverDiskOccupation.alertLimit;
    long warningLimit = serverDiskOccupation.warningLimit;
    boolean auditEnabled = DiskUsageService.getInstance().isAuditEnabled();
    boolean mailOnAlert = DiskUsageService.getInstance().isMailOnAlertEnabled();
    int auditInterval = DiskUsageService.getInstance().getAuditIntervalHours();

    this.addProperty(MessageTranslator.getString(
      "server.disksmonitor.server.usage.percentual.label", locale), NumberFormat
        .getPercentInstance(locale).format(usedSpacePerc), null);

    this.addProperty(MessageTranslator.getString(
      "server.disksmonitor.server.free.percentual.label", locale), NumberFormat
        .getPercentInstance(locale).format(freeSpacePerc), null);

    this.addProperty(MessageTranslator.getString(
      "server.disksmonitor.server.usage.gb.label", locale), MessageTranslator
        .getString("server.disksmonitor.server.usage.gb.value", new String[] {
            Long.toString(usedSpaceGb) }, locale), null);

    this.addProperty(MessageTranslator.getString(
      "server.disksmonitor.server.free.gb.label", locale), MessageTranslator
        .getString("server.disksmonitor.server.free.gb.value", new String[] {
            Long.toString(freeSpaceGb) }, locale), null);

    this.addProperty(MessageTranslator.getString(
      "server.disksmonitor.server.alert.percentual.label", locale),
      MessageTranslator.getString(
        "server.disksmonitor.server.alert.percentual.value", new String[] { Long
          .toString(alertLimit) }, locale), "DiskUsageService.alertPercentage");
    ;

    this.addProperty(MessageTranslator.getString(
      "server.disksmonitor.server.warning.percentual.label", locale),
      MessageTranslator.getString(
        "server.disksmonitor.server.warning.percentual.value", new String[] {
            Long.toString(warningLimit) }, locale),
      "DiskUsageService.warningPercentage");

    this.addProperty(MessageTranslator.getString(
      "server.disksmonitor.server.audit.enable", locale), Boolean.toString(
        auditEnabled), "DiskUsageService.enableAudit");

    this.addProperty(MessageTranslator.getString(
      "server.disksmonitor.server.mail.on.alert", locale), Boolean.toString(
        mailOnAlert), "DiskUsageService.mailOnAlert");

    this.addProperty(MessageTranslator.getString(
      "server.disksmonitor.server.audit.interval.label", locale),
      MessageTranslator.getString(
        "server.disksmonitor.server.audit.interval.value", new String[] {
            Integer.toString(auditInterval) }, locale),
      "DiskUsageService.auditIntervalHours");
  }
}
