/**
 * $Id: LogAdministrationService.java 176168 2016-09-22 21:12:51Z fpina $
 */
package csbase.server.services.logadministrationservice;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.exception.RequestInternalException;
import csbase.exception.PermissionException;
import csbase.logic.LogFile;
import csbase.logic.LogFileInfo;
import csbase.logic.User;
import csbase.logic.diagnosticservice.LogType;
import csbase.remote.LogAdministrationServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.ftcservice.FTCService;
import csbase.server.services.mailservice.MailService;

/**
 * A classe <code>LogAdministrationService</code> implementa o servio de
 * administrao dos arquivos de log do servidor.
 * 
 * @author Tecgraf
 */
public class LogAdministrationService extends Service implements
  LogAdministrationServiceInterface {

  /**
   * Tipos de logs e seus respctivos diretrios
   */
  Map<String, LogType> logType = new Hashtable<String, LogType>();

  /**
   * Construtor
   * @throws ServerException em caso de erro de construo do servio.
   */
  protected LogAdministrationService() throws ServerException {
    super(SERVICE_NAME);
  }

  /**
   * Obtm uma instncia do servio.
   * 
   * @return LogAdministrationService
   */
  public static LogAdministrationService getInstance() {
    return (LogAdministrationService) getInstance(SERVICE_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void initService() throws ServerException {
    Server.logInfoMessage("Iniciando o servio de Admnistrao de Log!");

    List<String> logTypes = getStringListProperty("log.type.name");
    List<String> logDirs = getStringListProperty("log.type.dir");

    if (logTypes.size() < logDirs.size()) {
      Server
        .logSevereMessage("Propriedade com o nome de um tipo de log no definida.");
    }

    if (logTypes.size() > logDirs.size()) {
      Server
        .logSevereMessage("Propriedade com o diretrio de um tipo de log no definida.");
    }
    try {
      for (int i = 0; i < Math.min(logTypes.size(), logDirs.size()); i++) {
        File dir = new File(logDirs.get(i));
        if (dir.exists()) {
          logType.put(logTypes.get(i),
            new LogType(logTypes.get(i), dir.getCanonicalPath()));
        }
        else {
          Server.logSevereMessage("Diretrio " + logDirs.get(i)
            + " no existe.");
        }
      }
    }
    catch (IOException e) {
      Server
        .logSevereMessage("Erro ao ler o diretrio da propriedade LogAdministrationService.log.type.dir.");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void shutdownService() throws ServerException {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return false;
  }

  /**
   * Criao do servio
   *
   * @throws ServerException em caso de falha de criao.
   */
  public static void createService() throws ServerException {
    new LogAdministrationService();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public LogType[] getLogTypes() throws RemoteException {
    List<String> typeList = getStringListProperty("log.type.name");
    List<LogType> logTypeList = new LinkedList<LogType>();

    for (String type : typeList) {
      if (logType.containsKey(type)) {
        logTypeList.add(logType.get(type));
      }
    }

    return logTypeList.toArray(new LogType[0]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public LogFileInfo[] listLogFiles(String type, String path)
    throws RemoteException {
    //Verifica se o path recebido viola a regra de apenas listar os diretrios
    //filhos do diretrio inicial (configurado na propriedade)
    String initialPath = logType.get(type).dir;
    if (initialPath.startsWith(path) && path.length() < initialPath.length()) {
      return null;
    }
    File dir = new File(path);

    List<LogFileInfo> filenames = new ArrayList<LogFileInfo>();
    for (File file : dir.listFiles()) {
      try {
        filenames.add(new LogFileInfo(file.getName(), file.isDirectory(), file
          .getCanonicalPath(), file.length(), file.lastModified()));
      }
      catch (IOException e) {
        Server.logSevereMessage("No foi possvel listar os arquivos de log");
        e.printStackTrace();
      }
    }
    return filenames.toArray(new LogFileInfo[filenames.size()]);
  }

  /**
   * Gera um arquivo zip a partir de uma lista de arquivos
   * 
   * @param zipFileName nome do arquivo zip a ser gerado
   * @param files lista de arquivos para adicionar no arquivo zip geado
   * @return arquivo zip gerado
   */
  private File zipFiles(final String zipFileName, final List<File> files) {

    try {
      ZipOutputStream zipOut =
        new ZipOutputStream(new FileOutputStream(zipFileName));
      for (File file : files) {
        addToZip(zipOut, file, "");
      }
      zipOut.close();
      return new File(zipFileName);
    }
    catch (IOException e) {
      throw new RequestInternalException("Erro no envio de logs", e);
    }
  }

  /**
   * Adiciona recursivamente arquivos ao um arquivo zip.
   * 
   * @param zipOut Stream de sada do arquivo zip
   * @param file arquivos a ser adicionado
   * @param relativePath caminho relativo do arquivo a ser adicionado ao arquivo
   *        zip
   * 
   * @throws IOException
   */
  private void addToZip(ZipOutputStream zipOut, File file, String relativePath)
    throws IOException {
    if (file.isDirectory()) {
      String[] children = file.list();
      if (children.length == 0) {
        zipOut.putNextEntry(new ZipEntry(relativePath + file.getName()
          + File.separator));
      }
      else {
        for (String child : children) {
          addToZip(zipOut, new File(file.getAbsolutePath() + File.separator
            + child), relativePath + file.getName() + File.separator);
        }
      }
    }
    else {
      FileInputStream in = new FileInputStream(file);
      zipOut.putNextEntry(new ZipEntry(relativePath + file.getName()));
      byte[] data = new byte[in.available()];
      in.read(data);
      zipOut.write(data);
      zipOut.closeEntry();
      in.close();
    }
  }

  /**
   * Gera o nome do arquivo zip contendo os arquivos de log. O formato do
   * arquivo segue o segunte padro: <NOME_DO_SERVIDOR>_<DATA>.zip, com DATA no
   * formato YYYYMMDDHHMMSS.
   * 
   * @return nome do arquivo zip
   */
  private String getZipFileName() {
    String zipFileName =
      Server.getInstance().getHostName() + "_"
        + csbase.logic.Utilities.getFormattedDate(new Date(), "yyyyMMddHHmmss");
    String tempDir = System.getProperty("java.io.tmpdir");
    if (!(tempDir.endsWith("/") || tempDir.endsWith("\\"))) {
      tempDir = tempDir.concat(System.getProperty("file.separator"));
    }
    zipFileName = tempDir.concat(zipFileName).concat(".zip");

    return zipFileName;
  }

  /**
   * Gera o arquivo que ser anexado ao email
   * 
   * @param filesPaths caminhos para os arquivos de log que devem ser anexados
   * @return lista de arquivos zipados
   */
  private File[] makeAttachement(String[] filesPaths) {
    List<File> logFiles = new ArrayList<File>();
    for (String filePath : filesPaths) {
      logFiles.add(new File(filePath));
    }
    String zipFileName = getZipFileName();
    File zipFile = zipFiles(zipFileName, logFiles);
    return new File[] { zipFile };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean sendlLogFilesToSuport(final String body,
    final String[] filesPaths) {
    MailService mailService = MailService.getInstance();
    if (filesPaths.length == 0) {
      return mailService.mailSupport(body);
    }
    else {
      return mailService.mailSupport(body, makeAttachement(filesPaths));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean sendlLogFilesToAdmin(final String body,
    final String[] filesPaths) {
    MailService mailService = MailService.getInstance();

    Vector<Object> ids = new Vector<Object>();
    try {
      ids.addAll(User.getAdminIds());
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro ao obter a lista de usurios adminstradores.", e);
    }

    if (filesPaths.length == 0) {
      return mailService.mailSomeUsers(null, ids.toArray(new Object[0]), body);
    }
    else {
      return mailService.mailSomeUsers(null, ids.toArray(new Object[0]), body,
        makeAttachement(filesPaths));
    }
  }

  /**
   * Check de download.
   * @throws PermissionException em caso de falha de permisso.
   */
  protected void checkDownloadLogFilePermission() throws PermissionException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public LogFile[] downloadLogFile(final String[] paths) throws RemoteException {
    checkDownloadLogFilePermission();

    List<LogFile> files = new LinkedList<LogFile>();
    for (String path : paths) {
      try {
        files.add(createLogFile(new File(path)));
      }
      catch (Exception e) {
        Server.logWarningMessage("Erro ao exportar arquivo de log " + path);
      }
    }
    return files.toArray(new LogFile[files.size()]);
  }

  /**
   * Cria um LogFile referente ao arquivo recebido.
   * 
   * @param file arquivo
   * 
   * @return LogFile criado
   * 
   * @throws Exception em caso de erro na criao do canal
   */
  LogFile createLogFile(File file) throws Exception {
    RemoteFileChannelInfo channel = null;
    LogFile[] children = new LogFile[0];

    if (file.isDirectory()) {
      List<LogFile> childrenList = new LinkedList<LogFile>();
      for (File f : file.listFiles()) {
        childrenList.add(createLogFile(f));
      }
      children = childrenList.toArray(new LogFile[childrenList.size()]);
    }
    else {
      FTCService ftcService = FTCService.getInstance();
      channel = ftcService.createFileChannelInfo(file, true);
    }

    return new LogFile(file.getName(), file.isDirectory(), channel, children);
  }
}
