/*
 * ORBHandler.java
 *
 * $Id: ORBHandler.java 176168 2016-09-22 21:12:51Z fpina $
 */
package csbase.server.services.sgaservice;

import java.util.Properties;
import java.util.Set;

import org.omg.CORBA.ORB;
import org.omg.CORBA.Policy;
import org.omg.CORBA.SetOverrideType;
import org.omg.PortableServer.IdAssignmentPolicyValue;
import org.omg.PortableServer.LifespanPolicyValue;
import org.omg.PortableServer.POA;
import org.omg.PortableServer.POAHelper;
import org.omg.PortableServer.Servant;

import sgaidl.SGADaemon;
import csbase.server.Server;
import csbase.server.services.sgaservice.ssl.SSLContext;

/**
 * A classe <code>ORBHandler</code> implementa um servidor CORBA persistente,
 * isto , um servidor cujos objetos TESTE possuem referncias persistentes.
 */
public class ORBHandler extends Thread {
  /** Instncia de ORB utilizada pelo servidor */
  private ORB orb = null;
  /** Root POA */
  private POA rootPOA = null;
  /** POA persistente para os objetos CORBA ativados pelo servidor */
  private POA persistentPOA = null;
  /**
   * Indica se o servidor CORBA est ativo. O servidor CORBA associado a uma
   * instncia de ORBHandler  considerado desativado (por precauo) aps a
   * execuo de um shutdown.
   */
  private boolean isActive;
  /**
   * Indica se o suporte a SSL est ativo.
   */
  private boolean enableSSL = false;

  /**
   * Inicializa o servidor CORBA persistente. Instancia um ORB para o servidor e
   * cria um POA persistente para os objetos CORBA que sero associados ao
   * servidor. O nome do servidor persistente, fornecido como parmetro, ser
   * utilizado para nomear o POA persistente. Em alguns ORBs (como o JacORB)
   * tanto o nome do servidor como o nome do POA so utilizados na composio
   * das referncias aos objetos CORBA. A atribuio de uma porta "bem
   * conhecida" ao servidor (fornecida tambm como parmetro) permite a
   * construo de referncias aos objetos CORBA ativados pelo servidor sem que
   * seja necessria a utilizao de um servio de nomes.
   *
   * @param props propriedades do ORB (configuradas externamente)
   *
   * @throws ORBException
   */
  private void init(final Properties props) throws ORBException {

    //Propriedade preventiva para evitar bug de no recuperao aps queda.
    props.setProperty(
      "jacorb.connection.client.disconnect_after_systemexception", "false");

    final Set<Object> keySet = props.keySet();
    for (Object key : keySet) {
      final String value = props.getProperty((String) key);
      Server.logInfoMessage("Propriedade ORB: " + key + " = " + value);
    }

    final String serverName = props.getProperty("jacorb.implname");

    // Instancia o ORB e cria um POA persistente
    try {
      String[] args = null;
      orb = ORB.init(args, props);
      rootPOA = POAHelper.narrow(orb.resolve_initial_references("RootPOA"));
      org.omg.CORBA.Policy[] policies = new org.omg.CORBA.Policy[2];
      policies[0] =
        rootPOA.create_id_assignment_policy(IdAssignmentPolicyValue.USER_ID);
      policies[1] =
        rootPOA.create_lifespan_policy(LifespanPolicyValue.PERSISTENT);
      persistentPOA =
        rootPOA.create_POA(serverName + "POA", rootPOA.the_POAManager(),
          policies);
      for (int i = 0; i < policies.length; i++) {
        policies[i].destroy();
      }
      persistentPOA.the_POAManager().activate();
      // Cria a thread fornecida ao ORB para atendimento s
      // requisies de servio aos objetos CORBA
      start();
    }
    catch (org.omg.CORBA.UserException e) {
      throw new ORBException("Erro na inicializao de servidor CORBA", e);
    }
  }

  /**
   * Inicia o ORB com suporte a SSL.
   *
   * @param props propriedades do ORB (configuradas externamente)
   *
   * @throws ORBException
   */
  private void initWithSSL(final Properties props) throws ORBException {
    init(props);
    SSLContext sslContext = SSLContext.getSSLContext(orb);
    sslContext.setORB(orb);
  }

  /**
   * Ativa um objeto CORBA associado ao servidor.
   *
   * @param objectID identificao do objeto
   * @param servant implementao do objeto (servant)
   * @return o IOR do objeto
   *
   * @throws ORBException em caso de falha no ORB
   */
  public String activateObject(String objectID, Servant servant)
    throws ORBException {
    // Verifica se o servidor est ativo
    if (!isActive) {
      throw new ORBException("Servidor CORBA desativado");
    }

    // Ativa o objeto
    try {
      persistentPOA.activate_object_with_id(objectID.getBytes(), servant);
      org.omg.CORBA.Object obj = persistentPOA.servant_to_reference(servant);
      return orb.object_to_string(obj);
    }
    catch (org.omg.CORBA.UserException e) {
      throw new ORBException("Erro na ativao de um objeto CORBA", e);
    }
  }

  /**
   * Dedica uma thread ao procedimento de atendimento a requisies de servio
   * do ORB. Ao receber um comando de shutdown, o ORB devolve o controle. Nesse
   * caso, a instncia de ORB ser destruda.
   */
  @Override
  public void run() {
    Server.logFineMessage("Thread iniciada: servidor");
    orb.run();
    orb.destroy();
    Server.logFineMessage("Thread terminada: servidor");
  }

  /**
   * Solicita a desativao do servidor CORBA. Comanda o shutdown do ORB
   * instanciado para o servidor, o que provocar a terminao da thread
   * dedicada ao ORB.
   *
   * @throws ORBException em caso de falha no ORB
   */
  public void shutdown() throws ORBException {
    // Verifica se o servidor est ativo
    if (!isActive) {
      throw new ORBException("Servidor CORBA desativado");
    }

    // Comanda o shutdown do ORB
    orb.shutdown(false);
    isActive = false;
  }

  /**
   * Configura o timeout sobre as chamadas no objeto CORBA.
   *
   * @param ref Referncia ao objeto.
   * @param timeout Tempo em segundos.
   *
   * @throws Exception Caso no seja possvel configurar o timeout.
   */
  void setTimeOut(SGADaemon ref, long timeout) throws Exception {
    // O tempo  dado unidades de 100 nanosegundos.
    Policy p =
      new org.jacorb.orb.policies.RelativeRoundtripTimeoutPolicy(
        10000000 * timeout);
    ref._set_policy_override(new Policy[] { p }, SetOverrideType.ADD_OVERRIDE);
  }

  /**
   * Constri um servidor CORBA persistente associado ao nome e porta fornecidos
   * como parmetros. Uma instncia de ORB e um POA persistente so criados para
   * este servidor.
   *
   * @param props Propriedades externas do ORB
   * @param enableSSL indica se o suporte a SSL deve ser ativado
   *
   * @throws ORBException em caso de falha.
   */
  public ORBHandler(final Properties props, boolean enableSSL)
    throws ORBException {
    orb = null;
    rootPOA = null;
    persistentPOA = null;
    this.enableSSL = enableSSL;
    if (this.enableSSL) {
      initWithSSL(props);
    }
    else {
      init(props);
    }
    isActive = true;
  }
}
