package csbase.server.services.opendreamsservice.opendreams.v2_0;

import csbase.exception.OperationFailureException;
import csbase.server.Server;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.server.services.schedulerservice.Base32IdGenerator;
import org.omg.PortableServer.POA;
import org.omg.PortableServer.POAPackage.ServantNotActive;
import org.omg.PortableServer.POAPackage.WrongPolicy;
import tecgraf.openbus.DRMAA.v2_0.*;
import tecgraf.openbus.DRMAA.v2_0.JobTemplatePackage.PARAMETRIC_INDEX;
import tecgraf.openbus.DRMAA.v2_0.JobTemplatePackage.PARAMETRIC_NUMBER_OF_JOBS;
import tecgraf.openbus.DRMAA.v2_0.UnsupportedOperationException;
import tecgraf.openbus.opendreams.v2_0.OpenDreamsJobTemplate;
import tecgraf.openbus.opendreams.v2_0.OpenDreamsJobTemplateImpl;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

public class OpenDreamsJobArray implements JobArrayOperations, Serializable {

	/**
	 *
	 */
	private String jobArrayId;

	/**
	 *
	 */
	private String sessionName;

	/**
	 *
	 */
	private OpenDreamsJobTemplate jt;

	/**
	 *
	 */
	private HashMap<String, OpenDreamsJob> jobs;

	/**
	 *
	 */
	private transient JobArray ref;

	/**
	*
	*/
	private transient Job[] jobRefs;

	/**
	 * @param sessionName
	 * @param jt
	 */
	public OpenDreamsJobArray(String sessionName, String userId, OpenDreamsJobTemplate jt) {
		this.sessionName = sessionName;
		this.jt = jt;
		this.jobArrayId = generateUniqueId(userId, sessionName);
		this.jobs = new HashMap<String, OpenDreamsJob>();
	}

	@Override
	public String jobArrayId() {
		return jobArrayId;
	}

	public OpenDreamsJob getJob(String jobId) {
		return jobs.get(jobId);
	}

	public HashMap<String, OpenDreamsJob> jobsMap() {
		return jobs;
	}

	@Override
	public Job[] jobs() {
		if (jobRefs == null) {
			int jobCounter = 0;
			jobRefs = new Job[jobs.entrySet().size()];
			for (Map.Entry<String, OpenDreamsJob> job : jobs.entrySet()) {
				try {
					jobRefs[jobCounter++] = job.getValue().corbaObjectReference();
				} catch (OperationFailureException | ServantNotActive | WrongPolicy e) {
					String msg = "Erro ao criar referncias do Job " + job.getValue().jobId()
							+ " que pertence ao JobArray " + jobArrayId;
					Server.logSevereMessage(msg, e);
					e.printStackTrace();
				}
			}
		}
		return jobRefs;
	}

	@Override
	public String sessionName() {
		return sessionName;
	}

	@Override
	public JobTemplate jobTemplate() {
		return jt;
	}

	public void execute(String userId, int beginIndex, int endIndex, int step)
			throws InvalidJobTemplateException, InvalidArgumentException, CloneNotSupportedException, ServantNotActive,
			WrongPolicy, OperationFailureException {

		System.out.println("Executing JobArray " + jobArrayId);
		validateIndexes(beginIndex, endIndex, step);

		int numberOfJobs = ((endIndex - beginIndex) / step) + 1;
		int jobCounter = 0;
		String baseOutputPath = jt.outputPath;
		jobRefs = new Job[numberOfJobs];

		for (int i = beginIndex; i <= endIndex; i += step) {
			OpenDreamsJobTemplate jtClone = (OpenDreamsJobTemplate) ((OpenDreamsJobTemplateImpl) jt).clone();
			// Check parametrized JobTemplate fields
			if (baseOutputPath != null && baseOutputPath.length() > 0) {
				System.out.println(baseOutputPath);
				jtClone.outputPath = baseOutputPath.replaceAll(Pattern.quote(PARAMETRIC_INDEX.value),
						String.valueOf(i));
				jtClone.outputPath = jtClone.outputPath.replaceAll(Pattern.quote(PARAMETRIC_NUMBER_OF_JOBS.value),
						String.valueOf(numberOfJobs));
				System.out.println(jtClone.outputPath);
			}
			if (jtClone.jobParameters != null) {
				for (String[] parameters : jtClone.jobParameters) {
					parameters[1] = parameters[1].replaceAll(Pattern.quote(PARAMETRIC_INDEX.value), String.valueOf(i));
					parameters[1] = parameters[1].replaceAll(Pattern.quote(PARAMETRIC_NUMBER_OF_JOBS.value),
							String.valueOf(numberOfJobs));
				}
			}
			OpenDreamsJob job = new OpenDreamsJob(sessionName, jtClone);
			job.execute(userId);
			jobs.put(job.jobId(), job);
			jobRefs[jobCounter++] = job.corbaObjectReference();
		}
	}

	private void validateIndexes(int beginIndex, int endIndex, int step) throws InvalidArgumentException {
		if (beginIndex < 1) {
			throw new InvalidArgumentException("O valor do argumento beginIndex deve ser maior que 1");
		}
		if (beginIndex > endIndex) {
			throw new InvalidArgumentException(
					"O valor do argumento beginIndex deve ser menor ou igual ao valor do argumento endIndex");
		}
		if (step <= 0) {
			throw new InvalidArgumentException("O valor do argumento step deve ser positivo");
		}
	}

	/**
	 * Criao de um jobArrayId nico.
	 * 
	 * @param userId
	 *            Identificador do usurio
	 * @param sessionName
	 *            Identificador da sesso
	 *
	 * @return uma string que representa um JobArray nico.
	 */
	private String generateUniqueId(String userId, String sessionName) {
		return String.format("%s@%s.%s.%s", userId, sessionName, Server.getInstance().getSystemName(),
				new Base32IdGenerator().generateId());
	}

	/**
	 * @return a stub object that references the actual server's object
	 * @throws OperationFailureException
	 * @throws ServantNotActive
	 * @throws WrongPolicy
	 */
	public JobArray corbaObjectReference() throws OperationFailureException, ServantNotActive, WrongPolicy {
		POA poa = OpenBusService.getInstance().getRootPOA();
		if (ref == null) {
			JobArrayPOATie tie = new JobArrayPOATie(this, poa);
			org.omg.CORBA.Object obj = poa.servant_to_reference(tie);
			ref = JobArrayHelper.narrow(obj);
		}
		return ref;
	}

	@Override
	public void suspend() throws UnsupportedOperationException {
		throw new UnsupportedOperationException("Operao no suportada.");
	}

	@Override
	public void resume() throws UnsupportedOperationException {
		throw new UnsupportedOperationException("Operao no suportada.");
	}

	@Override
	public void hold() throws UnsupportedOperationException {
		throw new UnsupportedOperationException("Operao no suportada.");
	}

	@Override
	public void release() throws UnsupportedOperationException {
		throw new UnsupportedOperationException("Operao no suportada.");
	}

	@Override
	public void terminate() throws DrmCommunicationException, DeniedByDrmsException, InternalException {
		for (Map.Entry<String, OpenDreamsJob> job : jobs.entrySet()) {
			job.getValue().terminate();
		}
	}

	@Override
	public void reap() throws UnsupportedOperationException {
		throw new UnsupportedOperationException("Operao no suportada.");
	}
}
