package csbase.server.services.schedulerservice;

import csbase.logic.CommandInfo;
import csbase.logic.SGAInfo;
import csbase.logic.SGASet;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.server.services.schedulerservice.filter.SGAFilterByCommandRequirements;

import java.rmi.RemoteException;
import java.util.*;
import java.util.Map.Entry;

public class NewPercCPUPolicy implements NewSchedulerPolicyInterface {

	List<CommandInfo> commands = new ArrayList<CommandInfo>();
	List<SGASet> servers = new ArrayList<SGASet>();
	List<SchedulerPolicyRestriction> restrictions = new ArrayList<SchedulerPolicyRestriction>();

	@Override
	public void init(List<CommandInfo> cmds, List<SGASet> sgas) {
		commands.clear();
		commands.addAll(cmds);
		Collections.sort(commands,
				Collections.reverseOrder(new GreedCommandComparator()));
		servers.clear();
		servers.addAll(sgas);
		
	}

	@Override
	public Entry<CommandInfo, List<SGASet>> getAllocation() {
		if (commands.isEmpty()||servers.isEmpty())
			return null;
		
		for (CommandInfo currCmd : commands) {
			List<SGASet> filteresSGAs = SGAFilterByCommandRequirements.filter(currCmd,
					servers);
			if (filteresSGAs == null || filteresSGAs.isEmpty()) {
				continue;
			}

			List<SGASet> availableServers = filterUnavailableServers(filteresSGAs);
			if (availableServers == null || availableServers.isEmpty()) {
				continue;
			}
			
			List<SGASet> orderedServers = getSortedSGAByCPU(availableServers);
			if (orderedServers.size() == 0) {
				continue;
			}

			return new AbstractMap.SimpleEntry<CommandInfo, List<SGASet>>(
					currCmd, orderedServers);
		}
		return null;
	}

	@Override
	public void addRestriction(SchedulerPolicyRestriction restriction) {
		restrictions.add(restriction);
	}

	@Override
	public void removeCommand(CommandInfo cmd) {
		commands.remove(cmd);
	}

	@Override
	public void notifySucessfulAllocation(CommandInfo cmd, SGASet server) {
		commands.remove(cmd);
		SGASet choosedServerCopy = new SGASet(updateSGAInfo(cmd,
				server.getMainInfo()), server.getName(),
				server.getEnabled(), server.getAlive(),
				server.hasDiskAccess(),
				server.isBackoffExpired(),
				server.getJobsInfo(),
				server.isCSFSEnabled(), null, System.currentTimeMillis());
		
		servers.remove(server);
		servers.add(choosedServerCopy);
	}
	
	private SGAInfo[] updateSGAInfo(CommandInfo cmd, SGAInfo info) {
		int ramInfo = 0;
		try {
			double sgaFreeRam = info.getRAMFreeMemoryMb();
			double commandUseRam = cmd.getConfigurator().getMemoryAmount();
			ramInfo = (int) (sgaFreeRam - commandUseRam);
		} catch (RemoteException e) {
		}
		SGAInfo newInfo = new SGAInfo(info.getHostName(), info.getPlatformId(), info.getNumProcessors(), info.getRAMMemoryInfoMb(), 0, info.getClockSpeedMHz(), info.getFileSeparator(), info.getProjectRootDirectory(), info.getAlgorithmRootDirectory(), info.getSandboxRootDirectory(), null);
		newInfo.setAlive(true);
		newInfo.setRAMFreeMemory(ramInfo);
		Set<String> requirements = info.getRequirements();
		for (String req:requirements)
			newInfo.addRequirement(req);
		SGAInfo[] infos = new SGAInfo[]{newInfo};
		//System.out.println("Memory: "+ramInfo);
		return infos;
	}

	private List<SGASet> getSortedSGAByCPU(List<SGASet> servers) {
		List<SGASet> sgasSets = new ArrayList<SGASet>();
		sgasSets.addAll(servers);
		Collections.sort(sgasSets, new SGASetCPUComparator());
		return sgasSets;
	}

	private List<SGASet> filterUnavailableServers(List<SGASet> servers) {
		List<SGASet> filteredServers = new ArrayList<SGASet>();
		for (SGASet sga : servers)
			if (checkServerAvailability(sga))
				filteredServers.add(sga);
		return filteredServers;
	}

	private boolean checkServerAvailability(SGASet server) {
		for (SchedulerPolicyRestriction restriction : restrictions)
			if (!restriction.isSGAAvailable(server))
				return false;
		return true;
	}

}

class GreedCommandComparator implements Comparator<CommandInfo> {
	@Override
	public int compare(CommandInfo o1, CommandInfo o2) {
		try {
			AlgorithmConfigurator conf1 = o1.getConfigurator();
			AlgorithmConfigurator conf2 = o2.getConfigurator();

			int prior1 = o1.getPriority().ordinal();
			int prior2 = o2.getPriority().ordinal();

			float mem1 = conf1.getMemoryAmount();
			float mem2 = conf2.getMemoryAmount();

			float cpu1 = conf1.getCpuAmount();
			float cpu2 = conf2.getCpuAmount();

			int numReq1 = conf1.getRequirements().size();
			int numReq2 = conf2.getRequirements().size();

			if (prior1 > prior2)
				return -1;
			else if (prior1 < prior2)
				return 1;
			else if (cpu1 < cpu2)
				return -1;
			else if (cpu1 > cpu2)
				return 1;
			else if (mem1 < mem2)
				return -1;
			else if (mem1 > mem2)
				return 1;
			else if (numReq1 < numReq2)
				return -1;
			else if (numReq1 > numReq2)
				return 1;
			return 0;

		} catch (RemoteException e) {
			return 0;
		}
	}
}
