/*
 * $Id: MailService.java 124798 2011-12-12 15:54:32 +0000 (Mon, 12 Dec 2011)
 * clinio $
 */
package csbase.server.services.mailservice;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.regex.Pattern;

import javax.activation.DataHandler;
import javax.activation.FileDataSource;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

import csbase.logic.User;
import csbase.remote.MailServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;

/**
 * A classe <code>MailService</code> implementa o servio de envio e-mails a
 * partir do servidor.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class MailService extends Service implements MailServiceInterface {

  /**
   * Nome da propriedade que define a porta para envio de e-mails SMTP.
   * 
   * O nome desta propriedade bate com o usado para o uso da sesso em
   * {@link Session#getDefaultInstance(Properties, javax.mail.Authenticator)}.
   * 
   * @see #mailHostPort
   */
  private static final String MAIL_SMTP_PORT_PROPERTY_NAME = "mail.smtp.port";

  /**
   * Nome da propriedade que define a mquina para envio de e-mails SMTP
   * 
   * O nome desta propriedade bate com o usado para o uso da sesso em
   * {@link Session#getDefaultInstance(Properties, javax.mail.Authenticator)}.
   * 
   * @see #mailHostName
   */
  private static final String MAIL_SMTP_HOST_PROPERTY_NAME = "mail.smtp.host";

  /**
   * Pattern representando um e-mail: <qualquer coisa diferente de
   * espao>@<qualquer coisa diferente de espao>
   */
  private static final Pattern _MAIL_PTTN = Pattern
    .compile("^[^\\s@]+@[^\\s@]+$");

  /**
   * Sesso junto ao servidor de mails que  definido com base na propriedade
   * ajustada pelas propriedades do servio.
   * 
   * @see #createSession()
   */
  private Session session = null;

  /**
   * Propriedade de e-mail de suporte
   */
  final private String[] supportTo;

  /**
   * Propriedade de remetente de e-mail de suporte.
   */
  final private String defaultSender;

  /**
   * Propriedade que define a mquina para envio de e-mails SMTP
   */
  final private String mailHostName;

  /**
   * Propriedade que define a porta para envio de e-mails SMTP
   */
  final private int mailHostPort;

  /**
   * Consulta ao email do destinatrio
   * 
   * @param userId o identificador do usurio
   * @return o email do usurio
   */
  private String[] getUserMails(final Object userId) {
    try {
      if (userId == null) {
        Server.logSevereMessage("Detectado id de usurio nulo.");
        return null;
      }
      final User user = User.getUser(userId);
      if (user == null) {
        Server.logSevereMessage("Detectado usurio nulo para: " + userId);
        return null;
      }
      return user.getEmails();
    }
    catch (Exception ue) {
      Server.logSevereMessage("Falha na deteco de email do usurio " + userId
        + "!");
      return null;
    }
  }

  /**
   * Criao da sesso no servidor de mail.
   * 
   * @return <code>true</code> se a sesso foi criada com sucesso
   * 
   * @see #session
   */
  private boolean createSession() {
    final String fmt = "Servidor de mail - %s:%d";
    final String msg = String.format(fmt, mailHostName, mailHostPort);
    Server.logInfoMessage(msg);

    final Properties props = new Properties();
    props.put(MAIL_SMTP_HOST_PROPERTY_NAME, mailHostName);
    // O objeto colocado precisa ser uma string, seno o session
    // no  inicializado corretamente (usa sempre o default 25).
    final String mailHostPortStr = String.format("%d", mailHostPort);
    props.put(MAIL_SMTP_PORT_PROPERTY_NAME, mailHostPortStr);
    session = Session.getDefaultInstance(props, null);
    if (session == null) {
      Server
        .logSevereMessage("No foi possvel instanciar uma sesso de e-mail");
      return false;
    }
    return true;
  }

  /**
   * Mtodo de acesso ao servidor de emails
   * 
   * @param from o remetente
   * @param addressees os destinatrios separados por vrgula.
   * @param subject o assunto
   * @param textContent a string enviada
   * @param replyTo endereo para resposta
   * @param copyTo destinatrios para envio de cpias da mensagem (campo "cc:")
   * @param attachments a lista de arquivos a ser anexada ao email
   * @param mimeType Tipo MIME do e-mail. 
   * 
   * @return flag indicativo de sucesso.
   */
  private boolean accessMailServer(final String from,
    final String[] addressees, final String subject, final String textContent,
    final String replyTo, final String[] copyTo, final File[] attachments, String mimeType) {
    if (session == null) {
      return false;
    }

    String sender = createValidSender(from);
    try {
      // Quebra a string contendo os destinatrios.
      List<InternetAddress> addresseesList = new ArrayList<InternetAddress>();
      for (String address : addressees) {
        if (address == null) {
          continue;
        }
        address = address.trim();
        if (address.length() == 0) {
          continue;
        }
        addresseesList.add(new InternetAddress(address));
      }
      if (addresseesList.size() == 0) {
        return false;
      }
      InternetAddress[] to =
        addresseesList.toArray(new InternetAddress[addresseesList.size()]);

      final Message msg = new MimeMessage(session);
      msg.setFrom(new InternetAddress(sender));
      msg.setRecipients(Message.RecipientType.TO, to);
      msg.setSubject(subject);
      msg.setSentDate(new Date());

      MimeBodyPart part = new MimeBodyPart();
      if(mimeType != null && !mimeType.isEmpty()) {
        part.setContent(textContent, mimeType);
      } else {
        part.setText(textContent);
      }
      MimeMultipart attachedContent = new MimeMultipart();
      attachedContent.addBodyPart(part);

      // Coloca os arquivos
      if (attachments != null) {
        for (File file : attachments) {
          part = new MimeBodyPart();
          FileDataSource source = new FileDataSource(file);
          part.setDataHandler(new DataHandler(source));
          part.setFileName(file.getName());
          part.setHeader("Content-ID", "<" + file.getName() + ">");
          attachedContent.addBodyPart(part);
        }
      }
      msg.setContent(attachedContent);

      if (replyTo != null && !replyTo.trim().isEmpty()) {
        final InternetAddress[] replies = { new InternetAddress(replyTo) };
        msg.setReplyTo(replies);
      }

      if (copyTo != null && copyTo.length > 0) {
        final ArrayList<InternetAddress> addresses =
          new ArrayList<InternetAddress>();
        for (int i = 0; i < copyTo.length; i++) {
          if (!copyTo[i].trim().isEmpty()) {
            addresses.add(new InternetAddress(copyTo[i]));
          }
        }
        final InternetAddress[] copies =
          addresses.toArray(new InternetAddress[addresses.size()]);
        msg.setRecipients(Message.RecipientType.CC, copies);
      }
      Transport.send(msg);
    }
    catch (MessagingException mex) {
      final String msg = "Falha no envio de email para [" + addressees + "]! ";
      final String err = "Exceo no servidor: " + mex + "!";
      Server.logSevereMessage(msg + err);
      return false;
    }
    return true;
  }

  /**
   * Verifica se o remetente  vlido, se no cria um utilizando o parmetro
   * passado.<br/>
   * Caso o parmetro passado seja nulo, o remetente default ser utilizado.<br/>
   * Caso o remetente esteja de acordo com a expresso regular
   * <code>'^[^\\s@]+@[^\\s@]+$'</code> , este  retornado, caso contrrio, 
   * criado um e-mail vlido usando o parmetro passado e o remetente default.<br/>
   * Ex.<br/>
   * seed = "infogrid"<br/>
   * remetente default = "default@tecgraf.puc-rio.br"<br/>
   * remetente final = "infogrid@no-reply.tecgraf.puc-rio.br"<br/>
   * 
   * @param seedArg remetente vlido ou uma semente para ser utilizada na
   *        criao de um vlido.
   * 
   * @return um remetente vlido.
   */
  private String createValidSender(String seedArg) {
    String seed = seedArg;
    if (null == seed) {
      seed = defaultSender;
    }
    if (_MAIL_PTTN.matcher(seed).matches()) {
      return seed;
    }

    // Alguns servidores de smtp checam se o e-mail do remetente est bem
    // formatado. Caso ele no esteja, criaremos um para ele baseado no
    // remetente passado como parmetro.
    return String.format("%1$s@no-reply.%2$s", seed.split("@")[0],
      defaultSender.split("@")[1]);
  }

  /**
   * Montagem de <i>subject</i> para e-mail.
   * 
   * @param sender remetente
   * 
   * @return o texto.
   */
  private String makeSubject(final String sender) {
    if (sender == null) {
      return Server.getInstance().getSystemName() + " - " + defaultSender;
    }
    else {
      return Server.getInstance().getSystemName() + " - " + sender;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public boolean sendMail(final String sender, final String[] addressees,
    final String content) {
    final String subject = makeSubject(sender);
    return sendMail(sender, addressees, content, subject, null, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public boolean sendMail(final String sender, final String[] addressees,
    final String content, final String subject, final String replyTo,
    final String[] copyTo) {
    return sendMail(sender, addressees, content, subject, replyTo, copyTo, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public boolean sendMail(final String senderName, final String[] to,
    final String content, final String subject, final String replyTo,
    final String[] copyTo, final File[] attachments) {
    return sendMail(senderName, to, content, subject, replyTo, copyTo, attachments, null);
  }
  
  @Override
  final public boolean sendMail(final String senderName, final String[] to,
    final String content, final String subject, final String replyTo,
    final String[] copyTo, final File[] attachments, String mimeType) {
    final String cnt = (content == null ? getString("no.content") : content);
    final String snd = (senderName == null ? defaultSender : senderName);
    final String sub = (subject == null ? makeSubject(senderName) : subject);
    return accessMailServer(snd, to, sub, cnt, replyTo, copyTo, attachments, mimeType);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public boolean mailTo(final String sender, final Object userId,
    final String content) {
    return mailUser(sender, userId, content, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean mailTo(final String senderName, final Object userId,
    final String content, final File[] attachments) {
    return mailUser(senderName, userId, content, null, attachments);
  }
  
  @Override
  public boolean mailTo(final String senderName, final Object userId,
    final String content, final File[] attachments, final String mimeType) {
    return mailUser(senderName, userId, content, mimeType, attachments);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public boolean mailSupport(final String content) {
    String systemName = Server.getInstance().getSystemName();
    String nowText = new Date().toString();
    String timePrefix = "\n\n" + systemName + "\n" + nowText + "\n\n";
    String fmtContent = timePrefix + content;
    return sendMail(defaultSender, supportTo, fmtContent);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public boolean mailSupport(final String textContent,
    final File[] attachments) {
    String systemName = Server.getInstance().getSystemName();
    String nowText = new Date().toString();
    String timePrefix = "\n\n" + systemName + "\n" + nowText + "\n\n";
    String fmtContent = timePrefix + textContent;
    return sendMail(defaultSender, supportTo, fmtContent, null, null, null,
      attachments);
  }

  /**
   * Manda um email para um usurio
   * 
   * @param sender a mensagem.
   * @param userId o identificador do usurio destinatrio.
   * @param content a mensagem.
   * @param mimeType Tipo MIME.
   * @param attachments a lista de arquivos para ser anexa ao email (opcional)
   * 
   * @return um indicativo de sucesso.
   */
  private boolean mailUser(final String sender, final Object userId,
    final String content, String mimeType, File... attachments) {
    final String[] addressee = getUserMails(userId);
    if ((addressee == null) || (addressee.length == 0)) {
      Server.logSevereMessage("Usurio [" + userId
        + "] no tem email relacionado!");
      return false;
    }
    return sendMail(sender, addressee, content, null, null, null, attachments, mimeType);
  }

  /**
   * Notifica alguns os usurios.
   * 
   * @param sender o nome remetente.
   * @param usersId um array de usurios
   * @param content a mensagem.
   * 
   * @return um indicativo de sucesso.
   */
  final public boolean mailSomeUsers(final String sender,
    final Object[] usersId, final String content) {
    boolean flag = true;
    if (usersId == null) {
      return true;
    }
    for (int i = 0; i < usersId.length; i++) {
      final Object id = usersId[i];
      flag &= mailUser(sender, id, content, null);
    }
    return flag;
  }

  /**
   * Notifica alguns os usurios.
   * 
   * @param sender o nome remetente.
   * @param usersId um array de usurios
   * @param content a mensagem.
   * @param attachments a lista de arquivos para anexar ao email
   * 
   * @return um indicativo de sucesso.
   */
  final public boolean mailSomeUsers(final String sender,
    final Object[] usersId, final String content, final File[] attachments) {
    boolean flag = true;
    if (usersId == null) {
      return true;
    }
    for (int i = 0; i < usersId.length; i++) {
      final Object id = usersId[i];
      flag &= mailUser(sender, id, content, null, attachments);
    }
    return flag;
  }

  /**
   * Notifica um conjunto usurios atravs de um servio
   * 
   * @param service o servio remetente.
   * @param usersId array de identificadores dos destinatrios.
   * @param content a mensagem.
   * 
   * @return flag indicativo de sucesso no envio.
   */
  final public boolean mailSomeUsersFromService(final Service service,
    final Object[] usersId, final String content) {
    if (service == null) {
      Server.logSevereMessage("Servio no identificado tentou enviar email!");
      return false;
    }

    String formatedContent = formatServiceMailContent(service, content);
    return mailSomeUsers(defaultSender, usersId, formatedContent);
  }

  /**
   * Notifica um usurio atravs de um servio
   * 
   * @param service o servio remetente.
   * @param userId o identificador do usurio destinatrio.
   * @param content a mensagem.
   * 
   * @return flag indicativo de sucesso no envio.
   */
  final public boolean mailUserFromService(final Service service,
    final Object userId, final String content) {
    if (service == null) {
      Server.logSevereMessage("Servio no identificado tentou enviar email!");
      return false;
    }
    String formatedContent = formatServiceMailContent(service, content);
    return mailUser(defaultSender, userId, formatedContent, null);
  }

  /**
   * Notifica um endereo de email atravs de um servio
   * 
   * @param service o servio remetente.
   * @param addressees os endereos de e-mail dos destinatrios separados por
   *        vrgula.
   * @param content a mensagem.
   * 
   * @return flag indicativo de sucesso no envio.
   */
  final public boolean mailFromService(Service service, String[] addressees,
    String content) {
    if (service == null) {
      String err = "Servio no identificado tentou enviar email!";
      Server.logSevereMessage(err);
      return false;
    }
    String formatedContent = formatServiceMailContent(service, content);
    return sendMail(defaultSender, addressees, formatedContent);
  }

  /**
   * Formata o contedo de um e-mail cujo remetente  um servio. Adiciona um
   * cabealho no e-mail indicando qual servio o gerou e de qual servidor.
   * 
   * @param service remetente
   * @param content contedo ainda no formatado do e-mail
   * @return o contedo do e-mail formatado para ser enviado.
   */
  private String formatServiceMailContent(final Service service,
    final String content) {
    String fmt = getString("MailService.services.mail.content");
    String systemName = Server.getInstance().getSystemName();
    String serviceName = service.getName();
    String newContent = String.format(fmt, systemName, serviceName, content);

    String timePrefix = "\n\n" + new Date().toString() + "\n\n";
    return timePrefix + newContent;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final protected boolean has2Update(final Object arg, final Object event) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public void shutdownService() {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public void initService() throws ServerException {
    Server.logInfoMessage("Endereo remetente: " + defaultSender);
    if (!createSession()) {
      throw new ServerException("Falha na construo da sesso de mail!");
    }
  }

  /**
   * Busca do objeto servio de e-mail no servidor
   * 
   * @return o objeto <code>MailService</code>
   */
  public static MailService getInstance() {
    return (MailService) getInstance(SERVICE_NAME);
  }

  /**
   * Constri a instncia do servio.
   * 
   * @throws ServerException se houver falha.
   */
  final public static void createService() throws ServerException {
    new MailService();
  }

  /**
   * Construtor padro.
   * 
   * @throws ServerException se houver falha.
   * 
   * @see Service#Service(String)
   */
  protected MailService() throws ServerException {
    super(SERVICE_NAME);

    supportTo = new String[] { getStringProperty("support") };
    defaultSender = getStringProperty("from");

    mailHostName = getStringProperty(MAIL_SMTP_HOST_PROPERTY_NAME);
    mailHostPort = getIntProperty(MAIL_SMTP_PORT_PROPERTY_NAME);
  }
}
