/*
 * $Id$
 */
package csbase.server.services.schedulerservice;

import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import csbase.exception.PermissionException;
import csbase.logic.CommandInfo;
import csbase.logic.FailureFinalizationType;
import csbase.logic.Priority;
import csbase.logic.SGASet;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.sgaservice.ProjectNotFoundException;
import csbase.server.services.sgaservice.SGANotAvailableException;
import csbase.server.services.sgaservice.SGAService;

/**
 * Classe responsvel por submeter os comandos ao <code>SGAService</code>.
 *
 * Para submeter um comando para um servidor, o <code>Scheduler</code> considera
 * os seguintes critrios: - peso do algoritmo (CPU e memria); -
 * disponibilidade da mquina; - permisso do usurio;
 *
 * Na configurao da execuo de um comando, o usurio escolhe manualmente os
 * servidores para execuo ou deixa que o <code>Scheduler</code> tome a deciso
 * de onde o comando ser executado.
 */
public class Scheduler {
  /** Thread do escalonador */
  private Thread schedulerThread;
  /** Indica o trmino do processamento da fila de comandos. */
  private boolean exitSchedulerThread;
  /** Fila que controla a prioridade dos processos */
  private final PriorityQueue commandQueue;
  /** Referncia para o servio. */
  private final SchedulerService service;
  /** SGAService */
  private final SGAService sgaService;
  /** Mapeamento dos sgas que acabaram de receber comandos a uma data */
  private final HashMap<String, Long> recentUsedServers;
  /**
   * Tempo que o SGA fica de quarentena depois que recebe um comando. O valor
   * default  zero. Neste caso no h quarentena.
   */
  private final int sgaQuarantine;
  /**
   * Intervalo de processamento da fila (milisegundos). O valor default  duas
   * vezes o intervalo de atualizao das informaes do SGA.
   */
  private final long queueProcessingInterval;

  /** Nmero de retentativas de submisso do comando */
  private final int MaxSubmissionRetries;

  /** Objeto que executa a poltica de escalonamento ativa */
  private SchedulerPolicyInterface schedPolicyInterface;

  /**
   * Identificador da poltica de escalonamento que deve ser empregada na
   * escolha dos melhores SGAs ativos
   */
  private final String schedPolicyName;

  /** Controlador de recursos dos sgas. */
  static private SchedulerResourcesController schedulerResourcesController;

  /**
   * Constri a representao do Scheduler.
   *
   * @param commandQueue
   *            A fila de comandos a ser processada.
   * @param queueProcessingInterval
   *            Intervalo de processamento da fila.
   * @param submissionRetries
   *            Nmero de retentativas de submisso do comando
   * @param sgaQuarantine
   *            Tempo de quarentena dos SGAs.
   * @param schedPolicyName
   *            Poltica de escalonamento ativa.
   * @param resourcesControl
   *            Indica se deve fazer o controle de recursos de cada sga.
   * @param backupFilePath
   *            Caminho do arquivo de backup dos recursos de cada sga.
   */
  protected Scheduler(PriorityQueue commandQueue,
    long queueProcessingInterval, int submissionRetries,
    int sgaQuarantine, String schedPolicyName,
    boolean resourcesControl, String backupFilePath) {
    this.commandQueue = commandQueue;
    this.queueProcessingInterval = queueProcessingInterval;
    this.MaxSubmissionRetries = submissionRetries;
    this.sgaQuarantine = sgaQuarantine;
    this.schedPolicyName = schedPolicyName;
    this.sgaService = SGAService.getInstance();
    this.service = SchedulerService.getInstance();
    this.recentUsedServers = new HashMap<String, Long>();
    schedulerResourcesController = new SchedulerResourcesController(
      resourcesControl, backupFilePath);
  }

  /**
   * Cria a thread de processamento da fila de comandos. A thread  acordada a
   * cada novo comando submetido ou no intervalo de
   * <code>queueProcessingInterval</code>. Para cada comando, escolhe o
   * servidor para execuo e submete o comando no servidor escolhido. Coloca
   * o servidor em quarentena, depois que ele  escolhido. Caso no encontre
   * servidor disponvel para executar, o comando permanece na fila.
   */

  protected void startScheduler() {
    loadSchedulerPolicy();

    schedulerThread = new Thread(new Runnable() {
      @Override
      public void run() {
        while (!exitSchedulerThread) {

          Server.logFineMessage("Incio do processamento, se houver comandos");

          CommandInfo[] cmds = commandQueue.getAndWaitForCommands();
          int numOfCommandsToSchedule = cmds.length;
          List<CommandInfo> commandsToSchedule = new LinkedList<CommandInfo>();
          for (int i = 0; i < cmds.length && !exitSchedulerThread; i++) {
            CommandInfo cmd = cmds[i];

            // Indica qual usurio est executando o comando.
            Service.setUserId(cmd.getUserId());

            switch (cmd.getStatus()) {
              case EXECUTING:
              case DOWNLOADING:
              case FINISHED:
              case SYSTEM_FAILURE:
                // Retira da fila comandos nos estados EXECUTING, DOWNLOADING,
                // FINISHED e SYSTEM_FAILURE
                commandQueue.remove(cmd.getId());
                numOfCommandsToSchedule--;
                break;
              case SCHEDULED:
                if (cmd.getSubmissionAttempts() > MaxSubmissionRetries) {
                  commandQueue.remove(cmd.getId());
                  numOfCommandsToSchedule--;
                  service.failCommand(
                    cmd,
                    FailureFinalizationType.MAX_SUBMISSION_RETRIES_REACHED);
                  try {
                    Server.logWarningMessage("Comando retirado da fila por excesso de tentativas de submisso. "
                      + "Plataforma "
                      + cmd.getPlatformFilter()
                      + " ID: "
                      + cmd.getId()
                      + "Algoritmo "
                      + cmd.getConfigurator()
                      .getAlgorithmName());
                  } catch (RemoteException e) {
                    e.printStackTrace();
                  }
                  continue;
                } else {
                  cmd.incSubmissionAttempts();
                  commandsToSchedule.add(cmd);
                }
                break;
              default:
                break;
            }
          }

          if (commandsToSchedule.isEmpty()) {
            if ((commandQueue.size() > 0) && (commandQueue.size() == numOfCommandsToSchedule)) {
              Server.logFineMessage("Ainda h comandos para processar. Fila em sleep.");
              commandQueue.sleep(queueProcessingInterval);
            }
            continue;
          }

          Map<CommandInfo, SGASet> serversChoosed = null;

          serversChoosed = allocateSGA(commandsToSchedule);

          if (serversChoosed == null) {
            Server.logFineMessage("Nenhum SGA alocado");
            if ((commandQueue.size() > 0) && (commandQueue.size() == numOfCommandsToSchedule)) {
              Server.logFineMessage("Ainda h comandos para processar. Fila em sleep.");
              commandQueue.sleep(queueProcessingInterval);
            }
            continue;
          }

          Set<Map.Entry<CommandInfo, SGASet>> allocationSet = serversChoosed
            .entrySet();

          for (Entry<CommandInfo, SGASet> entry : allocationSet) {
            CommandInfo cmd = entry.getKey();
            SGASet serverChoosed = entry.getValue();

            // Indica qual usurio est executando o comando.
            Service.setUserId(cmd.getUserId());

            if (isInQuarantine(serverChoosed.getName())) {
              continue;
            }

            Server.logFineMessage("Servidor aprovado: "
              + serverChoosed);

            // Submete o comando!
            try {
              cmd.setSGAName(serverChoosed.getName());
              schedulerResourcesController.makeReservation(cmd);
              /* Submete o comando para execuo */
              // TODO mjulia REVISAR
              // Estou sincronizando na fila e testando se j houve um
              // cancelamento que removeu.
              // Isso porque o comando por ter sido removido da fila aps o
              // array de comandos ser recuperado aqui.
              synchronized (commandQueue) {
                if (commandQueue.getCommand(cmd.getId()) == null) {
                  Server.logWarningMessage("O comando "
                    + cmd.getId()
                    + " no vai ser disparado porque j foi removido da fila");
                } else {
                  sgaService.executeCommand(cmd);
                  // Coloca os servidores na tabela de recm utilizado
                  Server.logFineMessage(
                      MessageFormat.format(
                          "Comando {0} enviado ao SGA {1}", cmd.getId(), serverChoosed.getName()));
                  recentUsedServers.put(
                    serverChoosed.getName(),
                    System.currentTimeMillis());
                  commandsToSchedule.remove(cmd);

                }
              }
            } catch (SGANotAvailableException e) {
              Server.logSevereMessage(String.format(
                "Erro ao submeter o comando %s - %s.",
                cmd.getId(), cmd.getTip()), e);

              service.failCommand(
                cmd,
                FailureFinalizationType.SGA_IS_NOT_AVAILABLE);
            } catch (ProjectNotFoundException e) {
              Server.logSevereMessage(String.format(
                "Erro ao submeter o comando %s - %s.",
                cmd.getId(), cmd.getTip()), e);
              // Retira da lista
              commandQueue.remove(cmd.getId());
              numOfCommandsToSchedule--;
              // Notifica que no disparou!
              service.failCommand(cmd,
                FailureFinalizationType.PROJECT_NOT_FOUND);
            } catch (PermissionException e) {
              Server.logSevereMessage(String.format(
                "Erro ao submeter o comando %s - %s.",
                cmd.getId(), cmd.getTip()), e);
              // Retira da lista
              commandQueue.remove(cmd.getId());
              numOfCommandsToSchedule--;
              // Notifica que no disparou!
              service.failCommand(
                cmd,
                FailureFinalizationType.USER_WITHOUT_PERMISSION_FOR_EXECUTION);
            } catch (Exception e) {
              Server.logSevereMessage(String.format(
                "Erro ao submeter o comando %s - %s.",
                cmd.getId(), cmd.getTip()), e);
              // Retira da lista
              commandQueue.remove(cmd.getId());
              numOfCommandsToSchedule--;
              // Notifica que no disparou!
              FailureFinalizationType.UNKNOWN.changeDescription(e
                .getMessage());
              service.failCommand(cmd,
                FailureFinalizationType.UNKNOWN);
            } finally {
              Service.setUserId(null);
            }
          }

          for (CommandInfo cmd : commandsToSchedule) {
            if (cmd.getPriority() == Priority.ROOT) {
              Service.setUserId(cmd.getUserId());
              // Retira da lista
              commandQueue.remove(cmd.getId());
              numOfCommandsToSchedule--;
              Server.logWarningMessage("Falha ao disparar comando com prioridade "
                + "ROOT. No foi encontrado um SGA disponvel para atender "
                + "a plataforma "
                + cmd.getPlatformFilter()
                + " do algoritmo usado no comando: "
                + cmd.getId());
              // Notifica que no disparou!
              service.failCommand(
                cmd,
                FailureFinalizationType.NO_SGA_AVAILABLE_TO_ROOT_COMMAND);
              Service.setUserId(null);
            }
          }

          // Espera um tempo para reprocessar os elementos que ainda esto
          // na fila. Mas, nesse meio tempo podem ter entrado outros processos.
          if ((commandQueue.size() > 0)
            && (commandQueue.size() == numOfCommandsToSchedule)) {
            Server.logFineMessage("Ainda h comandos para processar. Fila em sleep.");
            commandQueue.sleep(queueProcessingInterval);
          }
        }
      }

    });
    schedulerThread.setName(this.getClass().getSimpleName() + "::" + "SchedulerThread");
    schedulerThread.start();
  }

  private Map<CommandInfo, SGASet> allocateSGA(List<CommandInfo> cmds) {
    List<String> sgasNames = sgaService.getAllSGANames();
    List<String> serversFiltered = filterQuarantaineSGAs(sgasNames);

    if (serversFiltered == null || serversFiltered.isEmpty()) {
      return null;
    }

    List<SGASet> sgasSetsFiltered = new LinkedList<SGASet>();
    for (String name : serversFiltered) {
      sgasSetsFiltered.add(sgaService.getSGASet(name));
    }

    clearQuarantine();
    return schedPolicyInterface.chooseServer(cmds, sgasSetsFiltered);
  }

  /**
   * Interrompe a execuo da <code>Thread</code>.
   */
  protected void stopScheduler() {
    this.exitSchedulerThread = true;
  }

  /**
   * Instancia um objeto contendo a poltica de escalonamento ativa.
   *
   * @return Objeto com a poltica de escalonamento ativa.
   */
  private SchedulerPolicyInterface loadSchedulerPolicy() {
    try {
      Class<?> c = Class.forName(schedPolicyName);
      schedPolicyInterface = (SchedulerPolicyInterface) c.newInstance();
    } catch (Exception e) {
      schedPolicyInterface = new PercCPUPolicy();
      Server.logSevereMessage(String.format(
        "Erro ao ativar a poltica de escalonamento %s.",
        schedPolicyName), e);
    }

    Server.logInfoMessage("Classe de escalonamento instanciada: "
      + schedPolicyInterface.getClass().getName() + ".java.");

    return schedPolicyInterface;
  }

  /**
   * Retira os SGAs da quarentena se seus tempos j expiraram.
   */
  private void clearQuarantine() {
    Iterator<Long> iterator = recentUsedServers.values().iterator();
    while (iterator.hasNext()) {
      Long quarantineExpirationTime = iterator.next();
      // Verifica se a quarentena j expirou. Nesse caso, retira
      // o SGA da lista e retorna true.
      long now = System.currentTimeMillis();
      if (now - quarantineExpirationTime > sgaQuarantine) {
        iterator.remove();
      }
    }
  }

  private List<String> filterQuarantaineSGAs(List<String> sgaNames) {
    if (sgaNames == null) {
      return null;
    }
    List<String> filteredList = new ArrayList<String>();
    for (String sgaName : sgaNames) {
      if (!isInQuarantine(sgaName)) {
        filteredList.add(sgaName);
      }
    }
    return filteredList.isEmpty() ? null : filteredList;
  }

  /**
   * Verifica se o servidor est na quarentena.
   *
   * @param sgaName
   *            Nome do servidor.
   * @return <code>true</code> se o servidor est na quarentena,
   *         <code>false</code> caso contrrio.
   */
  private boolean isInQuarantine(String sgaName) {
    if (!recentUsedServers.containsKey(sgaName)) {
      return false;
    }
    // Verifica se a quarentena j expirou. Nesse caso, retira
    // o SGA da lista e retorna true.
    long now = System.currentTimeMillis();
    if ((sgaQuarantine > 0)
      && (now - recentUsedServers.get(sgaName) <= sgaQuarantine)) {
      return true;
    }
    recentUsedServers.remove(sgaName);
    return false;
  }

  public static SchedulerResourcesController getSchedulerResourcesController() {
    return schedulerResourcesController;
  }
}
