package csbase.server.services.schedulerservice.classadpolicies;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import csbase.logic.CommandInfo;
import csbase.logic.SGAInfo;
import csbase.logic.SGASet;
import csbase.server.services.schedulerservice.sgafilters.SGASimpleFilter;

public class ClassAdPolicyCommons {
	public static List<ClassAdMatchInfo> getMatchedAds(CommandInfo command,
			List<SGASet> servers, ClassAdManager manager) {
		List<SGASet> candidateServers = SGASimpleFilter
				.filter(command, servers);
		List<ClassAdMatchInfo> matchedSGAs = new ArrayList<ClassAdMatchInfo>();
		// cria ads para SGA's e faz match com ad do comando
		// coloca em matchedSGAs somente aqueles que tiveram match
		for (SGASet sga : candidateServers) {
			ClassAdMatchInfo s = manager.match(command, sga);
			if (s.match())
				matchedSGAs.add(s);
		}
		return matchedSGAs;
	}

	public static List<ClassAdMatchInfo> getMatchedAds(
			List<CommandInfo> commands, List<SGASet> servers,
			ClassAdManager manager) {
		List<ClassAdMatchInfo> matches = new ArrayList<ClassAdMatchInfo>();
		for (CommandInfo cmd : commands) {
			List<SGASet> candidateServers = SGASimpleFilter
					.filter(cmd, servers);
			for (SGASet sga : candidateServers) {
				ClassAdMatchInfo s = manager.match(cmd, sga);
				if (s.match())
					matches.add(s);
			}
		}
		return matches;
	}

	public static List<ClassAdMatchInfo> getMatchedAds(
			List<CommandInfo> commands, SGASet server, ClassAdManager manager) {
		List<ClassAdMatchInfo> matches = new ArrayList<ClassAdMatchInfo>();
		List<SGASet> servers = new LinkedList<SGASet>();
		servers.add(server);

		for (CommandInfo cmd : commands) {
			List<SGASet> candidateServers = SGASimpleFilter
					.filter(cmd, servers);
			for (SGASet sga : candidateServers) {
				ClassAdMatchInfo s = manager.match(cmd, sga);
				if (s.match())
					matches.add(s);
			}
		}
		return matches;
	}

	public static Map<SGASet, List<ClassAdMatchInfo>> getMatchedAdsMappedBySGA(
			List<CommandInfo> commands, List<SGASet> servers,
			ClassAdManager manager) {
		Map<SGASet, List<ClassAdMatchInfo>> map = new HashMap<SGASet, List<ClassAdMatchInfo>>();

		for (CommandInfo cmd : commands) {
			List<SGASet> candidateServers = SGASimpleFilter
					.filter(cmd, servers);
			for (SGASet sga : candidateServers) {
				ClassAdMatchInfo s = manager.match(cmd, sga);
				if (s.match()) {
					if (map.get(sga) == null)
						map.put(sga, new LinkedList<ClassAdMatchInfo>());
					map.get(sga).add(s);
				}
			}
		}
		return map;
	}

	public static Map<SGASet, List<ClassAdMatchInfo>> getMatchedAdsMappedBySGA(
			List<ClassAdMatchInfo> infos) {
		Map<SGASet, List<ClassAdMatchInfo>> map = new HashMap<SGASet, List<ClassAdMatchInfo>>();

		for (ClassAdMatchInfo info : infos) {
			if (info.match()) {
				SGASet sga = info.getSGA();
				if (map.get(sga) == null)
					map.put(sga, new LinkedList<ClassAdMatchInfo>());
				map.get(sga).add(info);
			}
		}
		return map;
	}

	public static Map<CommandInfo, List<ClassAdMatchInfo>> getMatchedAdsMappedByCommand(
			List<CommandInfo> commands, List<SGASet> servers,
			ClassAdManager manager) {
		Map<CommandInfo, List<ClassAdMatchInfo>> map = new HashMap<CommandInfo, List<ClassAdMatchInfo>>();

		for (CommandInfo cmd : commands) {
			List<SGASet> candidateServers = SGASimpleFilter
					.filter(cmd, servers);
			for (SGASet sga : candidateServers) {
				ClassAdMatchInfo info = manager.match(cmd, sga);
				if (info.match()) {
					if (map.get(cmd) == null)
						map.put(cmd, new LinkedList<ClassAdMatchInfo>());
					map.get(sga).add(info);
				}
			}
		}
		return map;
	}

	public static Map<CommandInfo, List<ClassAdMatchInfo>> getMatchedAdsMappedByCommand(
			List<ClassAdMatchInfo> infos) {
		Map<CommandInfo, List<ClassAdMatchInfo>> map = new HashMap<CommandInfo, List<ClassAdMatchInfo>>();

		for (ClassAdMatchInfo info : infos) {
			if (info.match()) {
				CommandInfo cmd = info.getCommand();
				if (map.get(cmd) == null)
					map.put(cmd, new LinkedList<ClassAdMatchInfo>());
				map.get(cmd).add(info);
			}
		}
		return map;
	}

	public static Map<SGASet, Integer> getMatchedAdsMappedBySGACount(
			Map<SGASet, List<ClassAdMatchInfo>> matchesBySGA) {
		Map<SGASet, Integer> count =  new HashMap<SGASet, Integer>();
		for(Entry<SGASet, List<ClassAdMatchInfo>> entry: matchesBySGA.entrySet())
			count.put(entry.getKey(), entry.getValue().size());
		return count;
	}
	
	public static Map<CommandInfo, Integer> getMatchedAdsMappedByCommandCount(
			Map<CommandInfo, List<ClassAdMatchInfo>> matchesByCommand) {
		Map<CommandInfo, Integer> count =  new HashMap<CommandInfo, Integer>();
		for(Entry<CommandInfo, List<ClassAdMatchInfo>> entry: matchesByCommand.entrySet())
			count.put(entry.getKey(), entry.getValue().size());
		return count;
	}

	public static List<SGASet> getSGAsSortedByDefaultClassAdRank(
			CommandInfo command, List<SGASet> servers, ClassAdManager manager) {
		List<ClassAdMatchInfo> matchedSGAs = getMatchedAds(command, servers,
				manager);
		List<SGASet> sgas = new ArrayList<SGASet>();
		Collections.sort(matchedSGAs, Collections.reverseOrder());
		for (ClassAdMatchInfo info : matchedSGAs) {
			sgas.add(info.getSGA());
		}
		return sgas;
	}

	public static List<SGASet> getSGAsSortedByCustomizedClassAdRank(
			CommandInfo command, List<SGASet> servers, ClassAdManager manager,
			Comparator<ClassAdMatchInfo> comparator) {
		List<ClassAdMatchInfo> matchedSGAs = getMatchedAds(command, servers,
				manager);
		List<SGASet> sgas = new ArrayList<SGASet>();
		Comparator<ClassAdMatchInfo> revComparator = Collections
				.reverseOrder(comparator);
		Collections.sort(matchedSGAs, revComparator);
		for (ClassAdMatchInfo info : matchedSGAs) {
			sgas.add(info.getSGA());
		}
		return sgas;
	}

	public static SGASet simulateResourceConsumption(CommandInfo cmd, SGASet server) {
		return new SGASet(createSGAInfoResourceConsumption(cmd, server.getMainInfo()),
				server.getName(), server.getEnabled(), server.getAlive(),
				server.hasDiskAccess(), server.isBackoffExpired(),
				server.getJobsInfo(), server.isCSFSEnabled(), null,
				System.currentTimeMillis());
	}
	
	public static SGASet simulateResourceRelease(CommandInfo cmd, SGASet server) {
		return new SGASet(createSGAInfoResourceRelease(cmd, server.getMainInfo()),
				server.getName(), server.getEnabled(), server.getAlive(),
				server.hasDiskAccess(), server.isBackoffExpired(),
				server.getJobsInfo(), server.isCSFSEnabled(), null,
				System.currentTimeMillis());
	}

	private static SGAInfo[] createSGAInfoResourceConsumption(CommandInfo cmd, SGAInfo info) {
		int ramInfo = (int) info.getRAMFreeMemoryMb();
		float ncpus = info.getNumProcessors();
		try {
			ramInfo -= cmd.getConfigurator().getMemoryAmount();
			ncpus -= cmd.getConfigurator().getCpuAmount();

		} catch (RemoteException e) {
		}
		SGAInfo[] infos = new SGAInfo[] { createSGAInfoCopy(info, ramInfo, ncpus) };
		return infos;
	}
	
	private static SGAInfo[] createSGAInfoResourceRelease(CommandInfo cmd, SGAInfo info) {
		int ramInfo = (int) info.getRAMFreeMemoryMb();
		float ncpus = info.getNumProcessors();
		try {
			ramInfo += cmd.getConfigurator().getMemoryAmount();
			ncpus += cmd.getConfigurator().getCpuAmount();;
		} catch (RemoteException e) {
		}
		SGAInfo[] infos = new SGAInfo[] { createSGAInfoCopy(info, ramInfo, ncpus) };
		return infos;
	}
	
	private static SGAInfo createSGAInfoCopy (SGAInfo info, int ramInfo, float ncpus)
	{
		SGAInfo newInfo = new SGAInfo(info.getHostName(), info.getPlatformId(),
				(int) ncpus, ramInfo, 0, info.getClockSpeedMHz(),
				info.getFileSeparator(), info.getProjectRootDirectory(),
				info.getAlgorithmRootDirectory(),
				info.getSandboxRootDirectory(), null);
		newInfo.setAlive(true);
		return newInfo;
	}
}
