package csbase.server.services.schedulerservice.heuristic;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import csbase.logic.CommandInfo;
import csbase.logic.SGASet;

/**
 * Heurstica baseada no artigo Simple and Integrated Heuristic Algorithms for
 * Scheduling Tasks with Time and Resource Constraints. Zhao, Wei; Ramamritham.
 * The Journal of Systems and Software 7, 195-207 (1987)
 * 
 * Se houver uma alocao factvel, retorna a melhor alocao factvel que a
 * heurstica encontrar, sem qualquer garantia de que a alocao seja tima.
 * 
 * Em caso de no haver alocao factvel, retorna uma alocao parcial,
 * garantidamente tima.
 * 
 */

public class BruteForceHeuristicApplyer {

	Comparator<CommandInfo> commandComparator; 
	Comparator<Allocation> allocationComparator; // TODO: avaliar se devemos ter
													// um default
	private List<CommandInfo> commands;
	private List<SGASet> servers;
	private List<SGASet> simulatedServers;
	private List<CommandInfo> remainingCommands = new LinkedList<CommandInfo>();
	private HashMap<String, SGASet> originalServers = new HashMap<String, SGASet>();
	private ResourceControllerInterface manager;
	private Allocation allocation;
	private AllocationStorage storage = new AllocationStorage();
	private boolean foundFeasibleAllocation = false;

	public BruteForceHeuristicApplyer(
			Comparator<CommandInfo> commandComparator,
			Comparator<Allocation> allocationComparator,
			List<CommandInfo> commands, List<SGASet> servers,
			ResourceControllerInterface manager) {
		this.commandComparator = commandComparator;
		this.allocationComparator = allocationComparator;
		this.commands = commands;
		this.servers = servers;
		this.simulatedServers = new ArrayList<SGASet>(servers);
		this.manager = manager;
		applyHeuristic();
	}

	public BruteForceHeuristicApplyer(
			Comparator<Allocation> allocationComparator,
			List<CommandInfo> commands, List<SGASet> servers,
			ResourceControllerInterface manager) {
		this.allocationComparator = allocationComparator;
		this.commands = commands;
		this.servers = servers;
		this.simulatedServers = new ArrayList<SGASet>(servers);
		this.manager = manager;
		applyHeuristic();
	}

	private void applyHeuristic() {

		Map<CommandInfo, SGASet> currentAllocation = new HashMap<CommandInfo, SGASet>();
		remainingCommands.addAll(commands);

		for (SGASet sga : servers) {
			originalServers.put(sga.getName(), sga);
		}
		if (chooseCommand(currentAllocation)) {
			this.allocation = new Allocation(currentAllocation,
					simulatedServers, remainingCommands, true, this.manager,
					this.originalServers);
		}
	}

	private boolean chooseCommand(Map<CommandInfo, SGASet> currentAllocationMap) {
		List<CommandInfo> cmdsCopy = new ArrayList<CommandInfo>(
				remainingCommands);
		if (this.commandComparator != null)
			Collections.sort(cmdsCopy, this.commandComparator);
		boolean successful = false;
		for (CommandInfo cmd : cmdsCopy)
		{
			boolean feasible = tryCommand(cmd, currentAllocationMap);
			if (!this.foundFeasibleAllocation)
				successful |= feasible;
			else if (!feasible)
				return false;
		}
		return successful;
	}

	private boolean tryCommand(CommandInfo cmd,
			Map<CommandInfo, SGASet> currentAllocationMap) {
		remainingCommands.remove(cmd);
		List<SGASet> possibleAllocations = manager
				.getServersThatMeetsRequirements(cmd, simulatedServers);
		if (possibleAllocations.isEmpty()) {
			System.out.println("No server found for command.");
			Allocation currentAllocation = new Allocation(currentAllocationMap,
					simulatedServers, remainingCommands, false, this.manager,
					this.originalServers);
			if (allocation == null
					|| allocation.compareTo(currentAllocation) == -1
					|| (allocation.compareTo(currentAllocation) == 0 && allocationComparator
							.compare(allocation, currentAllocation) == -1)) {
				allocation = currentAllocation;
			}
			remainingCommands.add(cmd);
			return false;
		}
		boolean successful = false;
		for (SGASet sga : possibleAllocations) {
			successful|= tryAllocation(currentAllocationMap, cmd, sga);
		}
		remainingCommands.add(cmd);
		return successful;
	}

	private boolean tryAllocation(
			Map<CommandInfo, SGASet> currentAllocationMap, CommandInfo cmd,
			SGASet server) {
		currentAllocationMap.put(cmd, originalServers.get(server.getName()));

		if (!storage.hasAllocation(currentAllocationMap)) {
			storage.addAllocation(currentAllocationMap);
			// simula uso de recursos
			SGASet consumedSGA = manager.simulateResourceConsumption(cmd,
					server);
			simulatedServers.remove(server);
			simulatedServers.add(consumedSGA);

			if (remainingCommands.isEmpty())
			{
				this.foundFeasibleAllocation  = true;
				Allocation currentAllocation = new Allocation(currentAllocationMap,
						simulatedServers, remainingCommands, true, this.manager,
						this.originalServers);
				if (allocation == null
						|| allocation.compareTo(currentAllocation) == -1
						|| (allocation.compareTo(currentAllocation) == 0 && allocationComparator
								.compare(allocation, currentAllocation) == -1)) {
					allocation = currentAllocation;
				}
			}

			if (remainingCommands.isEmpty()
					|| chooseCommand(currentAllocationMap))
				return true;

			// desfaz simulao de uso de recursos
			simulatedServers.remove(consumedSGA);
			simulatedServers.add(server);
			manager.undoResourceConsumptionSimulation(cmd, server);
		}
		currentAllocationMap.remove(cmd);
		return false;
	}

	public Allocation getAllocation() {
		return this.allocation;
	}

}