/**
 * $Id: AbstractServlet.java 156183 2014-09-11 13:32:20Z mjulia $
 */
package csbase.servlet;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Classe bsica para os servlets. Carrega a cada request o arquivo de
 * propriedades <code>WEB-INF/System.properties</code>, e disponibiliza acesso
 * s propriedades atravs de getters.
 * <p>
 * O arquivo <code>System.properties</code> deve ser copiado para o diretrio
 * <code>WEB-INF</code> pelo script de inicializao do sistema.
 */
abstract class AbstractServlet extends HttpServlet {

  /**
   * Diretrio dentro de <code>tomcat/webapps/[sistema]</code> para onde ser
   * copiado o {@link #PROPERTIES_FILE arquivo de propriedades}.
   */
  protected static final String PROPERTIES_DIR = "/WEB-INF";

  /**
   * Arquivo com as propriedades do sistema, mantido pelo administrador. Este
   * arquivo  copiado para dentro do diretrio {@link #PROPERTIES_DIR} pelo
   * script de inicializao do servidor.
   */
  protected static final String PROPERTIES_FILE = "System.properties";

  /**
   * Nome da propriedade que contm o nome do servidor.
   * 
   * @see #PROP_SERVER_HOST_ADDR
   */
  static final String PROP_SERVER_HOST_NAME = "Server.hostName";

  /**
   * Nome da propriedade que contm a porta RMI do servidor.
   * 
   * @see #DEFAULT_RMI_PORT
   */
  static final String PROP_SERVER_PORT_RMI = "Server.registryPort";

  /**
   * Nome da propriedade que contm o endereo IP do servidor.
   * 
   * @see #PROP_SERVER_HOST_NAME
   */
  static final String PROP_SERVER_HOST_ADDR = "Server.hostAddr";

  /**
   * Nome da propriedade que contm a porta pela qual o cliente deve exportar
   * seus objetos.
   * 
   * @see #PROP_CLIENT_PORT_RMI
   */
  static final String PROP_CLIENT_PORT_RMI = "Client.port.rmi";

  /**
   * Porta RMI default do servidor.
   * 
   * @see #PROP_SERVER_PORT_RMI
   */
  private static final String DEFAULT_RMI_PORT = "1099";

  /**
   * Porta RMI default do cliente.
   * 
   * @see #PROP_CLIENT_PORT_RMI
   */
  private static final String DEFAULT_CLIENT_RMI_PORT = "0";

  /**
   * Propriedades definidas no arquivo {@link #PROPERTIES_FILE}.
   */
  private Properties systemProperties;

  /**
   * {@inheritDoc}
   */
  @Override
  public void init() throws ServletException {
    try {
      loadPropertiesFile();
    }
    catch (Exception e) {
      throw new ServletException(e);
    }
  }

  /**
   * Retorna o valor associado  propriedade.
   * 
   * @param key nome da propriedade
   * @return valor associado  propriedade, ou <code>null</code> caso esta no
   *         tenha sido definida.
   */
  protected String getProperty(String key) {
    return systemProperties.getProperty(key);
  }

  /**
   * Retorna o valor associado  propriedade ou um valor default, caso esta no
   * tenha sido definida.
   * 
   * @param key nome da propriedade
   * @param defaultValue valor default
   * 
   * @return valor associado  propriedade ou o valor default caso esta no
   *         tenha sido definida
   */
  protected String getProperty(String key, String defaultValue) {
    return systemProperties.getProperty(key, defaultValue);
  }

  /**
   * Obtm o path RMI para o servidor no formato
   * <code>rmi://IP_SERVIDOR:PORTA_RMI/</code> .
   * 
   * @return path RMI para o servidor
   * 
   * @see #getServerAddress()
   * @see #getServerRMIport()
   */
  protected String getRMIpath() {
    return "rmi://" + getServerAddress() + ':' + getServerRMIport() + '/';
  }

  /**
   * Obtm o nome do servidor. Obtido da propriedade
   * {@value #PROP_SERVER_HOST_NAME}.
   * 
   * @return nome do servidor
   */
  protected String getServerName() {
    return getProperty(PROP_SERVER_HOST_NAME);
  }

  /**
   * Obtm o endereo IP do servidor. Obtido da propriedade
   * {@value #PROP_SERVER_HOST_ADDR}.
   * 
   * @return endereo IP do servidor
   */
  protected String getServerAddress() {
    return getProperty(PROP_SERVER_HOST_ADDR);
  }

  /**
   * Obtm a porta RMI para comunicao com o servidor. Obtido da propriedade
   * {@value #PROP_SERVER_PORT_RMI}. Como esta propriedade pode no ter sido
   * definida, temos um fallback para o default {@value #DEFAULT_RMI_PORT}.
   * 
   * @return porta para comunicao RMI com o servidor
   */
  protected String getServerRMIport() {
    return getProperty(PROP_SERVER_PORT_RMI, DEFAULT_RMI_PORT);
  }

  /**
   * Obtm a porta RMI para comunicao com o servidor. Obtido da propriedade
   * {@value #PROP_SERVER_PORT_RMI}. Como esta propriedade pode no ter sido
   * definida, temos um fallback para o default {@value #DEFAULT_RMI_PORT}.
   * 
   * @return porta para comunicao RMI com o servidor
   */
  protected String getClientRMIport() {
    return getProperty(PROP_CLIENT_PORT_RMI, DEFAULT_CLIENT_RMI_PORT);
  }

  /**
   * Carrega as propriedades definidas no arquivo de propriedade.
   * 
   * @return true se as propriedades foram carregadas e false caso contrrio.
   */
  protected boolean loadPropertiesFile() {
    String propsPath = PROPERTIES_DIR + '/' + PROPERTIES_FILE;
    InputStream stream = getServletContext().getResourceAsStream(propsPath);
    if (stream == null) {
      systemProperties = new Properties();
      return false;
    }
    else {
      systemProperties = new Properties();
      try {
        systemProperties.load(stream);
      }
      catch (IOException e) {
        return false;
      }
      return true;
    }
  }

  /**
   * Verifica se as proprieades foram carregadas.
   * 
   * @return true se as propriedades foram carregadas e false caso contrrio
   */
  protected boolean hasPropertiesFile() {
    return !systemProperties.isEmpty();
  }

  /**
   * Efetivamente processa o GET.
   * 
   * @param req solicitao feita ao servidor
   * @param resp resposta do servidor
   * @throws ServletException se o GET no foi tratado
   * @throws IOException se houve um erro de I/O durante o processamento do GET
   */
  @Override
  protected abstract void doGet(HttpServletRequest req, HttpServletResponse resp)
    throws IOException, ServletException;
}
