package csbase.sga.rest;

import java.util.Map;
import java.util.List;
import java.util.HashMap;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.logging.Logger;
import java.security.MessageDigest;

import org.omg.CORBA.IntHolder;

import csbase.server.plugin.service.sgaservice.ISGAService;

import csbase.server.plugin.service.IServiceManager;
import csbase.server.plugin.service.sgaservice.ISGAService;

import csbase.sga.rest.messages.parts.PersistentData;
import csbase.sga.rest.messages.parts.RetrievedJob;
import csbase.sga.rest.messages.parts.LostJob;

import sgaidl.InvalidSGAException;
import sgaidl.InvalidParameterException;
import sgaidl.NoPermissionException;
import sgaidl.SGAAlreadyRegisteredException;
import sgaidl.SGANotRegisteredException;
import sgaidl.InvalidCommandException;

import sgaidl.CompletedCommandInfo;
import sgaidl.Pair;
import sgaidl.RetrievedInfo;
import sgaidl.SGACommand;
import sgaidl.SGAProperties;

import sgaidl.SGA_FILE_SEPARATOR;
import sgaidl.SGA_NODE_RESOURCES_SEQ;
import sgaidl.SGA_PROJECT_ROOT_DIR;
import sgaidl.SGA_ALGORITHM_ROOT_DIR;
import sgaidl.SGA_SANDBOX_ROOT_DIR;
import sgaidl.SGA_NODE_NAME;
import sgaidl.SGA_NODE_PLATFORM_ID;
import sgaidl.SGA_NODE_NUM_PROCESSORS;
import sgaidl.SGA_NODE_MEMORY_RAM_INFO_MB;
import sgaidl.SGA_NODE_MEMORY_SWAP_INFO_MB;
import sgaidl.SGA_NODE_CLOCK_SPEED_MHZ;

import sgaidl.SGA_NODE_MEMORY_RAM_FREE_PERC;
import sgaidl.SGA_NODE_MEMORY_SWAP_FREE_PERC;
import sgaidl.SGA_NODE_LOAD_AVG_1MIN_PERC;
import sgaidl.SGA_NODE_LOAD_AVG_5MIN_PERC;
import sgaidl.SGA_NODE_LOAD_AVG_15MIN_PERC;
import sgaidl.SGA_NODE_NUMBER_OF_JOBS;

import sgaidl.SGA_HAS_DISK_ACCESS;
import sgaidl.SGA_BOOLEAN_TRUE;

public class CSBaseFacade {

   private ISGAService sgaService;
   private Logger logger;
   
   private class DaemonData {
      public DaemonData(SGARestDaemon daemon, String configHash, Map<String, String> sgaProperties) {
         this.daemon = daemon;
         this.configHash = configHash;
         this.sgaProperties = sgaProperties;
      }
      SGARestDaemon daemon;
      String configHash;
      Map<String, String> sgaProperties;
   }
   
   // FIXME DaemonData instances are never forgotten by this map.
   private Map<String, DaemonData> daemonData = new HashMap<String, DaemonData>();

   private SGARestDaemon getDaemon(String sgaName) {
      DaemonData data = daemonData.get(sgaName);
      return (data != null) ? data.daemon : null;
   }

     
   public CSBaseFacade(IServiceManager serviceManager) {
      this.logger = Logger.getLogger(this.getClass().getName());
      this.sgaService = ISGAService.class.cast(serviceManager.getService("SGAService"));
   }

   private Pair[] mapToPairs(Map<String, String> map) {
      Pair[] pairs = new Pair[map.size()];
      int i = 0;
      for (Object key : map.keySet()) {
         pairs[i] = new Pair((String) key, map.get(key));
         i++;
      }
      return pairs;
   }
   
   private String tryGet(Map<String, String> map, String field, String def) {
      String value = map.get(field);
      return (value != null) ? value : def;
   }
   
   private SGAProperties makeSGAProperties(Map<String, String> sgaProperties, Map<String, Map<String, String>> nodes) {
      // Default values for mandatory properties
      // FIXME don't use these hardcoded values for defaults
      if (sgaProperties.get(SGA_FILE_SEPARATOR.value)     == null) { sgaProperties.put(SGA_FILE_SEPARATOR.value,     tryGet(sgaProperties, "file_separator",     "/")); }
      if (sgaProperties.get(SGA_PROJECT_ROOT_DIR.value)   == null) { sgaProperties.put(SGA_PROJECT_ROOT_DIR.value,   tryGet(sgaProperties, "project_root_dir",   "/tmp")); }
      if (sgaProperties.get(SGA_ALGORITHM_ROOT_DIR.value) == null) { sgaProperties.put(SGA_ALGORITHM_ROOT_DIR.value, tryGet(sgaProperties, "algorithm_root_dir", "/tmp")); }
      if (sgaProperties.get(SGA_SANDBOX_ROOT_DIR.value)   == null) { sgaProperties.put(SGA_SANDBOX_ROOT_DIR.value,   tryGet(sgaProperties, "sandbox_root_dir",   "/tmp")); }
      // needs to be set or else SGA doesn't work...
      sgaProperties.put(SGA_HAS_DISK_ACCESS.value, SGA_BOOLEAN_TRUE.value);
      
      Pair[] sgaPropPairs = mapToPairs(sgaProperties);

      Pair[][] nodePropPairs = new Pair[nodes.size()][];
      int i = 0;
      for (String nodeName : nodes.keySet()) {
         Map<String, String> nodeProperties = nodes.get(nodeName);

         // static
         nodeProperties.put(SGA_NODE_NAME.value,                nodeName);
         nodeProperties.put(SGA_NODE_PLATFORM_ID.value,         tryGet(sgaProperties, "platform", "MisconfiguredPlatform"));
         
         nodeProperties.put(SGA_NODE_NUM_PROCESSORS.value,      tryGet(nodeProperties, "num_of_cpus", "-1"));
         nodeProperties.put(SGA_NODE_MEMORY_RAM_INFO_MB.value,  tryGet(nodeProperties, "ram_mb",      "-1"));
         nodeProperties.put(SGA_NODE_MEMORY_SWAP_INFO_MB.value, tryGet(nodeProperties, "swap_mb",     "-1"));
         nodeProperties.put(SGA_NODE_CLOCK_SPEED_MHZ.value,     tryGet(nodeProperties, "clock_mhz",   "-1"));
   
         // dynamic
         //nodeProperties.put(SGA_NODE_CPU_PERC.value, tryGet(nodeProperties, "cpu_perc", "100"));
         nodeProperties.put(SGA_NODE_MEMORY_RAM_FREE_PERC.value,  tryGet(nodeProperties, "ram_used_perc",       "100"));
         nodeProperties.put(SGA_NODE_MEMORY_SWAP_FREE_PERC.value, tryGet(nodeProperties, "swap_used_perc",      "100"));
         nodeProperties.put(SGA_NODE_NUMBER_OF_JOBS.value,        tryGet(nodeProperties, "number_of_jobs",      "1"));
         // TODO: remove load averages
         nodeProperties.put(SGA_NODE_LOAD_AVG_1MIN_PERC.value,    tryGet(nodeProperties, "load_avg_1min_perc",  "0"));
         nodeProperties.put(SGA_NODE_LOAD_AVG_5MIN_PERC.value,    tryGet(nodeProperties, "load_avg_5min_perc",  "0"));
         nodeProperties.put(SGA_NODE_LOAD_AVG_15MIN_PERC.value,   tryGet(nodeProperties, "load_avg_15min_perc", "0"));
         
         //Resources
         int resourceCount = 1;
         while(sgaProperties.containsKey("resource."+resourceCount)) {
            nodeProperties.put(SGA_NODE_RESOURCES_SEQ.value + "." + resourceCount, sgaProperties.get("resource."+resourceCount));
            resourceCount++;
         }

         nodePropPairs[i] = mapToPairs(nodeProperties);
         i++;
      }

      return new SGAProperties(sgaPropPairs, nodePropPairs);
   }

   /**
    * Hack to force an SGAAlreadyRegisteredException if
    * SGA is already registered... registerSGA should have thrown
    * SGAAlreadyRegisteredException in this case, but apparently it doesn't.
    */
   private void throwIfRegistered(SGARestDaemon daemon, String sgaName) throws SGAAlreadyRegisteredException {
      boolean ok = true;
      try {
         ok = sgaService.isRegistered(daemon, sgaName);
      } catch (Exception e) {
         ok = true;
      }
      if (!ok) {
         throw new SGAAlreadyRegisteredException();
      }
   }

   private void digestStringMap(MessageDigest md, Map<String, String> map) {
      SortedSet<String> keys = new TreeSet<String>(map.keySet());
      for (String key : keys) {
         try {
            md.update(key.getBytes("UTF-8"));
            md.update(new byte[] { 0 });
            md.update(map.get(key).getBytes("UTF-8"));
            md.update(new byte[] { 0 });
         } catch (java.io.UnsupportedEncodingException e) {
            throw new RuntimeException("Unsupported encoding UTF-8?! JVM is broken");
         }
      }
   }

   private void digestStringMapMap(MessageDigest md, Map<String, Map<String, String>> map) {
      SortedSet<String> keys = new TreeSet<String>(map.keySet());
      for (String key : keys) {
         try {
            md.update(key.getBytes("UTF-8"));
            md.update(new byte[] { 0 });
            digestStringMap(md, map.get(key));
            md.update(new byte[] { 0 });
         } catch (java.io.UnsupportedEncodingException e) {
            throw new RuntimeException("Unsupported encoding UTF-8?! JVM is broken");
         }
      }
   }

   final protected static char[] hexArray = "0123456789ABCDEF".toCharArray();
   public static String bytesToHex(byte[] bytes) {
       char[] hexChars = new char[bytes.length * 2];
       for ( int j = 0; j < bytes.length; j++ ) {
           int v = bytes[j] & 0xFF;
           hexChars[j * 2] = hexArray[v >>> 4];
           hexChars[j * 2 + 1] = hexArray[v & 0x0F];
       }
       return new String(hexChars);
   }

   private String hashConfiguration(Map<String, String> sgaProperties, Map<String, Map<String, String>> nodes, Map<String, String> actions) {
      try {
         MessageDigest md = MessageDigest.getInstance("MD5");
         digestStringMap(md, sgaProperties);
         digestStringMapMap(md, nodes);
         digestStringMap(md, actions);
         
         return bytesToHex(md.digest());
      } catch (java.security.NoSuchAlgorithmException e) {
         throw new RuntimeException("Unsupported algorithm MD5?! JVM is broken");
      }
   }

   private void reportRetrievedJobs(String sgaName, List<RetrievedJob> jobList, SGARestDaemon daemon, boolean isNewDaemon) {
      RetrievedInfo[] retrieved = new RetrievedInfo[jobList.size()];
      int i = 0;
      for (RetrievedJob rjob : jobList) {
         retrieved[i] = new RetrievedInfo();
         retrieved[i].cmdId = rjob.cmd_id;
         SGARestCommand cmdRef = null;
         if (!isNewDaemon) {
            cmdRef = daemon.getCommand(rjob.cmd_id);
            cmdRef.setActions(rjob.actions);
         }
         if (cmdRef == null) {
            cmdRef = daemon.createCommand(rjob.cmd_id, rjob.actions);
         }
         retrieved[i].cmdRef = cmdRef;
         i++;
      }
      try {
         sgaService.commandRetrieved(sgaName, retrieved);
      } catch (Exception e) {
         logger.severe("Failed reporting retrieved commands for " + sgaName);
      }
   }

   private void reportLostJobs(String sgaName, List<String> cmdIds) {
      for (String cmdId : cmdIds) {
         try {
            sgaService.commandLost(sgaName, cmdId);
         } catch (Exception e) {
            logger.severe("Failed reporting lost command " + cmdId + " for " + sgaName);
         }
      }
   }
   
   public boolean register(String sgaName,
                           Map<String, String> sgaProperties,
                           Map<String, Map<String, String>> nodes,
                           Map<String, String> actions,
                           PersistentData persistentData
                          ) throws InvalidParameterException, NoPermissionException, SGAAlreadyRegisteredException, InvalidSGAException {
   
      SGARestDaemon daemon = new SGARestDaemon(sgaName, actions);
      
      String configHash = hashConfiguration(sgaProperties, nodes, actions);
      logger.fine("SGA config hash: " + configHash);
      DaemonData oldData = daemonData.get(sgaName);
      boolean isNewDaemon = true;
      if (oldData != null) {
         
         if (configHash.equals(oldData.configHash) && sgaService.isRegistered(oldData.daemon, sgaName)) {
            logger.fine("SGA was found and configuration is identical. Reuse it.");
            throwIfRegistered(oldData.daemon, sgaName);
            daemon = oldData.daemon;
            isNewDaemon = false;
         } else {
            logger.info("SGA was found but configuration has changed. Unregister and re-register.");
            try {
               unregister(sgaName);
            } catch (SGANotRegisteredException e) {
               daemonData.remove(sgaName);
            }
         }
      }
      
      IntHolder heartbeatInterval = new IntHolder();
      if (isNewDaemon) {
         try {
            boolean ok = sgaService.registerSGA(daemon, sgaName, makeSGAProperties(sgaProperties, nodes), heartbeatInterval);
            throwIfRegistered(daemon, sgaName);
            if (!ok) {
               logger.severe("Error registering SGA; CSBase did not specify the error condition.");
               return false;
            }
         } catch (Exception e) {
            throwIfRegistered(daemon, sgaName);
            logger.severe("Error registering SGA: " + e + ": " + e.getMessage());
            throw e;
         }
      }
      
      reportRetrievedJobs(sgaName, persistentData.retrieved, daemon, isNewDaemon);
      reportLostJobs(sgaName, daemon.cleanupJobs(persistentData));
      
      if (isNewDaemon) {
         daemon.setHeartbeatInterval(heartbeatInterval.value);
         daemonData.put(sgaName, new DaemonData(daemon, configHash, sgaProperties));
      }
      
      return true;
   }

   public void unregister(String sgaName) throws NoPermissionException, SGANotRegisteredException {
      SGARestDaemon daemon = getDaemon(sgaName);
      if (daemon == null) {
         throw new SGANotRegisteredException();
      }
      sgaService.unregisterSGA(daemon, sgaName);
      daemonData.remove(sgaName);
   }
   
   /**
    * Returns the heartbeat interval for a registered SGA or -1 if the sgaName is invalid.
    */
   public int getHeartbeatInterval(String sgaName) {
      SGARestDaemon daemon = getDaemon(sgaName);
      if (daemon == null) {
         return -1;
      }
      return daemon.getHeartbeatInterval();
   }

   /**
    * sends a heartbeat to the server, asking for an SGA to be kept alive.
    */
   public boolean keepAlive(String sgaName) throws InvalidSGAException, NoPermissionException {
      SGARestDaemon daemon = getDaemon(sgaName);
      if (daemon == null) {
         return false;
      }
      try {
         boolean ok = sgaService.isRegistered(daemon, sgaName);
         if (!ok) {
            daemonData.remove(sgaName);
         }
         return ok;
      } catch (Exception e) {
         logger.severe("Error in SGA heartbeat: " + e + ": " + e.getMessage());
         throw e;
      }
   }

   public boolean updateStatus(String sgaName, Map<String, Map<String, String>> nodes) throws InvalidParameterException, NoPermissionException, SGANotRegisteredException {
      DaemonData data = daemonData.get(sgaName);
      if (data == null) {
         throw new SGANotRegisteredException();
      }
      try {
         sgaService.updateSGAInfo(data.daemon, sgaName, makeSGAProperties(data.sgaProperties, nodes));
      } catch (Exception e) {
         logger.severe("Error updating SGA status: " + e + ": " + e.getMessage());
         throw e;
      }
      return true;
   }

   public boolean commandCompleted(String sgaName, String commandId, int wallTimeSec, int userTimeSec, int systemTimeSec) throws InvalidSGAException, NoPermissionException, InvalidCommandException {
      SGARestDaemon daemon = getDaemon(sgaName);
      if (daemon == null) {
         return false;
      }
      try {
         SGACommand cmd = daemon.removeCommand(commandId);
         return sgaService.commandCompleted(sgaName, cmd, commandId, new CompletedCommandInfo(wallTimeSec, userTimeSec, systemTimeSec));
      } catch (Exception e) {
         throw e;
      }
   }

}
