package csbase.sga.rest;

import java.util.List;
import java.util.ArrayList;
import java.util.Set;
import java.util.TreeSet;
import java.util.Map;
import java.util.HashMap;
import java.util.Properties;
import java.util.logging.Logger;

import javax.ws.rs.core.Response;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.client.Invocation;

import csbase.server.plugin.service.sgaservice.ISGADaemon;
import csbase.server.plugin.service.sgaservice.SGADaemonException;
import csbase.sga.rest.messages.PathResponse;
import csbase.sga.rest.messages.JobRequest;
import csbase.sga.rest.messages.JobResponse;
import csbase.sga.rest.messages.CommandStatusResponse;
import csbase.sga.rest.messages.parts.PersistentData;
import csbase.sga.rest.messages.parts.RetrievedJob;
import csbase.sga.rest.messages.parts.LostJob;

import sgaidl.InvalidPathException;
import sgaidl.MissingParameterException;
import sgaidl.Pair;
import sgaidl.PathNotFoundException;
import sgaidl.SGACommand;
import sgaidl.SGAControlAction;
import sgaidl.SGAPath;
import sgaidl.SystemException;

/**
 * Instance of ISGADaemon that's registered in the server.
 */
public class SGARestDaemon extends AbstractDaemon {

   /**
    * Name this SGA is registered with.
    * This is the name the SGA itself used when registering,
    * and which is passed on to CSBase to be used as this SGA's identifier.
    */
   private String name;
   
   /**
    * Map relating actions to URLs.
    */
   private Map<String, String> actions;

   /**
    * Map relating command ids to commands.
    */
   private Map<String, SGARestCommand> commands;
   
   /** Heartbeat interval in seconds.
    */
   private int heartbeatInterval;
   
   private Client client;
   
   public SGARestDaemon(String name, Map<String, String> actions) {
      this.name = name;
      this.actions = actions;
      this.commands = new HashMap<String, SGARestCommand>();
      this.logger = Logger.getLogger(this.getClass().getName() + "." + name);
      this.client = ClientBuilder.newClient()
                                 .register(org.glassfish.jersey.jackson.JacksonFeature.class);
   }
   
   public void setHeartbeatInterval(int seconds) {
      heartbeatInterval = seconds;
   }

   public int getHeartbeatInterval() {
      return heartbeatInterval;
   }

   @Override
   public SGAPath getPath(String path) throws InvalidPathException, PathNotFoundException {
      String url = actions.get("path");
      if (url == null) {
         throw new InvalidPathException("Can't query paths");
      }
      logger.info("Will query path " + path + " via " + url);
      Response response = client.target(url)
                                .queryParam("name", path)
                                .request("application/json")
                                .get();
      int code = response.getStatus();
      if (code == Response.Status.OK.getStatusCode()) {
         PathResponse pr = response.readEntity(PathResponse.class);
         response.close();
         return new SGAPath(pr.path, pr.size_bytes / 1024, pr.is_dir, pr.is_symlink, pr.link_path, 
                            pr.is_readable, pr.is_writable, pr.is_executable, true);
      } else if (code == Response.Status.NOT_FOUND.getStatusCode()) {
         response.close();
         throw new PathNotFoundException(path);
      }
      logger.severe("Error querying path " + path + ": " + code + " - " + response.readEntity(String.class) );
      response.close();
      throw new InvalidPathException(path);
   }

   private Map<String, String> pairsToMap(Pair[] pairs) {
      Map<String, String> map = new HashMap<String, String>();
      for (Pair pair : pairs) {
         map.put(pair.key, pair.value);
      }
      return map;
   }
   
   private String pairsToString(Pair[] pairs) {
      StringBuilder sb = new StringBuilder();
      String sep = "{ ";
      for (Pair pair : pairs) {
         sb.append(sep);
         sb.append("\"" + pair.key + "\" = \"" + pair.value + "\"");
         sep = ", ";
      }
      sb.append(" }");
      return sb.toString();
      
   }

   public SGARestCommand createCommand(String commandId, Map<String, String> actions) {
      SGARestCommand cmd = new SGARestCommand(commandId, actions);
      commands.put(commandId, cmd);
      return cmd;
   }

   @Override
   public SGACommand executeCommand(final String commandString, final String commandId, Pair[] extraParams) throws SystemException, MissingParameterException {
      String url = actions.get("job");
      if (url == null) {
         throw new SystemException("Can't submit job");
      }
      logger.info("Will submit job: " + commandString + " id: " + commandId + " extraParams: " + pairsToString(extraParams) + " via " + url);
      Entity<JobRequest> entity = Entity.entity(new JobRequest(commandString, commandId, pairsToMap(extraParams)), "application/json");
      logger.info("entity: " + entity);
      Response response = client.target(url)
                                .request("application/json")
                                .post(entity);
      Response.Status code = Response.Status.fromStatusCode(response.getStatus());
      if (code == Response.Status.CREATED) {
         JobResponse jr = response.readEntity(JobResponse.class);
         SGARestCommand cmd = createCommand(commandId, jr.actions);
         response.close();
         return cmd;
      } else if (code == Response.Status.BAD_REQUEST) {
         response.close();
         throw new MissingParameterException();
      }
      response.close();
      logger.severe("Error submitting command for execution.");
      throw new SystemException(response.readEntity(String.class));
   }

   public SGARestCommand removeCommand(String commandId) {
      return commands.remove(commandId);
   }

   SGARestCommand getCommand(String commandId) {
      return commands.get(commandId);
   }
   
   /**
    * Inform command ids of SGACommand instances that were managed by the
    * plugin but are not reported by the SGA as being alive.
    * Essentially, this "collects garbage" in the daemon's
    * internal map of commands.
    * Since SGARestDaemon instances are recycled, this is important to
    * avoid memory leaks in the internal map of commands.
    *
    * @param persistentData the persistent data structure, containing a list of
    * retrieved jobs that are known to the SGA, and a list of jobs
    * reported by the SGA as being lost.
    * @return a list of commandIds of jobs that are either
    * reported by the SGA as being lost or were dangling in this
    * object's internal map.
    */
   List<String> cleanupJobs(PersistentData persistentData) {
      List<String> lostJobs = new ArrayList<String>();
      Set<String> alive = new TreeSet<String>();
      Set<String> dead = new TreeSet<String>();
      for (RetrievedJob rjob : persistentData.retrieved) {
         alive.add(rjob.cmd_id);
      }
      for (String commandId : commands.keySet()) {
         if (!alive.contains(commandId)) {
            lostJobs.add(commandId);
            commands.remove(commandId);
            dead.add(commandId);
         }
      }
      for (LostJob ljob : persistentData.lost) {
         if (!dead.contains(ljob.cmd_id)) {
            lostJobs.add(ljob.cmd_id);
         }
      }
      return lostJobs;
   }

   @Override
   public void control(SGAControlAction sca) {
      try {
         String action;
         if (sca.equals(SGAControlAction.SHUTDOWN)) {
            action = "shutdown";
         } else if (sca.equals(SGAControlAction.RESTART)) {
            action = "restart";
         } else {
            throw new Exception("Unknown control action: " + sca);
         }
         String url = actions.get(action);
         if (url == null) {
            throw new Exception("SGA didn't tell us URI for: " + action);
         }
         logger.info("Sending SGA control action '" + action + "' via " + url);
         Response response = client.target(url)
                                   .request()
                                   .get();
         int code = response.getStatus();
         response.close();
         if (code == Response.Status.OK.getStatusCode()) {
            logger.info("SGA control action '" + action + "' response: " + code);
         } else {
            throw new Exception("SGA control action " + action + " response: " + code);
         }
      } catch (Exception e) {
         logger.severe(e.getMessage());
      }
   }

}
