package csbase.sga.rest;

import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Map;
import java.util.List;
import java.util.LinkedList;
import java.util.logging.Logger;

import javax.ws.rs.core.Response;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Client;

import csbase.server.plugin.service.sgaservice.SGADaemonCommand;
import csbase.sga.rest.CSBaseFacade;
import csbase.sga.rest.messages.CommandStatusResponse;

import sgaidl.Pair;
import sgaidl.SGACommand;
import sgaidl.JobControlAction;
import sgaidl.InvalidActionException;
import sgaidl.ActionNotSupportedException;
import sgaidl.InvalidTransitionException;

public class SGARestCommand extends SGADaemonCommand {

   private final String commandId;
   
   private final Logger logger;

   private final Client client;

   /**
    * Map relating actions to URLs.
    */
   private Map<String, String> actions;
   
   public SGARestCommand(String commandId, Map<String, String> actions) {
      this.commandId = commandId;
      this.actions = actions;
      this.logger = Logger.getLogger(this.getClass().getName() + "." + commandId);
      this.client = ClientBuilder.newClient()
                                 .register(org.glassfish.jersey.jackson.JacksonFeature.class);

   }
   
   public void setActions(Map<String, String> actions) {
      this.actions = actions;
   }

   private sgaidl.RunningCommandInfo convertCommandStatus(CommandStatusResponse commandStatus) {
      List<Pair[]> processData = new LinkedList<Pair[]>();

      for (Map<String, String> ps : commandStatus.processes) {
         List<Pair> pDic = new LinkedList<Pair>();
         for (String key : ps.keySet()) {
            String value = ps.get(key);
            // Avoid crashes in CSBase by pre-checking the types of some values.
            if (key.endsWith("_sec")) {
               try {
                  int intValue = (int) Double.parseDouble(value);
                  value = "" + intValue;
               } catch (Exception e) {
                  value = "-1";
               }
            } else if (key.endsWith("_mb") || key.endsWith("_kb")) {
               try {
                  double doubleValue = (int) Double.parseDouble(value);
               } catch (Exception e) {
                  value = "-1";
               }
            } else if (key.startsWith("monitor_file_chunk")) {
               try {
                  value = new String(Base64.getDecoder().decode(value), StandardCharsets.ISO_8859_1);
               }
               catch (Exception e){
                  // Do nothing
               }
            }
            // TODO we really need to check that the set of mandatory fields is set here.
            pDic.add(new Pair("csbase_command_" + key, value));
         }
         processData.add(pDic.toArray(new Pair[0]));
      }
      // TODO force progress to be on the execution data structure for compatibility reasons
      List<Pair> executionData = new LinkedList<>();
      if (commandStatus.processes.get(0).containsKey("progresso")) {
         executionData.add(new Pair("progresso", commandStatus.processes.get(0).get("progresso")));
      }
      /*
      const string COMMAND_EXECUTION_PATH = "csbase_command_path";
      const string COMMAND_EXECUTION_OUTPUT_PATH = "csbase_command_output_path";
      const string COMMAND_EXECUTION_SANDBOX_PATHS_SEQ = "csbase_command_sandbox_paths";
      */
      return new sgaidl.RunningCommandInfo(processData.toArray(new Pair[0][]), executionData.toArray(new Pair[0]));
   }

   public sgaidl.RunningCommandInfo getRunningCommandInfo() {
      String url = actions.get("status");
      if (url == null) {
         logger.severe("Error, no 'status' action defined for command " + commandId);
      }
      logger.info("Will query status for command " + commandId +" via " + url);
      Response response;
      try {
         response = client.target(url)
                          .request("application/json")
                          .get();
      } catch (Exception e) {
         logger.severe("Failed querying " + url + " - " + e);
         return null;
      }
      logger.info("Queried status for command " + commandId +" via " + url);
      int code = response.getStatus();
      if (code != Response.Status.OK.getStatusCode()) {
         //logger.severe("Error reading status of command " + commandId + ": " + code + " - " + response.readEntity(String.class));
         return null;
      }
      CommandStatusResponse commandStatus = response.readEntity(CommandStatusResponse.class);
      return convertCommandStatus(commandStatus);
   }
   
   public void control(JobControlAction jca, String childJobId) throws InvalidActionException, ActionNotSupportedException, InvalidTransitionException {
      // childJobId is not sent by the server.
      try {
         String action;
         if (jca.equals(JobControlAction.TERMINATE)) {
            action = "terminate";
         } else if (jca.equals(JobControlAction.SUSPEND)) {
            action = "suspend";
         } else if (jca.equals(JobControlAction.RESUME)) {
            action = "resume";
         } else if (jca.equals(JobControlAction.HOLD)) {
            action = "hold";
         } else if (jca.equals(JobControlAction.RELEASE)) {
            action = "release";
         } else {
            throw new Exception("Unknown control action: " + jca);
         }
         String url = actions.get(action);
         if (url == null) {
            throw new Exception("SGA didn't tell us URI for: " + action);
         }
         logger.info("Sending job control action '" + action + "' via " + url);
         Response response = client.target(url)
                                   .request()
                                   .get();
         int code = response.getStatus();
         response.close();
         if (code == Response.Status.OK.getStatusCode()) {
            logger.info("Job control action " + action + " response: " + code);
         } else {
            throw new Exception("Job control action " + action + " response: " + code);
         }
      } catch (Exception e) {
         logger.severe(e.getMessage());
      }
   }

}
