package csbase.sga.rest.resources;

import csbase.sga.rest.messages.*;
import csbase.sga.rest.messages.parts.*;
import csbase.sga.rest.*;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.POST;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Response;
import javax.validation.Valid;

import java.util.Map;
import java.util.HashMap;

import sgaidl.InvalidSGAException;
import sgaidl.InvalidCommandException;
import sgaidl.InvalidParameterException;
import sgaidl.NoPermissionException;
import sgaidl.SGAAlreadyRegisteredException;
import sgaidl.SGANotRegisteredException;

/**
 * Root resource (exposed at "myresource" path)
 */
@Path("v1/sga")
public class SGA {

   private static final String root = "/v1/sga";

   @POST
   @Path("/")
   @Consumes("application/json")
   @Produces("application/json")
   public Response postIt(@Valid RegistrationRequest registration) {
      boolean ok = true;
      Response.Status code = Response.Status.CREATED;
      String error = null;
      Map<String, String> properties = new HashMap<String, String>();
      String sgaName = registration.name;
      properties.put("name", sgaName);
      properties.put("platform", registration.platform);
      properties.put("project_root_dir", registration.project_root_dir);
      properties.put("algorithm_root_dir", registration.algorithm_root_dir);
      properties.put("sandbox_root_dir", registration.sandbox_root_dir);
      if (registration.resources != null) {
         int resourceCount = 1;
         for (String resource : registration.resources) {
            properties.put("resource." + resourceCount, resource);
            resourceCount++;
         }
      }
      if(registration.extra_config != null) {
         properties.putAll(registration.extra_config);
      }
      try {
         ok = SGARestPlugin.csbase.register(registration.name, properties, registration.nodes, registration.actions, registration.persistent_data);
         if (!ok) {
            code = Response.Status.BAD_REQUEST;
            error = "Error registering. CSBase did not provide further diagnostics.";
         }
      } catch (InvalidSGAException e) {
         code = Response.Status.BAD_REQUEST;
         error = "Invalid SGA (?)";
      } catch (InvalidParameterException e) {
         code = Response.Status.BAD_REQUEST;
         error = "Invalid parameter";
      } catch (NoPermissionException e) {
         code = Response.Status.FORBIDDEN;
         error = "No permission";
      } catch (SGAAlreadyRegisteredException e) {
         code = Response.Status.FOUND;
         error = "SGA already registered";
         ok = true;
      }
      RegistrationResponse response;
      if (ok) {
         RegistrationResponseActions responseActions = new RegistrationResponseActions();
         responseActions.actions.heartbeat  = new HeartbeatAction(SGARestPlugin.csbase.getHeartbeatInterval(sgaName), RestServer.path(root, sgaName + "/heartbeat"));
         responseActions.actions.status     = new Action(RestServer.path(root, sgaName + "/status"));
         responseActions.actions.completion = new Action(RestServer.path(root, sgaName + "/completion"));
         response = responseActions;
      } else {
         assert error != null;
         response = new RegistrationResponseError(error);
      }
      return Response.status(code).entity(response).build();
   }

   @GET
   @Path("/{name:[A-Za-z0-9_]+}/heartbeat")
   @Consumes("*/*")
   @Produces("text/plain")
   public Response heartbeat(@PathParam("name") String sgaName) {
      Response.Status code = Response.Status.OK;
      try {
         boolean ok = SGARestPlugin.csbase.keepAlive(sgaName);
         if (!ok) {
            code = Response.Status.NOT_FOUND;
         }
      } catch (InvalidSGAException e) {
         code = Response.Status.BAD_REQUEST;
      } catch (NoPermissionException e) {
         code = Response.Status.FORBIDDEN;
      } 
      return Response.status(code).build();
   }

   @POST
   @Path("/{name:[A-Za-z0-9_]+}/status")
   @Consumes("application/json")
   @Produces("text/plain")
   public Response postIt(@PathParam("name") String sgaName, @Valid StatusRequest status) {
      Response.Status code = Response.Status.OK;
      try {
         boolean ok = SGARestPlugin.csbase.updateStatus(sgaName, status.nodes);
         if (!ok) {
            code = Response.Status.BAD_REQUEST;
         }
      } catch (InvalidParameterException e) {
         code = Response.Status.BAD_REQUEST;
      } catch (NoPermissionException e) {
         code = Response.Status.FORBIDDEN;
      } catch (SGANotRegisteredException e) {
         code = Response.Status.UNAUTHORIZED;
      }            
      return Response.status(code).build();
   }

   @POST
   @Path("/{name:[A-Za-z0-9_]+}/completion")
   @Consumes("application/json")
   @Produces("text/plain")
   public Response postIt(@PathParam("name") String sgaName, @Valid CompletionRequest cr) {
      Response.Status code = Response.Status.OK;
      boolean ok = false;
      try {
         ok = SGARestPlugin.csbase.commandCompleted(sgaName, cr.cmd_id, cr.walltime_s, cr.usertime_s, cr.systime_s);
         if (!ok) {
            code = Response.Status.NOT_FOUND;
         }
      } catch (InvalidSGAException e) {
         code = Response.Status.BAD_REQUEST;
      } catch (NoPermissionException e) {
         code = Response.Status.FORBIDDEN;
      } catch (InvalidCommandException e) {
         code = Response.Status.BAD_REQUEST;
      }            
      return Response.status(code).build();
   }

}
