/*
 * $Id:$
 */

package csbase.tools;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.StreamCorruptedException;

import tecgraf.javautils.core.lng.LNG;
import csbase.logic.CommonProjectInfo;
import csbase.logic.ProjectAttribute;

/**
 * Classe para verificao do contedo do atributo
 * {@link ProjectAttribute#LOCKING_AREA_SIZE} na hashtable de projetos.
 * 
 * Caso este atributo seja do tipo Integer,  emitida uma mensagem de erro no
 * console, indicando o projeto em questo.
 * 
 * Para maiores informaes, ver issue CSBASE-2426.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class IncorrectLockingAreaSizeIdentifier {

  /**
   * Main
   * 
   * @param args Argumentos nome do arquivo .csbase_project_info a ser testado.
   */
  static public void main(String[] args) {
    if (args.length < 1) {
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.configfile.undefined");
      System.out.println(err + "\n\n");
      printHelp();
      System.exit(-1);
    }

    if (args.length > 2) {
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.configfile.numargument");
      System.out.println(err);
      System.out.println(err + "\n\n");
      printHelp();
      System.exit(-2);
    }

    final String firstArg = args[0];
    if (firstArg == null || firstArg.trim().isEmpty()) {
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.configfile.null");
      System.out.println(err);
      System.exit(-3);
      return;
    }

    if (firstArg.trim().equalsIgnoreCase("--help")) {
      printHelp();
      return;
    }

    final File file = new File(firstArg);
    final String absPath = file.getAbsolutePath();
    if (!file.exists()) {
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.configfile.inexistent", new String[]{absPath});
      System.out.println(err);
      System.exit(-4);
    }

    if (file.isDirectory()) {
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.configfile.isadir", new String[]{absPath});
      System.out.println(err);
      System.exit(-5);
    }

    if (!file.canRead()) {
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.configfile.nopermission", new String[]{absPath});
      System.out.println(err);
      System.exit(-6);
    }

    // Testando caso especfico detectado em instalao de produo
    // do WebSintesi onde projetos (criados em verses antigas?)
    // ficam com tamanho zerado. Neste caso, o administrador
    // precisar rodar a ferramenta de recuperao.
    if (file.length() <= 0) {
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.configfile.zerolength", new String[]{absPath});
      System.out.println(err);
      System.exit(-7);
    }

    boolean forceCorrection = false;
    if (args.length == 2) {
      final String secArg = args[1];
      if (secArg.trim().equalsIgnoreCase("--fix")) {
        forceCorrection = true;
      }
    }

    try {
      processProjectInfo(file, forceCorrection);
    }
    catch (StreamCorruptedException e) {
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.configfile.notobjectstream", new String[]{absPath});
      System.out.println(err);
      System.exit(-7);
    }
    catch (IOException e) {
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.io", new String[]{e.getMessage()});
      System.out.println(err);
      System.exit(-8);
    }
    catch (ClassNotFoundException e) {
      final String msg = e.getMessage();
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.info.class.notfound", new String[]{msg});
      System.out.println(err);
      System.exit(-9);
    }
    catch (ClassCastException e) {
      final String msg = e.getMessage();
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.configfile.notfromproject", new String[]{absPath});
      System.out.println(err + " - " + msg);
      System.exit(-10);
    }
    catch (Throwable t) {
      final String msg = t.getMessage();
      final String err = LNG.get("IncorrectLockingAreaSizeIdentifier.error.internali1", new String[]{msg});
      System.out.println(err);
      t.printStackTrace();
      System.exit(-11);
    }
  }

  /**
   * Imprime mensagem de help.
   */
  private static void printHelp() {
    System.out.println(LNG.get("IncorrectLockingAreaSizeIdentifier.help"));
  }

  /**
   * L as informaes do projeto do seu arquivo de configurao, e o corrige
   * caso forceCorrection seja true.
   * 
   * @param cfgFile arquivo de configurao
   * @param forceCorrection Indica se o atributo deve ser consertado.
   * @throws IOException erro de I/O
   * @throws ClassNotFoundException no caso da clase no ser encontrada.
   * @see CommonProjectInfo
   */
  private static void processProjectInfo(File cfgFile,
    final boolean forceCorrection) throws IOException, ClassNotFoundException {
    final FileInputStream inFileStream = new FileInputStream(cfgFile);
    final BufferedInputStream inBuffStream =
      new BufferedInputStream(inFileStream);
    final DataInputStream inDataStream = new DataInputStream(inBuffStream);
    final ObjectInputStream in = new ObjectInputStream(inDataStream);
    final Object object = in.readObject();
    in.close();

    CommonProjectInfo info = CommonProjectInfo.class.cast(object);
    final ProjectAttribute prjAttr = ProjectAttribute.LOCKING_AREA_SIZE;
    final String prjAttrKey = prjAttr.getAttributeKey();
    //      System.out.println("Projeto " + info.name + ", usurio " + info.userId);
    final Object att = info.getAttribute(prjAttrKey);
    if (att instanceof Integer) {
      System.err.println(LNG.get("IncorrectLockingAreaSizeIdentifier.error.lockingareasize.isinteger", new String[]{info.name, info.userId.toString()}));

      if (forceCorrection) {
        final Integer lockingAreaSizeInt = (Integer) att;
        final Long lockingAreaSizeLong = lockingAreaSizeInt.longValue();
        info.setAttribute(prjAttrKey, lockingAreaSizeLong);

        final FileOutputStream outFileStream = new FileOutputStream(cfgFile);
        final BufferedOutputStream outBuffStream =
          new BufferedOutputStream(outFileStream);
        final DataOutputStream outDataStream =
          new DataOutputStream(outBuffStream);
        final ObjectOutputStream out = new ObjectOutputStream(outDataStream);
        out.writeObject(info);
        System.out.println(LNG.get("IncorrectLockingAreaSizeIdentifier.file.corrected", new String[]{cfgFile.getAbsolutePath()}));
        out.close();
      }
    }
  }
}
