package csbase.util.data.channel;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

import tecgraf.javautils.core.filter.IFilter;
import csbase.util.data.dispatcher.IDispatchListener;
import csbase.util.data.dispatcher.IDispatcher;

/**
 * Representa um canal de dados do tipo publish/subscribe.
 * 
 * @param <S> Tipo dos assinantes.
 * @param <D> Tipo dos dados sendo difundidos.
 * 
 * @author Tecgraf
 */
public class DataChannel<S, D> {

  /**
   * Estratgia de entrega dos dados para os assinantes.
   */
  private IDispatcher<S, D> dispatcher;
  /**
   * Mtodo chamado pelo despachante para indicar o status das entregas.
   */
  private IDispatchListener<S, D> dispatchListener;

  /**
   * <p>
   * Lista de todos os assinantes.<br>
   * Um {@link DataObserver observador de dados}  um encapsulador de assinante.
   * </p>
   * <p>
   * Utilizada pelo mtodo {@link #unsubscribe(Object)} para descobrir qual
   * assinante deve ser removido da {@link #dataNotifier estrutura}.
   * </p>
   */
  private List<DataObserver> observers;
  /**
   * Estrutura responsvel por notificar todos os {@link DataObserver
   * observadores} quando dados forem enviados.
   */
  private Observable dataNotifier;

  /**
   * Construtor.
   * 
   * @param dispatcher Estratgia utilizada para entregar os dados aos
   *        assinantes.
   * @param dispatchListener Ouvinte de entrega de dados.
   */
  public DataChannel(IDispatcher<S, D> dispatcher,
    IDispatchListener<S, D> dispatchListener) {
    this.dispatcher = dispatcher;
    this.dispatchListener = dispatchListener;

    this.observers = new ArrayList<DataObserver>();

    this.dataNotifier = new Observable() {
      /**
       * Chama o mtodo {@link Observable#setChanged()} e repassa a mensagem
       * para o {@link #notifyObservers()} do pai, fazendo com que elas sejam
       * disparadas automaticamente a cada nova notificao.
       */
      @Override
      public void notifyObservers(Object arg) {
        setChanged();
        super.notifyObservers(arg);
      }
    };
  }

  /**
   * Publica dados no canal.
   * 
   * @param data Dados a serem publicados.
   */
  public void publish(D... data) {
    if (data == null) {
      throw new NullPointerException("data");
    }
    if (data.length > 0) {
      dataNotifier.notifyObservers(data);
    }
  }

  /**
   * Adiciona um assinante ao canal.
   * 
   * @param subscriber Assinante interessado em receber dados.
   * @param selector Responsvel por selecionar os dados de interesse do
   *        assinante.
   * 
   * @return o nmero de assinantes restantes.
   * 
   * @throws IllegalArgumentException Se o assinante j estiver cadastrado.
   */
  public synchronized int subscribe(S subscriber, IFilter<D> selector)
    throws IllegalArgumentException {

    if (subscriber == null) {
      throw new NullPointerException("subscriber == null");
    }
    if (selector == null) {
      throw new NullPointerException("selector == null");
    }
    for (DataObserver anObserver : observers) {
      if (anObserver.subscriber.equals(subscriber)) {
        throw new IllegalArgumentException("Assinante j cadastrado.");
      }
    }

    DataObserver observer = new DataObserver(subscriber, selector);
    observers.add(observer);
    dataNotifier.addObserver(observer);
    return observers.size();
  }

  /**
   * <p>
   * Remove um determinado assinante do canal.
   * </p>
   * <p>
   * Uma vez removido, o assinante no ir mais receber dados a menos que seja
   * inserido de novamente atravs do mtodo {@link #subscribe(Object, IFilter)}
   * .
   * </p>
   * 
   * @param subscriber Assinante a ser removido.
   * 
   * @return o nmero de assinantes restantes.
   * 
   * @see #subscribe(Object, IFilter)
   */
  public synchronized int unsubscribe(S subscriber) {
    if (subscriber == null) {
      throw new NullPointerException("listener == null");
    }

    Iterator<DataObserver> iter = observers.iterator();
    while (iter.hasNext()) {
      DataObserver observer = iter.next();
      if (observer.subscriber.equals(subscriber)) {
        iter.remove();
        this.dataNotifier.deleteObserver(observer);
        break;
      }
    }
    return observers.size();
  }

  /**
   * Remove todos os assinantes do canal.
   */
  public void unsubscribeAll() {
    observers.clear();
    dataNotifier.deleteObservers();
  }

  /**
   * Obtm o nmero de assinantes cadastrados neste canal.
   * 
   * @return o nmero de assinantes cadastrados neste canal.
   */
  public int countListeners() {
    return observers.size();
  }

  /**
   * Observa um {@link DataChannel canal de dados} e repassa as mensagens
   * selecionadas para um determinado assinante.
   */
  private class DataObserver implements Observer {

    /**
     * Assinante do canal.
     */
    private S subscriber;
    /**
     * Responsvel por selecionar as mensagens que interesso ao assinante.
     */
    private IFilter<D> selector;

    /**
     * Construtor.
     * 
     * @param subscriber Assinante do canal.
     * @param selector Responsvel por selecionar as mensagens que interesso ao
     *        assinante.
     */
    DataObserver(S subscriber, IFilter<D> selector) {
      super();
      this.subscriber = subscriber;
      this.selector = selector;
    }

    /**
     * Escuta os dados repassados pelo {@link #dataNotifier}, filtra eles com o
     * {@link #selector seletor} e repassa para o {@link #subscriber assinante}.
     * 
     * @param o {@link #dataNotifier notificador de dados}.
     * @param arg os dados recebidos.
     */
    @Override
    @SuppressWarnings("unchecked")
    public void update(Observable o, Object arg) {
      D[] data = (D[]) arg;
      // Verifica quais dados interessam ao assinante. 
      List<D> accepteds = new ArrayList<D>();
      for (D aData : data) {
        if (selector.accept(aData)) {
          accepteds.add(aData);
        }
      }
      // Se o assinante no tem interesse em nenhum dado...
      if (accepteds.size() == 0) {
        // ...ignora os dados.
        return;
      }
      // Se o assinante tem interesse em todos os dados...
      if (accepteds.size() == data.length) {
        // ...repassa o array de dados que j existe. 
        dispatcher.dispatch(dispatchListener, subscriber, data);
      }
      else {
        /*
         * Caso contrrio,  necessrio criar um novo array contendo apenas os
         * dados de interesse.
         */

        // Obtm o tipo dos dados do array.
        Class<?> dataType = data.getClass().getComponentType();
        // Cria um novo array para armazenar os dados aceitos.
        D[] acceptedsArray =
          (D[]) Array.newInstance(dataType, accepteds.size());
        // Copia os dados aceitos para aquele array.
        accepteds.toArray(acceptedsArray);
        dispatcher.dispatch(dispatchListener, subscriber, acceptedsArray);
      }
    }
  }
}