package csbase.util.data.dispatcher;

import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import tecgraf.javautils.core.lng.LNG;

/**
 * Um {@link IDispatchListener} que loga o resultado das entregas.
 * 
 * @param <T> Tipo do destino.
 * @param <D> Tipo dos dados despachados.
 * 
 * @author Tecgraf
 */
public class DispatchLogger<T, D> implements IDispatchListener<T, D> {

  /**
   * Nome da classe (utilizado para internacionalizao a priori).
   */
  final static String className = DispatchLogger.class.getSimpleName();
	
  /**
   * Utilizado para <i>logar</i> informaes no sistema.
   */
  private final Logger LOGGER;

  /**
   * Despachante sendo decorado.
   */
  private IDispatchListener<T, D> decorated;

  /**
   * Construtor.
   */
  public DispatchLogger() {
    this(new IDispatchListener<T, D>() {
      @Override
      public void onDataDelivered(T destination, D... data) {}
      @Override
      public void onExceptionThrown(Exception e, T destination, D... data) {}
    });
  }

  /**
   * Constri um {@link DispatchLogger} que decora um {@link IDispatchListener}
   * logando o resultado das entregas antes de repassar a ele a informao.
   * 
   * @param decorated Ouvinte de entregas sendo decrado.
   */
  public DispatchLogger(IDispatchListener<T, D> decorated) {
    this.decorated = decorated;
    LOGGER = Logger.getLogger(getClass().getName());
    LOGGER.setLevel(Level.ALL);
  }

  /**
   * <p>
   * Altera o nvel do log.
   * </p>
   * <p>
   * O nvel indica que mensagens sero logadas.<br>
   * As mensagens de erro de entrega so logadas no nvel {@link Level#WARNING}
   * e as de entrega bem sucedida so logadas no nvel {@link Level#FINER}.
   * </p>
   * 
   * @param level novo nvel do log.
   * 
   * @see Level
   */
  public void setLevel(Level level) {
    LOGGER.setLevel(Level.ALL);
  }

  /**
   * Loga o erro de entrega no nvel {@link Level#WARNING}. {@inheritDoc}
   */
  @Override
  public void onExceptionThrown(Exception e, T destination, D... data) {
    LogRecord record =
      new LogRecord(Level.WARNING, createExceptionThrown(destination, data));
    record.setThrown(e);
    LOGGER.log(record);
    
    decorated.onExceptionThrown(e, destination, data);
  }

  /**
   * Loga a entrega no nvel {@link Level#FINER}. {@inheritDoc}
   */
  @Override
  public void onDataDelivered(T destination, D... data) {
    LOGGER.log(Level.FINER, createDataDeliveredMessage(destination, data));
    
    decorated.onDataDelivered(destination, data);
  }

  /**
   * Cria a mensagem que ser logada quando houver erro na entrega dos dados.
   * 
   * @param destination Destino dos dados.
   * @param data Dados que no foram entregues.
   * 
   * @return a mensagem que ser logada quando houver erro na entrega dos dados.
   */
  protected String createExceptionThrown(T destination, D... data) {
    return String
      .format(LNG.get(className + ".deliverytodestination.inputerror", new Object[]{data.length, data[0].getClass().getName(), destination.getClass().getName()}));
  }

  /**
   * Cria a mensagem que ser logada quando os dados forem entregues com
   * sucesso.
   * 
   * @param destination Destino dos dados.
   * @param data Dados entregues.
   * 
   * @return a mensagem que ser logada quando os dados forem entregues com
   *         sucesso.
   */
  protected String createDataDeliveredMessage(T destination, D... data) {
    return String.format(LNG.get(className + ".deliverytodestination.success", new Object[]{data.length, data[0].getClass().getName(), destination.getClass()
            .getName()}));
  }
}
