/**
 * $Id$
 */
package validations;

import java.io.BufferedInputStream;
import java.io.Console;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.lang.reflect.Modifier;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Level;

import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;

import csbase.util.FileSystemUtils;
import validations.AbstractValidation.Status;
import validations.util.ValidationsLogger;
import validations.util.ValidatorUtils;

/**
 * Classe que executa validaes sobre uma instalao do sistema.
 *
 * Todas as subclasses de {@link AbstractValidation} implementadas no pacote
 * {@code validations} do CSBase e no pacote {@code validations_sys}
 * do projeto sero automaticamente executadas.
 *
 * Os parmetros da linha de comando so processados e definem a configurao de
 * alguns parmetros da execuo que podem ser consultados pelas validaes. O
 * nico parmetro obrigatrio  o path para o arquivo que contm as
 * propriedades globais definidas para a instalao em questo do sistema. Deste
 * arquivo so extradas propriedades obrigatrias para o processo de validao:

 * <ul>
 * <li>{@code ProjectService.baseProjectDir} path para o diretrio
 * {@code project} do CSBase. Pode ser absoluto ou relativo ao diretrio
 * onde se encontra o arquivo de propriedades.
 * <li>{@code Validator.tempDir} path para o diretrio temporrio onde
 * sero criados os backups e onde ser armazenado o log da validao (
 * {@code validacoes.log}).
 * </ul>
 * Alm das propriedades obrigatrias, existem algumas propriedades opcionais
 * que podem ser usadas para ajustes finos:
 *
 * <ul>
 * <li>{@code Validator.logPath} path para o log da validao.
 * </ul>
 *
 * As validaes recebem o validador como parmetro e podem us-lo p.ex. para
 * consultar as propriedades lidas do arquivo de propriedades.
 *
 * Um logger especfico  criado por default, para registro do processo tanto no
 * console quanto no arquivo {@code validacoes.log}. O nvel de logging
 * para o arquivo  definido como {@link Level#FINE} enquanto que o nvel
 * default para o console  {@link Level#INFO}.
 *
 * As validaes so instanciadas por reflexividade e ordenadas de acordo com
 * seus nomes. Normalmente a ordem entre as validaes no deve ser um fator
 * relevante mas, caso seja, pode-se adotar uma poltica semelhante  usada para
 * os scripts de inicializao UNIX (p.ex. {@code Val05XXX},
 * {@code Val10YYY} etc.).
 *
 * O validador pode ser executado em <i>modo validao</i> (parmetro
 * {@code -V}) ou em <i>modo patch</i> (default). No modo validao os
 * backups no so criados, e nenhuma alterao  feita. No modo patch, caso a
 * aplicao de um dos patches falhe, este e todos os demais aplicadas antes
 * dele so desfeitos, na ordem inversa em que foram executados.
 *
 * Os diretrios de backup so removidos apenas em duas situaes:
 * <ol>
 * <li>quando todos os patches foram aplicados com sucesso
 * <li>quando algum patch falhou mas todos os <i>rollbacks</i> foram
 * bem-sucedidos
 * </ol>
 * Em todos os demais casos, os diretrios de backup so preservados para
 * restaurao "manual" dos dados alterados (que deve ser feita sob superviso
 * da equipe de desenvolvimento). As validaes no so executadas se seus
 * respectivos diretrios de backup no existem, i.e. uma tentativa de
 * instalao que tenha sido mal-sucedida e tenha deixado para trs diretrios
 * de backup impedir instalaes subsequentes at que o problema tenha sido
 * corrigido e os diretrios de backup tenham sido removidos.
 *
 * Instalaes bem sucedidas podem ser repetidas sem efeito colateral, i.e.
 * patches aplicados com sucesso no sero aplicados novamente em instalaes
 * subsequentes.
 * 
 * @see #processArgs(String[])
 * @see #isValidatingOnly()
 * @see #isVerbose()
 * @see #getSystemProperty(String)
 * @see #getLogger()
 * @see #rollback(AbstractValidation)
 * @see #run()
 * @see ValidationsLogger
 * @see AbstractValidation
 * @see ValidatorUtils
 * 
 * @author Tecgraf
 */
public class Validator {
  /**
   * Extenso para os marcadores das validaes.
   */
  private static final String VALIDATION_MARKER_EXTENSION = ".ok";

  /**
   * Nome do pacote que contm as validaes do CSBase.
   */
  private static final String FRAMEWORK_VALIDATIONS_PACKAGE = "validations";

  /**
   * Nome do pacote que contm as validaes do sistema.
   */
  private static final String SYSTEM_VALIDATIONS_PACKAGE = "validations_sys";

  /**
   * Chave para a propriedade que contm o path para o diretrio temporrio da
   * instalao.
   */
  public static final String VALIDATIONS_TEMP_DIR = "Validator.tempDir";

  /**
   * Nome do arquivo default de propriedades.
   */
  private static final String DEFAULT_PROPERTIES_FILE = "System.properties";

  /**
   * Nome da propriedade (opcional) que contm o path para o arquivo de log da
   * instalao.
   */
  private static final String LOGPATH_PROPERTY_NAME = "Validator.logPath";

  /**
   * Nome default para o arquivo de log que ser criado no diretrio temporrio
   * caso o usurio no especifique um arquivo de logs.
   */
  private static final String DEFAULT_LOGFILE_NAME = "validacoes.log";

  /**
   * Nome da propriedade que armazena o caminho para o diretrio de execuo do
   * sistema. Pode ser absoluto ou relativo ao diretrio de instalao do
   * sistema.
   */
  private static final String SYSTEM_EXECDIR_PROP = "Server.system.exec.dir";

  /**
   * Nome da propriedade que armazena o caminho para o diretrio de projetos do
   * sistema. Pode ser absoluto ou relativo ao diretrio de execuo do sistema.
   */
  private static final String PRJ_BASEDIR_PROP =
    "ProjectService.base.project.dir";

  /**
   * Nome da propriedade que armazena o caminho para o diretrio de algoritmos
   * do sistema. Pode ser absoluto ou relativo ao diretrio de execuo do
   * sistema.
   */
  private static final String ALGO_BASEDIR_PROP =
    "AlgorithmService.base.algorithm.dir";

  /**
   * Nome do arquivo de exemplo com valores default para as configuraes. Caso
   * o usurio no fornea o path para o arquivo de propriedades, ele ter a
   * chance de criar um arquivo de propriedades a partir deste.
   */
  public static final String SAMPLE_PROPERTIES_FILE =
    SYSTEM_VALIDATIONS_PACKAGE + File.separatorChar + DEFAULT_PROPERTIES_FILE
      + ".default";

  /**
   * Referncia para o repositrio de projetos.
   */
  private File baseProjectDir;

  /**
   * Referncia para o repositrio de algoritmos.
   */
  private File baseAlgorithmDir;

  /**
   * Referncia para o diretrio de execuo do sistema.
   */
  private File baseExecutionDir;

  /**
   * Lista de validaes.
   * 
   * @see #registerValidations()
   */
  private List<AbstractValidation> validations;

  /**
   * Flag que indica se o usurio deseja apenas validar a instalao, sem
   * aplicar as correes.
   */
  private boolean validateOnly;

  /**
   * Propriedades do sistema.
   */
  private Properties systemProperties;

  /**
   * Logger para registro de mensagens. Possui sempre um {@link ConsoleHandler},
   * mas pode possuir tambm um {@link FileHandler} para cada validao.
   */
  private ValidationsLogger logger;

  /**
   * Path absoluto do diretrio corrente.
   */
  private String currDir;

  /**
   * Flag que indica se o validador foi instanciado com sucesso e est pronto
   * para ser executado.
   */
  private boolean readyToRun;

  /**
   * Path para o diretrio temporrio da instalao.
   */
  private String instTempDirPath;

  /**
   * Flag que indica se estamos em modo verbose.
   */
  private boolean verboseMode;

  /**
   * Conjunto das validaes que no foram executadas.
   * 
   * @see #hasAlreadyRun(AbstractValidation)
   */
  private final Set<AbstractValidation> skippedValidations =
    new HashSet<AbstractValidation>();

  /**
   * Classe auxiliar usada para processamento dos parmetros da linha de comando
   * pelo args4j. S  necessria porque nem todos os parmetros correspondem a
   * campos de Validator (seno poderamos usar a prpria).
   */
  private static class CLOptions {
    /**
     * Apenas valida a instalao, no aplica nenhuma correo.
     */
    @Option(name = "-V", usage = "apenas valida a instalao (no faz mudanas)")
    boolean validateOnly = false;
    /**
     * Liga o modo verbose (opcional).
     */
    @Option(name = "-v", usage = "modo verbose")
    boolean verboseMode = false;
    /**
     * Path para o arquivo de log da validao (opcional).
     */
    @Option(name = "-l", usage = "path para o arquivo de log da validao")
    File logFile = null;
    /**
     * Path para o arquivo de propriedades (obrigatrio -- se no for fornecido,
     * ser obtido interativamente).
     */
    @Option(name = "-p", usage = "path para o arquivo global de propriedades")
    File propertiesFile = null;
  }

  /**
   * Construtor. Cria o logger e processa os parmetros da linha de comando.
   * 
   * @param args - parmetros da linha de comando
   */
  private Validator(String[] args) {
    currDir = System.getProperty("user.dir");
    logger = new ValidationsLogger();
    readyToRun = processArgs(args) && registerValidations();
  }

  /**
   * Obtm o path para o diretrio temporrio do arquivo de propriedades, e
   * garante que o diretrio correspondente existe.
   * 
   * @return <code>true</code> se o diretrio temporrio existe ou, se no
   *         existia, se foi criado com sucesso
   */
  private boolean prepareTempDir() {
    instTempDirPath = getMandatorySystemProperty(VALIDATIONS_TEMP_DIR);
    if (instTempDirPath == null) {
      return false;
    }
    File tempDirFile = new File(instTempDirPath);
    if (!tempDirFile.exists()) {
      if (!tempDirFile.mkdir()) {
        logger.severe("No foi possvel criar o diretrio " + instTempDirPath
          + " para backups");
        return false;
      }
      logger.fine("Diretrio " + instTempDirPath + " criado para backups");
    }
    else if (!tempDirFile.isDirectory()) {
      logger.severe(instTempDirPath + " no  um diretrio");
      return false;
    }
    // TODO reclamar se o diretrio no estiver vazio (i.e. se uma instalao
    // prvia no foi concluda)
    logger.fine("Diretrio temporrio: " + tempDirFile.getAbsolutePath());
    return true;
  }

  /**
   * Verifica se os diretrios do sistema necessrios para as validaes existem
   * e so vlidos (diretrio de execuo do sistema, repositrio de projetos e
   * repositrio de algoritmos). Permite a criao dos diretrios necessrios,
   * caso possvel.
   * 
   * @return verdadeiro se os diretrios do sistema existem e so vlidos ou
   *         falso, caso contrrio.
   */
  private final boolean checkSystemDirs() {
    /*
     * verifica se o diretrio de execuo do sistema foi configurado
     * corretamente.
     */
    if (!checkExecutionDir()) {
      logger.severe("Diretrio de execuo invlido.");
      return false;
    }

    /*
     * verifica se o console  interativo, para que se possa perguntar para o
     * usurio se ele deseja criar os repositrios de projetos e algoritmos,
     * caso no existam.
     */
    Console console = System.console();
    boolean askToCreateDir;
    if (console == null) {
      askToCreateDir = false;
    }
    else {
      askToCreateDir = true;
    }

    /*
     * verifica a configurao do repositrio de projetos, permitindo que ele
     * seja criado, caso no exista.
     */
    if (!checkOrCreateProjectDir(askToCreateDir)) {
      logger.severe("Repositrio de projetos invlido.");
      return false;
    }

    /*
     * verifica a configurao do repositrio de algoritmos, permitindo que ele
     * seja criado, caso no exista.
     */
    if (!checkOrCreateAlgorithmDir(askToCreateDir)) {
      logger.severe("Repositrio de algoritmos invlido.");
      return false;
    }

    return true;
  }

  /**
   * Verifica a configurao do diretrio de execuo do CSBase.
   * <p>
   * Caso o diretrio de execuo especificado no exista ou no seja vlido,
   * retorna <code>false</code>.
   * 
   * @return <code>true</code> se o diretrio de execuo foi configurado
   *         corretamente ou <code>false</code>, caso contrrio.
   */
  private final boolean checkExecutionDir() {

    logger.info("Verificando o diretrio de execuo do sistema");

    /*
     * obtemos o path para o diretrio de execuo do arquivo de propriedades
     */
    String execDirPath = getMandatorySystemProperty(SYSTEM_EXECDIR_PROP);
    if (execDirPath == null) {
      return false;
    }

    /*
     * determina se  um diretrio de execuo vlido
     */
    File execDir = new File(execDirPath);
    if (!isValidExecutionDir(execDir)) {
      logger
        .severe("Diretrio de execuo invlido (no  diretrio ou no  do CSBase)");
      return false;
    }

    this.baseExecutionDir = execDir;
    logger.info("Diretrio de execuo: OK");
    return true;
  }

  /**
   * Testes bsicos para garantir que o diretrio especificado  realmente um
   * diretrio de execuo de uma aplicao CSBase.
   * <p>
   * Caso o diretrio especificado existe e contenha os arquivos
   * <code>runserver</code> e <code>runclient</code>, assumimos que trata-se de
   * um diretrio vlido.
   * 
   * @param execDir diretrio de execuo do sistema
   * @return <code>true</code> se  um diretrio de execuo vlido
   */
  private boolean isValidExecutionDir(File execDir) {
    if (FileSystemUtils.dirExists(execDir)
      && ValidatorUtils.hasChildFile(execDir, "runserver")
      && ValidatorUtils.hasChildFile(execDir, "runclient")) {
      return true;
    }
    return false;
  }

  /**
   * Verifica a configurao do respositrio de algoritmos do CSBase, caso ele
   * exista.
   * <p>
   * Caso o repositrio especificado no exista, podemos perguntar ao usurio se
   * ele deseja criar o diretrio (se a flag <code>askToCreateDir</code> for
   * verdadeira).
   * 
   * @param askToCreateDir se o repositrio de algoritmos no existir, esta flag
   *        determina se devemos perguntar para o usurio se ele deseja criar o
   *        diretrio.
   * @return <code>true</code> se o repositrio de algoritmos foi configurado
   *         corretamente ou <code>false</code>, caso no exista o diretrio e
   *         no se queira ou no se possa cri-lo.
   */
  private final boolean checkOrCreateAlgorithmDir(boolean askToCreateDir) {

    logger.info("Verificando o repositrio de algoritmos");

    /*
     * obtemos o path para o diretrio de algoritmos do arquivo de propriedades
     */
    String baseAlgorithmPath = getMandatorySystemProperty(ALGO_BASEDIR_PROP);
    if (baseAlgorithmPath == null) {
      return false;
    }

    /*
     * obtm a referncia para o diretrio do algoritmos: se seu caminho for
     * relativo, o obtemos a partir do diretrio de execuo, seno utilizamos
     * diretamente seu caminho absoluto.
     */
    File tmpDir = new File(baseAlgorithmPath);
    File algorithmDir = null;
    if (tmpDir.isAbsolute()) {
      algorithmDir = tmpDir;
    }
    else if (this.baseExecutionDir != null) {
      algorithmDir = new File(this.baseExecutionDir, baseAlgorithmPath);
    }
    else {
      logger
        .severe("A verificao do diretrio de algoritmos depende de uma configurao vlida para o diretrio de execuo");
      return false;
    }

    if (!assureDirExists(algorithmDir, askToCreateDir)) {
      logger
        .severe("Diretrio de algoritmos invlido (no  diretrio ou no  do CSBase)");
      return false;
    }

    logger.info("Repositrio de algoritmos: OK");
    this.baseAlgorithmDir = algorithmDir;
    return true;
  }

  /**
   * Verifica a configurao do respositrio de projetos do CSBase, caso ele
   * exista.
   * <p>
   * Caso o repositrio especificado no exista, podemos perguntar ao usurio se
   * ele deseja criar o diretrio (se a flag <code>askToCreateDir</code> for
   * verdadeira).
   * 
   * @param askToCreateDir se o repositrio de projetos no existir, esta flag
   *        determina se devemos perguntar para o usurio se ele deseja criar o
   *        diretrio.
   * @return <code>true</code> se o repositrio de projetos foi configurado
   *         corretamente ou <code>false</code>, caso no exista o diretrio e
   *         no se queira ou no se possa cri-lo.
   */
  private final boolean checkOrCreateProjectDir(boolean askToCreateDir) {

    logger.info("Verificando o repositrio de projetos");

    /*
     * obtemos o path para o diretrio de projetos do arquivo de propriedades
     */
    String baseProjectPath = getMandatorySystemProperty(PRJ_BASEDIR_PROP);
    if (baseProjectPath == null) {
      return false;
    }

    /*
     * obtm a referncia para o diretrio do projeto: se seu caminho for
     * relativo, o obtemos a partir do diretrio de execuo, seno utilizamos
     * diretamente seu caminho absoluto.
     */
    File tmpDir = new File(baseProjectPath);
    File projectDir = null;
    if (tmpDir.isAbsolute()) {
      projectDir = tmpDir;
    }
    else if (this.baseExecutionDir != null) {
      projectDir = new File(this.baseExecutionDir, baseProjectPath);
    }
    else {
      logger
        .severe("A verificao do diretrio de projetos depende de uma configurao vlida para o diretrio de execuo");
      return false;
    }

    if (!assureDirExists(projectDir, askToCreateDir)) {
      logger
        .severe("Diretrio de projetos invlido (no  diretrio ou no  do CSBase)");
      return false;
    }

    logger.info("Repositrio de projetos: OK");
    this.baseProjectDir = projectDir;
    return true;
  }

  /**
   * Testes bsicos para garantir que o diretrio especificado existe.
   * <p>
   * Caso o diretrio especificado no exista, podemos perguntar ao usurio se
   * ele deseja criar o diretrio (se a flag <code>askToCreateDir</code> for
   * verdadeira).
   * 
   * @param dir o diretrio a ser verificado
   * @param askToCreateDir se o diretrio no existir, esta flag determina se
   *        devemos perguntar para o usurio se ele deseja criar o diretrio.
   * @return verdadeiro se  um diretrio que j existia ou se trata-se de uma
   *         nova instalao e o diretrio foi criado
   */
  private boolean assureDirExists(File dir, boolean askToCreateDir) {
    if (FileSystemUtils.dirExists(dir)) {
      return true;
    }

    logger.warning("O diretrio especificado no existe");
    boolean createDir = askToCreateDir && askUserToCreateDir(dir);
    if (createDir) {
      // Criamos o diretrio com todo o path necessrio.
      logger.fine("Criando o diretrio: " + dir.getAbsolutePath());
      final boolean created = dir.mkdirs();
      return created;
    }
    return false;
  }

  /**
   * Pergunta para o usurio se o diretrio deve ser criado. Retorna verdadeiro
   * se o usurio indicar que o diretrio deve ser criado ou falso, caso o
   * usurio indique que no quer criar o diretrio ou caso no seja possvel
   * perguntar para o usurio (console no  interativo).
   * 
   * @param dir o diretrio a ser criado
   * @return verdadeiro se o diretrio deve ser criado ou falso, caso contrrio.
   */
  private boolean askUserToCreateDir(File dir) {
    Console console = System.console();
    if (console == null) {
      /*
       * no conseguimos uma referncia para o console, o que provavelmente
       * significa que no  um shell interativo
       */
      logger.severe("Este terminal no  interativo");
      return false;
    }
    logger.warning("Deseja criar o diretrio ");
    logger.warning(dir.getAbsolutePath() + " ?");

    boolean createDir =
      ValidatorUtils.readConfirmation(console, ValidatorUtils.Option.YES);
    System.out.println();

    return createDir;
  }

  /**
   * Retorna uma referncia para o repositrio de projetos do CSBase.
   * 
   * @return a referncia para o repositrio de projetos.
   */
  protected final File getProjectDir() {
    return this.baseProjectDir;
  }

  /**
   * Retorna uma referncia para o repositrio de algoritmos do CSBase.
   * 
   * @return a referncia para o repositrio de algoritmos.
   */

  protected final File getAlgorithmDir() {
    return this.baseAlgorithmDir;
  }

  /**
   * Cadastra as validaes que sero executadas. Instancia programaticamente as
   * classes dos pacotes <code>validations</code> e <code>validations_sys</code>
   * que derivam de {@link AbstractValidation} e no so abstratas. A lista de
   * validaes  ordenada alfabeticamente pelo nome das respectivas classes,
   * para possibilitar a priorizao de validaes em funo dos nomes das
   * mesmas, se desejado.
   * 
   * @return <code>true</code> se todas as validaes foram registradas com
   *         sucesso
   */
  private boolean registerValidations() {
    validations = new ArrayList<AbstractValidation>();
    if (!instantiateValidations(FRAMEWORK_VALIDATIONS_PACKAGE)
      || !instantiateValidations(SYSTEM_VALIDATIONS_PACKAGE)) {
      /*
       * o erro j foi informado
       */
      return false;
    }
    /*
     * ordenamos a lista de validaes em funo dos nomes das respectivas
     * classes
     */
    Collections.sort(validations, new Comparator<AbstractValidation>() {

      @Override
      public int compare(AbstractValidation v1, AbstractValidation v2) {
        return getValidationName(v1).compareTo(getValidationName(v2));
      }
    });
    /*
     * registramos no arquivo de log quais validaes sero aplicadas, e em qual
     * ordem
     */
    logger.fine("Validaes:");
    for (AbstractValidation val : validations) {
      logger.fine("    " + getValidationName(val));
    }
    return true;
  }

  /**
   * Instancia as validaes de um determinado pacote. Apenas as classes que
   * derivam de {@link AbstractValidation} e no so abstratas so instanciadas.
   * 
   * @param pkg pacote que contm as validaes
   * @return <code>true</code> se todas as validaes foram instanciadas com
   *         sucessos
   */
  private boolean instantiateValidations(String pkg) {
    for (String className : getValidationsNames(pkg)) {
      try {
        Class<?> valClass = Class.forName(className);
        if (Modifier.isAbstract(valClass.getModifiers())
          || !AbstractValidation.class.isAssignableFrom(valClass)) {
          /*
           * a classe em questo era abstrata ou no era subclasse de
           * AbstractValidation, vamos ignor-la
           */
          logger.warning("Ignorando validao " + className);
          continue;
        }
        validations.add((AbstractValidation) valClass.newInstance());
      }
      catch (Exception e) {
        logger.severe("No foi possvel instanciar validao " + className);
        logger.severe("Erro: " + e.getClass().getSimpleName());
        return false;
      }
    }
    /*
     * por enquanto, vamos sinalizar como erro a ausncia de validaes
     */
    if (validations.isEmpty()) {
      logger.severe("No foram encontradas validaes para esta instalao");
      return false;
    }

    return true;
  }

  /**
   * Obtm uma lista com os nomes dos arquivos <code>.class</code> do pacote
   * especificado.
   * 
   * @param pkg nome do pacote que contm as validaes
   * 
   * @return lista com os nomes dos arquivos <code>.class</code> do pacote
   *         especificado
   */
  private List<String> getValidationsNames(final String pkg) {
    final List<String> namesList = new ArrayList<String>();
    File directory = new File(currDir + File.separatorChar + pkg);
    if (!directory.exists()) {
      return namesList;
    }

    directory.listFiles(new FilenameFilter() {
      @Override
      public boolean accept(File dir, String name) {
        /*
         * queremos apenas os arquivos .class que no so inner-classes (i.e.
         * no possuem '$' no nome)
         */
        if (name.endsWith(".class") && name.indexOf('$') == -1) {
          String n = name.substring(0, name.indexOf('.'));
          namesList.add(pkg + '.' + n);
        }
        return false;
      }

    });
    return namesList;
  }

  /**
   * Processa os argumentos fornecidos na linha de comando. Aceita os seguintes
   * parmetros:
   * <ul>
   * <li>-p path : path para o arquivo com as propriedades do sistema
   * (obrigatrio)
   * <li>-v : verbose (opcional). Exibe informaes durante o processamento.
   * <li>-V : validateOnly (opcional). Indica que apenas a validao deve ser
   * executada; as correes no sero aplicadas, mesmo que o validador seja
   * capaz de faz-lo.
   * <li>-l path (opcional): path para o log
   * </ul>
   * 
   * Qualquer argumento comeando com '-' diferente destes abortar o processo,
   * o mesmo valendo caso no haja nenhum argumento sem o '-'. Caso haja
   * mltiplos argumentos sem o '-', o ltimo ser usado como o path para o
   * arquivo de propriedades.
   * 
   * @param args - argumentos fornecidos na linha de comando
   * 
   * @return <code>true</code> se os parmetros foram processados com sucesso
   */
  private boolean processArgs(String[] args) {
    CLOptions options = new CLOptions();
    CmdLineParser parser = new CmdLineParser(options);
    try {
      parser.parseArgument(args);
    }
    catch (CmdLineException e) {
      System.err.println(e.getMessage());
      parser.printUsage(System.err);
      return false;
    }
    verboseMode = options.verboseMode;
    validateOnly = options.validateOnly;
    File logFile = options.logFile;
    File propertiesFile = options.propertiesFile;

    if (propertiesFile == null) {
      propertiesFile = getPropertiesFile();
      // propertiesFile ainda pode ser null, se houve algum erro
    }
    if (propertiesFile != null && !FileSystemUtils.fileExists(propertiesFile)) {
      logger.severe("Arquivo de propriedades " + propertiesFile.getPath()
        + " no existe");
      return false;
    }
    /*
     * l o arquivo de propriedades especificado. Neste ponto o arquivo pode ser
     * null, mas isto  tratado pelo mtodo que l as propriedades
     */
    if (!readSystemDefaultProperties(propertiesFile)) {
      return false;
    }

    /*
     * verifica se os diretrios do sistema necessrios para a validao existem
     * e so vlidos. Diretrios que no existem podem ser criados, se for
     * possvel e o usurio assim quiser.
     */
    if (!checkSystemDirs()) {
      return false;
    }
    /*
     * garante que existe o diretrio temporrio para backups
     */
    if (!prepareTempDir()) {
      return false;
    }
    /*
     * configura o nvel de log
     */
    if (!configureLogger(verboseMode, logFile)) {
      return false;
    }
    logger.blank(Level.FINE);
    logger.fine(Calendar.getInstance().getTime().toString());
    return true;
  }

  /**
   * Cria um handler para log em arquivo, e configura os nveis de logging. O
   * nvel de log para o console  controlado pelo usurio (defaul == INFO), mas
   * o log para arquivo  sempre FINE.
   * 
   * @param isVerbosed indica se o usurio escolheu o modo verbose
   * @param logFile arquivo de log. Pode ser null; neste caso tentamos obter o
   *        path do arquivo de propriedades, seno criamos um log default no
   *        diretrio temporrio
   * @return <code>true</code> se o handler foi criado com sucesso
   * 
   * @see #LOGPATH_PROPERTY_NAME
   */
  private boolean configureLogger(boolean isVerbosed, File logFile) {
    logger.setConsoleLevel(isVerbosed ? Level.FINE : Level.INFO);

    String logFilePath;
    if (logFile != null) {
      if (!FileSystemUtils.fileExists(logFile)) {
        logger.severe("Arquivo de log " + logFile.getPath() + " no existe");
        return false;
      }
      logFilePath = logFile.getAbsolutePath();
    }
    else {
      /*
       * usurio no forneceu o path na linha de comando, verificamos se foi
       * definido como propriedade
       */
      logFilePath = getSystemProperty(LOGPATH_PROPERTY_NAME);
      if (logFilePath == null) {
        /*
         * tambm no definiu como propriedade, criamos um com um nome default
         * no diretrio temporrio
         */
        logFilePath =
          getTempDirPath() + File.separatorChar + DEFAULT_LOGFILE_NAME;
      }
    }
    return logger.setFileHandler(logFilePath);
  }

  /**
   * Obtm o path para o arquivo de propriedades default do sistema. Busca
   * primeiro por arquivos j existentes, na seguinte ordem:
   * <ul>
   * <li><code>../System.properties</code>
   * <li><code>validations_sys/System.properties.default</code> (se este arquivo
   * existe,  copiado para <code>../System.properties</code>, mediante
   * confirmao do usurio)
   * </ul>
   * Se nenhuma das opes acima for bem-sucedida, obtm um path
   * interativamente.
   * 
   * @return o arquivo de propriedades, ou null caso ocorra algum erro
   */
  private File getPropertiesFile() {
    logger.warning("No foi fornecido path para o arquivo de propriedades.");
    Console console = System.console();
    if (console == null) {
      /*
       * no conseguimos uma referncia para o console, o que provavelmente
       * significa que no  um shell interativo
       */
      logger.severe("Este terminal no  interativo");
      return null;
    }
    /*
     * agora verificamos se existe "validations_sys/System.properties.default".
     * Se existe, perguntamos se o usurio quer us-lo como referncia
     */
    boolean copySample = false;
    if (FileSystemUtils.fileExists(SAMPLE_PROPERTIES_FILE)) {
      logger
        .warning("Deseja criar um novo arquivo de propriedades a partir de ");
      logger.warning(SAMPLE_PROPERTIES_FILE + " ?");
      copySample =
        ValidatorUtils.readConfirmation(console, ValidatorUtils.Option.YES);
      System.out.println();
    }
    /*
     * obtemos o path para o arquivo de propriedades interativamente
     */
    logger.warning("Fornea o path para o arquivo de propriedades.");
    logger.warning("O path deve ser absoluto, ou relativo ao diretrio");
    logger.warning(currDir);
    logger.warning("e deve incluir o nome do arquivo.");
    String path = ValidatorUtils.readLine(console, "  > ");
    System.out.println();
    if (copySample) {
      if (FileSystemUtils.fileExists(path)) {
        logger.warning("O arquivo j existe. Deseja sobrescrev-lo?");
        boolean overwrite =
          ValidatorUtils.readConfirmation(console, ValidatorUtils.Option.YES);
        if (!overwrite) {
          logger.severe("Operao abortada pelo usurio");
          return null;
        }
        logger.info("Sobrescrevendo o arquivo de propriedades " + path);
      }
      else {
        logger.info("Criando o arquivo de propriedades " + path);
      }
      if (!ValidatorUtils.copyFile(SAMPLE_PROPERTIES_FILE, path, logger)) {
        return null;
      }
    }
    return new File(path);
  }

  /**
   * Carrega as propriedades default de um arquivo especfico.
   * 
   * @param propertiesFile - o arquivo com as propriedades
   * 
   * @return <code>true</code> se a carga foi bem-sucedida
   */
  private boolean readSystemDefaultProperties(File propertiesFile) {
    if (propertiesFile == null) {
      return false;
    }
    if (!propertiesFile.exists()) {
      logger
        .severe("Arquivo de propriedades " + propertiesFile + " no existe");
      return false;
    }

    BufferedInputStream in = null;
    try {
      in = new BufferedInputStream(new FileInputStream(propertiesFile));
      systemProperties = new Properties();
      systemProperties.load(in);
    }
    catch (Exception e) {
      logger.severe("Erro lendo arquivo de propriedades do sistema");
      return false;
    }
    finally {
      if (in != null) {
        try {
          in.close();
        }
        catch (IOException e) {
          return false;
        }
      }
    }
    logger.info("Arquivo de propriedades: " + propertiesFile.getPath());
    return true;
  }

  /**
   * Executa todas as validaes cadastradas. Para cada validao, cria um log
   * handler para o arquivo especificado pela validao (se este no for
   * <code>null</code>) e o registra no logger global. Aps o processamento, o
   * handler da validao  descadastrado.
   * 
   * @return <code>true</code> se todas as validaes foram aplicadas com
   *         sucesso
   */
  private boolean run() {
    try {
      if (!readyToRun) {
        logger.separator(Level.SEVERE);
        logger.severe("*** ERRO NA CONFIGURAO DAS VALIDAES");
        return false;
      }

      boolean allOK = true;
      AbstractValidation lastValidation = null;
      try {
        for (AbstractValidation validation : validations) {
          /*
           * se a validao deve executar apenas uma vez verificamos se j foi
           * executada
           */
          if (validation.runsOnlyOnce()) {
            if (hasAlreadyRun(validation)) {
              /*
               * validao j foi executada, pulamos para a prxima
               */
              skippedValidations.add(validation);
              continue;
            }
          }
          boolean succeeded = validation.run(this);
          validation.reportStatus();
          lastValidation = validation;
          if (succeeded) {
            if (validation.runsOnlyOnce()) {
              /*
               * sinalizamos que a validao j foi executada
               */
              createValidationMarker(validation);
            }
          }
          else {
            allOK = false;
            break;
          }
        }
      }
      catch (Exception e) {
        logger.exception(e);
        allOK = false;
      }

      if (allOK) {
        logger.info("Validao da instalao: OK");
        /*
         * todos os patches foram bem-sucedidos, podemos remover todos os
         * backups
         */
        removeBackups();
      }
      else {
        logger.severe("Validao da instalao: *** ERRO ***");
        Status status = lastValidation.getStatus();
        if (status == Status.BACKUP_FAILED) {
          /*
           * como o erro foi na gerao dos backups, podemos (e devemos)
           * remov-los
           */
          removeBackups();
        }
        else if (!validateOnly && status == Status.PATCH_FAILED) {
          /*
           * algum patch falhou; se no estvamos apenas validando, precisamos
           * fazer os rollbacks na ordem inversa, e remover os marcadores das
           * validaes
           */
          removeValidationMarkers(lastValidation);
          if (rollback(lastValidation)) {
            logger.separator(Level.SEVERE);
            logger
              .severe("*** TODAS AS ATUALIZAES FORAM DESFEITAS COM SUCESSO");
            logger
              .severe("*** O SISTEMA NO PODE SER EXECUTADO NA VERSO ATUAL");
            logger
              .severe("*** CORRIJA O PROBLEMA OU REINSTALE A VERSO ANTERIOR");
          }
          else {
            logger.severe("*** ALGUMAS ATUALIZAES NO PUDERAM SER DESFEITAS");
            logger
              .severe("*** O SISTEMA NO PODE SER EXECUTADO NA VERSO ATUAL");
            logger.severe("*** NO  POSSVEL REINSTALAR A VERSO ANTERIOR");
            logger.severe("*** CONTACTE A EQUIPE DE DESENVOLVIMENTO");
          }
        }
        System.err.println();
        logger.toConsole("O registro das converses foi anexado ao arquivo ");
        logger.toConsole(logger.getLogFilePath());
      }
      return allOK;
    }
    finally {
      logger.close();
    }
  }

  /**
   * Obtm o nome de uma validao a partir da sua classe. Corresponde ao nome
   * da classe, sem o pacote.
   * 
   * @param validation validao
   * @return nome da validao
   */
  private String getValidationName(AbstractValidation validation) {
    return validation.getClass().getSimpleName();
  }

  /**
   * Obtm o arquivo correspondente ao marcador para uma validao (arquivo com
   * o nome da classe mais a extenso {@link #VALIDATION_MARKER_EXTENSION}). O
   * arquivo aponta para o diretrio temporrio das execues.
   * 
   * @param validation validao
   * @return arquivo correspondente ao marcador da validao
   * 
   * @see #VALIDATIONS_TEMP_DIR
   * @see #VALIDATION_MARKER_EXTENSION
   */
  private File getValidationMarkerFile(AbstractValidation validation) {
    return new File(getTempDirPath(), getValidationName(validation)
      + VALIDATION_MARKER_EXTENSION);
  }

  /**
   * Verifica se uma validao j foi executada, buscando pelo seu respectivo
   * marcador. Se a validao j foi executada, exibe a data da sua ltima
   * execuo (data da ltima modificao do marcador).
   * 
   * @param validation validao
   * @return <code>true</code> se a validao j foi executada
   * @see #getValidationMarkerFile(AbstractValidation)
   */
  private boolean hasAlreadyRun(AbstractValidation validation) {
    File markerFile = getValidationMarkerFile(validation);
    if (!markerFile.exists()) {
      return false;
    }
    Date lastModified = new Date(markerFile.lastModified());
    String timestamp =
      DateFormat.getDateInstance(DateFormat.MEDIUM).format(lastModified);
    String validationName = getValidationName(validation);
    logger.info(String.format("Validao j executada em %s : %s", timestamp,
      validationName));
    return true;
  }

  /**
   * Cria um marcador para uma validao.
   * 
   * @param validation validao
   * 
   * @see #getValidationMarkerFile(AbstractValidation)
   */
  private void createValidationMarker(AbstractValidation validation) {
    File validationFile = getValidationMarkerFile(validation);
    try {
      validationFile.createNewFile();
    }
    catch (IOException e) {
      String validationName = getValidationName(validation);
      logger.exception("Erro criando marcador para " + validationName, e);
    }
  }

  /**
   * Remove todos os marcadores criados para validaes executadas antes de uma
   * determinada validao (inclusive).
   * 
   * @param lastValidation ltima validao executada
   */
  private void removeValidationMarkers(AbstractValidation lastValidation) {
    int i = validations.indexOf(lastValidation);
    for (; i >= 0; i--) {
      AbstractValidation val = validations.get(i);
      if (!val.runsOnlyOnce()) {
        continue;
      }
      File validationMarker = getValidationMarkerFile(val);
      if (validationMarker.exists()) {
        if (!validationMarker.delete()) {
          logger.severe("Erro ao remover o marcador de "
            + getValidationName(val));
        }
      }
    }
  }

  /**
   * Removemos os backups das validaes que foram executadas.
   * 
   * @see #hasAlreadyRun(AbstractValidation)
   */
  private void removeBackups() {
    try {
      for (AbstractValidation validation : validations) {
        if (!skippedValidations.contains(validation)) {
          validation.removeBackupDir();
        }
      }
    }
    catch (Exception e) {
      logger.exception("Erro removendo backups", e);
    }
  }

  /**
   * Percorre a lista de validaes a partir de uma validao que falhou at o
   * incio, desfazendo as alteraes na ordem inversa em que foram aplicadas.
   * Caso algum rollback falhe o processo ser interrompido, pois o rollback N-1
   * pode no ser possvel se o rollback N no for bem-sucedido.
   * 
   * @param lastValidation ltima validao aplicada (neste caso, a que falhou)
   * @return <code>true</code> se todas as modificaes foram desfeitas com
   *         sucesso
   */
  private boolean rollback(AbstractValidation lastValidation) {
    try {
      logger.sectionSeparator(Level.INFO);
      int i = validations.indexOf(lastValidation);
      for (; i >= 0; i--) {
        AbstractValidation val = validations.get(i);
        if (skippedValidations.contains(val)) {
          /*
           * no precisamos fazer rollback de uma validao que no foi
           * executada
           */
          continue;
        }
        String validationName = getValidationName(val);
        logger.info(validationName + " : desfazendo alteraes");
        if (!val.rollback()) {
          logger.severe(validationName + " : ERRO");
          return false;
        }
        logger.info(validationName + " : OK");
        /*
         * rollback bem-sucedido, removemos o respectivo backup
         */
        val.removeBackupDir();
      }

    }
    catch (Exception e) {
      logger.exception("Erro restaurando backups de "
        + lastValidation.getClass().getSimpleName(), e);
      return false;
    }
    return true;
  }

  /**
   * Mtodo de entrada. Para cada execuo, cria uma nova instncia do validador
   * e executa todas as validaes.
   * <p>
   * Se todas as validaes forem aplicadas com sucesso, encerra a JVM com
   * cdigo 0, seno encerra com cdigo 1.
   * 
   * @param args - parmetros da linha de comando
   */
  public static void main(String[] args) {
    System.out.println();
    Validator validator = new Validator(args);
    int status = validator.run() ? 0 : 1;
    System.exit(status);
  }

  /**
   * Obtm o path para o diretrio temporrio da instalao.
   * 
   * @return path para o diretrio temporrio da instalao
   */
  public String getTempDirPath() {
    return instTempDirPath;
  }

  /**
   * Obtm o logger.
   * 
   * @return logger
   */
  public ValidationsLogger getLogger() {
    return logger;
  }

  /**
   * Retorna o valor associado a uma propriedade do sistema, ou
   * <code>null</code> caso esta no exista.
   * 
   * @param propName nome da propriedade
   * 
   * @return valor associado  propriedade do sistema, ou <code>null</code> caso
   *         esta no exista
   * 
   * @see #getMandatorySystemProperty(String)
   */
  public String getSystemProperty(String propName) {
    final String value = systemProperties.getProperty(propName);
    if (value == null) {
      return null;
    }
    final String trimmedValue = value.trim();
    return trimmedValue;
  }

  /**
   * Retorna o valor associado a uma propriedade do sistema. Caso esta no
   * exista, sinaliza no log e retorna <code>null</code>.
   * 
   * @param propName nome da propriedade
   * @return valor associado  propriedade do sistema, ou <code>null</code> caso
   *         esta no exista
   * 
   * @see #getSystemProperty(String)
   */
  public String getMandatorySystemProperty(String propName) {
    final String value = systemProperties.getProperty(propName);
    if (value == null) {
      final String msg = "Prop. obrigatria [" + propName + "] indefinida.";
      logger.severe(msg);
      return null;
    }
    final String trimmedValue = value.trim();
    return trimmedValue;
  }

  /**
   * Verifica se estamos apenas validando (i.e. no ser feito nenhum ajuste).
   * As validaes devem usar este mtodo para configurar seu modo de operao.
   * 
   * @return <code>true</code> se estamos apenas validando
   */
  public boolean isValidatingOnly() {
    return validateOnly;
  }

  /**
   * Indica se estamos em modo verbose. As validaes devem usar este mtodo
   * para configurar seu modo de operao.
   * 
   * @return <code>true</code> se estamos em modo verbose
   */
  public boolean isVerbose() {
    return verboseMode;
  }
}
