#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <string>
#include <ctype.h>

#include <iup.h>
#include <iuptree.h>
#include <iupmatrix.h>

#include <wioapi.h>
#include "wiofiledlg.h"

/** Macro de indicao de que no h evento pendente. */
#define NO_EVENT (-1)

/** Maior texto possvel de uma matriz */
#define MAX_TEXT_SIZE 256

/** Maior texto que pode representar um atributo numrico na forma X:X */
#define MAX_INTEGER_SIZE 16

/** Maior texto que pode representar um atributo filtro*/
#define MAX_FILTER_SIZE 256

/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */

using namespace wioapi;
using namespace std;

/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */

/**
 * Estrutura para armazenamento dos dados do arquivo.
 * @author Camilo Freire.
 */
typedef struct _Tfile {

   /** Nome do arquivo */
   char* name; 

   /** Estrutura de data de criao do arquivo */
   WIODateTimeInfo* date; 

   /** Tamanho em bytes do arquivo */
   long long size; 

   /** Tipo do arquivo */
   int type;
} Tfile;

/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */

/**
 * Internacionalizao simples.
 * @param key string-chave de idiomas (ingles, portugues).
 * @author Camilo Freire
 */
static const char* translate(const char *key) {
  /** Tabela de chaves e tradues */
  static char* stringTable[][3] = {
   {"OK", "OK", "Confirmar"},
   {"CANCEL", "Cancel", "Cancelar"},
   {"OPEN", "Open", "Abrir"}, 
   {"SAVE", "Save", "Salvar"},
   {"FILES", "Files", "Arquivos"},
   {"SELECTION", "Selection", "Seleo"},
   {"TYPES", "Types", "Tipos"},
   {"RENAME", "Rename", "Renomear"},
   {"CURRENT_DIRECTORY", "Current Directory:", "Diretrio Corrente:"},
   {"DIRECTORY_NAME", "Directory Name:", "Nome para o Diretrio:"},
   {"ALL_FILES", "All files|*.*", "Todos arquivos|*.*"},
   {"ATTENTION", "Warning", "Ateno"},
   {"FILE_EXISTS", "This file already exists!\nReplace existing file?",
    "J existe uma arquivo com esse mesmo nome!.\nDeseja sobreescreve-lo?"
   },
   {"YES", "Yes", "Sim"},
   {"NO", "No", "No"},
   {"FILE", "File", "Arquivo"},
   {"TYPE", "Type", "Tipo"},
   {"DATE_MODIFIED", "Last Modification", "ltima Modificao"},
   {"SIZE", "Size", "Tamanho"},
   {"FILE_FOLDER", "File folder", "Pasta"},
   {"TITLE_OPEN", "Select File in Project ", "Selecionar Arquivo no Projeto "},
   {"TITLE_SAVE", "Save File in Project ", "Salvar Arquivo no Projeto "},
   {"TITLE_DIR", "Select Folder in Project ", "Selecionar Diretrio no Projeto "},
   {"NEW_FOLDER", "New_folder", "Nova_pasta"}
  };
  
  for (unsigned int i = 0; i < sizeof(stringTable)/(sizeof(char*)*3); i++) {
    if (strcmp(key, stringTable[i][0]) == 0) {
       if (strcmp(IupGetLanguage(), IUP_ENGLISH) == 0) {
          return stringTable[i][1];
       }
       else { 
          // IUP_PORTUGUESE
          return stringTable[i][2];
       }
    }
  }
  
  // Caso no exista essa chave exista, fica sem traduo..
  return key;
}

/* ....................................................................... */

/**
 * Funo de comparao de strings case insensitive (compatibilizao
 * com a funo no-ANSI stricmp().
 * @param str1 string A
 * @param str2 string B
 * @return o mesmo resultado de strcmp tpico.
 */
static int mystricmp(char* str1, char* str2) {
  if (str1 == NULL || str2 == NULL) return strcmp(str1, str2);
  char* a = strdup(str1);
  char* b = strdup(str2);
  int na = strlen(a);
  int nb = strlen(b);
  for (int i = 0; i < na; i++) a[i] = tolower(a[i]);
  for (int j = 0; j < nb; j++) b[j] = tolower(b[j]);
  int r = strcmp(a, b);
  free(a);
  free(b);
  return r;
}

/* ....................................................................... */
/**
 * Funo para filtar arquivos (paths) por tipos.
 * @param path o caminho lgico do arquivo.
 * @param type tipos desejados.
 * @return flag indicativo.
 */
static int filterFile(char* path, char* types) {
  if (types == NULL) return 1;

  char* types_copy = strdup(types);
  char* ext = strrchr(path, '.');
  int ret = 0;

  for (char* s = strtok(types_copy, ";"); s != NULL; s = strtok(NULL, ";")) {
      char* typeExt = strrchr(s, '.') + 1;
      if (strcmp(typeExt, "*") == 0) {
         ret = 1;
         break;
      }
      if (ext != NULL && mystricmp(ext+1, typeExt) == 0) {
         ret = 1;
         break;
      }
  }

  delete types_copy;
  return ret;
}

/* ....................................................................... */
/**
 * Funo de ordenao de dois objetos do tipo <code>Tfile</code>
 * utilizada na funo <code>qsort</code>.
 * @param elem1 elemento A
 * @param elem2 elemento B.
 * @return indicativo de ordem: -1 ou 1.
 */
static int sortData(const void *elem1, const void *elem2) {
  Tfile* file1 = (*(Tfile**)elem1);
  Tfile* file2 = (*(Tfile**)elem2);
  char *name1 = file1->name;
  char *name2 = file2->name;
  int type1 = file1->type;
  int type2 = file2->type;

  if (type1 == 1 && type2 == 0) return -1;
  else if (type1 == 0 && type2 == 1) return 1;
  else if (strcmp(name1, "..") == 0) return -1;
  else if (strcmp(name2, "..") == 0) return 1;
  else if (mystricmp(name1, name2) > 0) return 1;
  else if (mystricmp(name1, name2) < 0) return -1;
  else return 0;
}

/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */
/**
 * Classe que representa a caixa de dilogo de 
 * arquivos da rea de projetos WIO
 * @author Camilo Freire.
 */
class WIOFileDialog {

  /** Caixa de texto de diretrio */
  Ihandle* directory;

  /** Caixa de filtros de arquivo */
  Ihandle* filter;

  /** Caixa do nome do arquivo selecionado */
  Ihandle* selection;

  /** Boto para subir um diretrio */
  Ihandle* up;

  /** Boto para ir ao <i>home</i> do projeto*/
  Ihandle* home;

  /** Boto para criar um diretrio */
  Ihandle* createDir;

  /** Boto que aciona a operao desejada: abrir ou salvar */
  Ihandle* actionButton;

  /** Matriz com os arquivos e diretrios do diretrio corrente */
  Ihandle* matrix;

  /** rvore com os diretrios que fazem parte do projeto */
  Ihandle* tree;

  /** Boto para cancelar a operao */
  Ihandle* cancelButton;

  /** String com a especificao do filtro */
  char* extFilter;

  /** String com o filtro default. */
  char* defaultExt;

  /** ndice do ltimo elemento selecionado na lista de diretrios */
  int lastEventIndex;

  /** Projeto aberto */
  WIOProject* project;

  /** Diretrio remoto corrente */
  WIOFile* currentDir;

  /** dados dos arquivos e diretrios que conpe o diretrio corrente */
  Tfile** data;

  /** Tamanho do dado */
  int datasize;

  /** Tamanho do buffer interno */
  int bufsize;

  /** Quantidade de dados. */
  int ndata;

  /** Indica a ocorrncia de erro na execuo */
  bool error;

  /** Separator de diretrios do servidor */
  char* separatorChar;

  static int directories(WIOFileDialog* wioFileDialog, char* dirClicked);
  
  static int action_cb(Ihandle* self);
  static int cancel_cb(Ihandle* self);
  static int cancelDir_cb(Ihandle* self);
  static int home_cb(Ihandle* self);
  static int up_cb(Ihandle* self);
  static int createDir_cb(Ihandle* self);
  static int filter_cb(Ihandle* self, char *t, int i, int v);
  static int k_any_cb(Ihandle* self, int c);
  static int edition_cb(Ihandle* self, int lin, int col, int mode);
  static int value_edit_cb(Ihandle* self, int lin, int col, char *newVal);
  static int click_cb(Ihandle* self, int lin, int col, char *r);
  static int rename_cb(Ihandle* self);
  static int selection_cb(Ihandle* h, int c, char *after);
  static int treeSelection_cb(Ihandle* h, int id,int status);

  static int bgcolor_cb(Ihandle* self, int lin, int col, 
         unsigned int *r, unsigned int *g, unsigned int *b);
  static int matrixAction_cb(Ihandle* self, int c, int lin, 
         int col, int active, char* after);

  static char* value_cb(Ihandle* self, int lin, int col);

  static void popupMenu(Ihandle* self);

  /** Callback chamado quando o usurio clica na lista de arquivos. */
  // static int files_cb(Ihandle * self, char *fileClicked, int index,
    // int isSelected);

 public:
  void SetExtFilter(const char* flt) throw (wioapi::Exception*);
  void SetDefaultExt(const char* defExt) throw (wioapi::Exception*);
  
  void fillLists(void) throw (wioapi::Exception*);
  void fillTree(const char* diri, int level) throw (wioapi::Exception*);
  void fillFilterList(void) throw (wioapi::Exception*);
  
  Ihandle* createFileDialog(enum WioFileDlgType type,
    const char* parent = NULL) throw ();
  Ihandle* createDirDialog(const char* parent = NULL) throw ();
  
  char* getPath(void) throw ();
  char* getDir(void) throw ();

  WIOFileDialog(WIOFileSystem* filesystem, WIOProject* project,
    enum WioFileDlgType type) throw (wioapi::Exception*);

  ~WIOFileDialog();
};
/* ....................................................................... */
/**
 * Construtor padro.
 * @param filesystem Representa um ponto de acesso  rea de projetos
 * @param project Projeto aberto
 * @param type Tipo de dilogo de arquivos: para abrir ou salvar arquivos
 * @throws wioapi::Exception
 */  
WIOFileDialog::WIOFileDialog(WIOFileSystem* filesystem, WIOProject* project,
enum WioFileDlgType type) throw (wioapi::Exception*) {
  lastEventIndex = NO_EVENT;
  error = false;
  extFilter = NULL;
  defaultExt = NULL;
  separatorChar = filesystem->getSeparatorChar();
  this->project = project;
  this->currentDir = project->getRootFile();

  if (type != DIR) {
     bufsize = 80;
     datasize = bufsize;
     data = (Tfile**) malloc(sizeof(Tfile *)*datasize);
  } else {
     data = NULL;
  }

  // Criao dos widgets
  directory = IupText("do_nothing");
  IupSetAttribute(directory, IUP_READONLY, IUP_YES);
  IupSetAttribute(directory, IUP_VALUE, separatorChar);
  cancelButton = IupButton((char*)translate("CANCEL"), "wiofiledlg_cancel_cb");

  if (type == DIR) {
     tree = IupTree();
  }
  else {
     selection = IupText("wiofiledlg_selection_cb");
     up = IupButton("", "wiofiledlg_up_cb");
     home = IupButton("", "wiofiledlg_home_cb");
     createDir = IupButton("", "wiofiledlg_createDir_cb");
     matrix = IupMatrix("wiofiledlg_matrixAction_cb");

     if (type == OPEN) IupSetAttribute(matrix, "DIALTYPE", "0");
     else IupSetAttribute(matrix, "DIALTYPE", "1");

     filter = IupList("wiofiledlg_filter_cb"); 
  }

  if (type == OPEN) {
    // Quando o dilogo for para abrir arquivo, 
    // o usurio no pode digitar o seu nome.
    actionButton = IupButton((char*)translate("OPEN"), "do_nothing");
    IupSetAttribute(selection, IUP_READONLY, IUP_YES);
  }
  else {
    actionButton = IupButton((char*)translate("SAVE"), "do_nothing");
  }

  // Define os callbacks. Os nomes aqui so pr-fixados com 
  // wiofiledlg para evitar conflitos de nomes no IUP
  IupSetFunction("wiofiledlg_action_cb", (Icallback) action_cb);
  IupSetAttribute(actionButton, IUP_ACTION, "wiofiledlg_action_cb");

  if (type != DIR) {
     IupSetFunction("wiofiledlg_selection_cb", (Icallback) selection_cb);
     IupSetAttribute(selection, IUP_ACTION, "wiofiledlg_selection_cb");

     IupSetFunction("wiofiledlg_home_cb", (Icallback) home_cb);
     IupSetAttribute(home, IUP_ACTION, "wiofiledlg_home_cb");

     IupSetFunction("wiofiledlg_up_cb", (Icallback) up_cb);
     IupSetAttribute(up, IUP_ACTION, "wiofiledlg_up_cb");

     IupSetFunction("wiofiledlg_createDir_cb", (Icallback) createDir_cb);
     IupSetAttribute(createDir, IUP_ACTION, "wiofiledlg_createDir_cb");

     IupSetFunction("wiofiledlg_matrixAction_cb", (Icallback) matrixAction_cb);

     IupSetFunction("wiofiledlg_k_any_cb", (Icallback) k_any_cb);
     IupSetAttribute(matrix, IUP_K_ANY, "wiofiledlg_k_any_cb");

     IupSetFunction("wiofiledlg_edition_cb", (Icallback) edition_cb);
     IupSetAttribute(matrix, IUP_EDITION_CB, "wiofiledlg_edition_cb");

     IupSetFunction("wiofiledlg_value_cb", (Icallback) value_cb);
     IupSetAttribute(matrix, IUP_VALUE_CB, "wiofiledlg_value_cb");

     IupSetFunction("wiofiledlg_value_edit_cb", (Icallback) value_edit_cb);
     IupSetAttribute(matrix, IUP_VALUE_EDIT_CB, "wiofiledlg_value_edit_cb");

     IupSetFunction("wiofiledlg_click_cb", (Icallback) click_cb);
     IupSetAttribute(matrix, IUP_CLICK_CB, "wiofiledlg_click_cb");

     IupSetFunction("wiofiledlg_bgcolor_cb", (Icallback) bgcolor_cb);
     IupSetAttribute(matrix, "BGCOLOR_CB", "wiofiledlg_bgcolor_cb");

     IupSetFunction("wiofiledlg_filter_cb", (Icallback) filter_cb);
     IupSetAttribute(filter, IUP_ACTION, "wiofiledlg_filter_cb");

     IupSetFunction("wiofiledlg_cancel_cb", (Icallback) cancel_cb);
     IupSetAttribute(cancelButton, IUP_ACTION, "wiofiledlg_cancel_cb");
  }
  else {
     IupSetFunction("wiofiledlg_treeSelection_act", 
         (Icallback) treeSelection_cb);
     IupSetAttribute(tree, IUP_SELECTION_CB, "wiofiledlg_treeSelection_act") ;

     IupSetFunction("wiofiledlg_cancel_cb", (Icallback) cancelDir_cb);
     IupSetAttribute(cancelButton, IUP_ACTION, "wiofiledlg_cancel_cb");
  }  

  // Para evitar o uso de variveis globais, inserimos a prpria 
  // classe como um atributo do widget
  IupSetAttribute(actionButton, "WIO_FILE_DIALOG", (char*)this);
  IupSetAttribute(cancelButton, "WIO_FILE_DIALOG", (char*)this);
  if (type != DIR) IupSetAttribute(matrix, "WIO_FILE_DIALOG", (char*)this);
}

/* ....................................................................... */
/**
 * Destrutor que libera e aterra os atributos internos.
 */
WIOFileDialog::~WIOFileDialog() {
  if (separatorChar != NULL) delete[] separatorChar;
  if (currentDir != NULL) delete currentDir;
  if (data != NULL) free(data);
  data = NULL;
  currentDir = NULL;
  separatorChar = NULL;
}

/* ....................................................................... */
/**
 * Cria um dilogo de arquivo em IUP.
 * @return O handle do dilogo criado.
 */
Ihandle *WIOFileDialog::createFileDialog(enum WioFileDlgType type,
   const char *parent) throw () {
  IupSetAttribute(matrix, IUP_RESIZEMATRIX, IUP_YES);
  IupSetAttribute(matrix, IUP_EXPAND, IUP_YES);
  IupSetAttribute(matrix, IUP_NUMLIN_VISIBLE, "10");
  IupSetAttribute(matrix, IUP_NUMCOL, "4");
  IupSetAttribute(matrix, IUP_NUMCOL_VISIBLE, "4");
  IupSetAttribute(matrix, IUP_SCROLLBAR, IUP_VERTICAL);
  IupSetAttribute(matrix, "HIDEFOCUS", IUP_YES);
  IupSetAttribute(matrix, "WIDTH1", "160");
  IupSetAttribute(matrix, "WIDTH2", "60");
  IupSetAttribute(matrix, "WIDTH3", "80");
  IupSetAttribute(matrix, "WIDTH4", "60");
  IupSetAttribute(matrix, "HEIGHT0", "8");
  IupSetAttribute(matrix, "ALIGNMENT1", IUP_ALEFT);
  IupSetAttribute(matrix, IUP_FRAMECOLOR, "255 255 255");
  IupSetAttribute(actionButton, IUP_RASTERSIZE, "100x");
  IupSetAttribute(cancelButton, IUP_RASTERSIZE, "100x");

  Ihandle *dirFrame = IupFrame(matrix);

#include "up.ih"
#include "up_p.ih"
#include "newDir.ih"
#include "newDir_p.ih"
#include "home.ih"
#include "home_p.ih"

  IupSetAttribute(directory, IUP_EXPAND, IUP_HORIZONTAL);
  IupSetAttribute(selection, IUP_EXPAND, IUP_HORIZONTAL);
  IupSetAttribute(filter, IUP_DROPDOWN, IUP_YES);
  IupSetAttribute(filter, IUP_EXPAND, IUP_HORIZONTAL);
  IupSetAttribute(up, IUP_RASTERSIZE, "23x22");
  IupSetAttribute(up, IUP_IMAGE, "upButImg");
  IupSetAttribute(up, IUP_IMPRESS, "up_pButImg");
  IupSetAttribute(home, IUP_RASTERSIZE, "23x22");
  IupSetAttribute(home, IUP_IMAGE, "homeButImg");
  IupSetAttribute(home, IUP_IMPRESS, "home_pButImg");
  IupSetAttribute(createDir, IUP_RASTERSIZE, "23x22");
  IupSetAttribute(createDir, IUP_IMAGE, "newDirButImg");
  IupSetAttribute(createDir, IUP_IMPRESS, "newDir_pButImg");
  IupSetAttribute(actionButton, IUP_ACTIVE, IUP_NO);

  Ihandle *selLbl = IupLabel((char*)translate("SELECTION"));
  IupSetAttribute(selLbl, IUP_SIZE, "40");
  Ihandle *selBox = IupHbox(selLbl, selection, NULL);
  IupSetAttribute(selBox, IUP_ALIGNMENT, IUP_ACENTER);

  Ihandle *filterLbl = IupLabel((char*)translate("TYPES"));
  IupSetAttribute(filterLbl, IUP_SIZE, "40");
  Ihandle *filterBox = IupHbox(filterLbl, filter, NULL);
  IupSetAttribute(filterBox, IUP_ALIGNMENT, IUP_ACENTER);

  Ihandle *selVbox = IupVbox(selBox, filterBox, NULL);
  Ihandle *dirLbl = IupLabel((char*)translate("CURRENT_DIRECTORY"));
  Ihandle *dirBox = IupHbox(dirLbl, directory, up, home, createDir, NULL);
  IupSetAttribute(dirBox, IUP_ALIGNMENT, IUP_ACENTER);
  Ihandle *dirVbox = IupVbox(dirBox, NULL);

  Ihandle *buttonsHbox = IupHbox(IupFill(), actionButton, 
       cancelButton, IupFill(), NULL);

  Ihandle *dialog = IupDialog(
     IupVbox(
        IupFill(), dirVbox, IupFill(),
        IupHbox(IupFill(), dirFrame, IupFill(), NULL),
        IupFill(), selVbox, IupFill(),
        buttonsHbox, IupFill(), NULL
  ));

  IupSetAttribute(dialog, IUP_MARGIN, "3x3");
  IupSetAttribute(dialog, IUP_GAP, "4");
  IupSetAttribute(dialog, IUP_RESIZE, IUP_YES);
  IupSetAttribute(dialog, IUP_MAXBOX, IUP_NO);
  IupSetAttribute(dialog, IUP_MINBOX, IUP_NO);
  IupSetAttribute(dialog, IUP_DEFAULTENTER, "Open");
  IupSetAttribute(dialog, IUP_DEFAULTESC, "Cancel");
  char str[256];
  if (type ==  OPEN) {
    strcpy(str, (char*)translate("TITLE_OPEN"));
  } else {
    strcpy(str, (char*)translate("TITLE_SAVE"));
  } 
  strcat(str, project->getId());
  IupStoreAttribute(dialog, IUP_TITLE, str);

  IupSetAttribute(dialog, IUP_PARENTDIALOG, (char*) parent);
  IupSetAttribute(dialog, "DIALTYPE", IupGetAttribute(matrix, "DIALTYPE"));
  
  // Fechar a janela deve ser o mesmo que pressionar o boto de
  // cancelamento.
  IupSetAttribute(dialog, IUP_CLOSE_CB, "wiofiledlg_cancel_cb");
  IupSetAttribute(dialog, "WIO_FILE_DIALOG", (char*)this);

  return dialog;
}

/* ....................................................................... */
/**
 * Criao de um dilogo de busca de diretrios.
 */
Ihandle* WIOFileDialog::createDirDialog(const char *parent) throw () {
  IupSetAttribute(tree, IUP_FONT, IUP_TIMES_NORMAL_10);
  IupSetAttribute(tree, IUP_CTRL, IUP_NO);
  IupSetAttribute(tree, IUP_SHIFT, IUP_NO);
  IupSetAttribute(tree, IUP_ADDEXPANDED, IUP_NO);
  IupSetAttribute(actionButton, IUP_RASTERSIZE, "100x");
  IupSetAttribute(cancelButton, IUP_RASTERSIZE, "100x");

  Ihandle* dirFrame = IupFrame(tree);

  IupSetAttribute(directory, IUP_EXPAND, IUP_HORIZONTAL);

  Ihandle *dirLbl = IupLabel((char *)translate("CURRENT_DIRECTORY"));
  Ihandle *dirBox = IupHbox(dirLbl, directory, NULL);
  IupSetAttribute(dirBox, IUP_ALIGNMENT, IUP_ACENTER);
  Ihandle *dirVbox = IupVbox(dirBox, NULL);

  Ihandle *buttonsHbox = IupHbox(
        IupFill(), actionButton, cancelButton, IupFill(), NULL
   );

  Ihandle *dialog = IupDialog(
     IupVbox(
        IupFill(),
        dirVbox,
        IupFill(),
        IupHbox(IupFill(), dirFrame, IupFill(), NULL),
        IupFill(),
        buttonsHbox,
        IupFill(),
        NULL
     )
  );

  IupSetAttribute(dialog, IUP_GAP, "4");
  IupSetAttribute(dialog, IUP_RESIZE, IUP_NO);
  IupSetAttribute(dialog, IUP_MAXBOX, IUP_NO);
  IupSetAttribute(dialog, IUP_MINBOX, IUP_NO);
  IupSetAttribute(dialog, IUP_DEFAULTENTER, "Open");
  IupSetAttribute(dialog, IUP_DEFAULTESC, "Cancel");
  char str[256];
  strcpy(str, (char*)translate("TITLE_DIR"));
  strcat(str, project->getId());
  IupStoreAttribute(dialog, IUP_TITLE, str);

  IupSetAttribute(dialog, IUP_PARENTDIALOG, (char*) parent);
  
  // Fechar a janela deve ser o mesmo que pressionar o botao cancel
  IupSetAttribute(dialog, IUP_CLOSE_CB, "wiofiledlg_cancel_cb");
  IupSetAttribute(dialog, "WIO_FILE_DIALOG", (char *)this);

  return dialog;
}

/* ....................................................................... */
/**
 * Ajuste do filtro de seleo.
 * @param flt o filtro desejado.
 */
void WIOFileDialog::SetExtFilter(const char* flt) throw (wioapi::Exception*) {
  if (extFilter != NULL) free(extFilter);
  if (flt != NULL) extFilter = strdup(flt);
  else extFilter = NULL;
}

/* ....................................................................... */
/**
 * Ajuste da extenso default.
 * @param defExt a extenso desejada.
 */
void WIOFileDialog::SetDefaultExt(const char* defExt) 
throw (wioapi::Exception*) {
  if (defaultExt != NULL) free(defaultExt);
  if (defExt != NULL) defaultExt = strdup(defExt);
  else defaultExt = NULL;
}

/* ....................................................................... */
/**
 * Retorna o caminho absoluto do arquivo selecionado tendo como 
 * raiz, o raiz do projeto.
 * @return O caminho do arquivo selecionado.
 */
char* WIOFileDialog::getPath(void) throw () {
  if (error) return NULL;

  string dirname = IupGetAttribute(directory, "VALUE");
  string filename = IupGetAttribute(selection, "VALUE");
  string temp = dirname + filename;
  
  char* cstr = new char[temp.length()+1];
  strcpy(cstr, temp.c_str());

  return cstr;
}

/* ....................................................................... */

char* WIOFileDialog::getDir(void) throw () {
  if (error) return NULL;

  string dirname = IupGetAttribute(directory, "VALUE");
  
  char* cstr = new char[dirname.length()+1];
  strcpy(cstr, dirname.c_str());

  return cstr;
}

/* ....................................................................... */

void WIOFileDialog::fillTree(const char* initialDir, int level) 
throw (wioapi::Exception*) {
   WIOFile* currentDir = NULL;

   if (initialDir == NULL || strlen(initialDir) == 0) {
      initialDir = ".";
      currentDir = project->getRootFile();
   }
   else {
      currentDir = project->getFile(initialDir + 1);
   }
 
   int numfiles = currentDir->getNumFiles();
   for (int i = 0; i < numfiles; i++) {
       WIOFile *file = currentDir->getFile(i);
       char* path = file->getPath();
       char* fileName = strrchr(path, separatorChar[0]);

       if (fileName == NULL) fileName = path;
       else fileName += 1;

       if (file->isDirectory()) {
          if (strcmp(".", fileName) != 0 && strcmp("..", fileName) != 0) {
             char branch[25]; sprintf(branch,"ADDBRANCH%d",level);
             IupSetAttribute(tree,branch,fileName);

             char* subDir = new char[MAX_TEXT_SIZE];
             sprintf(subDir, "%s%s%s", initialDir, separatorChar, fileName);

             IupTreeSetUserId(tree,level+1,(void*)strdup(subDir));
             IupSetAttribute(tree,IUP_ADDEXPANDED,IUP_NO);

             // Go to next level of trees.
             fillTree(subDir, (level+1));
             delete []subDir;
         }
       }
       delete file;
   }
}

/* ....................................................................... */
/**
 * Preenche as listas de diretrios e arquivos a partir do diretrio corrente.
 */
void WIOFileDialog::fillLists(void) throw (wioapi::Exception*) {
  int fileCounter = 0;

  char str[MAX_TEXT_SIZE];
  sprintf(str, "F%s", IupGetAttribute(filter, IUP_VALUE));
  char* types = IupGetAttribute(filter, str);

  // Verifica se  o raiz do projeto para no por o item ".." na 
  // lista de diretrios
  if (strcmp(currentDir->getPath(), ".") != 0) {
     data[fileCounter] = new Tfile;
     data[fileCounter]->type = 1;
     data[fileCounter]->name = "..";
     fileCounter++;
  }

  int numfiles = currentDir->getNumFiles();
  for (int i = 0; i < numfiles; i++) {
      if (fileCounter >= datasize) {
         datasize = datasize + bufsize;
         data = (Tfile**) realloc(data, sizeof(Tfile *)*datasize);
      }

      WIOFile *file = currentDir->getFile(i);
      if (file->isDirectory()) {
         data[fileCounter] = new Tfile;
         data[fileCounter]->type = 1;
         char sep0 = separatorChar[0];
         if (strrchr(file->getPath(), sep0) != NULL) {
            sprintf(str, "%s", strrchr(file->getPath(), sep0) + 1);
         }
         else {
            sprintf(str, "%s", file->getPath());
         }
         data[fileCounter]->name = strdup(str);
         fileCounter++;
      }
      else if (filterFile(file->getPath(), types)) {
         data[fileCounter] = new Tfile;
         data[fileCounter]->type = 0;
         char sep0 = separatorChar[0];
         if (strrchr(file->getPath(), sep0) != NULL) {
            sprintf(str, "%s", strrchr(file->getPath(), sep0) + 1);
         }
         else {
            sprintf(str, "%s", file->getPath());
         }
         data[fileCounter]->name = strdup(str);
         data[fileCounter]->date = file->lastModificationTime();
         data[fileCounter]->size = file->size();
         fileCounter++;
      }
      delete file;
  }

  ndata = fileCounter;
  
  if (ndata <= 10) IupSetAttribute(matrix, IUP_NUMLIN, "11");
  else {
    char num_attr[16];
    sprintf(num_attr, "%d", ndata);
    IupStoreAttribute(matrix, IUP_NUMLIN, num_attr);
  }

  qsort(data, ndata, sizeof(Tfile*), sortData);
}

/* ....................................................................... */

void WIOFileDialog::fillFilterList(void) throw (wioapi::Exception*) {
  IupSetAttribute(filter, "1", NULL);
  if (extFilter == NULL) return;

  char* ext = strdup(extFilter);
  char* p = NULL;
  int i = -9;

  for (p = strtok(ext, "|"), i = 1; p != NULL; p = strtok(NULL, "|"), i++) {
     char str[MAX_FILTER_SIZE];
     strcpy(str, p);
     strcat(str, " (");
     p = strtok(NULL, "|");
     strcat(str, p);
     strcat(str, ")");
     
     char atti_name[MAX_INTEGER_SIZE];
     sprintf(atti_name, "%d", i);
     IupStoreAttribute(filter, atti_name, str);

     char attf_name[MAX_INTEGER_SIZE];
     sprintf(attf_name, "F%d", i);
     IupStoreAttribute(filter, attf_name, p);
  }

  char attend_name[MAX_INTEGER_SIZE];
  sprintf(attend_name, "%d", i);
  IupSetAttribute(filter, attend_name, NULL);
  IupSetAttribute(filter, IUP_VALUE, "1");

  delete ext;
}

/* ....................................................................... */

int WIOFileDialog::directories(WIOFileDialog *wioFileDialog, 
char *dirClicked) {
  if (strcmp(dirClicked, "..") == 0) {
     // Pega o caminho do diretrio pai do diretorio corrente
     char* directoryName = IupGetAttribute(
           wioFileDialog->directory, "VALUE");
    
     // Acha o caminho do diretrio pai manipulando o caminho 
     // do diretrio corrente
     char sep0 = wioFileDialog->separatorChar[0];
     char* t = strrchr(directoryName, sep0);
     t[0] = '\0';
     t = strrchr(directoryName, sep0);
     t[1] = '\0';

     WIOFile* parent = NULL;
     // Caso estejamos buscando o diretrio raiz, usamos getRootFile()
     if (strcmp(directoryName, wioFileDialog->separatorChar) == 0) {
        parent = wioFileDialog->project->getRootFile();
     }
    else {
       //usamos "directoryName+1" para tirar a barra inicial
       parent = wioFileDialog->project->getFile(directoryName + 1);
    }

    delete wioFileDialog->currentDir;
    wioFileDialog->currentDir = parent;

    IupStoreAttribute(wioFileDialog->directory, "VALUE", directoryName);
    wioFileDialog->fillLists();
  }
  else {
    string tempstr = IupGetAttribute(wioFileDialog->directory, "VALUE");
    tempstr += string(dirClicked) + wioFileDialog->separatorChar;

    // A converso de (const char*) para (char*)  possivel porque
    // IupStoreAttribute duplica a string.
    IupStoreAttribute(wioFileDialog->directory, "VALUE",
      (char*)tempstr.c_str());

    delete wioFileDialog->currentDir;
    wioFileDialog->currentDir =
         wioFileDialog->project->getFile(tempstr.c_str()+1);
    wioFileDialog->fillLists();
  }

  // limpa a caixa de nome de arquivo quando entramos em um diretrio novo
  IupSetAttribute(wioFileDialog->selection, "VALUE", "");
  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 *
 */
void WIOFileDialog::popupMenu(Ihandle* self) {
  Ihandle* item1 = IupItem((char*) translate("RENAME"), "rename_cb");
  Ihandle* menu = IupMenu(item1, NULL);

  IupSetAttribute(menu, "MATRIX", (char *) self);
  IupSetFunction("rename_cb", (Icallback) rename_cb);

  IupPopup(menu, IUP_MOUSEPOS, IUP_MOUSEPOS);

  IupDestroy(menu); 
}

/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */
/* Declarao de callbacks */
/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */

/* ....................................................................... */
/**
 * Callback chamado quando o usurio clica no boto de cancelar. 
 */
int WIOFileDialog::cancel_cb(Ihandle * self) {
  WIOFileDialog *wioFileDialog =
    (WIOFileDialog *) IupGetAttribute(self, "WIO_FILE_DIALOG");

  Ihandle* dial = IupGetDialog(self);
  IupSetAttribute(dial, "CANCELED", "1");

  // Fora o retorno de "" (string vazia) em getPath()
  IupSetAttribute(wioFileDialog->selection, "VALUE", "");
  IupSetAttribute(wioFileDialog->directory, "VALUE", "");

  return IUP_CLOSE;
}

/** 
 * Callback chamado quando o usurio clica no boto de cancelar;
 * usado quando o type de seleo for DIRETORIO 
 */
int WIOFileDialog::cancelDir_cb(Ihandle* self) {
  WIOFileDialog *wioFileDialog =
    (WIOFileDialog *) IupGetAttribute(self, "WIO_FILE_DIALOG");

  // Fora o retorno de "" (string vazia) em getPath()
  IupSetAttribute(wioFileDialog->directory, "VALUE", "");

  return IUP_CLOSE;
}


/* ....................................................................... */
/**
 * Callback chamado quando o usurio clica no boto HOME;
 * incializa como driretrio corrente HOME 
 */
int WIOFileDialog::home_cb(Ihandle* self) {
  WIOFileDialog *wioFileDialog =
    (WIOFileDialog *) IupGetAttribute(self, "WIO_FILE_DIALOG");

  int sel = IupGetInt(wioFileDialog->matrix, "SELECTED");

  delete wioFileDialog->currentDir;
  wioFileDialog->currentDir = wioFileDialog->project->getRootFile();
  IupSetAttribute(wioFileDialog->directory, IUP_VALUE, 
       wioFileDialog->separatorChar);

  wioFileDialog->fillLists();

  if (sel != 0) {
     IupSetAttribute(self, "SELECTED", NULL);
  }

  IupSetAttribute(wioFileDialog->selection, "VALUE", "");
  IupSetAttribute(wioFileDialog->actionButton, IUP_ACTIVE, IUP_NO);
  IupSetAttribute(wioFileDialog->matrix, IUP_REDRAW, "ALL");
  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Callback chamado quando o usurio clica no boto UP;
 * inicializa co corrente o diretrio imediatamente acima.
 */
int WIOFileDialog::up_cb(Ihandle* self) {
  WIOFileDialog *wioFileDialog =
    (WIOFileDialog *) IupGetAttribute(self, "WIO_FILE_DIALOG");

  int sel = IupGetInt(wioFileDialog->matrix, "SELECTED");

  // Pega o caminho do diretorio pai do diretorio corrente
  char* directoryName = IupGetAttribute(wioFileDialog->directory, "VALUE");
  const char* sepChar = wioFileDialog->separatorChar;
 
  if (strcmp(directoryName, sepChar) == 0) return IUP_DEFAULT;

  // Acha o caminho do diretrio pai manipulando o caminho 
  // do diretrio corrente
  const char sep0 = sepChar[0];
  char *t = strrchr(directoryName, sep0);
  t[0] = '\0';
  t = strrchr(directoryName, sep0);
  t[1] = '\0';

  delete wioFileDialog->currentDir;

  // Caso estejamos buscando o diretorio raiz, usamos getRootFile()
  const WIOProject* proj = wioFileDialog->project;
  if (strcmp(directoryName, sepChar) == 0) {
     wioFileDialog->currentDir = proj->getRootFile();
  }
  else {
    //usamos "directoryName+1" para tirar a barra inicial
    wioFileDialog->currentDir = proj->getFile(directoryName + 1);
  }
  IupStoreAttribute(wioFileDialog->directory, IUP_VALUE, directoryName);

  wioFileDialog->fillLists();

  if (sel != 0) IupSetAttribute(self, "SELECTED", NULL);
  IupSetAttribute(wioFileDialog->selection, "VALUE", "");
  IupSetAttribute(wioFileDialog->actionButton, IUP_ACTIVE, IUP_NO);

  IupSetAttribute(wioFileDialog->matrix, IUP_REDRAW, "ALL");

  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Callback chamado quando o usurio clica no boto "<i>New Folder</i>";
 * cria um novo diretrio no diretrio corrente 
 */
int WIOFileDialog::createDir_cb(Ihandle * self) {
  WIOFileDialog *wioFileDialog =
    (WIOFileDialog *) IupGetAttribute(self, "WIO_FILE_DIALOG");

  char newPath[MAX_TEXT_SIZE];
  char str[MAX_INTEGER_SIZE];

  char* directoryName = IupGetAttribute(wioFileDialog->directory, "VALUE");
  int sel = IupGetInt(wioFileDialog->matrix, "SELECTED");

  sprintf(newPath, "%s%s", directoryName, (char*) translate("NEW_FOLDER"));
  WIOProject* proj = wioFileDialog->project;
  for (int i = 2; proj->fileExists(newPath+1) == true; i++) {
      char* nf = (char*) translate("NEW_FOLDER");
      sprintf(newPath, "%s%s%d", directoryName, nf, i);
  }
  proj->createDirectory(newPath+1);

  Tfile** _data = wioFileDialog->data;
  int _ndata = wioFileDialog->ndata;
  _data[_ndata] = new Tfile;
  _data[_ndata]->type = 1;
  _data[_ndata]->name = strdup(newPath+1);

  wioFileDialog->ndata++;

  if (sel != 0) IupSetAttribute(self, "SELECTED", NULL);
  IupSetAttribute(wioFileDialog->selection, "VALUE", "");

  char sel_value[MAX_INTEGER_SIZE];
  sprintf(sel_value, "%d", wioFileDialog->ndata);
  IupStoreAttribute(wioFileDialog->matrix, "SELECTED", sel_value);
  IupSetAttribute(wioFileDialog->actionButton, IUP_ACTIVE, IUP_NO);
// IupSetAttribute(wioFileDialog->matrix, IUP_REDRAW, "ALL");

  IupSetFocus(wioFileDialog->matrix);
  sprintf(str, "%d:1", wioFileDialog->ndata);
  IupStoreAttribute(wioFileDialog->matrix, IUP_FOCUS_CELL, str);

  IupSetAttribute(wioFileDialog->matrix, "NOEDITION", "0");
  IupSetAttribute(wioFileDialog->matrix, IUP_EDIT_MODE, IUP_YES);

  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Callback chamado quando o usurio clica no boto de abrir ou salvar.
 */
int WIOFileDialog::action_cb(Ihandle* self) {

  if (IupGetInt(self, "DIALTYPE") == SAVE) {
     WIOFileDialog *wioFileDialog =
       (WIOFileDialog *) IupGetAttribute(self, "WIO_FILE_DIALOG");
 
     int appendExt = 0;
     char path[MAX_TEXT_SIZE];
     strcpy(path, wioFileDialog->getPath());
     char *c = strrchr(path, '.');
     if (wioFileDialog->defaultExt != NULL && 
        (c==NULL || mystricmp(c+1, wioFileDialog->defaultExt)!=0)) {
        strcat(path, ".");
        strcat(path, wioFileDialog->defaultExt);
        appendExt = 1;
     }

     if (wioFileDialog->project->fileExists(path+1)==true) {
        char str[MAX_TEXT_SIZE];
        char* fexists = (char*) translate("FILE_EXISTS");
        sprintf(str, (char*) translate("%s\n%s"), path, fexists);
        int ret = IupAlarm((char*)translate("ATTENTION"), str, 
            (char*) translate("YES"),
            (char*) translate("NO"), NULL);
        if (ret==2) return IUP_IGNORE;
     }

     if (appendExt) {
        strcpy(path, IupGetAttribute(wioFileDialog->selection, IUP_VALUE));
        strcat(path, ".");
        strcat(path, wioFileDialog->defaultExt);
        IupSetAttribute(wioFileDialog->selection, IUP_VALUE, strdup(path));
     }
  }

  Ihandle* dial = IupGetDialog(self);
  IupSetAttribute(dial, "CANCELED", "0");
  return IUP_CLOSE;
}

/* ....................................................................... */
/**
 * Callback
 */
int WIOFileDialog::selection_cb(Ihandle* self, int c, char *after) {
   WIOFileDialog* wioFileDialog = (WIOFileDialog*) 
      IupGetAttribute(self, "WIO_FILE_DIALOG");

   if (after == NULL || after[0] == '\0') {
      IupSetAttribute(wioFileDialog->actionButton, IUP_ACTIVE, IUP_NO);
   }
   else {
      IupSetAttribute(wioFileDialog->actionButton, IUP_ACTIVE, IUP_YES);
   }
   return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Callback
 */
int WIOFileDialog::treeSelection_cb(Ihandle* self, int id,int status) {
   WIOFileDialog* wioFileDialog = (WIOFileDialog*) 
      IupGetAttribute(self, "WIO_FILE_DIALOG");

   if (status == 0) return IUP_DEFAULT;

   char* dir = (char*) IupTreeGetUserId(self,id);
   IupSetAttribute(wioFileDialog->directory, IUP_VALUE, strdup(dir));

   return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Callback
 */
int WIOFileDialog::rename_cb(Ihandle* self) {
  Ihandle* matrix = (Ihandle *) IupGetAttribute(self, "MATRIX");

  IupSetAttribute(matrix, "NOEDITION", "0");
  IupSetAttribute(matrix, IUP_EDIT_MODE, IUP_YES);

  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 *
 */
int WIOFileDialog::click_cb(Ihandle* self, int lin, int col, char *r) {
   WIOFileDialog *wioFileDialog = (WIOFileDialog*) 
      IupGetAttribute(self, "WIO_FILE_DIALOG");

   int sel = IupGetInt(self, "SELECTED");
   if (isbutton3(r) && sel==lin) {
      popupMenu(self);
      return IUP_DEFAULT;
   }

   if (!isbutton1(r)) return IUP_DEFAULT;

   IupSetAttribute(self, "NOEDITION", "1");

   if (lin == 0 || col == 0 || lin > wioFileDialog->ndata) {
      IupSetAttribute(self, "SELECTED", NULL);
      IupSetAttribute(wioFileDialog->selection, "VALUE", "");
      IupSetAttribute(wioFileDialog->actionButton, IUP_ACTIVE, IUP_NO);
      return IUP_DEFAULT;
   }

   Tfile* file = wioFileDialog->data[lin-1];
   Ihandle* acButton = wioFileDialog->actionButton;
   if (file->type == 0) IupSetAttribute(acButton, IUP_ACTIVE, IUP_YES);
   else IupSetAttribute(acButton, IUP_ACTIVE, IUP_NO);

   char selected_attr[MAX_INTEGER_SIZE];
   sprintf(selected_attr, "%d", lin);
   IupStoreAttribute(self, "SELECTED", selected_attr);

   if (file->type == 0) {
      static char filename[MAX_TEXT_SIZE];
      sprintf(filename, "%s", file->name);
      IupStoreAttribute(wioFileDialog->selection, "VALUE", filename);
   }
   else {
      IupSetAttribute(wioFileDialog->selection, "VALUE", "");
   }

   if (isdouble(r) && file->type == 1) {
      char* dirClicked = file->name;
      directories(wioFileDialog, dirClicked);
      IupSetAttribute(self, "SELECTED", NULL);
      IupSetAttribute(self, "NOEDITION", "1");
      IupSetAttribute(self, IUP_EDIT_MODE, IUP_NO);
      IupSetAttribute(wioFileDialog->actionButton, IUP_ACTIVE, IUP_NO);
   }

   IupSetAttribute(self, IUP_REDRAW, "ALL");
   return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 *
 */
int WIOFileDialog::bgcolor_cb(Ihandle *self, int lin, int col, 
unsigned int *r, unsigned int *g, unsigned int *b) {
   if (lin == 0) return IUP_IGNORE;

   int sel = IupGetInt(self, "SELECTED");
   unsigned int color = ((sel == lin) ? color = 230 : 255);

   *r = *g = *b = color;
   return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 *
 */
int WIOFileDialog::value_edit_cb(Ihandle *self, int lin, int col, char *nv) {
   WIOFileDialog *wioFileDialog =
     (WIOFileDialog *) IupGetAttribute(self, "WIO_FILE_DIALOG");

   if (IupGetInt(self, "NOEDITION") == 1) return IUP_DEFAULT;

   char str[MAX_INTEGER_SIZE];
   char fpath[MAX_TEXT_SIZE];
   char newPath[MAX_TEXT_SIZE];
   char newVal[MAX_TEXT_SIZE];
   char *v = NULL;

   strcpy(newVal, nv);
   char *path = IupGetAttribute(wioFileDialog->directory, "VALUE");
   sprintf(str, "%d:%d", lin, col);
   v = IupGetAttribute(wioFileDialog->createDir, "NEW_FOLDER_NAME");
   sprintf(fpath,"%s%s" , path, v);
   sprintf(newPath,"%s%s" , path, newVal);
   if (strcmp(fpath, newPath) != 0) {
      wioFileDialog->project->renameFile(fpath+1, newVal);
   }
   wioFileDialog->fillLists();

   return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 *
 */
char* WIOFileDialog::value_cb(Ihandle *self, int lin, int col) {
  static char str[MAX_TEXT_SIZE];

  WIOFileDialog* wioFileDialog = (WIOFileDialog*) 
      IupGetAttribute(self, "WIO_FILE_DIALOG");

  char* title[] = {"FILE", "TYPE", "DATE_MODIFIED", "SIZE"};
  char* sizeExt[] = {"B", "KB", "MB", "GB"};

  if (lin > wioFileDialog->ndata) return " ";

  if (lin == 0) {
     if (col == 0) return "";
     return (char*) translate(title[col-1]);
  }

  Tfile* _data = wioFileDialog->data[lin-1];
  switch(col) {
    case 1: {
       char* name = _data->name;
       char sep0 = wioFileDialog->separatorChar[0];
       char* filename = strrchr(name, sep0);
       if (filename != NULL) {
         sprintf(str, "%s", filename+1);
       }
       else {
          sprintf(str, "%s", name);
       }
       return str;
    }
    case 2: {
       if (_data->type == 1) return (char*)translate("FILE_FOLDER");
       else return "";
    }
    case 3: {
       if (_data->type == 1) break;
       WIODateTimeInfo* dt = _data->date;
       sprintf(str, "%02d/%02d/%02d %02d:%02d", 
          dt->day, dt->month, dt->year,
          dt->hour, dt->minute);
       return str;
    }
    case 4: {
       int i;
       if (_data->type == 1) break;
       long long size = _data->size;
       for(i = 0; size > 1024 && i < 4; i++) size = size/1024;
       sprintf(str, "%ld %s", (long)size, sizeExt[i]);
       return str;
    }
    default: break;
  }
  return " ";
}

/* ....................................................................... */
/**
 * Callback chamada quando o usurio clica na MATRIZ.
 */
int WIOFileDialog::matrixAction_cb(Ihandle* self, int c, int lin, int col, 
int active, char* after) {
  if (c == 27) {
     IupSetAttribute(self, "NOEDITION", "1");
     IupSetAttribute(self, IUP_EDIT_MODE, IUP_NO);
     return IUP_IGNORE;
  }

  if (c == 13) {
     IupSetAttribute(self, "NOEDITION", "0");
     IupSetAttribute(self, IUP_EDIT_MODE, IUP_NO);
     return IUP_IGNORE;
  }

  IupSetAttribute(self, "NOEDITION", "0");

  if (!(c>=48 && c<=57) && !(c>=65 && c<=90) && !(c>=97 && c<=122) &&
      c!=46 && c!=95 && c!=8 && c!=339 && c!=331 && c!=333) {
      return IUP_IGNORE;
  }
  return c;
}

/* ....................................................................... */
/**
 * Callback chamada quando com o foco na matriz, o usurio clicar numa tecla.
 */
int WIOFileDialog::k_any_cb(Ihandle * self, int c) {
  WIOFileDialog *wioFileDialog = (WIOFileDialog*) 
      IupGetAttribute(self, "WIO_FILE_DIALOG");

  int sel = IupGetInt(self, "SELECTED");
  if (sel < 1) return IUP_IGNORE;

  if (c == K_DEL) {
     char* mode = IupGetAttribute(self, IUP_EDIT_MODE);
     if (strcmp(mode, IUP_NO) == 0) {
       char fpath[MAX_TEXT_SIZE];
       char str[MAX_INTEGER_SIZE];
       char* path = IupGetAttribute(wioFileDialog->directory, "VALUE");
       sprintf(str, "%d:1", sel);

       char* v = IupGetAttribute(self, str);
       sprintf(fpath,"%s%s" , path, v);
       wioFileDialog->project->deleteFile(fpath+1);
       IupSetAttribute(self, "SELECTED", NULL);
       IupSetAttribute(wioFileDialog->selection, "VALUE", "");
       IupSetAttribute(wioFileDialog->actionButton, IUP_ACTIVE, IUP_NO);
       wioFileDialog->fillLists();
       IupSetAttribute(wioFileDialog->matrix, IUP_REDRAW, "ALL");
     } 
  }
  else if (c == K_F2) {
     IupSetAttribute(self, "NOEDITION", "0");
     IupSetAttribute(self, IUP_EDIT_MODE, IUP_YES);
  }
  return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Callback de edio da matriz.
 */
int WIOFileDialog::edition_cb(Ihandle* self, int lin, int col, int mode) {
   if (col != 1) return IUP_IGNORE;

   if (mode == 1) {
      if (IupGetInt(self, "NOEDITION") == 1) return IUP_IGNORE;
      char str[MAX_INTEGER_SIZE];
      sprintf(str, "%d:%d", lin, col);
      char* v = IupGetAttribute(self, str);
      if (strcmp(v, "..") == 0) return IUP_IGNORE;
      int n = strlen(v);

      char caret_attr[MAX_INTEGER_SIZE];
      sprintf(caret_attr, "%d", n+1);
      IupStoreAttribute(self, IUP_CARET, caret_attr);

      char selection_attr[MAX_INTEGER_SIZE];
      sprintf(selection_attr, "1:%d", n+1);
      IupStoreAttribute(self, IUP_SELECTION, selection_attr);

      WIOFileDialog *wioFileDialog = 
        (WIOFileDialog*) IupGetAttribute(self, "WIO_FILE_DIALOG");
      IupStoreAttribute(
        wioFileDialog->createDir, "NEW_FOLDER_NAME", v);
   }

   return IUP_DEFAULT;
}

/* ....................................................................... */
/**
 * Callback chamada quando o usurio clica na lista <code>FILTER</code>. 
 */
int WIOFileDialog::filter_cb(Ihandle * self, char *t, int i, int v) {
  WIOFileDialog *wioFileDialog =
    (WIOFileDialog *) IupGetAttribute(self, "WIO_FILE_DIALOG");

  if (v == 1) wioFileDialog->fillLists();
  IupSetAttribute(wioFileDialog->matrix, IUP_REDRAW, "ALL");
  return IUP_DEFAULT;
}

/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */
/* ----------------------------------------------------------------------- */

/* ....................................................................... */
// Funo exportada: documentao no arquivo de interface.
char* wioDirDlg(const char *server, const char *port, const char *user, 
const char *pwd, const char *projectId, const char* owner, 
const char *initialDir, const char *parent) throw (wioapi::Exception*) {
   const char* ior = WIOLocator::createIOR(server, port);
   WIOFileSystem* fs = WIOLocator::buildFileSystem(ior);

   WIOProject* project = NULL;
   if (owner==NULL || strcmp(owner, user)==0) {
      project = fs->openProject(user, pwd, projectId);
   }
   else {
      project = fs->openProjectFromOthers(user, pwd, owner, projectId);
   }
    
   char* path = wioDirDlg(fs, project, initialDir, parent);
   delete project;    
   delete fs;
   return path;
}

/* ....................................................................... */
// Funo exportada: documentao no arquivo de interface.
char* wioDirDlg(wioapi::WIOFileSystem* filesystem,
wioapi::WIOProject* project, const char* initialDir, 
const char* parent) throw (wioapi::Exception*) {

  WIOFileDialog wioFileDialog(filesystem, project, DIR);

  Ihandle* dialog = wioFileDialog.createDirDialog(parent);
  wioFileDialog.fillTree(initialDir, 0);

  IupPopup(dialog, IUP_CENTER, IUP_CENTER);
  
  // Na destruio do dilogo, a string com path tambm  deletada.
  char* path = wioFileDialog.getDir();
  IupDestroy(dialog);
  return path;
}

/* ....................................................................... */
// Funo exportada: documentao no arquivo de interface.
char* wioFileDlg(const char *server, const char *port, const char *user, 
const char *pwd, const char *projectId, const char* owner, 
enum WioFileDlgType type, const char *defaultExt, const char *filterExt, 
const char *parent) throw (wioapi::Exception*) {
   const char* ior = WIOLocator::createIOR(server, port);
   WIOFileSystem* fs = WIOLocator::buildFileSystem(ior);

   WIOProject* project = NULL;
   if (owner == NULL || strcmp(owner, user) == 0) {
      project = fs->openProject(user, pwd, projectId);
   }
   else {
      project = fs->openProjectFromOthers(user, pwd, owner, projectId);
   }
    
   char* path = wioFileDlg(fs, project, type, defaultExt, filterExt, parent);
   delete project;    
   delete fs;
   return path;
}

/* ....................................................................... */
// Funo exportada: documentao no arquivo de interface.
char* wioFileDlg(wioapi::WIOFileSystem* filesystem, 
wioapi::WIOProject* project, enum WioFileDlgType type, 
const char* defaultExt, const char* extFilter, const char* parent) 
throw (wioapi::Exception*) {
  WIOFileDialog wioFileDialog(filesystem, project, type);
  Ihandle* dialog = NULL;

  if (type != DIR) {
     wioFileDialog.SetExtFilter(extFilter);
     wioFileDialog.SetDefaultExt(defaultExt);
     wioFileDialog.fillFilterList();
     wioFileDialog.fillLists();
     dialog = wioFileDialog.createFileDialog(type, parent);
  }
  else {
     wioFileDialog.fillTree(NULL, 0);
     dialog = wioFileDialog.createDirDialog(parent);
  }

  IupPopup(dialog, IUP_CENTER, IUP_CENTER);

  if (IupGetInt(dialog, "CANCELED") == 1) return NULL;
  
  // Na destruio do dilogo, a string com path tambm  deletada.
  char* path = wioFileDialog.getPath();
  IupDestroy(dialog);
  return path;
}

/* ....................................................................... */
// Funo exportada: documentao no arquivo de interface.
char* wioFileDlg(const char *server, const char *port, const char *user, 
const char *pwd, const char *projectId, const char* owner, 
enum WioFileDlgType type, const char *parent) throw (wioapi::Exception*) {
   return wioFileDlg(server, port, user, pwd, projectId, 
          owner, type, NULL, NULL, parent);
}

/* ....................................................................... */
// Funo exportada: documentao no arquivo de interface.
char* wioFileDlg(wioapi::WIOFileSystem* filesystem, 
wioapi::WIOProject* project, enum WioFileDlgType type, const char* parent) 
throw (wioapi::Exception*) {
   return wioFileDlg(filesystem, project, type, NULL, 
          (char*)translate("ALL_FILES"), parent);
}

