/**
 * @mainpage 
 * @section intro Introduo
 * Existem dois grupos de funcionalidades:
 * <ul>
 * <li> API bsica WIO - um conjunto de classes para acesso aos dados 
 *      do sistema de arquivos remoto.
 * <li> UTIL - Conjunto de funes utilitrias para 
 *      download/upload de arquivos.
 * </ul>
 *
 * <p>
 * Cada um desses dois grupos et definido em um arquivo de interface
 * (<code>.h</code>) distinto: <code>wioapi.h</code> e <code>util.h</code>.
 */

/**
 * @file wioapi.h
 * Arquivo de definio da interface exportada do WIO (binding C++).
 * $Id$
 * @author Aldo de Araujo Nogueira
 * @author Tatiana Almeida Souza Coelho
 * @author Andr Luiz Clinio
 */

#ifndef __WIOAPI_H__
#define __WIOAPI_H__

#include <stddef.h>

/** 
 * define que representa a declarao dos mtodos para 
 * gerao de DLL's em ambiente Windows (se necessrio).
 */
#if !defined(WIO_API)
  #if defined(WIN32)
    #if defined(WIO_DLL_MAKE_EXPORT)
      #define WIO_API __declspec(dllexport)
    #elif defined(WIO_DLL_MAKE_IMPORT)
      #define WIO_API __declspec(dllimport)
    #else
      #define WIO_API
    #endif
  #else
    #define WIO_API
  #endif
#endif

/**
 * Namespace da interface exportada do sistema WIO (Web I/O).
 */
namespace wioapi {

class WIOFileSystem;
class WIOProject;
class WIOFile;


// =======================================================================
/**
 * Enumerao de flag de abertura (modo) de arquivo.
 * Copiado de wio.h
 */
enum WIO_API WIOFileMode {
   /** Arquivo no foi aberto remotamente. */   
   NOT_OPENED = 0, 

   /** Arquivo remoto aberto em modo de leitura. */   
   MODE_READ,      

   /** Arquivo remoto aberto em modo de escrita. */   
   MODE_WRITE      
};

/**
 * Enumerao de tipos de seek nos arquivos remotos.
 */
enum WIO_API SeekType {
   /** Seek em posio determinada.*/
   SEEK_TYPE_SET = 0, 

   /** Seek a partir da posio corrente do arquivo.*/
   SEEK_TYPE_CUR,     

   /** Seek em relao ao final do arquivo.*/
   SEEK_TYPE_END      
};


// =======================================================================
/**
 * Data e hora de um arquivo
 * classes WIO.
 */
struct WIO_API WIODateTimeInfo {
    char internal_string[128];

    int day;
    int month;
    int year;
    int hour;
    int minute;
    int second;

  public: 

    char* dump(void);

    virtual ~WIODateTimeInfo();

    int getDay(void);
    int getMonth(void);
    int getYear(void);
    int getHour(void);
    int getMinute(void);
    int getSecond(void);

    WIODateTimeInfo();
    WIODateTimeInfo(int _day, int _month, int _year, 
          int _hour, int _minute, int _second);
};

// =======================================================================
/**
 * Lista de projetos (informaes) compatilhados.
 * classes WIO.
 */
struct WIO_API SharedProjectInfo {
   public: 
     const char* ownerId;     /**< Identificador do dono do projeto */
     const char* ownerName;   /**< Nome do dono do projeto */
     const char* projectId;   /**< Identificador do projeto */
};


// =======================================================================
/**
 *
 *
 */
struct WIO_API WIOAlgorithmParameter {
      const char* name;
      const char* value;
};

// =======================================================================
/**
 *
 *
 */
struct WIO_API WIOAlgorithmInfo {
      const char* algorithmName;
      long algorithmParameterAmount;
      WIOAlgorithmParameter* algorithmParameters;
};

// =======================================================================
/**
 * Exceo lanada se algo der errado em qualquer mtodo das 
 * classes WIO.
 */
class WIO_API Exception {

  private:

     /** Nome do mtodo remoto envolvido */
     char* method_name;

     /** A mensagem do servidor descrevendo o que ocorreu. */
     char* server_message;

     /** A mensagem do cliente descrevendo o que ocorreu. */
     char* client_message;

  public:

     /**
      * Construtor com o motivo da exceo.
      * @param method_name nome do mtodo remoto envolvido.
      * @param client_message a mensagem do servidor.
      * @param service_message a mensagem do servidor.
      */
     Exception(const char* method_name, const char* client_message, 
     const char* server_message);

     /**
      * Destrutor padro.
      */
     ~Exception();

   
     /**
      * Obtm a mensagem do servidor descrevendo essa exceo.
      * @return a mensagem.
      */
      const char* getServerMessage();

     /**
      * Obtm a mensagem do cliente descrevendo essa exceo.
      * @return a mensagem.
      */
      const char* getClientMessage();

     /**
      * Obtm o nome do mtodo remoto envolvido
      * @return o nome do mtodo.
      */
      const char* getMethodName();
};

// =======================================================================
/**
 * Classe que representa o localizador ORB do sistema de arquivos
 * remoto (WIOFileSystem).
 */
class WIO_API WIOLocator {

    /** Instncia do ORB utilizado. */
    static void* orb;
                
  public:
    /**
     * Finaliza o ORB utilizado no WIO.
     * @return um flag booleano indicativo de sucesso.
     */
    static bool finishORB(void);

    /** 
     * Ajusta o ORB utilizado pelo WIO.
     * @param userdata um ponteiro que representa a refrencia ao ORB a
     *        ser utilizado pelo WIO.
     */
    static void setORB(void* userdata);

    /**
     * Cria automaticamente um ORB que pode ser utilizado pelo WIO.
     * @param argc a quantidade de argumentos recebidos pela aplicao.
     * @param argv os argumentos recebidos pela aplicao.
     * @return um ponteiro que representa a refrencia ao ORB.
     */
    static void* startORB(int argc, char* argv[]);

    /** 
     * Consulta ao ORB corrente utilizado pelo WIO.
     * @return um ponteiro que representa a refrencia ao ORB.
     */
    static void* getORB(void);

    /**
     * Monta um texto que represente uma referncia IOR a um sistema
     * de arquivos remotos WIO.
     * @param server o nome da mquina aonde se localiza o servidor.
     * @param port a porta utilizada pelo servidor.
     * @return a referncia textual IOR do sistema de arquivos remoto.
     */
    static const char* createIOR(const char* server, const char* port);

    /**
     * Constri de um sistema de arquivos remoto.
     * @param ior o texo da referncia remota (IOR).
     * @return o sistema de arquivos remoto.
     */
    static WIOFileSystem* buildFileSystem(const char* ior);
};


// =======================================================================
/**
 * Classe que representa um sistema de arquivos remoto.
 */
class WIO_API WIOFileSystem {

  public:

     /**
      * Consulta o IOR definido para o objeto remoto file system.
      * @return o valor do IOR
      */
     char* getIOR(void);

     /**
      * Obtm a identificao da verso usada pelo servidor.
      * @return uma string contendo a identificao da verso
      */
     virtual char* getServerVersion(void) const throw (Exception*) = 0;

     /**
      * Obtm o caracter separador de diretrios do sistema de arquivos remoto.
      * @return uma string contendo o separador de diretrios.
      */
     virtual char* getSeparatorChar(void) const throw (Exception*) = 0;

     /**
      * Obtm o tamanho mximo do pacote em um wiofile::read().
      * @return o tamanho em Kbytes.
      */
     virtual long getMaxReadKb(void) const throw (Exception*) = 0;

     /**
      * Obtm o tamanho mximo do pacote em um wiofile::write().
      * @return o tamanho em Kbytes.
      */
     virtual long getMaxWriteKb(void) const throw (Exception*) = 0;

     /**
      * Consulta ao nmero de projetos abertos no servidor.
      * @return o nmero de projetos abertos.
      */
     virtual long getNumOpenedProjects() const throw (Exception*) = 0;

     /**
      * Consulta ao mximo de projetos abertos por servidor.
      * @return o nmero mximo de projetos.
      */
     virtual long getMaxOpenedProjects() const throw (Exception*) = 0;

     /**
      * Obtm a lista de projetos compartilhados pelo usurio
      * @param userId o login do usurio.
      * @param passwd a senha do usurio.
      * @param qtd a referncia para retorno da quantidade de projetos achada.
      * @return a lista com os nomes dos projetos.
      */
     virtual SharedProjectInfo* getProjectsFromOthers(const char* userId, 
             const char* passwd, long& qtd) const  throw (Exception*) = 0;

     /**
      * Obtm a lista de projetos nesse servidor.
      * @param userId o login do usurio.
      * @param passwd a senha do usurio.
      * @param qtd a referncia para retorno da quantidade de projetos achada.
      * @return a lista com os nomes dos projetos.
      */
     virtual char** getProjects(const char* userId, const char* passwd, 
             long& qtd) const  throw (Exception*) = 0;

     /**
      * Abre um determinado projeto.
      * @param userId o login do usurio
      * @param passwd o senha do usurio
      * @param projectId o identificador do projeto.
      * @return o projeto remoto escolhido.
      */
     virtual WIOProject* openProject(const char* userId, const char* passwd, 
             const char* projectId) const throw (Exception*) = 0;

     /**
      * Abre um determinado projeto de outro usurio.
      * @param userId o login do usurio
      * @param passwd o senha do usurio
      * @param ownerId o login do owner dos projetos desejados.
      * @param projectId o identificador do projeto.
      * @return o projeto remoto escolhido.
      */
     virtual WIOProject* openProjectFromOthers(const char* userId, 
             const char* passwd, const char* ownerId, 
             const char* projectId) const throw (Exception*) = 0;

    /**
     *   Executa o pr-login de um usurio no servidor CSBase e retorna
     * uma URL para acesso ao sistema com esse usurio.
     *   Quando executar o sistema com essa URL de acesso, o projeto
     * informado (pertencente ao usurio) ser aberto e o arquivo informado
     * ser executado.
     *
     * @param login O login do usurio.
     * @param password A senha do usurio.
     * @param projectName O nome de um projeto a ser aberto ao se usar a URL de
     *        acesso.
     * @param fileToExecute O nome de um arquivo a ser executado ao se usar a
     *        URL de acesso.
     *
     * @return Uma URL para acesso ao sistema.
     */
     virtual char* preLogin2Web(const char* login, const char* password,
             const char* projectName,
             const WIOAlgorithmInfo& algorithmInfo)
             const throw (Exception*) = 0;

     /**
      * Destrutor padro, que, devido ao polimorfismo, precisa ser virtual.
      */
     virtual ~WIOFileSystem() = 0;
};




// =======================================================================
/**
 * Classe que modela um projeto remoto.
 */
class WIO_API WIOProject {

  public:

    /**
     * Destrutor.
     */
    virtual ~WIOProject() = 0;
   
   /**
    * Obtm o id (nome) do projeto.
    * @return o id do projeto.
    */
   virtual char* getId() const throw (Exception*) = 0;
    
   /**
    * Obtm o login (id) de um usurio
    * @return o id do usurio.
    */
   virtual char* getUser() const throw (Exception*) = 0;
    
   /**
    * Obtm o dirtrio raiz do projeto
    * @return o diretrio remoto raiz do projeto, 
    *         sob o qual a rvore do projeto est construda.
    */
   virtual WIOFile* getRootFile() const throw (Exception*) = 0;
    
   /**
    * Consulta ao nmero de arquivos abertos no projeto.
    * @return o nmero de arquivos abertos.
    */
   virtual long getNumOpenedFiles() const throw (Exception*) = 0;

   /**
    * Consulta ao mximo de arquivos abertos por projeto.
    * @return o nmero mximo de arquivos.
    */
   virtual long getMaxOpenedFiles() const throw (Exception*) = 0;

   /**
    * Obtm um arquivo remoto do projeto.
    * @param path o caminho do arquivo que se deseja recuperar
    * @return o arquivo remoto recuperado.
    */
   virtual WIOFile* getFile(const char* path) const throw (Exception*) = 0;
    
   /**
    * Verifica se o arquivo existe no projeto remoto.
    * @param path o caminho do arquivo 
    * @return <tt>true</tt>, se o arquivo especificado existe. 
    *         <tt>false</tt>, caso contrrio
    */
   virtual bool fileExists(const char* path) const throw (Exception*) = 0;
    
   /**
    * Renomeia o arquivo remoto.
    * @param path o caminho do arquivo a ser renomeado
    * @param newName o novo nome do arquivo
    * @return <tt>true</tt>, se a operao foi feita com sucesso; 
    *         <tt>false</tt>, caso contrrio
    */
   virtual void renameFile(const char* path, const char* newName) 
   throw (Exception*) = 0;
    
   /**
    * Copia o arquivo remoto.
    * @param fromPath o caminho do arquivo origem
    * @param toPath o caminho do arquivo destino
    * @return <tt>true</tt>, se o arquivo foi apagado com sucesso; 
    *         <tt>false</tt>, caso contrrio
    */
   virtual void copyFile(const char* fromPath, const char* toPath) 
   throw (Exception*) = 0;
    
   /**
    * Move o arquivo  remoto.
    * @param fromPath o caminho do arquivo origem
    * @param toPath o caminho do arquivo destino
    * @return <tt>true</tt>, se o arquivo foi apagado com sucesso; 
    *         <tt>false</tt>, caso contrrio
    */
   virtual void moveFile(const char* fromPath, const char* toPath) 
   throw (Exception*) = 0;
    
   /**
    * Apaga o arquivo remoto.
    * @param path o caminho do arquivo a ser apagado
    * @return <tt>true</tt>, se o arquivo foi apagado com sucesso; 
    *         <tt>false</tt>, caso contrrio
    */
   virtual void deleteFile(const char* path) throw (Exception*) = 0;
    
   /**
    * Cria um novo arquivo com o caminho especificado 
    * @param path caminho do arquivo a ser criado
    * @param type tipos do arquivo no servidor.
    * @param userId usurio que criou o arquivo (em projeto multi-usurio).
    * @return arquivo remoto criado
    */
   virtual WIOFile* createFile(const char* path, const char* type, 
   const char* userId) throw (Exception*) = 0;
    
   /**
    * Cria um novo diretrio com o caminho especificado
    * @param path o caminho do diretrio a ser criado
    * @return o diretrio remoto criado
    */
   virtual WIOFile* createDirectory(const char* path) throw (Exception*) = 0;

   /**
    * Destroi o projeto remoto (chamado automaticamente pelo destrutor).
    */
   virtual void destroy(void) throw (Exception*) = 0;
    
};

// =======================================================================
// =======================================================================
// =======================================================================

/**
 * Classe que representa um arquivo remoto.
 */
class WIO_API WIOFile {

  public:

   /**
    * Destrutor.
    */
   virtual ~WIOFile() = 0;

   /**
    * Obtm o caminho do arquivo
    * @return o caminho
    */
   virtual char* getPath() const throw (Exception*) = 0;
   
   /**
    * Obtm a descrio.
    * @return a descrio.
    */
   virtual char* getDescription() const throw (Exception*) = 0;
   
   /**
    * Appenda a descrio.
    * @param desc a descrio.
    */
   virtual void appendDescription(const char* desc) 
   const throw (Exception*) = 0;
   
   /**
    * Ajusta a descrio.
    * @param desc a descrio.
    */
   virtual void setDescription(const char* desc) const throw (Exception*) = 0;
   
   /**
    * Obtm o identificador de quem criou o arquivo
    * @return o identificador como string
    */
   virtual char* whoCreated(void) const throw (Exception*) = 0;
   
   /**
    * Obtm o diretrio-pai do arquivo.
    * @return o diretrio-pai remoto.
    */
   virtual WIOFile* getParent(void) const throw (Exception*) = 0;

   /**
    * Obtm o projeto ao qual o arquivo est vinculado
    * @return o projeto remoto.
    */
   virtual WIOProject* getProject(void) const throw (Exception*) = 0;

   /**
    * Verifica se o arquivo pode ser lido
    * @return <tt>true</tt>, se o arquivo pode ser lido; 
    * <tt>false</tt>, caso contrrio
    */
   virtual bool canRead(void) const throw (Exception*) = 0;

   /**
    * Verifica se o arquivo pode ser alterado
    * @return <tt>true</tt>, se o arquivo pode ser alterado; 
    * <tt>false</tt>, caso contrrio
    */
   virtual bool canWrite(void) const throw (Exception*) = 0;

   /**
    * Obtm o modo de abertura do arquivo.
    * @return o modo de abertura do arquivo
   */
   virtual WIOFileMode getMode(void) const throw (Exception*) = 0;

   /**
    * Verifica se o arquivo  um diretrio
    * @return <tt>true</tt>, se for um diretrio; 
    *         <tt>false</tt>, caso contrrio
    */
   virtual bool isDirectory(void) const throw (Exception*) = 0;

   /**
    * Retorna o tipo do arquivo.
    * @return o tipo do arquivo.
   */
   virtual char* getFileType(void) const throw (Exception*) = 0;

   /**
    * Para o caso de diretrios, obtm o nmero de arquivos neste 
    * diretrio (sejam eles arquivos ou outros diretrios)
    * @return nmero de arquivos/diretrios 
    */
   virtual long getNumFiles(void) const throw (Exception*) = 0;

   /**
    * Obtm o i-simo arquivo deste diretrio.
    * @param i o ndice desejado dentro do diretrio remoto.
    * @return i-simo arquivo
    */
   virtual WIOFile* getFile(long i) const throw (Exception*) = 0;

   /**
    * Abre o arquivo corrente no modo especificado.
    * @param mode o modo de abertura do arquivo
    * @return <tt>true</tt>, se o arquivo foi aberto com sucesso. 
    * <tt>false</tt>, caso contrrio
    */
   virtual void open(WIOFileMode mode) throw (Exception*) = 0;

   /**
    * Fecha o arquivo corrente
    * @return <tt>true</tt>, se o arquivo foi fechado com sucesso. 
    * <tt>false</tt>, caso contrrio
    */
   virtual void close(void) throw (Exception*) = 0;

   /**
    * Leitura de bytes do arquivo. 
    * Caso a leitura ultrapasse o fim do arquivo,  lido apenas o que 
    * estiver disponvel. 
    * @param size a quantindade de bytes desejada na leitura
    * @param buffer O endereco do buffer previamente alocado. 
    *         assumido que este aponta para uma regio
    *        que tem espaco suficiente para o tamanho requerido pelo usurio.
    * @param chunk o tamanho mximo do pacote de leitura. Este parmetro 
    *        opcional e seu valor default  o maior pacote permitido pelo
    *        servidor (file system remoto).
    * @return o nmero de bytes lidos.
    */
   virtual long read(long size, char* buffer, long chunk = -1) const 
   throw (Exception*) = 0;

   /**
    * Obtm o tamanho do arquivo
    * @return o tamanho do arquivo
    */
   virtual long long size(void) const throw (Exception*) = 0;

   /**
    * Obtm a data da ltima modificao do arquivo.
    * @return uma struct especfica que deve ser desalocada pela aplicao.
    */
   virtual WIODateTimeInfo* lastModificationTime() const throw (Exception*) = 0;

   /**
    * Obtm a data da criao do arquivo.
    * @return uma struct especfica que deve ser desalocada pela aplicao.
    */
   virtual WIODateTimeInfo* creationTime() const throw (Exception*) = 0;

   /**
    * Verifica se o arquivo est em construo.
    * @return um flag correspondente
    */
   virtual bool isUnderConstruction() const throw (Exception*) = 0;

   /**
    * Verifica se o arquivo est publicado.
    * @return um flag correspondente
    */
   virtual bool isPublished() const throw (Exception*) = 0;

   /**
    * Altera o tamanho do arquivo.
    * @param size O novo tamanho do arquivo.
    * @return <tt>true</tt>, se o arquivo foi alterado. 
    * <tt>false</tt>, caso contrrio.
    */
   virtual void resize(long long size) throw (Exception*) = 0;

   /**
    * Escreve bytes no arquivo.
    * @param content o array de char contendo os bytes.
    * @param size o tamanho em bytes desse array.
    * @param chunk o tamanho mximo do pacote de escrita. Este parmetro 
    *        opcional e seu valor default  o maior pacote permitido pelo
    *        servidor (file system remoto).
    * @return a quantidade de bytes foram escritos.
    */
   virtual long write(const char* content, long size, long chunk = -1) 
   throw (Exception*) = 0;

   /**
    * Posiciona o ponteiro do arquivo.
    * @param offset a nova posio do ponteiro.
    * @param type a referncia da nova posio.
    * @return <tt>true</tt>, se o ponteiro foi alterado; 
    *         <tt>false</tt>, caso contrrio.
    * @see SeekType
    */
   virtual void seek(long long offset, SeekType type) throw (Exception*) = 0;

   /**
    * Informa a posio atual do ponteiro no arquivo.
    * @return a posio em bytes.
    */
   virtual long long tell() const throw (Exception*) = 0;

   /**
    * Verifica se  fim de arquivo
    * @return <tt>true</tt>, se  fim de arquivo; 
    *         <tt>false</tt>, caso contrrio
    */
   virtual bool eof() const throw (Exception*) = 0;

   /**
    * Destroi o arquivo remoto (chamado automaticamente pelo destrutor).
    */
   virtual void destroy(void) throw (Exception*) = 0;
    
};


} //namespace wioapi

#endif //__WIOAPI_H__
