#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <iostream>

#include "util.h"
#include "wioapiimpl.h"

#ifdef _WINDOWS
#include <io.h>
#endif

// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------

// ..........................................................................
/**
 *
 */
static long long getLocalFileSize(FILE* file) {
  long long localSize;
#ifdef _WINDOWS
   int localHandle = _fileno(file);
   if (localHandle == -1) return 0;
   _lseeki64(localHandle, 0, SEEK_END);
   localSize = _telli64(localHandle);
   _lseeki64(localHandle, 0, SEEK_SET);
#else
   fseeko(file, 0, SEEK_END);
   localSize = ftello(file);
   fseeko(file, 0, SEEK_SET);
#endif
  return localSize;
}

// ..........................................................................

static char* strlower(char* str) {
   int i;
   if (str == NULL) return NULL;
   for (i = 0; i < (int)strlen(str); i++) {
       str[i] = (char) tolower((int)str[i]);
   }
   return str;
}


// ..........................................................................
/**
 * @param filename O nome do arquivo.
 * @param type A extensao.
 * @return <tt>true</tt> se o arquivo tem a extensao dada.
 */
static bool isType(const char* filename, const char* type) {
	char *filenameLwr = strlower(strdup(filename));
	char *typeLwr = strlower(strdup(type));
	bool result;
   char* temp = filenameLwr;

	while (true) {
		temp = strstr(temp, typeLwr);

		if (temp == NULL) {
			result = false;
			break;
		}
		else {
		  // a extensao estah no fim?
        if ((int)(temp - filename) == (int)(strlen(filename) - strlen(temp))) {
           result = true;
           break;
        }
      }
		temp++;
	}

	free(filenameLwr);
	free(typeLwr);

	return result;
}


// ..........................................................................
/**
 * Retorna o tipo do arquivo do CSBASE de acordo com a extensao.
 * @param filename O nome do arquivo.
 * @return A string do tipo do arquivo.
 */
static const char *getType(const char *filename) {
	if(isType(filename, ".txt")) {
		return "TEXT";
	}
	else if(isType(filename, ".html") || isType(filename, ".htm")) {
		return "HTML";
	}
	else if(isType(filename, ".pdf")) {
		return "PDF";
	}
	else if(isType(filename, ".jdf")) {
		return "JDF";
	}
	else if(isType(filename, ".z")) {
		return "COMPRESS";
	}
	else if(isType(filename, ".algpar")) {
		return "ALGPAR";
	}
	else if(isType(filename, ".zip")) {
		return "ZIP";
	}
	else if(isType(filename, ".jpg") || isType(filename, ".jpeg")) {
		return "JPG";
	}
	else if(isType(filename, ".gif")) {
		return "GIF";
	}
	else if(isType(filename, ".doc")) {
		return "DOC";
	}
	else {
		return "UNKNOWN";
	}
}


// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
//
using namespace std;

namespace wioapi {


// ..........................................................................
/**
 *
 */
IdleTransfer::IdleTransfer(Mode mode, const char *localPath,
const char *remotePath, WIOProject* project, const char* userId, 
long packSize) throw(wioapi::Exception *) {
    packSize = (packSize <= 0 ? 1024 : packSize);
    this->mode = mode;
    this->packSize = packSize;
    this->project = project;
    this->userId = userId;

    transferred = 0;
    localFile = NULL;
    remoteFile = NULL;
    buff = NULL;

    try {
      openFiles(localPath, remotePath);
      initTransfer();
    }
    catch(wioapi::Exception*) {
      closeFiles();
      throw;
    }
}

// ..........................................................................
/**
 *
 */
IdleTransfer::~IdleTransfer() {
  closeFiles();
}

// ..........................................................................
/**
 *
 */
void IdleTransfer::openFiles(const char *localPath, const char *remotePath) 
throw(wioapi::Exception *) {
  bool isRemoteOpened = false;
  if (mode == PUT) {
     localFile = fopen(localPath, "rb");
     remoteFile = project->createFile(remotePath, getType(localPath), 
           this->userId);  
     remoteFile->open(MODE_WRITE); // ACCOXX
     isRemoteOpened = true;
  }
  else {
     localFile = fopen(localPath, "wb");
     remoteFile = project->getFile(remotePath);
     remoteFile->open(MODE_READ); // ACCOXX
     isRemoteOpened = true;
  }

  if (localFile == NULL) {
     cerr << localPath << endl;
     throw new wioapi::Exception("IdleTransfer::openFiles", 
        "Arquivo local no pde ser aberto.", "");
  }

  if (!isRemoteOpened) {
     delete remoteFile;
     remoteFile = NULL;
     throw new wioapi::Exception("IdleTransfer::openFiles",
        "Arquivo remoto no pde ser aberto.", "Sem mensagem do servidor" );
  }
}

// ..........................................................................
/**
 *
 */
void IdleTransfer::closeFiles() throw (wioapi::Exception*) {
  delete []buff;
  buff = NULL;

  if (localFile != NULL) {
     fclose(localFile);
     localFile = NULL;
  }

  if (remoteFile != NULL) {
     _TRY(remoteFile->close(), "IdleTransfer::closeFiles");
     _TRY(delete remoteFile, "IdleTransfer::closeFiles");
     remoteFile = NULL;
  }
}

// ..........................................................................
/**
 *
 */
void IdleTransfer::initTransfer() 
throw (wioapi::Exception *) {
  buff = new char[packSize];
  if (mode == PUT) {
     size = getLocalFileSize(localFile);
  }
  else {
     _TRY(size = remoteFile->size();, "IdleTransfer::initTransfer")
  }
}

// ..........................................................................
/**
 *
 */
void IdleTransfer::transfer() throw(wioapi::Exception *) {
   // Se a escrita por package ultrapassar o tamanho pedido, truncamo-na.
   long long _bytes = minLongLong(size - transferred, packSize);
   long bytes = ((long)_bytes);

   long len;
   try {
     if (mode == PUT) {
        // leitura
        long blocks = fread(buff, bytes, 1, localFile);
        if (blocks != 1) throw new wioapi::Exception(
           "IdleTransfer::transfer()", "leitura de arq. local", "");

        // escrita
        len = remoteFile->write(buff, bytes);
        if (len != bytes) {
           printf("  + WRITE: %ld written %ld tried\n", len, bytes);
           throw new wioapi::Exception(
              "IdleTransfer::transfer()", "escrita de arquivo remoto.", "");
        }
      }
      else {
        // leitura
        len = remoteFile->read(bytes, buff);
        if (len != bytes) {
           printf("  + READ: %ld read %ld tried\n", len, bytes);
           throw new wioapi::Exception(
              "IdleTransfer::transfer()", "leitura de arq. remoto", "");
        }

        // escrita
        long blocks = fwrite(buff, bytes, 1, localFile);
        if (blocks != 1) throw new wioapi::Exception(
           "IdleTransfer::transfer()", "escrita de arq. local.", "");
      }
    }
    catch(wioapi::Exception *) {
      closeFiles();
      throw;
    }
    transferred += len;
    if (transferred == size) closeFiles();
}

// ..........................................................................
/**
 *
 */
long long IdleTransfer::getSize() {
   return size;
}

// ..........................................................................
/**
 *
 */
void IdleTransfer::appendDescription(const char *description) {
   remoteFile->appendDescription(description);
}

// ..........................................................................
/**
 *
 */
long long IdleTransfer::getTransferred() {
   return transferred;
}

// ..........................................................................
/**
 *
 */
WIOProject* IdleTransfer::getProject() {
   return project;
}

// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------

// ..........................................................................
/**
 *
 */
bool executeOperation(const char* localFile, const char* remoteFile,
WIOProject* proj, enum Mode mode, const char* owner, long packSize,
const char *desc) {
   try {
     IdleTransfer idleTransfer(mode, localFile, 
         remoteFile, proj, owner, packSize);

     if (mode==PUT && desc!=NULL) idleTransfer.appendDescription(desc);

     while (idleTransfer.getTransferred() < idleTransfer.getSize()) {
        idleTransfer.transfer();
     }
   }
   catch(wioapi::Exception* ex) {
     printf( 
        "executeOperation: %s / %s\n", 
        ex->getServerMessage(), ex->getClientMessage());
     return false;
   }
   return true;
}

// ..........................................................................
/**
 *
 */
bool put(const char *localFile, const char *remoteFile, const char *server,
const char *port, const char *user, const char *pwd, const char* owner, 
const char *projectId) throw() {
   const char* ior = WIOLocator::createIOR(server, port);
   WIOFileSystem* filesystem = WIOLocator::buildFileSystem(ior);

   WIOProject* project = NULL;
   if (owner != NULL) {
     project = filesystem->openProjectFromOthers(
         (char*)user, (char*)pwd, (char*)owner, (char*)projectId);
   } 
   else {
     project = filesystem->openProject(
         (char*)user, (char*)pwd, (char*)projectId);
   }

   long packSize = filesystem->getMaxReadKb() * 1024;
   bool result = put(localFile, remoteFile, project, user, packSize);

   delete filesystem;
   delete project;

   return result;
}

// ..........................................................................
/**
 *
 */
bool put(const char *localFile, const char *remoteFile, const char *server,
const char *port, const char *user, const char *pwd, const char* owner, 
const char *projectId, const char *desc) throw() {
   const char* ior = WIOLocator::createIOR(server, port);
   WIOFileSystem* filesystem = WIOLocator::buildFileSystem(ior);

   WIOProject* project = NULL;
   if (owner != NULL) {
     project = filesystem->openProjectFromOthers(
         (char*)user, (char*)pwd, (char*)owner, (char*)projectId);
   } 
   else {
     project = filesystem->openProject(
         (char*)user, (char*)pwd, (char*)projectId);
   }

   long packSize = filesystem->getMaxReadKb() * 1024;
   bool result = put(localFile, remoteFile, project, user, packSize, desc);

   delete filesystem;
   delete project;

   return result;
}

// ..........................................................................
/**
 *
 */
bool get(const char *localFile, const char *remoteFile, const char *server,
const char *port, const char *user, const char *pwd, const char* owner,
const char *projectId) throw() {
   const char* ior = WIOLocator::createIOR(server, port);
   WIOFileSystem* filesystem = WIOLocator::buildFileSystem(ior);

   WIOProject* project = NULL;
   if (owner != NULL) {
     project = filesystem->openProjectFromOthers(
         (char*)user, (char*)pwd, (char*)owner, (char*)projectId);
   } 
   else {
     project = filesystem->openProject(
         (char*)user, (char*)pwd, (char*)projectId);
   }

   long packSize = filesystem->getMaxWriteKb() * 1024;
   bool result = get(localFile, remoteFile, project, owner, packSize);
   delete filesystem;
   delete project;

   return result;
}

// ..........................................................................
/**
 *
 */
bool put(const char *localFile, const char *remoteFile, 
WIOProject* proj, const char* owner, long packSize) throw() {
   return executeOperation(localFile, remoteFile, proj, PUT, owner, packSize, NULL);
}

// ..........................................................................
/**
 *
 */
bool put(const char *localFile, const char *remoteFile, 
WIOProject* proj, const char* owner, long packSize, const char *desc) throw() {
   return executeOperation(localFile, remoteFile, proj, PUT, owner, packSize, desc);
}

// ..........................................................................
/**
 *
 */
bool get(const char *localFile, const char *remoteFile, 
WIOProject* proj, const char* owner, long packSize) throw() {
   return executeOperation(localFile, remoteFile, proj, GET, owner, packSize, NULL);
}

void appendDescription(char *remoteFile, WIOProject* proj, const char *desc) throw() {
   WIOFile *f = proj->getFile(remoteFile);
   f->appendDescription(desc); 
}

bool fileExists(char *remoteFile, char *server, char *port, char *user, char *pwd,
                char* proj, char *owner) throw() {
   const char* ior = WIOLocator::createIOR(server, port);
   WIOFileSystem* filesystem = WIOLocator::buildFileSystem(ior);

   WIOProject* project = NULL;
   if (owner != NULL) {
     project = filesystem->openProjectFromOthers(
         (char*)user, (char*)pwd, (char*)owner, (char*)proj);
   } 
   else {
     project = filesystem->openProject(
         (char*)user, (char*)pwd, (char*)proj);
   }

   bool ret =  project->fileExists(remoteFile);

   delete project;

   return ret;
}


} // namespace wioapi

