package csdk.v2.api.application;

import javax.swing.ImageIcon;

import csdk.v2.api.core.IContext;

/**
 * Define os mtodos disponveis para acesso da aplicao ao seu registro
 * durante a sua execuo.
 */
public interface IApplicationContext extends IContext {

  /**
   * Obtm o identificador da aplicao. Este id  comum a todas as instncias
   * de um mesmo tipo de aplicao.
   * 
   * @return o id da aplicao.
   */
  public String getApplicationId();

  /**
   * Obtm o identificador nico da instncia da aplicao.
   * 
   * @return o id da instncia.
   */
  public String getInstanceId();

  /**
   * Obtm o nome da aplicao.
   * 
   * @return o nome da aplicao.
   */
  public String getApplicationName();

  /**
   * Obtm a verso da aplicao.
   * 
   * @return a verso como string.
   */
  public String getVersion();

  /**
   * Obtm o nome do autor da aplicao.
   * 
   * @return o autor da aplicao.
   */
  public String getAuthor();

  /**
   * Obtm o endereo de contato do autor da aplicao.
   * 
   * @return o endereo.
   */
  public String getAuthorEmail();

  /**
   * Obtm a descrio da aplicao.
   * 
   * @return a descrio.
   */
  public String getApplicationDescription();

  /**
   * Indica se a aplicao s pode ter uma nica instncia ativa.
   * 
   * @return <code>true</code> se a aplicao s pode ter uma nica instncia ou
   *         <code>false</code>, caso contrrio.
   */
  public boolean isSingleton();

  /**
   * Indica se a aplicao s pode ser executada com um projeto aberto.
   * 
   * @return <code>true</code> se a aplicao precisa de um projeto para
   *         executar ou <code>false</code>, caso contrrio.
   */
  public boolean requiresProject();

  /**
   * Indica se a propriedade especificada est sem valor.
   * 
   * @param propName nome da propriedade.
   * @return <code>true</code> se a propriedade est sem valor definido ou
   *         <code>false</code> caso contrrio.
   */
  public boolean isPropertyNull(String propName);

  /**
   * Obtm o valor da propriedade especificada.
   * 
   * @param propName nome da propriedade.
   * @return o valor da propriedade.
   * 
   * @throws IllegalStateException se a propriedade no foi definida.
   */
  public String getProperty(String propName);

  /**
   * Indica se a aplicao faz uso do pacote de internacionalizao.
   * 
   * @return <code>true</code> se a aplicao precisa do pacote ou
   *         <code>false</code> caso contrrio.
   */
  public boolean isBundleRequired();

  /**
   * Consulta da existncia de uma string de idioma no pacote de
   * internacionalizao.
   * 
   * [ATENO] Este mtodo s deve ser chamado por aplicaes que usam o pacote
   * de internacionalizao (onde {@link #isBundleRequired()} retorna
   * <code>true</code>).
   * 
   * @param key a chave de busca.
   * @return <code>true</code> se a chave foi definida no pacote de
   *         internacionalizao ou <code>false</code> caso contrrio.
   */
  public boolean hasString(String key);

  /**
   * Obtm uma string de idioma do pacote de internacionalizao.
   * 
   * [ATENO] Este mtodo s deve ser chamado por aplicaes que usam o pacote
   * de internacionalizao (onde {@link #isBundleRequired()} retorna
   * <code>true</code>).
   * 
   * @param key a chave de busca.
   * @param args os argumentos de formatao da string.
   * @return o texto no idioma correto ou uma string com a chave, caso a chave
   *         no seja encontrada no pacote de internacionalizao (ex:
   *         {@code <<key>>}).
   */
  public String getString(String key, Object... args);

  /**
   * Obtm uma imagem do diretrio padro da aplicao, no caminho especificado.
   * 
   * @param path o caminho para o arquivo de imagem, relativo ao diretrio de
   *        imagens padro da aplicao.
   * @return a imagem ou <code>null</code> caso a imagem no seja encontrada no
   *         caminho especificado.
   */
  public ImageIcon getImageIcon(String[] path);

  /**
   * Obtm o cone reduzido da aplicao.
   * 
   * @return o cone reduzido da aplicao.
   */
  public ImageIcon getSmallApplicationIcon();

  /**
   * Obtm o cone da aplicao.
   * 
   * @return o cone da aplicao.
   */
  public ImageIcon getApplicationIcon();

  /**
   * Obtm os tipos de arquivo associados  aplicao.
   * 
   * @return os tipos.
   */
  public String[] getApplicationFileTypes();

}