package csdk.v2.helper;

import java.awt.Window;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;

import csdk.v2.api.application.ApplicationException;
import csdk.v2.api.core.ICSDKEnvironment;

/**
 * Classe abstrata para facilitar o desenvolvimento de aplicaes que acessam a
 * rea de projetos com CSDK.
 */
public abstract class AbstractCSDKWindowApplication extends
AbstractCSDKApplication {

  /**
   * Janela principal da aplicao.
   */
  private JFrame applicationFrame;

  /** Vetor de janelas */
  private final ArrayList<Window> dependentWindows = new ArrayList<Window>();

  /**
   * Observador para o fechamento da janela principa;.
   */
  private WindowCloseListener windowCloseListener;

  /**
   * Construtor.
   *
   * @param csdkInterface interface padro para o ambiente CSDK.
   */
  public AbstractCSDKWindowApplication(ICSDKEnvironment csdkInterface) {
    super(csdkInterface);
    this.applicationFrame = new JFrame();
    this.applicationFrame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
    this.windowCloseListener = new WindowCloseListener();
    this.applicationFrame.addWindowListener(windowCloseListener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final void onApplicationStart() throws ApplicationException {
    Runnable runnable = new Runnable() {
      @Override
      public void run() {
        try {
          ImageIcon icon = getSmallApplicationIcon();
          if (icon != null) {
            applicationFrame.setIconImage(icon.getImage());
          }
          applicationFrame.setTitle(getApplicationName());
          applicationStarted(applicationFrame);
          applicationFrame.setVisible(true);
        }
        catch (Exception e) {
          handleException(e, applicationFrame);
        }
      }
    };
    invoke(runnable);
  }

  /**
   * Mtodo de inicializao dos componentes visuais na janela da aplicao.
   *
   * @param frame a janela principal da aplicao.
   * @throws ApplicationException em caso de erro.
   */
  protected abstract void applicationStarted(JFrame frame)
    throws ApplicationException;

  /**
   * Mtodo de finalizao dos componentes visuais na janela da aplicao.
   *
   * @throws ApplicationException em caso de erro.
   */
  protected abstract void applicationEnded() throws ApplicationException;

  /**
   * {@inheritDoc}
   */
  @Override
  public final void onApplicationEnd() throws ApplicationException {
    Runnable runnable = new Runnable() {
      @Override
      public void run() {
        try {
          applicationEnded();
          closeDependentWindows();
        }
        catch (Exception e) {
          handleException(e, applicationFrame);
        }
        finally {
          if (applicationFrame != null) {
            applicationFrame.removeWindowListener(windowCloseListener);
            windowCloseListener = null;
            applicationFrame.dispose();
            applicationFrame = null;
          }
          disposeContexts();
        }
      }
    };
    invoke(runnable);
  }

  /**
   * Obtm a janela principal da aplicao.
   *
   * @return a janela principal.
   */
  public final JFrame getApplicationFrame() {
    return this.applicationFrame;
  }

  /**
   * Registra uma janela cliente na janela principal de modo que, ao fechar a
   * janela pricipal, todas as janelas registradas tambm so fechadas.
   *
   * @param win uma nova janela registrada
   */
  final public void addWindow(final Window win) {
    this.dependentWindows.add(win);
  }

  /**
   * Descadastra uma janela da cliente janela principal.
   *
   * @param win a janela que deve ser removida da janela principal.
   */
  final public void removeWindow(final Window win) {
    this.dependentWindows.remove(win);
  }

  /**
   * Fecha todas as janelas dependentes da janela principal da aplicao.
   */
  final public void closeDependentWindows() {
    final List<Window> aux = new ArrayList<Window>();
    aux.addAll(dependentWindows);
    for (Window window : aux) {
      window.setVisible(false);
      window.dispose();
    }
    dependentWindows.clear();
  }

  /**
   * Executa o cdigo fornecido dentro da EDT.
   *
   * @param code Cdigo a ser executado dentro da EDT.
   */
  private void invoke(Runnable code) {
    try {
      if (SwingUtilities.isEventDispatchThread()) {
        code.run();
      }
      else {
        SwingUtilities.invokeAndWait(code);
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Observador para fechar a aplicao quando a janela principal for fechada.
   */
  private final class WindowCloseListener extends WindowAdapter {
    /**
     * {@inheritDoc}
     */
    @Override
    public void windowClosing(WindowEvent arg) {
      finishApplication();
    }
  }

}
