package csdk.v2.rest.helper;

import com.fasterxml.jackson.databind.ObjectMapper;
import csdk.v2.rest.helper.exception.JSONException;
import csdk.v2.rest.helper.exception.RESTException;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Implementao de um tratador de mensagens JSON
 *
 * @author Tecgraf/PUC-Rio
 */
public class JSONMap extends HashMap<String,Object> {

    /**
     * Utilitario para tratamento de objetos JSON
     */
    private static ObjectMapper objectMapper = new ObjectMapper();

    /**
     * Contrutor do Mapa
     *
     * @param map Mapa a ser incluido nos valores iniciais
     */
    public JSONMap(Map<String,Object> map) {
        this.putAll(map);
    }

    /**
     * Construtor do mapa
     *
     * @param jsondata Dados json a serem incluidos nos valores
     *
     * @throws JSONException Em caso de erro de sintaxe no dado JSON
     */
    public JSONMap(String jsondata) throws JSONException {
        try {
            HashMap map = objectMapper.readValue(jsondata, HashMap.class);
            this.putAll(map);
        } catch (IOException e) {
            throw new JSONException("Error parsing JSON",e);
        }
    }

    /**
     * Monta a mensagem de erro padro para campo faltando
     *
     * @param field Nome do campo
     * @return Mensagem de erro
     */
    private String missingFieldMessage(String field) {
        final String fmt = "Required message field %s not found";
        final String msg = String.format(fmt, field);
        return msg;
    }

    /**
     * Testa e converte campos da mensagem do tipo double
     *
     * @param field campos
     * @param optional indica se o parmetro  opcional ou obrigatrio
     * @return valor
     */
    public Double asDouble(String field, boolean optional) throws JSONException {

        final Object object = this.get(field);

        if (object == null) {
            if (!optional) {
                throw new JSONException(missingFieldMessage(field));
            }
            return null;
        }

        Double d = null;

        if (object instanceof Double) {
            d = (Double) object;
        }
        else if (object instanceof Integer) {
            d = new Double((Integer) object);
        }
        else if (object instanceof String) {
            d = Double.parseDouble((String) object);
        }
        else if (!optional) {

            final String fmt = "Cannot convert field %s to double (value = %s)";
            final String msg = String.format(fmt, field, object.toString());

            throw new JSONException(msg);
        }
        return d;
    }

    /**
     * Testa e converte campos da mensagem do tipo inteiro
     *
     * @param field campos
     * @param optional indica se o parmetro  opcional ou obrigatrio
     * @return valor
     */
    public Integer asInteger(String field, boolean optional) throws JSONException {
        final Object object = this.get(field);

        if (object == null) {
            if (!optional) {
                throw new JSONException(missingFieldMessage(field));
            }
            return null;
        }

        Integer i = null;

        if (object instanceof Integer) {
            i = ((Integer) object);
        }
        else if (object instanceof String) {
            i = Integer.parseInt((String) object);
        }
        else if (!optional) {
            final String fmt = "Cannot convert field %s to integer (value = %s)";
            final String msg = String.format(fmt, field, object.toString());
            throw new JSONException(msg);
        }
        return i;
    }

    /**
     * Testa e converte campos da mensagem do tipo double
     *
     * @param field campos
     * @param optional indica se o parmetro  opcional ou obrigatrio
     * @return valor
     */
    public double[] asDoubleArray(String field, boolean optional) throws JSONException {

        final Object object = this.get(field);

        if (object == null) {
            if (!optional) {
                throw new JSONException(missingFieldMessage(field));
            }
            return null;
        }

        double[] da = null;

        if (object instanceof double[]) {
            da = (double[]) object;
        }
        else if (object instanceof int[]) {
            int[] ia = (int[]) object;
            da = new double[ia.length];
            for (int i = 0; i < ia.length; i++) {
                da[i] = ia[i];
            }
        }
        else if (object instanceof List) {
            @SuppressWarnings("unchecked")
            List<Object> list = ((List<Object>) object);
            da = new double[list.size()];
            for (int i = 0; i < list.size(); i++) {
                Object value = list.get(i);
                try {
                    da[i] = Double.parseDouble(value.toString());
                }
                catch (NumberFormatException e) {
                    final String fmt =
                            "Field %s is an array but contains non-numeric value: %s";
                    final String msg = String.format(fmt, field, value);
                    throw new JSONException(msg);
                }
            }
        }
        else if (!optional) {
            final String fmt = "Cannot convert field %s to double array (value = %s)";
            final String msg = String.format(fmt, field, object.toString());
            throw new JSONException(msg);
        }
        return da;
    }

    /**
     * Testa e converte campos da mensagem do tipo double
     *
     * @param field campos
     * @param optional indica se o parmetro  opcional ou obrigatrio
     * @return valor
     */
    public String asString(String field, boolean optional) throws JSONException {
        final Object object = this.get(field);

        if (object == null) {
            if (!optional) {
                throw new JSONException(missingFieldMessage(field));
            }
            return null;
        }

        if (object instanceof String) {
            return (String) object;
        }
        else if (!optional) {
            final String fmt = "Cannot recognize field %s as string (value = %s)";
            final String msg = String.format(fmt, field, object.toString());
            throw new JSONException(msg);
        }
        return null;
    }

    /**
     * Testa e converte campos da mensagem do tipo JSONMap
     *
     * @param field campo
     * @param optional indica se o parmetro  opcional ou obrigatrio
     * @return mapa
     */
    public JSONMap asJSONMap(String field, boolean optional) throws JSONException {
        final Object object = this.get(field);

        if (object == null) {
            if (!optional) {
                throw new JSONException(missingFieldMessage(field));
            }
            return null;
        }

        if (object instanceof Map) {
            return new JSONMap((Map)object);
        }
        else if (!optional) {
            final String fmt = "Cannot recognize field %s as map (value = %s)";
            final String msg = String.format(fmt, field, object.toString());
            throw new JSONException(msg);
        }
        return null;
    }

    /**
     * Testa e converte campos da mensagem do tipo List
     *
     * @param field campo
     * @param optional indica se o parmetro  opcional ou obrigatrio
     * @return mapa
     */
    public List<?> asList(String field, boolean optional) throws JSONException {
        final Object object = this.get(field);

        if (object == null) {
            if (!optional) {
                throw new JSONException(missingFieldMessage(field));
            }
            return null;
        }

        if (object instanceof List) {
            return (List) object;
        }
        else if (!optional) {
            final String fmt = "Cannot recognize field %s as list (value = %s)";
            final String msg = String.format(fmt, field, object.toString());
            throw new JSONException(msg);
        }
        return null;
    }

}
